/***

The Anti-Grain Geometry Project
A high quality rendering engine for C++
http://antigrain.com

Anti-Grain Geometry has dual licensing model. The Modified BSD 
License was first added in version v2.4 just for convenience.
It is a simple, permissive non-copyleft free software license, 
compatible with the GNU GPL. It's well proven and recognizable.
See http://www.fsf.org/licensing/licenses/index_html#ModifiedBSD
for details.

Note that the Modified BSD license DOES NOT restrict your rights 
if you choose the Anti-Grain Geometry Public License.




Anti-Grain Geometry Public License
====================================================

Anti-Grain Geometry - Version 2.4 
Copyright (C) 2002-2005 Maxim Shemanarev (McSeem) 

Permission to copy, use, modify, sell and distribute this software 
is granted provided this copyright notice appears in all copies. 
This software is provided "as is" without express or implied
warranty, and with no claim as to its suitability for any purpose.





Modified BSD License
====================================================
Anti-Grain Geometry - Version 2.4 
Copyright (C) 2002-2005 Maxim Shemanarev (McSeem) 

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions 
are met:

  1. Redistributions of source code must retain the above copyright 
     notice, this list of conditions and the following disclaimer. 

  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in 
     the documentation and/or other materials provided with the 
     distribution. 

  3. The name of the author may not be used to endorse or promote 
     products derived from this software without specific prior 
     written permission. 

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.

****/
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_ALPHA_MASK_U8_INCLUDED
#define AGG_ALPHA_MASK_U8_INCLUDED

#include <cstring>
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_BASICS_INCLUDED
#define AGG_BASICS_INCLUDED

#include <cmath>
#ifndef AGG_CONFIG_INCLUDED
#define AGG_CONFIG_INCLUDED

 

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 


 
 
 
 
 
 
 
 
 
 
 
 
 

#endif

 
#ifdef AGG_CUSTOM_ALLOCATOR
#include "agg_allocator.h"
#else
namespace agg
{
     
     
     
     
     
     
     
     
    template<class T> struct pod_allocator
    {
        static T*   allocate(unsigned num)       { return new T [num]; }
        static void deallocate(T* ptr, unsigned) { delete [] ptr;      }
    };

     
     
     
     
     
     
     
     
    template<class T> struct obj_allocator
    {
        static T*   allocate()         { return new T; }
        static void deallocate(T* ptr) { delete ptr;   }
    };
}
#endif


 
 
 
 
 
 
#ifndef AGG_INT8
#define AGG_INT8 signed char
#endif

#ifndef AGG_INT8U
#define AGG_INT8U unsigned char
#endif

#ifndef AGG_INT16
#define AGG_INT16 short
#endif

#ifndef AGG_INT16U
#define AGG_INT16U unsigned short
#endif

#ifndef AGG_INT32
#define AGG_INT32 int
#endif

#ifndef AGG_INT32U
#define AGG_INT32U unsigned
#endif

#ifndef AGG_INT64
#if defined(_MSC_VER) || defined(__BORLANDC__)
#define AGG_INT64 signed __int64
#else
#define AGG_INT64 signed long long
#endif
#endif

#ifndef AGG_INT64U
#if defined(_MSC_VER) || defined(__BORLANDC__)
#define AGG_INT64U unsigned __int64
#else
#define AGG_INT64U unsigned long long
#endif
#endif

 
#if __cplusplus >= 201703L
#define AGG_FALLTHROUGH [[fallthrough]];
#else
#define AGG_FALLTHROUGH ;
#endif

 
#if defined(_MSC_VER)
#pragma warning(disable:4786)  
#endif

#if defined(_MSC_VER)
#define AGG_INLINE __forceinline
#else
#define AGG_INLINE inline
#endif

namespace agg
{
     
    typedef AGG_INT8   int8;          
    typedef AGG_INT8U  int8u;         
    typedef AGG_INT16  int16;         
    typedef AGG_INT16U int16u;        
    typedef AGG_INT32  int32;         
    typedef AGG_INT32U int32u;        
    typedef AGG_INT64  int64;         
    typedef AGG_INT64U int64u;        

#if defined(AGG_FISTP)
#pragma warning(push)
#pragma warning(disable : 4035)  
    AGG_INLINE int iround(double v)               
    {
        int t;
        __asm fld   qword ptr [v]
        __asm fistp dword ptr [t]
        __asm mov eax, dword ptr [t]
    }
    AGG_INLINE unsigned uround(double v)          
    {
        unsigned t;
        __asm fld   qword ptr [v]
        __asm fistp dword ptr [t]
        __asm mov eax, dword ptr [t]
    }
#pragma warning(pop)
    AGG_INLINE int ifloor(double v)
    {
        return int(floor(v));
    }
    AGG_INLINE unsigned ufloor(double v)          
    {
        return unsigned(floor(v));
    }
    AGG_INLINE int iceil(double v)
    {
        return int(ceil(v));
    }
    AGG_INLINE unsigned uceil(double v)           
    {
        return unsigned(ceil(v));
    }
#elif defined(AGG_QIFIST)
    AGG_INLINE int iround(double v)
    {
        return int(v);
    }
    AGG_INLINE int uround(double v)
    {
        return unsigned(v);
    }
    AGG_INLINE int ifloor(double v)
    {
        return int(std::floor(v));
    }
    AGG_INLINE unsigned ufloor(double v)
    {
        return unsigned(std::floor(v));
    }
    AGG_INLINE int iceil(double v)
    {
        return int(std::ceil(v));
    }
    AGG_INLINE unsigned uceil(double v)
    {
        return unsigned(std::ceil(v));
    }
#else
    AGG_INLINE int iround(double v)
    {
        return int((v < 0.0) ? v - 0.5 : v + 0.5);
    }
    AGG_INLINE int uround(double v)
    {
        return unsigned(v + 0.5);
    }
    AGG_INLINE int ifloor(double v)
    {
        int i = int(v);
        return i - (i > v);
    }
    AGG_INLINE unsigned ufloor(double v)
    {
        return unsigned(v);
    }
    AGG_INLINE int iceil(double v)
    {
        return int(std::ceil(v));
    }
    AGG_INLINE unsigned uceil(double v)
    {
        return unsigned(std::ceil(v));
    }
#endif

     
    template<int Limit> struct saturation
    {
        AGG_INLINE static int iround(double v)
        {
            if(v < double(-Limit)) return -Limit;
            if(v > double( Limit)) return  Limit;
            return agg::iround(v);
        }
    };

     
    template<unsigned Shift> struct mul_one
    {
        AGG_INLINE static unsigned mul(unsigned a, unsigned b)
        {
            unsigned q = a * b + (1 << (Shift-1));
            return (q + (q >> Shift)) >> Shift;
        }
    };

     
    typedef unsigned char cover_type;     
    enum cover_scale_e
    {
        cover_shift = 8,                  
        cover_size  = 1 << cover_shift,   
        cover_mask  = cover_size - 1,     
        cover_none  = 0,                  
        cover_full  = cover_mask          
    };

     
     
     
     
     
     
    enum poly_subpixel_scale_e
    {
        poly_subpixel_shift = 8,                       
        poly_subpixel_scale = 1<<poly_subpixel_shift,  
        poly_subpixel_mask  = poly_subpixel_scale-1    
    };

     
    enum filling_rule_e
    {
        fill_non_zero,
        fill_even_odd
    };

     
    const double pi = 3.14159265358979323846;

     
    inline double deg2rad(double deg)
    {
        return deg * pi / 180.0;
    }

     
    inline double rad2deg(double rad)
    {
        return rad * 180.0 / pi;
    }
 
     
    template<class T> struct rect_base
    {
        typedef T            value_type;
        typedef rect_base<T> self_type;
        T x1, y1, x2, y2;

        rect_base() {}
        rect_base(T x1_, T y1_, T x2_, T y2_) :
            x1(x1_), y1(y1_), x2(x2_), y2(y2_) {}

        void init(T x1_, T y1_, T x2_, T y2_) 
        {
            x1 = x1_; y1 = y1_; x2 = x2_; y2 = y2_; 
        }

        const self_type& normalize()
        {
            T t;
            if(x1 > x2) { t = x1; x1 = x2; x2 = t; }
            if(y1 > y2) { t = y1; y1 = y2; y2 = t; }
            return *this;
        }

        bool clip(const self_type& r)
        {
            if(x2 > r.x2) x2 = r.x2;
            if(y2 > r.y2) y2 = r.y2;
            if(x1 < r.x1) x1 = r.x1;
            if(y1 < r.y1) y1 = r.y1;
            return x1 <= x2 && y1 <= y2;
        }

        bool is_valid() const
        {
            return x1 <= x2 && y1 <= y2;
        }

        bool hit_test(T x, T y) const
        {
            return (x >= x1 && x <= x2 && y >= y1 && y <= y2);
        }
        
        bool overlaps(const self_type& r) const
        {
            return !(r.x1 > x2 || r.x2 < x1
                  || r.y1 > y2 || r.y2 < y1);
        }
    };

     
    template<class Rect> 
    inline Rect intersect_rectangles(const Rect& r1, const Rect& r2)
    {
        Rect r = r1;

         
         
         
         
         
        if(r.x2 > r2.x2) r.x2 = r2.x2; 
        if(r.y2 > r2.y2) r.y2 = r2.y2;
        if(r.x1 < r2.x1) r.x1 = r2.x1;
        if(r.y1 < r2.y1) r.y1 = r2.y1;
        return r;
    }


     
    template<class Rect> 
    inline Rect unite_rectangles(const Rect& r1, const Rect& r2)
    {
        Rect r = r1;
        if(r.x2 < r2.x2) r.x2 = r2.x2;
        if(r.y2 < r2.y2) r.y2 = r2.y2;
        if(r.x1 > r2.x1) r.x1 = r2.x1;
        if(r.y1 > r2.y1) r.y1 = r2.y1;
        return r;
    }

    typedef rect_base<int>    rect_i;  
    typedef rect_base<float>  rect_f;  
    typedef rect_base<double> rect_d;  

     
    enum path_commands_e
    {
        path_cmd_stop     = 0,         
        path_cmd_move_to  = 1,         
        path_cmd_line_to  = 2,         
        path_cmd_curve3   = 3,         
        path_cmd_curve4   = 4,         
        path_cmd_curveN   = 5,         
        path_cmd_catrom   = 6,         
        path_cmd_ubspline = 7,         
        path_cmd_end_poly = 0x0F,      
        path_cmd_mask     = 0x0F       
    };

     
    enum path_flags_e
    {
        path_flags_none  = 0,          
        path_flags_ccw   = 0x10,       
        path_flags_cw    = 0x20,       
        path_flags_close = 0x40,       
        path_flags_mask  = 0xF0        
    };

     
    inline bool is_vertex(unsigned c)
    {
        return c >= path_cmd_move_to && c < path_cmd_end_poly;
    }

     
    inline bool is_drawing(unsigned c)
    {
        return c >= path_cmd_line_to && c < path_cmd_end_poly;
    }

     
    inline bool is_stop(unsigned c)
    { 
        return c == path_cmd_stop;
    }

     
    inline bool is_move_to(unsigned c)
    {
        return c == path_cmd_move_to;
    }

     
    inline bool is_line_to(unsigned c)
    {
        return c == path_cmd_line_to;
    }

     
    inline bool is_curve(unsigned c)
    {
        return c == path_cmd_curve3 || c == path_cmd_curve4;
    }

     
    inline bool is_curve3(unsigned c)
    {
        return c == path_cmd_curve3;
    }

     
    inline bool is_curve4(unsigned c)
    {
        return c == path_cmd_curve4;
    }

     
    inline bool is_end_poly(unsigned c)
    {
        return (c & path_cmd_mask) == path_cmd_end_poly;
    }

     
    inline bool is_close(unsigned c)
    {
        return (c & ~(+path_flags_cw | path_flags_ccw)) ==
               (+path_cmd_end_poly | path_flags_close); 
    }

     
    inline bool is_next_poly(unsigned c)
    {
        return is_stop(c) || is_move_to(c) || is_end_poly(c);
    }

     
    inline bool is_cw(unsigned c)
    {
        return (c & path_flags_cw) != 0;
    }

     
    inline bool is_ccw(unsigned c)
    {
        return (c & path_flags_ccw) != 0;
    }

     
    inline bool is_oriented(unsigned c)
    {
        return (c & (+path_flags_cw | path_flags_ccw)) != 0; 
    }

     
    inline bool is_closed(unsigned c)
    {
        return (c & path_flags_close) != 0; 
    }

     
    inline unsigned get_close_flag(unsigned c)
    {
        return c & path_flags_close; 
    }

     
    inline unsigned clear_orientation(unsigned c)
    {
        return c & ~(+path_flags_cw | path_flags_ccw);
    }

     
    inline unsigned get_orientation(unsigned c)
    {
        return c & (+path_flags_cw | path_flags_ccw);
    }

     
    inline unsigned set_orientation(unsigned c, unsigned o)
    {
        return clear_orientation(c) | o;
    }

     
    template<class T> struct point_base
    {
        typedef T value_type;
        T x,y;
        point_base() {}
        point_base(T x_, T y_) : x(x_), y(y_) {}
    };
    typedef point_base<int>    point_i;  
    typedef point_base<float>  point_f;  
    typedef point_base<double> point_d;  

     
    template<class T> struct vertex_base
    {
        typedef T value_type;
        T x,y;
        unsigned cmd;
        vertex_base() {}
        vertex_base(T x_, T y_, unsigned cmd_) : x(x_), y(y_), cmd(cmd_) {}
    };
    typedef vertex_base<int>    vertex_i;  
    typedef vertex_base<float>  vertex_f;  
    typedef vertex_base<double> vertex_d;  

     
    template<class T> struct row_info
    {
        int x1, x2;
        T* ptr;
        row_info() {}
        row_info(int x1_, int x2_, T* ptr_) : x1(x1_), x2(x2_), ptr(ptr_) {}
    };

     
    template<class T> struct const_row_info
    {
        int x1, x2;
        const T* ptr;
        const_row_info() {}
        const_row_info(int x1_, int x2_, const T* ptr_) : 
            x1(x1_), x2(x2_), ptr(ptr_) {}
    };

     
    template<class T> inline bool is_equal_eps(T v1, T v2, T epsilon)
    {
	bool neg1 = v1 < 0.0;
	bool neg2 = v2 < 0.0;

	if (neg1 != neg2)
	    return std::fabs(v1) < epsilon && std::fabs(v2) < epsilon;

        int int1, int2;
	std::frexp(v1, &int1);
	std::frexp(v2, &int2);
	int min12 = int1 < int2 ? int1 : int2;

	v1 = std::ldexp(v1, -min12);
	v2 = std::ldexp(v2, -min12);

	return std::fabs(v1 - v2) < epsilon;
    }
}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_RENDERING_BUFFER_INCLUDED
#define AGG_RENDERING_BUFFER_INCLUDED

#include <cstring>
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_ARRAY_INCLUDED
#define AGG_ARRAY_INCLUDED

#include <cstddef>
#include <cstring>

namespace agg
{

     
    template<class T> class pod_array_adaptor
    {
    public:
        typedef T value_type;
        pod_array_adaptor(T* array, unsigned size) : 
            m_array(array), m_size(size) {}

        unsigned size() const { return m_size; }
        const T& operator [] (unsigned i) const { return m_array[i]; }
              T& operator [] (unsigned i)       { return m_array[i]; }
        const T& at(unsigned i) const           { return m_array[i]; }
              T& at(unsigned i)                 { return m_array[i]; }
        T  value_at(unsigned i) const           { return m_array[i]; }

    private:
        T*       m_array;
        unsigned m_size;
    };


     
    template<class T, unsigned Size> class pod_auto_array
    {
    public:
        typedef T value_type;
        typedef pod_auto_array<T, Size> self_type;

        pod_auto_array() {}
        explicit pod_auto_array(const T* c)
        {
            std::memcpy(m_array, c, sizeof(T) * Size);
        }

        const self_type& operator = (const T* c)
        {
            std::memcpy(m_array, c, sizeof(T) * Size);
            return *this;
        }

        static unsigned size() { return Size; }
        const T& operator [] (unsigned i) const { return m_array[i]; }
              T& operator [] (unsigned i)       { return m_array[i]; }
        const T& at(unsigned i) const           { return m_array[i]; }
              T& at(unsigned i)                 { return m_array[i]; }
        T  value_at(unsigned i) const           { return m_array[i]; }

    private:
        T m_array[Size];
    };


     
    template<class T, unsigned Size> class pod_auto_vector
    {
    public:
        typedef T value_type;
        typedef pod_auto_vector<T, Size> self_type;

        pod_auto_vector() : m_size(0) {}

        void remove_all()            { m_size = 0; }
        void clear()                 { m_size = 0; }
        void add(const T& v)         { m_array[m_size++] = v; }
        void push_back(const T& v)   { m_array[m_size++] = v; }
        void inc_size(unsigned size) { m_size += size; }
        
        unsigned size() const { return m_size; }
        const T& operator [] (unsigned i) const { return m_array[i]; }
              T& operator [] (unsigned i)       { return m_array[i]; }
        const T& at(unsigned i) const           { return m_array[i]; }
              T& at(unsigned i)                 { return m_array[i]; }
        T  value_at(unsigned i) const           { return m_array[i]; }

    private:
        T m_array[Size];
        unsigned m_size;
    };


     
    template<class T> class pod_array
    {
    public:
        typedef T value_type;
        typedef pod_array<T> self_type;

        ~pod_array() { pod_allocator<T>::deallocate(m_array, m_size); }
        pod_array() : m_array(0), m_size(0) {}

        pod_array(unsigned size) : 
            m_array(pod_allocator<T>::allocate(size)), 
            m_size(size) 
        {}

        pod_array(const self_type& v) : 
            m_array(pod_allocator<T>::allocate(v.m_size)), 
            m_size(v.m_size) 
        {
            std::memcpy(m_array, v.m_array, sizeof(T) * m_size);
        }

        void resize(unsigned size)
        {
            if(size != m_size)
            {
                pod_allocator<T>::deallocate(m_array, m_size);
                m_array = pod_allocator<T>::allocate(m_size = size);
            }
        }
        const self_type& operator = (const self_type& v)
        {
            resize(v.size());
            std::memcpy(m_array, v.m_array, sizeof(T) * m_size);
            return *this;
        }

        unsigned size() const { return m_size; }
        const T& operator [] (unsigned i) const { return m_array[i]; }
              T& operator [] (unsigned i)       { return m_array[i]; }
        const T& at(unsigned i) const           { return m_array[i]; }
              T& at(unsigned i)                 { return m_array[i]; }
        T  value_at(unsigned i) const           { return m_array[i]; }

        const T* data() const { return m_array; }
              T* data()       { return m_array; }
    private:
        T*       m_array;
        unsigned m_size;
    };



     
     
     
     
    template<class T> class pod_vector
    {
    public:
        typedef T value_type;

        ~pod_vector() { pod_allocator<T>::deallocate(m_array, m_capacity); }
        pod_vector() : m_size(0), m_capacity(0), m_array(0) {}
        pod_vector(unsigned cap, unsigned extra_tail=0);

         
        pod_vector(const pod_vector<T>&);
        const pod_vector<T>& operator = (const pod_vector<T>&);

         
        void capacity(unsigned cap, unsigned extra_tail=0);
        unsigned capacity() const { return m_capacity; }

         
         
        void allocate(unsigned size, unsigned extra_tail=0);

         
        void resize(unsigned new_size);

        void zero()
        {
            std::memset(m_array, 0, sizeof(T) * m_size);
        }

        void add(const T& v)         { m_array[m_size++] = v; }
        void push_back(const T& v)   { m_array[m_size++] = v; }
        void insert_at(unsigned pos, const T& val);
        void inc_size(unsigned size) { m_size += size; }
        unsigned size()      const   { return m_size; }
        unsigned byte_size() const   { return m_size * sizeof(T); }
        void serialize(int8u* ptr) const;
        void deserialize(const int8u* data, unsigned byte_size);
        const T& operator [] (unsigned i) const { return m_array[i]; }
              T& operator [] (unsigned i)       { return m_array[i]; }
        const T& at(unsigned i) const           { return m_array[i]; }
              T& at(unsigned i)                 { return m_array[i]; }
        T  value_at(unsigned i) const           { return m_array[i]; }

        const T* data() const { return m_array; }
              T* data()       { return m_array; }

        void remove_all()         { m_size = 0; }
        void clear()              { m_size = 0; }
        void cut_at(unsigned num) { if(num < m_size) m_size = num; }

    private:
        unsigned m_size;
        unsigned m_capacity;
        T*       m_array;
    };

     
    template<class T> 
    void pod_vector<T>::capacity(unsigned cap, unsigned extra_tail)
    {
        m_size = 0;
        if(cap > m_capacity)
        {
            pod_allocator<T>::deallocate(m_array, m_capacity);
            m_capacity = cap + extra_tail;
            m_array = m_capacity ? pod_allocator<T>::allocate(m_capacity) : 0;
        }
    }

     
    template<class T> 
    void pod_vector<T>::allocate(unsigned size, unsigned extra_tail)
    {
        capacity(size, extra_tail);
        m_size = size;
    }


     
    template<class T> 
    void pod_vector<T>::resize(unsigned new_size)
    {
        if(new_size > m_size)
        {
            if(new_size > m_capacity)
            {
                T* data = pod_allocator<T>::allocate(new_size);
                std::memcpy(data, m_array, m_size * sizeof(T));
                pod_allocator<T>::deallocate(m_array, m_capacity);
                m_array = data;
            }
        }
        else
        {
            m_size = new_size;
        }
    }

     
    template<class T> pod_vector<T>::pod_vector(unsigned cap, unsigned extra_tail) :
        m_size(0), 
        m_capacity(cap + extra_tail), 
        m_array(pod_allocator<T>::allocate(m_capacity)) {}

     
    template<class T> pod_vector<T>::pod_vector(const pod_vector<T>& v) :
        m_size(v.m_size),
        m_capacity(v.m_capacity),
        m_array(v.m_capacity ? pod_allocator<T>::allocate(v.m_capacity) : 0)
    {
        std::memcpy(m_array, v.m_array, sizeof(T) * v.m_size);
    }

     
    template<class T> const pod_vector<T>& 
    pod_vector<T>::operator = (const pod_vector<T>&v)
    {
        allocate(v.m_size);
        if(v.m_size) std::memcpy(m_array, v.m_array, sizeof(T) * v.m_size);
        return *this;
    }

     
    template<class T> void pod_vector<T>::serialize(int8u* ptr) const
    { 
        if(m_size) std::memcpy(ptr, m_array, m_size * sizeof(T)); 
    }

     
    template<class T> 
    void pod_vector<T>::deserialize(const int8u* data, unsigned byte_size)
    {
        byte_size /= sizeof(T);
        allocate(byte_size);
        if(byte_size) std::memcpy(m_array, data, byte_size * sizeof(T));
    }

     
    template<class T> 
    void pod_vector<T>::insert_at(unsigned pos, const T& val)
    {
        if(pos >= m_size) 
        {
            m_array[m_size] = val;
        }
        else
        {
            std::memmove(m_array + pos + 1, m_array + pos, (m_size - pos) * sizeof(T));
            m_array[pos] = val;
        }
        ++m_size;
    }

     
     
     
     
     
     
     
     
     
     
     
    template<class T, unsigned S=6> class pod_bvector
    {
    public:
        enum block_scale_e
        {   
            block_shift = S,
            block_size  = 1 << block_shift,
            block_mask  = block_size - 1
        };

        typedef T value_type;

        ~pod_bvector();
        pod_bvector();
        pod_bvector(unsigned block_ptr_inc);

         
        pod_bvector(const pod_bvector<T, S>& v);
        const pod_bvector<T, S>& operator = (const pod_bvector<T, S>& v);

        void remove_all() { m_size = 0; }
        void clear()      { m_size = 0; }
        void free_all()   { free_tail(0); }
        void free_tail(unsigned size);
        void add(const T& val);
        void push_back(const T& val) { add(val); }
        void modify_last(const T& val);
        void remove_last();

        int allocate_continuous_block(unsigned num_elements);

        void add_array(const T* ptr, unsigned num_elem)
        {
            while(num_elem--)
            {
                add(*ptr++);
            }
        }

        template<class DataAccessor> void add_data(DataAccessor& data)
        {
            while(data.size())
            {
                add(*data);
                ++data;
            }
        }

        void cut_at(unsigned size)
        {
            if(size < m_size) m_size = size;
        }

        unsigned size() const { return m_size; }

        const T& operator [] (unsigned i) const
        {
            return m_blocks[i >> block_shift][i & block_mask];
        }

        T& operator [] (unsigned i)
        {
            return m_blocks[i >> block_shift][i & block_mask];
        }

        const T& at(unsigned i) const
        { 
            return m_blocks[i >> block_shift][i & block_mask];
        }

        T& at(unsigned i) 
        { 
            return m_blocks[i >> block_shift][i & block_mask];
        }

        T value_at(unsigned i) const
        { 
            return m_blocks[i >> block_shift][i & block_mask];
        }

        const T& curr(unsigned idx) const
        {
            return (*this)[idx];
        }

        T& curr(unsigned idx)
        {
            return (*this)[idx];
        }

        const T& prev(unsigned idx) const
        {
            return (*this)[(idx + m_size - 1) % m_size];
        }

        T& prev(unsigned idx)
        {
            return (*this)[(idx + m_size - 1) % m_size];
        }

        const T& next(unsigned idx) const
        {
            return (*this)[(idx + 1) % m_size];
        }

        T& next(unsigned idx)
        {
            return (*this)[(idx + 1) % m_size];
        }

        const T& last() const
        {
            return (*this)[m_size - 1];
        }

        T& last()
        {
            return (*this)[m_size - 1];
        }

        unsigned byte_size() const;
        void serialize(int8u* ptr) const;
        void deserialize(const int8u* data, unsigned byte_size);
        void deserialize(unsigned start, const T& empty_val, 
                         const int8u* data, unsigned byte_size);

        template<class ByteAccessor> 
        void deserialize(ByteAccessor data)
        {
            remove_all();
            unsigned elem_size = data.size() / sizeof(T);

            for(unsigned i = 0; i < elem_size; ++i)
            {
                int8u* ptr = (int8u*)data_ptr();
                for(unsigned j = 0; j < sizeof(T); ++j)
                {
                    *ptr++ = *data;
                    ++data;
                }
                ++m_size;
            }
        }

        template<class ByteAccessor>
        void deserialize(unsigned start, const T& empty_val, ByteAccessor data)
        {
            while(m_size < start)
            {
                add(empty_val);
            }

            unsigned elem_size = data.size() / sizeof(T);
            for(unsigned i = 0; i < elem_size; ++i)
            {
                int8u* ptr;
                if(start + i < m_size)
                {
                    ptr = (int8u*)(&((*this)[start + i]));
                }
                else
                {
                    ptr = (int8u*)data_ptr();
                    ++m_size;
                }
                for(unsigned j = 0; j < sizeof(T); ++j)
                {
                    *ptr++ = *data;
                    ++data;
                }
            }
        }

        const T* block(unsigned nb) const { return m_blocks[nb]; }

    private:
        void allocate_block(unsigned nb);
        T*   data_ptr();

        unsigned        m_size;
        unsigned        m_num_blocks;
        unsigned        m_max_blocks;
        T**             m_blocks;
        unsigned        m_block_ptr_inc;
    };


     
    template<class T, unsigned S> pod_bvector<T, S>::~pod_bvector()
    {
        if(m_num_blocks)
        {
            T** blk = m_blocks + m_num_blocks - 1;
            while(m_num_blocks--)
            {
                pod_allocator<T>::deallocate(*blk, block_size);
                --blk;
            }
        }
        pod_allocator<T*>::deallocate(m_blocks, m_max_blocks);
    }


     
    template<class T, unsigned S> 
    void pod_bvector<T, S>::free_tail(unsigned size)
    {
        if(size < m_size)
        {
            unsigned nb = (size + block_mask) >> block_shift;
            while(m_num_blocks > nb)
            {
                pod_allocator<T>::deallocate(m_blocks[--m_num_blocks], block_size);
            }
            if(m_num_blocks == 0)
            {
                pod_allocator<T*>::deallocate(m_blocks, m_max_blocks);
                m_blocks = 0;
                m_max_blocks = 0;
            }
            m_size = size;
        }
    }


     
    template<class T, unsigned S> pod_bvector<T, S>::pod_bvector() :
        m_size(0),
        m_num_blocks(0),
        m_max_blocks(0),
        m_blocks(0),
        m_block_ptr_inc(block_size)
    {
    }


     
    template<class T, unsigned S> 
    pod_bvector<T, S>::pod_bvector(unsigned block_ptr_inc) :
        m_size(0),
        m_num_blocks(0),
        m_max_blocks(0),
        m_blocks(0),
        m_block_ptr_inc(block_ptr_inc)
    {
    }


     
    template<class T, unsigned S> 
    pod_bvector<T, S>::pod_bvector(const pod_bvector<T, S>& v) :
        m_size(v.m_size),
        m_num_blocks(v.m_num_blocks),
        m_max_blocks(v.m_max_blocks),
        m_blocks(v.m_max_blocks ? 
                 pod_allocator<T*>::allocate(v.m_max_blocks) : 
                 0),
        m_block_ptr_inc(v.m_block_ptr_inc)
    {
        unsigned i;
        for(i = 0; i < v.m_num_blocks; ++i)
        {
            m_blocks[i] = pod_allocator<T>::allocate(block_size);
            std::memcpy(m_blocks[i], v.m_blocks[i], block_size * sizeof(T));
        }
    }


     
    template<class T, unsigned S> 
    const pod_bvector<T, S>& 
    pod_bvector<T, S>::operator = (const pod_bvector<T, S>& v)
    {
        unsigned i;
        for(i = m_num_blocks; i < v.m_num_blocks; ++i)
        {
            allocate_block(i);
        }
        for(i = 0; i < v.m_num_blocks; ++i)
        {
            std::memcpy(m_blocks[i], v.m_blocks[i], block_size * sizeof(T));
        }
        m_size = v.m_size;
        return *this;
    }


     
    template<class T, unsigned S>
    void pod_bvector<T, S>::allocate_block(unsigned nb)
    {
        if(nb >= m_max_blocks) 
        {
            T** new_blocks = pod_allocator<T*>::allocate(m_max_blocks + m_block_ptr_inc);

            if(m_blocks)
            {
                std::memcpy(new_blocks, 
                       m_blocks, 
                       m_num_blocks * sizeof(T*));

                pod_allocator<T*>::deallocate(m_blocks, m_max_blocks);
            }
            m_blocks = new_blocks;
            m_max_blocks += m_block_ptr_inc;
        }
        m_blocks[nb] = pod_allocator<T>::allocate(block_size);
        m_num_blocks++;
    }



     
    template<class T, unsigned S>
    inline T* pod_bvector<T, S>::data_ptr()
    {
        unsigned nb = m_size >> block_shift;
        if(nb >= m_num_blocks)
        {
            allocate_block(nb);
        }
        return m_blocks[nb] + (m_size & block_mask);
    }



     
    template<class T, unsigned S> 
    inline void pod_bvector<T, S>::add(const T& val)
    {
        *data_ptr() = val;
        ++m_size;
    }


     
    template<class T, unsigned S> 
    inline void pod_bvector<T, S>::remove_last()
    {
        if(m_size) --m_size;
    }


     
    template<class T, unsigned S> 
    void pod_bvector<T, S>::modify_last(const T& val)
    {
        remove_last();
        add(val);
    }


     
    template<class T, unsigned S> 
    int pod_bvector<T, S>::allocate_continuous_block(unsigned num_elements)
    {
        if(num_elements < block_size)
        {
            data_ptr();  
            unsigned rest = block_size - (m_size & block_mask);
            unsigned index;
            if(num_elements <= rest)
            {
                 
                 
                index = m_size;
                m_size += num_elements;
                return index;
            }

             
             
            m_size += rest;
            data_ptr();
            index = m_size;
            m_size += num_elements;
            return index;
        }
        return -1;  
    }


     
    template<class T, unsigned S> 
    unsigned pod_bvector<T, S>::byte_size() const
    {
        return m_size * sizeof(T);
    }


     
    template<class T, unsigned S> 
    void pod_bvector<T, S>::serialize(int8u* ptr) const
    {
        unsigned i;
        for(i = 0; i < m_size; i++)
        {
            std::memcpy(ptr, &(*this)[i], sizeof(T));
            ptr += sizeof(T);
        }
    }

     
    template<class T, unsigned S> 
    void pod_bvector<T, S>::deserialize(const int8u* data, unsigned byte_size)
    {
        remove_all();
        byte_size /= sizeof(T);
        for(unsigned i = 0; i < byte_size; ++i)
        {
            T* ptr = data_ptr();
            std::memcpy(ptr, data, sizeof(T));
            ++m_size;
            data += sizeof(T);
        }
    }


     
     
    template<class T, unsigned S> 
    void pod_bvector<T, S>::deserialize(unsigned start, const T& empty_val, 
                                        const int8u* data, unsigned byte_size)
    {
        while(m_size < start)
        {
            add(empty_val);
        }

        byte_size /= sizeof(T);
        for(unsigned i = 0; i < byte_size; ++i)
        {
            if(start + i < m_size)
            {
                std::memcpy(&((*this)[start + i]), data, sizeof(T));
            }
            else
            {
                T* ptr = data_ptr();
                std::memcpy(ptr, data, sizeof(T));
                ++m_size;
            }
            data += sizeof(T);
        }
    }


     
     
     
     
     
     
     
     
    class block_allocator
    {
        struct block_type
        {
            int8u*   data;
            unsigned size;
        };

    public:
        void remove_all()
        {
            if(m_num_blocks)
            {
                block_type* blk = m_blocks + m_num_blocks - 1;
                while(m_num_blocks--)
                {
                    pod_allocator<int8u>::deallocate(blk->data, blk->size);
                    --blk;
                }
                pod_allocator<block_type>::deallocate(m_blocks, m_max_blocks);
            }
            m_num_blocks = 0;
            m_max_blocks = 0;
            m_blocks = 0;
            m_buf_ptr = 0;
            m_rest = 0;
        }

        ~block_allocator()
        {
            remove_all();
        }

        block_allocator(unsigned block_size, unsigned block_ptr_inc=256-8) :
            m_block_size(block_size),
            m_block_ptr_inc(block_ptr_inc),
            m_num_blocks(0),
            m_max_blocks(0),
            m_blocks(0),
            m_buf_ptr(0),
            m_rest(0)
        {
        }
       

        int8u* allocate(unsigned size, unsigned alignment=1)
        {
            if(size == 0) return 0;
            if(size <= m_rest)
            {
                int8u* ptr = m_buf_ptr;
                if(alignment > 1)
                {
                    unsigned align = 
                        (alignment - unsigned((std::size_t)ptr) % alignment) % alignment;

                    size += align;
                    ptr += align;
                    if(size <= m_rest)
                    {
                        m_rest -= size;
                        m_buf_ptr += size;
                        return ptr;
                    }
                    allocate_block(size);
                    return allocate(size - align, alignment);
                }
                m_rest -= size;
                m_buf_ptr += size;
                return ptr;
            }
            allocate_block(size + alignment - 1);
            return allocate(size, alignment);
        }


    private:
        void allocate_block(unsigned size)
        {
            if(size < m_block_size) size = m_block_size;
            if(m_num_blocks >= m_max_blocks) 
            {
                block_type* new_blocks = 
                    pod_allocator<block_type>::allocate(m_max_blocks + m_block_ptr_inc);

                if(m_blocks)
                {
                    std::memcpy(new_blocks, 
                           m_blocks, 
                           m_num_blocks * sizeof(block_type));
                    pod_allocator<block_type>::deallocate(m_blocks, m_max_blocks);
                }
                m_blocks = new_blocks;
                m_max_blocks += m_block_ptr_inc;
            }

            m_blocks[m_num_blocks].size = size;
            m_blocks[m_num_blocks].data = 
                m_buf_ptr =
                pod_allocator<int8u>::allocate(size);

            m_num_blocks++;
            m_rest = size;
        }

        unsigned    m_block_size;
        unsigned    m_block_ptr_inc;
        unsigned    m_num_blocks;
        unsigned    m_max_blocks;
        block_type* m_blocks;
        int8u*      m_buf_ptr;
        unsigned    m_rest;
    };








     
    enum quick_sort_threshold_e
    {
        quick_sort_threshold = 9
    };

    
     
    template<class T> inline void swap_elements(T& a, T& b)
    {
        T temp = a;
        a = b;
        b = temp;
    }


     
    template<class Array, class Less>
    void quick_sort(Array& arr, Less less)
    {
        if(arr.size() < 2) return;

        typename Array::value_type* e1;
        typename Array::value_type* e2;

        int  stack[80];
        int* top = stack; 
        int  limit = arr.size();
        int  base = 0;

        for(;;)
        {
            int len = limit - base;

            int i;
            int j;
            int pivot;

            if(len > quick_sort_threshold)
            {
                 
                pivot = base + len / 2;
                swap_elements(arr[base], arr[pivot]);

                i = base + 1;
                j = limit - 1;

                 
                e1 = &(arr[j]); 
                e2 = &(arr[i]);
                if(less(*e1, *e2)) swap_elements(*e1, *e2);

                e1 = &(arr[base]); 
                e2 = &(arr[i]);
                if(less(*e1, *e2)) swap_elements(*e1, *e2);

                e1 = &(arr[j]); 
                e2 = &(arr[base]);
                if(less(*e1, *e2)) swap_elements(*e1, *e2);

                for(;;)
                {
                    do i++; while( less(arr[i], arr[base]) );
                    do j--; while( less(arr[base], arr[j]) );

                    if( i > j )
                    {
                        break;
                    }

                    swap_elements(arr[i], arr[j]);
                }

                swap_elements(arr[base], arr[j]);

                 
                if(j - base > limit - i)
                {
                    top[0] = base;
                    top[1] = j;
                    base   = i;
                }
                else
                {
                    top[0] = i;
                    top[1] = limit;
                    limit  = j;
                }
                top += 2;
            }
            else
            {
                 
                j = base;
                i = j + 1;

                for(; i < limit; j = i, i++)
                {
                    for(; less(*(e1 = &(arr[j + 1])), *(e2 = &(arr[j]))); j--)
                    {
                        swap_elements(*e1, *e2);
                        if(j == base)
                        {
                            break;
                        }
                    }
                }
                if(top > stack)
                {
                    top  -= 2;
                    base  = top[0];
                    limit = top[1];
                }
                else
                {
                    break;
                }
            }
        }
    }




     
     
     
     
    template<class Array, class Equal>
    unsigned remove_duplicates(Array& arr, Equal equal)
    {
        if(arr.size() < 2) return arr.size();

        unsigned i, j;
        for(i = 1, j = 1; i < arr.size(); i++)
        {
            typename Array::value_type& e = arr[i];
            if(!equal(e, arr[i - 1]))
            {
                arr[j++] = e;
            }
        }
        return j;
    }

     
    template<class Array> void invert_container(Array& arr)
    {
        int i = 0;
        int j = arr.size() - 1;
        while(i < j)
        {
            swap_elements(arr[i++], arr[j--]);
        }
    }

     
    template<class Array, class Value, class Less>
    unsigned binary_search_pos(const Array& arr, const Value& val, Less less)
    {
        if(arr.size() == 0) return 0;

        unsigned beg = 0;
        unsigned end = arr.size() - 1;

        if(less(val, arr[0])) return 0;
        if(less(arr[end], val)) return end + 1;

        while(end - beg > 1)
        {
            unsigned mid = (end + beg) >> 1;
            if(less(val, arr[mid])) end = mid; 
            else                    beg = mid;
        }

         
         

        return end;
    }

     
    template<class Array> class range_adaptor
    {
    public:
        typedef typename Array::value_type value_type;

        range_adaptor(Array& array, unsigned start, unsigned size) :
            m_array(array), m_start(start), m_size(size)
        {}

        unsigned size() const { return m_size; }
        const value_type& operator [] (unsigned i) const { return m_array[m_start + i]; }
              value_type& operator [] (unsigned i)       { return m_array[m_start + i]; }
        const value_type& at(unsigned i) const           { return m_array[m_start + i]; }
              value_type& at(unsigned i)                 { return m_array[m_start + i]; }
        value_type  value_at(unsigned i) const           { return m_array[m_start + i]; }

    private:
        Array& m_array;
        unsigned m_start;
        unsigned m_size;
    };

     
    inline bool int_less(int a, int b) { return a < b; }

     
    inline bool int_greater(int a, int b) { return a > b; }

     
    inline bool unsigned_less(unsigned a, unsigned b) { return a < b; }

     
    inline bool unsigned_greater(unsigned a, unsigned b) { return a > b; }
}

#endif

namespace agg
{

     
    template<class T> class row_accessor
    {
    public:
        typedef const_row_info<T> row_data;

         
        row_accessor() :
            m_buf(0),
            m_start(0),
            m_width(0),
            m_height(0),
            m_stride(0)
        {
        }

         
        row_accessor(T* buf, unsigned width, unsigned height, int stride) :
            m_buf(0),
            m_start(0),
            m_width(0),
            m_height(0),
            m_stride(0)
        {
            attach(buf, width, height, stride);
        }


         
        void attach(T* buf, unsigned width, unsigned height, int stride)
        {
            m_buf = m_start = buf;
            m_width = width;
            m_height = height;
            m_stride = stride;
            if(stride < 0) 
            { 
                m_start = m_buf - (AGG_INT64)(height - 1) * stride;
            }
        }

         
        AGG_INLINE       T* buf()          { return m_buf;    }
        AGG_INLINE const T* buf()    const { return m_buf;    }
        AGG_INLINE unsigned width()  const { return m_width;  }
        AGG_INLINE unsigned height() const { return m_height; }
        AGG_INLINE int      stride() const { return m_stride; }
        AGG_INLINE unsigned stride_abs() const 
        {
            return (m_stride < 0) ? unsigned(-m_stride) : unsigned(m_stride); 
        }

         
        AGG_INLINE       T* row_ptr(int, int y, unsigned) 
        { 
            return m_start + y * (AGG_INT64)m_stride; 
        }
        AGG_INLINE       T* row_ptr(int y)       { return m_start + y * (AGG_INT64)m_stride; }
        AGG_INLINE const T* row_ptr(int y) const { return m_start + y * (AGG_INT64)m_stride; }
        AGG_INLINE row_data row    (int y) const 
        { 
            return row_data(0, m_width-1, row_ptr(y)); 
        }

         
        template<class RenBuf>
        void copy_from(const RenBuf& src)
        {
            unsigned h = height();
            if(src.height() < h) h = src.height();
        
            unsigned l = stride_abs();
            if(src.stride_abs() < l) l = src.stride_abs();
        
            l *= sizeof(T);

            unsigned y;
            unsigned w = width();
            for (y = 0; y < h; y++)
            {
                std::memcpy(row_ptr(0, y, w), src.row_ptr(y), l);
            }
        }

         
        void clear(T value)
        {
            unsigned y;
            unsigned w = width();
            unsigned stride = stride_abs();
            for(y = 0; y < height(); y++)
            {
                T* p = row_ptr(0, y, w);
                unsigned x;
                for(x = 0; x < stride; x++)
                {
                    *p++ = value;
                }
            }
        }

    private:
         
        T*            m_buf;     
        T*            m_start;   
        unsigned      m_width;   
        unsigned      m_height;  
        int           m_stride;  
    };




     
    template<class T> class row_ptr_cache
    {
    public:
        typedef const_row_info<T> row_data;

         
        row_ptr_cache() :
            m_buf(0),
            m_rows(),
            m_width(0),
            m_height(0),
            m_stride(0)
        {
        }

         
        row_ptr_cache(T* buf, unsigned width, unsigned height, int stride) :
            m_buf(0),
            m_rows(),
            m_width(0),
            m_height(0),
            m_stride(0)
        {
            attach(buf, width, height, stride);
        }

         
        void attach(T* buf, unsigned width, unsigned height, int stride)
        {
            m_buf = buf;
            m_width = width;
            m_height = height;
            m_stride = stride;
            if(height > m_rows.size())
            {
                m_rows.resize(height);
            }

            T* row_ptr = m_buf;

            if(stride < 0)
            {
                row_ptr = m_buf - (AGG_INT64)(height - 1) * stride;
            }

            T** rows = &m_rows[0];

            while(height--)
            {
                *rows++ = row_ptr;
                row_ptr += stride;
            }
        }

         
        AGG_INLINE       T* buf()          { return m_buf;    }
        AGG_INLINE const T* buf()    const { return m_buf;    }
        AGG_INLINE unsigned width()  const { return m_width;  }
        AGG_INLINE unsigned height() const { return m_height; }
        AGG_INLINE int      stride() const { return m_stride; }
        AGG_INLINE unsigned stride_abs() const 
        {
            return (m_stride < 0) ? unsigned(-m_stride) : unsigned(m_stride); 
        }

         
        AGG_INLINE       T* row_ptr(int, int y, unsigned) 
        { 
            return m_rows[y]; 
        }
        AGG_INLINE       T* row_ptr(int y)       { return m_rows[y]; }
        AGG_INLINE const T* row_ptr(int y) const { return m_rows[y]; }
        AGG_INLINE row_data row    (int y) const 
        { 
            return row_data(0, m_width-1, m_rows[y]); 
        }

         
        T const* const* rows() const { return &m_rows[0]; }

         
        template<class RenBuf>
        void copy_from(const RenBuf& src)
        {
            unsigned h = height();
            if(src.height() < h) h = src.height();
        
            unsigned l = stride_abs();
            if(src.stride_abs() < l) l = src.stride_abs();
        
            l *= sizeof(T);

            unsigned y;
            unsigned w = width();
            for (y = 0; y < h; y++)
            {
                std::memcpy(row_ptr(0, y, w), src.row_ptr(y), l);
            }
        }

         
        void clear(T value)
        {
            unsigned y;
            unsigned w = width();
            unsigned stride = stride_abs();
            for(y = 0; y < height(); y++)
            {
                T* p = row_ptr(0, y, w);
                unsigned x;
                for(x = 0; x < stride; x++)
                {
                    *p++ = value;
                }
            }
        }

    private:
         
        T*            m_buf;         
        pod_array<T*> m_rows;        
        unsigned      m_width;       
        unsigned      m_height;      
        int           m_stride;      
    };




     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
#ifdef AGG_RENDERING_BUFFER
    typedef AGG_RENDERING_BUFFER rendering_buffer;
#else
 
    typedef row_accessor<int8u> rendering_buffer;
#endif

}


#endif

namespace agg
{
     
    struct one_component_mask_u8
    {
        static unsigned calculate(const int8u* p) { return *p; }
    };
    

     
    template<unsigned R, unsigned G, unsigned B>
    struct rgb_to_gray_mask_u8
    {
        static unsigned calculate(const int8u* p) 
        { 
            return (p[R]*77 + p[G]*150 + p[B]*29) >> 8; 
        }
    };

     
    template<unsigned Step=1, unsigned Offset=0, class MaskF=one_component_mask_u8>
    class alpha_mask_u8
    {
    public:
        typedef int8u cover_type;
        typedef alpha_mask_u8<Step, Offset, MaskF> self_type;
        enum cover_scale_e
        { 
            cover_shift = 8,
            cover_none  = 0,
            cover_full  = 255
        };

        alpha_mask_u8() : m_rbuf(0) {}
        explicit alpha_mask_u8(rendering_buffer& rbuf) : m_rbuf(&rbuf) {}

        void attach(rendering_buffer& rbuf) { m_rbuf = &rbuf; }

        MaskF& mask_function() { return m_mask_function; }
        const MaskF& mask_function() const { return m_mask_function; }

        
         
        cover_type pixel(int x, int y) const
        {
            if(x >= 0 && y >= 0 && 
               x < (int)m_rbuf->width() && 
               y < (int)m_rbuf->height())
            {
                return (cover_type)m_mask_function.calculate(
                                        m_rbuf->row_ptr(y) + x * Step + Offset);
            }
            return 0;
        }

         
        cover_type combine_pixel(int x, int y, cover_type val) const
        {
            if(x >= 0 && y >= 0 && 
               x < (int)m_rbuf->width() && 
               y < (int)m_rbuf->height())
            {
                return (cover_type)((cover_full + val * 
                                     m_mask_function.calculate(
                                        m_rbuf->row_ptr(y) + x * Step + Offset)) >> 
                                     cover_shift);
            }
            return 0;
        }


         
        void fill_hspan(int x, int y, cover_type* dst, int num_pix) const
        {
            int xmax = m_rbuf->width() - 1;
            int ymax = m_rbuf->height() - 1;

            int count = num_pix;
            cover_type* covers = dst;

            if(y < 0 || y > ymax)
            {
                std::memset(dst, 0, num_pix * sizeof(cover_type));
                return;
            }

            if(x < 0)
            {
                count += x;
                if(count <= 0) 
                {
                    std::memset(dst, 0, num_pix * sizeof(cover_type));
                    return;
                }
                std::memset(covers, 0, -x * sizeof(cover_type));
                covers -= x;
                x = 0;
            }

            if(x + count > xmax)
            {
                int rest = x + count - xmax - 1;
                count -= rest;
                if(count <= 0) 
                {
                    std::memset(dst, 0, num_pix * sizeof(cover_type));
                    return;
                }
                std::memset(covers + count, 0, rest * sizeof(cover_type));
            }

            const int8u* mask = m_rbuf->row_ptr(y) + x * Step + Offset;
            do
            {
                *covers++ = (cover_type)m_mask_function.calculate(mask);
                mask += Step;
            }
            while(--count);
        }


         
        void combine_hspan(int x, int y, cover_type* dst, int num_pix) const
        {
            int xmax = m_rbuf->width() - 1;
            int ymax = m_rbuf->height() - 1;

            int count = num_pix;
            cover_type* covers = dst;

            if(y < 0 || y > ymax)
            {
                std::memset(dst, 0, num_pix * sizeof(cover_type));
                return;
            }

            if(x < 0)
            {
                count += x;
                if(count <= 0) 
                {
                    std::memset(dst, 0, num_pix * sizeof(cover_type));
                    return;
                }
                std::memset(covers, 0, -x * sizeof(cover_type));
                covers -= x;
                x = 0;
            }

            if(x + count > xmax)
            {
                int rest = x + count - xmax - 1;
                count -= rest;
                if(count <= 0) 
                {
                    std::memset(dst, 0, num_pix * sizeof(cover_type));
                    return;
                }
                std::memset(covers + count, 0, rest * sizeof(cover_type));
            }

            const int8u* mask = m_rbuf->row_ptr(y) + x * Step + Offset;
            do
            {
                *covers = (cover_type)((cover_full + (*covers) * 
                                       m_mask_function.calculate(mask)) >> 
                                       cover_shift);
                ++covers;
                mask += Step;
            }
            while(--count);
        }

         
        void fill_vspan(int x, int y, cover_type* dst, int num_pix) const
        {
            int xmax = m_rbuf->width() - 1;
            int ymax = m_rbuf->height() - 1;

            int count = num_pix;
            cover_type* covers = dst;

            if(x < 0 || x > xmax)
            {
                std::memset(dst, 0, num_pix * sizeof(cover_type));
                return;
            }

            if(y < 0)
            {
                count += y;
                if(count <= 0) 
                {
                    std::memset(dst, 0, num_pix * sizeof(cover_type));
                    return;
                }
                std::memset(covers, 0, -y * sizeof(cover_type));
                covers -= y;
                y = 0;
            }

            if(y + count > ymax)
            {
                int rest = y + count - ymax - 1;
                count -= rest;
                if(count <= 0) 
                {
                    std::memset(dst, 0, num_pix * sizeof(cover_type));
                    return;
                }
                std::memset(covers + count, 0, rest * sizeof(cover_type));
            }

            const int8u* mask = m_rbuf->row_ptr(y) + x * Step + Offset;
            do
            {
                *covers++ = (cover_type)m_mask_function.calculate(mask);
                mask += m_rbuf->stride();
            }
            while(--count);
        }

         
        void combine_vspan(int x, int y, cover_type* dst, int num_pix) const
        {
            int xmax = m_rbuf->width() - 1;
            int ymax = m_rbuf->height() - 1;

            int count = num_pix;
            cover_type* covers = dst;

            if(x < 0 || x > xmax)
            {
                std::memset(dst, 0, num_pix * sizeof(cover_type));
                return;
            }

            if(y < 0)
            {
                count += y;
                if(count <= 0) 
                {
                    std::memset(dst, 0, num_pix * sizeof(cover_type));
                    return;
                }
                std::memset(covers, 0, -y * sizeof(cover_type));
                covers -= y;
                y = 0;
            }

            if(y + count > ymax)
            {
                int rest = y + count - ymax - 1;
                count -= rest;
                if(count <= 0) 
                {
                    std::memset(dst, 0, num_pix * sizeof(cover_type));
                    return;
                }
                std::memset(covers + count, 0, rest * sizeof(cover_type));
            }

            const int8u* mask = m_rbuf->row_ptr(y) + x * Step + Offset;
            do
            {
                *covers = (cover_type)((cover_full + (*covers) * 
                                       m_mask_function.calculate(mask)) >> 
                                       cover_shift);
                ++covers;
                mask += m_rbuf->stride();
            }
            while(--count);
        }


    private:
        alpha_mask_u8(const self_type&);
        const self_type& operator = (const self_type&);

        rendering_buffer* m_rbuf;
        MaskF             m_mask_function;
    };
    

    typedef alpha_mask_u8<1, 0> alpha_mask_gray8;    

    typedef alpha_mask_u8<3, 0> alpha_mask_rgb24r;   
    typedef alpha_mask_u8<3, 1> alpha_mask_rgb24g;   
    typedef alpha_mask_u8<3, 2> alpha_mask_rgb24b;   

    typedef alpha_mask_u8<3, 2> alpha_mask_bgr24r;   
    typedef alpha_mask_u8<3, 1> alpha_mask_bgr24g;   
    typedef alpha_mask_u8<3, 0> alpha_mask_bgr24b;   

    typedef alpha_mask_u8<4, 0> alpha_mask_rgba32r;  
    typedef alpha_mask_u8<4, 1> alpha_mask_rgba32g;  
    typedef alpha_mask_u8<4, 2> alpha_mask_rgba32b;  
    typedef alpha_mask_u8<4, 3> alpha_mask_rgba32a;  

    typedef alpha_mask_u8<4, 1> alpha_mask_argb32r;  
    typedef alpha_mask_u8<4, 2> alpha_mask_argb32g;  
    typedef alpha_mask_u8<4, 3> alpha_mask_argb32b;  
    typedef alpha_mask_u8<4, 0> alpha_mask_argb32a;  

    typedef alpha_mask_u8<4, 2> alpha_mask_bgra32r;  
    typedef alpha_mask_u8<4, 1> alpha_mask_bgra32g;  
    typedef alpha_mask_u8<4, 0> alpha_mask_bgra32b;  
    typedef alpha_mask_u8<4, 3> alpha_mask_bgra32a;  

    typedef alpha_mask_u8<4, 3> alpha_mask_abgr32r;  
    typedef alpha_mask_u8<4, 2> alpha_mask_abgr32g;  
    typedef alpha_mask_u8<4, 1> alpha_mask_abgr32b;  
    typedef alpha_mask_u8<4, 0> alpha_mask_abgr32a;  

    typedef alpha_mask_u8<3, 0, rgb_to_gray_mask_u8<0, 1, 2> > alpha_mask_rgb24gray;   
    typedef alpha_mask_u8<3, 0, rgb_to_gray_mask_u8<2, 1, 0> > alpha_mask_bgr24gray;   
    typedef alpha_mask_u8<4, 0, rgb_to_gray_mask_u8<0, 1, 2> > alpha_mask_rgba32gray;  
    typedef alpha_mask_u8<4, 1, rgb_to_gray_mask_u8<0, 1, 2> > alpha_mask_argb32gray;  
    typedef alpha_mask_u8<4, 0, rgb_to_gray_mask_u8<2, 1, 0> > alpha_mask_bgra32gray;  
    typedef alpha_mask_u8<4, 1, rgb_to_gray_mask_u8<2, 1, 0> > alpha_mask_abgr32gray;  



     
    template<unsigned Step=1, unsigned Offset=0, class MaskF=one_component_mask_u8>
    class amask_no_clip_u8
    {
    public:
        typedef int8u cover_type;
        typedef amask_no_clip_u8<Step, Offset, MaskF> self_type;
        enum cover_scale_e
        { 
            cover_shift = 8,
            cover_none  = 0,
            cover_full  = 255
        };

        amask_no_clip_u8() : m_rbuf(0) {}
        explicit amask_no_clip_u8(rendering_buffer& rbuf) : m_rbuf(&rbuf) {}

        void attach(rendering_buffer& rbuf) { m_rbuf = &rbuf; }

        MaskF& mask_function() { return m_mask_function; }
        const MaskF& mask_function() const { return m_mask_function; }


         
        cover_type pixel(int x, int y) const
        {
            return (cover_type)m_mask_function.calculate(
                                   m_rbuf->row_ptr(y) + x * Step + Offset);
        }

        
         
        cover_type combine_pixel(int x, int y, cover_type val) const
        {
            return (cover_type)((cover_full + val * 
                                 m_mask_function.calculate(
                                    m_rbuf->row_ptr(y) + x * Step + Offset)) >> 
                                 cover_shift);
        }


         
        void fill_hspan(int x, int y, cover_type* dst, int num_pix) const
        {
            const int8u* mask = m_rbuf->row_ptr(y) + x * Step + Offset;
            do
            {
                *dst++ = (cover_type)m_mask_function.calculate(mask);
                mask += Step;
            }
            while(--num_pix);
        }



         
        void combine_hspan(int x, int y, cover_type* dst, int num_pix) const
        {
            const int8u* mask = m_rbuf->row_ptr(y) + x * Step + Offset;
            do
            {
                *dst = (cover_type)((cover_full + (*dst) * 
                                    m_mask_function.calculate(mask)) >> 
                                    cover_shift);
                ++dst;
                mask += Step;
            }
            while(--num_pix);
        }


         
        void fill_vspan(int x, int y, cover_type* dst, int num_pix) const
        {
            const int8u* mask = m_rbuf->row_ptr(y) + x * Step + Offset;
            do
            {
                *dst++ = (cover_type)m_mask_function.calculate(mask);
                mask += m_rbuf->stride();
            }
            while(--num_pix);
        }


         
        void combine_vspan(int x, int y, cover_type* dst, int num_pix) const
        {
            const int8u* mask = m_rbuf->row_ptr(y) + x * Step + Offset;
            do
            {
                *dst = (cover_type)((cover_full + (*dst) * 
                                    m_mask_function.calculate(mask)) >> 
                                    cover_shift);
                ++dst;
                mask += m_rbuf->stride();
            }
            while(--num_pix);
        }

    private:
        amask_no_clip_u8(const self_type&);
        const self_type& operator = (const self_type&);

        rendering_buffer* m_rbuf;
        MaskF             m_mask_function;
    };
    

    typedef amask_no_clip_u8<1, 0> amask_no_clip_gray8;    

    typedef amask_no_clip_u8<3, 0> amask_no_clip_rgb24r;   
    typedef amask_no_clip_u8<3, 1> amask_no_clip_rgb24g;   
    typedef amask_no_clip_u8<3, 2> amask_no_clip_rgb24b;   

    typedef amask_no_clip_u8<3, 2> amask_no_clip_bgr24r;   
    typedef amask_no_clip_u8<3, 1> amask_no_clip_bgr24g;   
    typedef amask_no_clip_u8<3, 0> amask_no_clip_bgr24b;   

    typedef amask_no_clip_u8<4, 0> amask_no_clip_rgba32r;  
    typedef amask_no_clip_u8<4, 1> amask_no_clip_rgba32g;  
    typedef amask_no_clip_u8<4, 2> amask_no_clip_rgba32b;  
    typedef amask_no_clip_u8<4, 3> amask_no_clip_rgba32a;  

    typedef amask_no_clip_u8<4, 1> amask_no_clip_argb32r;  
    typedef amask_no_clip_u8<4, 2> amask_no_clip_argb32g;  
    typedef amask_no_clip_u8<4, 3> amask_no_clip_argb32b;  
    typedef amask_no_clip_u8<4, 0> amask_no_clip_argb32a;  

    typedef amask_no_clip_u8<4, 2> amask_no_clip_bgra32r;  
    typedef amask_no_clip_u8<4, 1> amask_no_clip_bgra32g;  
    typedef amask_no_clip_u8<4, 0> amask_no_clip_bgra32b;  
    typedef amask_no_clip_u8<4, 3> amask_no_clip_bgra32a;  

    typedef amask_no_clip_u8<4, 3> amask_no_clip_abgr32r;  
    typedef amask_no_clip_u8<4, 2> amask_no_clip_abgr32g;  
    typedef amask_no_clip_u8<4, 1> amask_no_clip_abgr32b;  
    typedef amask_no_clip_u8<4, 0> amask_no_clip_abgr32a;  

    typedef amask_no_clip_u8<3, 0, rgb_to_gray_mask_u8<0, 1, 2> > amask_no_clip_rgb24gray;   
    typedef amask_no_clip_u8<3, 0, rgb_to_gray_mask_u8<2, 1, 0> > amask_no_clip_bgr24gray;   
    typedef amask_no_clip_u8<4, 0, rgb_to_gray_mask_u8<0, 1, 2> > amask_no_clip_rgba32gray;  
    typedef amask_no_clip_u8<4, 1, rgb_to_gray_mask_u8<0, 1, 2> > amask_no_clip_argb32gray;  
    typedef amask_no_clip_u8<4, 0, rgb_to_gray_mask_u8<2, 1, 0> > amask_no_clip_bgra32gray;  
    typedef amask_no_clip_u8<4, 1, rgb_to_gray_mask_u8<2, 1, 0> > amask_no_clip_abgr32gray;  


}



#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_ARC_INCLUDED
#define AGG_ARC_INCLUDED


namespace agg
{

     
     
     
     
    class arc
    {
    public:
        arc() : m_scale(1.0), m_initialized(false) {}
        arc(double x,  double y, 
            double rx, double ry, 
            double a1, double a2, 
            bool ccw=true);

        void init(double x,  double y, 
                  double rx, double ry, 
                  double a1, double a2, 
                  bool ccw=true);

        void approximation_scale(double s);
        double approximation_scale() const { return m_scale;  }

        void rewind(unsigned);
        unsigned vertex(double* x, double* y);

    private:
        void normalize(double a1, double a2, bool ccw);

        double   m_x;
        double   m_y;
        double   m_rx;
        double   m_ry;
        double   m_angle;
        double   m_start;
        double   m_end;
        double   m_scale;
        double   m_da;
        bool     m_ccw;
        bool     m_initialized;
        unsigned m_path_cmd;
    };


}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_ARROWHEAD_INCLUDED
#define AGG_ARROWHEAD_INCLUDED


namespace agg
{

     
     
     
     
    class arrowhead
    {
    public:
        arrowhead();

        void head(double d1, double d2, double d3, double d4)
        {
            m_head_d1 = d1;
            m_head_d2 = d2;
            m_head_d3 = d3;
            m_head_d4 = d4;
            m_head_flag = true;
        }

        void head()    { m_head_flag = true; }
        void no_head() { m_head_flag = false; }

        void tail(double d1, double d2, double d3, double d4)
        {
            m_tail_d1 = d1;
            m_tail_d2 = d2;
            m_tail_d3 = d3;
            m_tail_d4 = d4;
            m_tail_flag = true;
        }

        void tail()    { m_tail_flag = true;  }
        void no_tail() { m_tail_flag = false; }

        void rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        double   m_head_d1;
        double   m_head_d2;
        double   m_head_d3;
        double   m_head_d4;
        double   m_tail_d1;
        double   m_tail_d2;
        double   m_tail_d3;
        double   m_tail_d4;
        bool     m_head_flag;
        bool     m_tail_flag;
        double   m_coord[16];
        unsigned m_cmd[8];
        unsigned m_curr_id;
        unsigned m_curr_coord;
    };

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_BEZIER_ARC_INCLUDED
#define AGG_BEZIER_ARC_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CONV_TRANSFORM_INCLUDED
#define AGG_CONV_TRANSFORM_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_TRANS_AFFINE_INCLUDED
#define AGG_TRANS_AFFINE_INCLUDED

#include <cmath>

namespace agg
{
    const double affine_epsilon = 1e-14; 

     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
    struct trans_affine
    {
        double sx, shy, shx, sy, tx, ty;

         
         
        trans_affine() :
            sx(1.0), shy(0.0), shx(0.0), sy(1.0), tx(0.0), ty(0.0)
        {}

         
        trans_affine(double v0, double v1, double v2, 
                     double v3, double v4, double v5) :
            sx(v0), shy(v1), shx(v2), sy(v3), tx(v4), ty(v5)
        {}

         
        explicit trans_affine(const double* m) :
            sx(m[0]), shy(m[1]), shx(m[2]), sy(m[3]), tx(m[4]), ty(m[5])
        {}

         
        trans_affine(double x1, double y1, double x2, double y2, 
                     const double* parl)
        {
            rect_to_parl(x1, y1, x2, y2, parl);
        }

         
        trans_affine(const double* parl, 
                     double x1, double y1, double x2, double y2)
        {
            parl_to_rect(parl, x1, y1, x2, y2);
        }

         
        trans_affine(const double* src, const double* dst)
        {
            parl_to_parl(src, dst);
        }

         
         
         
         
         
         
         
         
         
         
         
        const trans_affine& parl_to_parl(const double* src, 
                                         const double* dst);

        const trans_affine& rect_to_parl(double x1, double y1, 
                                         double x2, double y2, 
                                         const double* parl);

        const trans_affine& parl_to_rect(const double* parl, 
                                         double x1, double y1, 
                                         double x2, double y2);


         
         
        const trans_affine& reset();

         
        const trans_affine& translate(double x, double y);
        const trans_affine& rotate(double a);
        const trans_affine& scale(double s);
        const trans_affine& scale(double x, double y);

         
        const trans_affine& multiply(const trans_affine& m);

         
        const trans_affine& premultiply(const trans_affine& m);

         
        const trans_affine& multiply_inv(const trans_affine& m);

         
        const trans_affine& premultiply_inv(const trans_affine& m);

         
         
         
        const trans_affine& invert();

         
        const trans_affine& flip_x();

         
        const trans_affine& flip_y();

         
         
        void store_to(double* m) const
        {
            *m++ = sx; *m++ = shy; *m++ = shx; *m++ = sy; *m++ = tx; *m++ = ty;
        }

         
        const trans_affine& load_from(const double* m)
        {
            sx = *m++; shy = *m++; shx = *m++; sy = *m++; tx = *m++;  ty = *m++;
            return *this;
        }

         
        
         
        const trans_affine& operator *= (const trans_affine& m)
        {
            return multiply(m);
        }

         
        const trans_affine& operator /= (const trans_affine& m)
        {
            return multiply_inv(m);
        }

         
         
        trans_affine operator * (const trans_affine& m) const
        {
            return trans_affine(*this).multiply(m);
        }

         
         
        trans_affine operator / (const trans_affine& m) const
        {
            return trans_affine(*this).multiply_inv(m);
        }

         
        trans_affine operator ~ () const
        {
            trans_affine ret = *this;
            return ret.invert();
        }

         
        bool operator == (const trans_affine& m) const
        {
            return is_equal(m, affine_epsilon);
        }

         
        bool operator != (const trans_affine& m) const
        {
            return !is_equal(m, affine_epsilon);
        }

         
         
        void transform(double* x, double* y) const;

         
        void transform_2x2(double* x, double* y) const;

         
         
         
        void inverse_transform(double* x, double* y) const;

         
         
        double determinant() const
        {
            return sx * sy - shy * shx;
        }

         
        double determinant_reciprocal() const
        {
            return 1.0 / (sx * sy - shy * shx);
        }

         
         
         
        double scale() const;

         
        bool is_valid(double epsilon = affine_epsilon) const;

         
        bool is_identity(double epsilon = affine_epsilon) const;

         
        bool is_equal(const trans_affine& m, double epsilon = affine_epsilon) const;

         
         
        double rotation() const;
        void   translation(double* dx, double* dy) const;
        void   scaling(double* x, double* y) const;
        void   scaling_abs(double* x, double* y) const;
    };

     
    inline void trans_affine::transform(double* x, double* y) const
    {
        double tmp = *x;
        *x = tmp * sx  + *y * shx + tx;
        *y = tmp * shy + *y * sy  + ty;
    }

     
    inline void trans_affine::transform_2x2(double* x, double* y) const
    {
        double tmp = *x;
        *x = tmp * sx  + *y * shx;
        *y = tmp * shy + *y * sy;
    }

     
    inline void trans_affine::inverse_transform(double* x, double* y) const
    {
        double d = determinant_reciprocal();
        double a = (*x - tx) * d;
        double b = (*y - ty) * d;
        *x = a * sy - b * shx;
        *y = b * sx - a * shy;
    }

     
    inline double trans_affine::scale() const
    {
        double x = 0.70710678118654752440 * sx  + 0.70710678118654752440 * shx;
        double y = 0.70710678118654752440 * shy + 0.70710678118654752440 * sy;
        return std::sqrt(x*x + y*y);
    }

     
    inline const trans_affine& trans_affine::translate(double x, double y) 
    { 
        tx += x;
        ty += y; 
        return *this;
    }

     
    inline const trans_affine& trans_affine::rotate(double a) 
    {
        double ca = std::cos(a); 
        double sa = std::sin(a);
        double t0 = sx  * ca - shy * sa;
        double t2 = shx * ca - sy * sa;
        double t4 = tx  * ca - ty * sa;
        shy = sx  * sa + shy * ca;
        sy  = shx * sa + sy * ca; 
        ty  = tx  * sa + ty * ca;
        sx  = t0;
        shx = t2;
        tx  = t4;
        return *this;
    }

     
    inline const trans_affine& trans_affine::scale(double x, double y) 
    {
        double mm0 = x;  
        double mm3 = y; 
        sx  *= mm0;
        shx *= mm0;
        tx  *= mm0;
        shy *= mm3;
        sy  *= mm3;
        ty  *= mm3;
        return *this;
    }

     
    inline const trans_affine& trans_affine::scale(double s) 
    {
        double m = s;  
        sx  *= m;
        shx *= m;
        tx  *= m;
        shy *= m;
        sy  *= m;
        ty  *= m;
        return *this;
    }

     
    inline const trans_affine& trans_affine::premultiply(const trans_affine& m)
    {
        trans_affine t = m;
        return *this = t.multiply(*this);
    }

     
    inline const trans_affine& trans_affine::multiply_inv(const trans_affine& m)
    {
        trans_affine t = m;
        t.invert();
        return multiply(t);
    }

     
    inline const trans_affine& trans_affine::premultiply_inv(const trans_affine& m)
    {
        trans_affine t = m;
        t.invert();
        return *this = t.multiply(*this);
    }

     
    inline void trans_affine::scaling_abs(double* x, double* y) const
    {
         
         
         
        *x = std::sqrt(sx  * sx  + shx * shx);
        *y = std::sqrt(shy * shy + sy  * sy);
    }

     
     
     
     
     
    class trans_affine_rotation : public trans_affine
    {
    public:
        trans_affine_rotation(double a) : 
          trans_affine(std::cos(a), std::sin(a), -std::sin(a), std::cos(a), 0.0, 0.0)
        {}
    };

     
     
    class trans_affine_scaling : public trans_affine
    {
    public:
        trans_affine_scaling(double x, double y) : 
          trans_affine(x, 0.0, 0.0, y, 0.0, 0.0)
        {}

        trans_affine_scaling(double s) : 
          trans_affine(s, 0.0, 0.0, s, 0.0, 0.0)
        {}
    };

     
     
    class trans_affine_translation : public trans_affine
    {
    public:
        trans_affine_translation(double x, double y) : 
          trans_affine(1.0, 0.0, 0.0, 1.0, x, y)
        {}
    };

     
     
    class trans_affine_skewing : public trans_affine
    {
    public:
        trans_affine_skewing(double x, double y) : 
          trans_affine(1.0, std::tan(y), std::tan(x), 1.0, 0.0, 0.0)
        {}
    };


     
     
     
    class trans_affine_line_segment : public trans_affine
    {
    public:
        trans_affine_line_segment(double x1, double y1, double x2, double y2, 
                                  double dist)
        {
            double dx = x2 - x1;
            double dy = y2 - y1;
            if(dist > 0.0)
            {
                multiply(trans_affine_scaling(std::sqrt(dx * dx + dy * dy) / dist));
            }
            multiply(trans_affine_rotation(std::atan2(dy, dx)));
            multiply(trans_affine_translation(x1, y1));
        }
    };


     
     
     
     
    class trans_affine_reflection_unit : public trans_affine
    {
    public:
        trans_affine_reflection_unit(double ux, double uy) :
          trans_affine(2.0 * ux * ux - 1.0, 
                       2.0 * ux * uy, 
                       2.0 * ux * uy, 
                       2.0 * uy * uy - 1.0, 
                       0.0, 0.0)
        {}
    };


     
     
     
     
    class trans_affine_reflection : public trans_affine_reflection_unit
    {
    public:
        trans_affine_reflection(double a) :
          trans_affine_reflection_unit(std::cos(a), std::sin(a))
        {}


        trans_affine_reflection(double x, double y) :
          trans_affine_reflection_unit(x / std::sqrt(x * x + y * y), y / std::sqrt(x * x + y * y))
        {}
    };

}


#endif


namespace agg
{

     
    template<class VertexSource, class Transformer=trans_affine> class conv_transform
    {
    public:
        conv_transform(VertexSource& source, Transformer& tr) :
            m_source(&source), m_trans(&tr) {}
        void attach(VertexSource& source) { m_source = &source; }

        void rewind(unsigned path_id) 
        { 
            m_source->rewind(path_id); 
        }

        unsigned vertex(double* x, double* y)
        {
            unsigned cmd = m_source->vertex(x, y);
            if(is_vertex(cmd))
            {
                m_trans->transform(x, y);
            }
            return cmd;
        }

        void transformer(Transformer& tr)
        {
            m_trans = &tr;
        }

    private:
        conv_transform(const conv_transform<VertexSource>&);
        const conv_transform<VertexSource>& 
            operator = (const conv_transform<VertexSource>&);

        VertexSource*      m_source;
        Transformer* m_trans;
    };


}

#endif

namespace agg
{

     
    void arc_to_bezier(double cx, double cy, double rx, double ry, 
                       double start_angle, double sweep_angle,
                       double* curve);


     
     
     
     
    class bezier_arc
    {
    public:
         
        bezier_arc() : m_vertex(26), m_num_vertices(0), m_cmd(path_cmd_line_to) {}
        bezier_arc(double x,  double y, 
                   double rx, double ry, 
                   double start_angle, 
                   double sweep_angle)
        {
            init(x, y, rx, ry, start_angle, sweep_angle);
        }

         
        void init(double x,  double y, 
                  double rx, double ry, 
                  double start_angle, 
                  double sweep_angle);

         
        void rewind(unsigned)
        {
            m_vertex = 0;
        }

         
        unsigned vertex(double* x, double* y)
        {
            if(m_vertex >= m_num_vertices) return path_cmd_stop;
            *x = m_vertices[m_vertex];
            *y = m_vertices[m_vertex + 1];
            m_vertex += 2;
            return (m_vertex == 2) ? unsigned(path_cmd_move_to) : m_cmd;
        }

         
         
         
        unsigned  num_vertices() const { return m_num_vertices; }
        const double* vertices() const { return m_vertices;     }
        double*       vertices()       { return m_vertices;     }
 
    private:
        unsigned m_vertex;
        unsigned m_num_vertices;
        double   m_vertices[26];
        unsigned m_cmd;
    };



     
     
     
     
     
     
     
     
     
     
     
    class bezier_arc_svg
    {
    public:
         
        bezier_arc_svg() : m_arc(), m_radii_ok(false) {}

        bezier_arc_svg(double x1, double y1, 
                       double rx, double ry, 
                       double angle,
                       bool large_arc_flag,
                       bool sweep_flag,
                       double x2, double y2) : 
            m_arc(), m_radii_ok(false)
        {
            init(x1, y1, rx, ry, angle, large_arc_flag, sweep_flag, x2, y2);
        }

         
        void init(double x1, double y1, 
                  double rx, double ry, 
                  double angle,
                  bool large_arc_flag,
                  bool sweep_flag,
                  double x2, double y2);

         
        bool radii_ok() const { return m_radii_ok; }

         
        void rewind(unsigned)
        {
            m_arc.rewind(0);
        }

         
        unsigned vertex(double* x, double* y)
        {
            return m_arc.vertex(x, y);
        }

         
         
         
        unsigned  num_vertices() const { return m_arc.num_vertices(); }
        const double* vertices() const { return m_arc.vertices();     }
        double*       vertices()       { return m_arc.vertices();     }

    private:
        bezier_arc m_arc;
        bool       m_radii_ok;
    };




}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_BITSET_ITERATOR_INCLUDED
#define AGG_BITSET_ITERATOR_INCLUDED


namespace agg
{
    
    class bitset_iterator
    {
    public:
        bitset_iterator(const int8u* bits, unsigned offset = 0) :
            m_bits(bits + (offset >> 3)),
            m_mask(0x80 >> (offset & 7))
        {}

        void operator ++ ()
        {
            m_mask >>= 1;
            if(m_mask == 0)
            {
                ++m_bits;
                m_mask = 0x80;
            }
        }

        unsigned bit() const
        {
            return (*m_bits) & m_mask;
        }

    private:
        const int8u* m_bits;
        int8u        m_mask;
    };

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_BOUNDING_RECT_INCLUDED
#define AGG_BOUNDING_RECT_INCLUDED


namespace agg
{

     
    template<class VertexSource, class GetId, class CoordT>
    bool bounding_rect(VertexSource& vs, GetId& gi, 
                       unsigned start, unsigned num, 
                       CoordT* x1, CoordT* y1, CoordT* x2, CoordT* y2)
    {
        unsigned i;
        double x;
        double y;
        bool first = true;

        *x1 = CoordT(1);
        *y1 = CoordT(1);
        *x2 = CoordT(0);
        *y2 = CoordT(0);

        for(i = 0; i < num; i++)
        {
            vs.rewind(gi[start + i]);
            unsigned cmd;
            while(!is_stop(cmd = vs.vertex(&x, &y)))
            {
                if(is_vertex(cmd))
                {
                    if(first)
                    {
                        *x1 = CoordT(x);
                        *y1 = CoordT(y);
                        *x2 = CoordT(x);
                        *y2 = CoordT(y);
                        first = false;
                    }
                    else
                    {
                        if(CoordT(x) < *x1) *x1 = CoordT(x);
                        if(CoordT(y) < *y1) *y1 = CoordT(y);
                        if(CoordT(x) > *x2) *x2 = CoordT(x);
                        if(CoordT(y) > *y2) *y2 = CoordT(y);
                    }
                }
            }
        }
        return *x1 <= *x2 && *y1 <= *y2;
    }


     
    template<class VertexSource, class CoordT> 
    bool bounding_rect_single(VertexSource& vs, unsigned path_id,
                              CoordT* x1, CoordT* y1, CoordT* x2, CoordT* y2)
    {
        double x;
        double y;
        bool first = true;

        *x1 = CoordT(1);
        *y1 = CoordT(1);
        *x2 = CoordT(0);
        *y2 = CoordT(0);

        vs.rewind(path_id);
        unsigned cmd;
        while(!is_stop(cmd = vs.vertex(&x, &y)))
        {
            if(is_vertex(cmd))
            {
                if(first)
                {
                    *x1 = CoordT(x);
                    *y1 = CoordT(y);
                    *x2 = CoordT(x);
                    *y2 = CoordT(y);
                    first = false;
                }
                else
                {
                    if(CoordT(x) < *x1) *x1 = CoordT(x);
                    if(CoordT(y) < *y1) *y1 = CoordT(y);
                    if(CoordT(x) > *x2) *x2 = CoordT(x);
                    if(CoordT(y) > *y2) *y2 = CoordT(y);
                }
            }
        }
        return *x1 <= *x2 && *y1 <= *y2;
    }


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_BSPLINE_INCLUDED
#define AGG_BSPLINE_INCLUDED


namespace agg
{
     
     
     
     
     
     
     
     
     
     
     
     
     
    class bspline 
    {
    public:
        bspline();
        bspline(int num);
        bspline(int num, const double* x, const double* y);

        void   init(int num);
        void   add_point(double x, double y);
        void   prepare();

        void   init(int num, const double* x, const double* y);

        double get(double x) const;
        double get_stateful(double x) const;
    
    private:
        bspline(const bspline&);
        const bspline& operator = (const bspline&);

        static void bsearch(int n, const double *x, double x0, int *i);
        double extrapolation_left(double x) const;
        double extrapolation_right(double x) const;
        double interpolation(double x, int i) const;

        int               m_max;
        int               m_num;
        double*           m_x;
        double*           m_y;
        pod_array<double> m_am;
        mutable int       m_last_idx;
    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CLIP_LIANG_BARSKY_INCLUDED
#define AGG_CLIP_LIANG_BARSKY_INCLUDED


namespace agg
{

     
    enum clipping_flags_e
    {
        clipping_flags_x1_clipped = 4,
        clipping_flags_x2_clipped = 1,
        clipping_flags_y1_clipped = 8,
        clipping_flags_y2_clipped = 2,
        clipping_flags_x_clipped = clipping_flags_x1_clipped | clipping_flags_x2_clipped,
        clipping_flags_y_clipped = clipping_flags_y1_clipped | clipping_flags_y2_clipped
    };

     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
    template<class T>
    inline unsigned clipping_flags(T x, T y, const rect_base<T>& clip_box)
    {
        return  (x > clip_box.x2) |
               ((y > clip_box.y2) << 1) |
               ((x < clip_box.x1) << 2) |
               ((y < clip_box.y1) << 3);
    }

     
    template<class T>
    inline unsigned clipping_flags_x(T x, const rect_base<T>& clip_box)
    {
        return  (x > clip_box.x2) | ((x < clip_box.x1) << 2);
    }


     
    template<class T>
    inline unsigned clipping_flags_y(T y, const rect_base<T>& clip_box)
    {
        return ((y > clip_box.y2) << 1) | ((y < clip_box.y1) << 3);
    }


     
    template<class T>
    inline unsigned clip_liang_barsky(T x1, T y1, T x2, T y2,
                                      const rect_base<T>& clip_box,
                                      T* x, T* y)
    {
        const double nearzero = 1e-30;

        double deltax = x2 - x1;
        double deltay = y2 - y1; 
        double xin;
        double xout;
        double yin;
        double yout;
        double tinx;
        double tiny;
        double toutx;
        double touty;  
        double tin1;
        double tin2;
        double tout1;
        unsigned np = 0;

        if(deltax == 0.0) 
        {   
             
            deltax = (x1 > clip_box.x1) ? -nearzero : nearzero;
        }

        if(deltay == 0.0) 
        { 
             
            deltay = (y1 > clip_box.y1) ? -nearzero : nearzero;
        }
        
        if(deltax > 0.0) 
        {                
             
            xin  = clip_box.x1;
            xout = clip_box.x2;
        }
        else 
        {
            xin  = clip_box.x2;
            xout = clip_box.x1;
        }

        if(deltay > 0.0) 
        {
             
            yin  = clip_box.y1;
            yout = clip_box.y2;
        }
        else 
        {
            yin  = clip_box.y2;
            yout = clip_box.y1;
        }
        
        tinx = (xin - x1) / deltax;
        tiny = (yin - y1) / deltay;
        
        if (tinx < tiny) 
        {
             
            tin1 = tinx;
            tin2 = tiny;
        }
        else
        {
             
            tin1 = tiny;
            tin2 = tinx;
        }
        
        if(tin1 <= 1.0) 
        {
            if(0.0 < tin1) 
            {
                *x++ = (T)xin;
                *y++ = (T)yin;
                ++np;
            }

            if(tin2 <= 1.0)
            {
                toutx = (xout - x1) / deltax;
                touty = (yout - y1) / deltay;
                
                tout1 = (toutx < touty) ? toutx : touty;
                
                if(tin2 > 0.0 || tout1 > 0.0) 
                {
                    if(tin2 <= tout1) 
                    {
                        if(tin2 > 0.0) 
                        {
                            if(tinx > tiny) 
                            {
                                *x++ = (T)xin;
                                *y++ = (T)(y1 + tinx * deltay);
                            }
                            else 
                            {
                                *x++ = (T)(x1 + tiny * deltax);
                                *y++ = (T)yin;
                            }
                            ++np;
                        }

                        if(tout1 < 1.0) 
                        {
                            if(toutx < touty) 
                            {
                                *x++ = (T)xout;
                                *y++ = (T)(y1 + toutx * deltay);
                            }
                            else 
                            {
                                *x++ = (T)(x1 + touty * deltax);
                                *y++ = (T)yout;
                            }
                        }
                        else 
                        {
                            *x++ = x2;
                            *y++ = y2;
                        }
                        ++np;
                    }
                    else 
                    {
                        if(tinx > tiny) 
                        {
                            *x++ = (T)xin;
                            *y++ = (T)yout;
                        }
                        else 
                        {
                            *x++ = (T)xout;
                            *y++ = (T)yin;
                        }
                        ++np;
                    }
                }
            }
        }
        return np;
    }


     
    template<class T>
    bool clip_move_point(T x1, T y1, T x2, T y2, 
                         const rect_base<T>& clip_box, 
                         T* x, T* y, unsigned flags)
    {
       T bound;

       if(flags & clipping_flags_x_clipped)
       {
           if(x1 == x2)
           {
               return false;
           }
           bound = (flags & clipping_flags_x1_clipped) ? clip_box.x1 : clip_box.x2;
           *y = (T)(double(bound - x1) * (y2 - y1) / (x2 - x1) + y1);
           *x = bound;
       }

       flags = clipping_flags_y(*y, clip_box);
       if(flags & clipping_flags_y_clipped)
       {
           if(y1 == y2)
           {
               return false;
           }
           bound = (flags & clipping_flags_y1_clipped) ? clip_box.y1 : clip_box.y2;
           *x = (T)(double(bound - y1) * (x2 - x1) / (y2 - y1) + x1);
           *y = bound;
       }
       return true;
    }

     
     
     
     
     
    template<class T>
    unsigned clip_line_segment(T* x1, T* y1, T* x2, T* y2,
                               const rect_base<T>& clip_box)
    {
        unsigned f1 = clipping_flags(*x1, *y1, clip_box);
        unsigned f2 = clipping_flags(*x2, *y2, clip_box);
        unsigned ret = 0;

        if((f2 | f1) == 0)
        {
             
            return 0;
        }

        if((f1 & clipping_flags_x_clipped) != 0 && 
           (f1 & clipping_flags_x_clipped) == (f2 & clipping_flags_x_clipped))
        {
             
            return 4;
        }

        if((f1 & clipping_flags_y_clipped) != 0 && 
           (f1 & clipping_flags_y_clipped) == (f2 & clipping_flags_y_clipped))
        {
             
            return 4;
        }

        T tx1 = *x1;
        T ty1 = *y1;
        T tx2 = *x2;
        T ty2 = *y2;
        if(f1) 
        {   
            if(!clip_move_point(tx1, ty1, tx2, ty2, clip_box, x1, y1, f1)) 
            {
                return 4;
            }
            if(*x1 == *x2 && *y1 == *y2) 
            {
                return 4;
            }
            ret |= 1;
        }
        if(f2) 
        {
            if(!clip_move_point(tx1, ty1, tx2, ty2, clip_box, x2, y2, f2))
            {
                return 4;
            }
            if(*x1 == *x2 && *y1 == *y2) 
            {
                return 4;
            }
            ret |= 2;
        }
        return ret;
    }


}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_COLOR_GRAY_INCLUDED
#define AGG_COLOR_GRAY_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_COLOR_RGBA_INCLUDED
#define AGG_COLOR_RGBA_INCLUDED

#include <cmath>
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_GAMMA_LUT_INCLUDED
#define AGG_GAMMA_LUT_INCLUDED

#include <cmath>
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_GAMMA_FUNCTIONS_INCLUDED
#define AGG_GAMMA_FUNCTIONS_INCLUDED


namespace agg
{
     
    struct gamma_none
    {
        double operator()(double x) const { return x; }
    };


     
    class gamma_power
    {
    public:
        gamma_power() : m_gamma(1.0) {}
        gamma_power(double g) : m_gamma(g) {}

        void gamma(double g) { m_gamma = g; }
        double gamma() const { return m_gamma; }

        double operator() (double x) const
        {
            return pow(x, m_gamma);
        }

    private:
        double m_gamma;
    };


     
    class gamma_threshold
    {
    public:
        gamma_threshold() : m_threshold(0.5) {}
        gamma_threshold(double t) : m_threshold(t) {}

        void threshold(double t) { m_threshold = t; }
        double threshold() const { return m_threshold; }

        double operator() (double x) const
        {
            return (x < m_threshold) ? 0.0 : 1.0;
        }

    private:
        double m_threshold;
    };


     
    class gamma_linear
    {
    public:
        gamma_linear() : m_start(0.0), m_end(1.0) {}
        gamma_linear(double s, double e) : m_start(s), m_end(e) {}

        void set(double s, double e) { m_start = s; m_end = e; }
        void start(double s) { m_start = s; }
        void end(double e) { m_end = e; }
        double start() const { return m_start; }
        double end() const { return m_end; }

        double operator() (double x) const
        {
            if(x < m_start) return 0.0;
            if(x > m_end) return 1.0;
            return (x - m_start) / (m_end - m_start);
        }

    private:
        double m_start;
        double m_end;
    };


     
    class gamma_multiply
    {
    public:
        gamma_multiply() : m_mul(1.0) {}
        gamma_multiply(double v) : m_mul(v) {}

        void value(double v) { m_mul = v; }
        double value() const { return m_mul; }

        double operator() (double x) const
        {
            double y = x * m_mul;
            if(y > 1.0) y = 1.0;
            return y;
        }

    private:
        double m_mul;
    };

    inline double sRGB_to_linear(double x)
    {
        return (x <= 0.04045) ? (x / 12.92) : pow((x + 0.055) / (1.055), 2.4);
    }

    inline double linear_to_sRGB(double x)
    {
        return (x <= 0.0031308) ? (x * 12.92) : (1.055 * pow(x, 1 / 2.4) - 0.055);
    }
}

#endif




namespace agg
{
    template<class LoResT=int8u, 
             class HiResT=int8u, 
             unsigned GammaShift=8, 
             unsigned HiResShift=8> class gamma_lut
    {
    public:
        typedef gamma_lut<LoResT, HiResT, GammaShift, HiResShift> self_type;

        enum gamma_scale_e
        {
            gamma_shift = GammaShift,
            gamma_size  = 1 << gamma_shift,
            gamma_mask  = gamma_size - 1
        };

        enum hi_res_scale_e
        {
            hi_res_shift = HiResShift,
            hi_res_size  = 1 << hi_res_shift,
            hi_res_mask  = hi_res_size - 1
        };

        ~gamma_lut()
        {
            pod_allocator<LoResT>::deallocate(m_inv_gamma, hi_res_size);
            pod_allocator<HiResT>::deallocate(m_dir_gamma, gamma_size);
        }

        gamma_lut() : 
            m_gamma(1.0), 
            m_dir_gamma(pod_allocator<HiResT>::allocate(gamma_size)),
            m_inv_gamma(pod_allocator<LoResT>::allocate(hi_res_size))
        {
            unsigned i;
            for(i = 0; i < gamma_size; i++)
            {
                m_dir_gamma[i] = HiResT(i << (+hi_res_shift - gamma_shift));
            }

            for(i = 0; i < hi_res_size; i++)
            {
                m_inv_gamma[i] = LoResT(i >> (+hi_res_shift - gamma_shift));
            }
        }

        gamma_lut(double g) :
            m_gamma(1.0), 
            m_dir_gamma(pod_allocator<HiResT>::allocate(gamma_size)),
            m_inv_gamma(pod_allocator<LoResT>::allocate(hi_res_size))
        {
            gamma(g);
        }

        void gamma(double g) 
        {
            m_gamma = g;

            unsigned i;
            for(i = 0; i < gamma_size; i++)
            {
                m_dir_gamma[i] = (HiResT)
                    uround(std::pow(i / double(gamma_mask), m_gamma) * double(hi_res_mask));
            }

            double inv_g = 1.0 / g;
            for(i = 0; i < hi_res_size; i++)
            {
                m_inv_gamma[i] = (LoResT)
                    uround(std::pow(i / double(hi_res_mask), inv_g) * double(gamma_mask));
            }
        }

        double gamma() const
        {
            return m_gamma;
        }

        HiResT dir(LoResT v) const 
        { 
            return m_dir_gamma[unsigned(v)]; 
        }

        LoResT inv(HiResT v) const 
        { 
            return m_inv_gamma[unsigned(v)];
        }

    private:
        gamma_lut(const self_type&);
        const self_type& operator = (const self_type&);

        double m_gamma;
        HiResT* m_dir_gamma;
        LoResT* m_inv_gamma;
    };

     
     
     

     
     
     
    template<class LinearType>
    class sRGB_lut_base
    {
    public:
        LinearType dir(int8u v) const
        {
            return m_dir_table[v];
        }

        int8u inv(LinearType v) const
        {
             
            int8u x = 0;
            if (v > m_inv_table[128]) x = 128;
            if (v > m_inv_table[x + 64]) x += 64;
            if (v > m_inv_table[x + 32]) x += 32;
            if (v > m_inv_table[x + 16]) x += 16;
            if (v > m_inv_table[x + 8]) x += 8;
            if (v > m_inv_table[x + 4]) x += 4;
            if (v > m_inv_table[x + 2]) x += 2;
            if (v > m_inv_table[x + 1]) x += 1;
            return x;
        }

    protected:
        LinearType m_dir_table[256];
        LinearType m_inv_table[256];

         
        sRGB_lut_base() 
        {
        }
    };


     
     
    template<class LinearType>
    class sRGB_lut;

    template<>
    class sRGB_lut<float> : public sRGB_lut_base<float>
    {
    public:
		sRGB_lut()
		{
			 
			m_dir_table[0] = 0;
			m_inv_table[0] = 0;
			for (unsigned i = 1; i <= 255; ++i)
			{
				 
				m_dir_table[i] = float(sRGB_to_linear(i / 255.0));
				m_inv_table[i] = float(sRGB_to_linear((i - 0.5) / 255.0));
			}
		}
    };

    template<>
    class sRGB_lut<int16u> : public sRGB_lut_base<int16u>
    {
    public:
        sRGB_lut()
        {
             
            m_dir_table[0] = 0;
            m_inv_table[0] = 0;
            for (int i = 1; i <= 255; ++i)
            {
                 
                m_dir_table[i] = uround(65535.0 * sRGB_to_linear(i / 255.0));
                m_inv_table[i] = uround(65535.0 * sRGB_to_linear((i - 0.5) / 255.0));
            }
        }
    };

    template<>
    class sRGB_lut<int8u> : public sRGB_lut_base<int8u>
    {
    public:
        sRGB_lut()
        {
             
            m_dir_table[0] = 0;
            m_inv_table[0] = 0;
            for (int i = 1; i <= 255; ++i)
            {
                 
                m_dir_table[i] = uround(255.0 * sRGB_to_linear(i / 255.0));
                m_inv_table[i] = uround(255.0 * linear_to_sRGB(i / 255.0));
            }
        }

        int8u inv(int8u v) const
        {
             
            return m_inv_table[v];
        }
    };

     
     
    template<class T>
    class sRGB_conv_base
    {
    public:
        static T rgb_from_sRGB(int8u x)
        {
            return lut.dir(x);
        }

        static int8u rgb_to_sRGB(T x)
        {
            return lut.inv(x);
        }

    private:
        static sRGB_lut<T> lut;
    };

     
     
    template<class T>
    sRGB_lut<T> sRGB_conv_base<T>::lut;

     
     
    template<class T>
    class sRGB_conv;

    template<>
    class sRGB_conv<float> : public sRGB_conv_base<float>
    {
    public:
        static float alpha_from_sRGB(int8u x)
        {
            static const double y = 1 / 255.0;
            return float(x * y);
        }

        static int8u alpha_to_sRGB(float x)
        {
            if (x < 0) return 0;
            if (x > 1) return 255;
            return int8u(0.5 + x * 255);
        }
    };

    template<>
    class sRGB_conv<int16u> : public sRGB_conv_base<int16u>
    {
    public:
        static int16u alpha_from_sRGB(int8u x)
        {
            return (x << 8) | x;
        }

        static int8u alpha_to_sRGB(int16u x)
        {
            return x >> 8;
        }
    };

    template<>
    class sRGB_conv<int8u> : public sRGB_conv_base<int8u>
    {
    public:
        static int8u alpha_from_sRGB(int8u x)
        {
            return x;
        }

        static int8u alpha_to_sRGB(int8u x)
        {
            return x;
        }
    };
}

#endif

namespace agg
{
     
     
    struct order_rgb  { enum rgb_e  { R=0, G=1, B=2, N=3 }; };
    struct order_bgr  { enum bgr_e  { B=0, G=1, R=2, N=3 }; };
    struct order_rgba { enum rgba_e { R=0, G=1, B=2, A=3, N=4 }; };
    struct order_argb { enum argb_e { A=0, R=1, G=2, B=3, N=4 }; };
    struct order_abgr { enum abgr_e { A=0, B=1, G=2, R=3, N=4 }; };
    struct order_bgra { enum bgra_e { B=0, G=1, R=2, A=3, N=4 }; };

     
    struct linear {};
    struct sRGB {};

     
    struct rgba
    {
        typedef double value_type;

        double r;
        double g;
        double b;
        double a;

         
        rgba() {}

         
        rgba(double r_, double g_, double b_, double a_=1.0) :
            r(r_), g(g_), b(b_), a(a_) {}

         
        rgba(const rgba& c, double a_) : r(c.r), g(c.g), b(c.b), a(a_) {}

         
        rgba& clear()
        {
            r = g = b = a = 0;
			return *this;
        }

         
        rgba& transparent()
        {
            a = 0;
            return *this;
        }

         
        rgba& opacity(double a_)
        {
            if (a_ < 0) a = 0;
            else if (a_ > 1) a = 1;
            else a = a_;
            return *this;
        }

         
        double opacity() const
        {
            return a;
        }

         
        rgba& premultiply()
        {
            r *= a;
            g *= a;
            b *= a;
            return *this;
        }

         
        rgba& premultiply(double a_)
        {
            if (a <= 0 || a_ <= 0)
            {
                r = g = b = a = 0;
            }
            else
            {
                a_ /= a;
                r *= a_;
                g *= a_;
                b *= a_;
                a  = a_;
            }
            return *this;
        }

         
        rgba& demultiply()
        {
            if (a == 0)
            {
                r = g = b = 0;
            }
            else
            {
                double a_ = 1.0 / a;
                r *= a_;
                g *= a_;
                b *= a_;
            }
            return *this;
        }


         
        rgba gradient(rgba c, double k) const
        {
            rgba ret;
            ret.r = r + (c.r - r) * k;
            ret.g = g + (c.g - g) * k;
            ret.b = b + (c.b - b) * k;
            ret.a = a + (c.a - a) * k;
            return ret;
        }

        rgba& operator+=(const rgba& c)
        {
            r += c.r;
            g += c.g;
            b += c.b;
            a += c.a;
            return *this;
        }

        rgba& operator*=(double k)
        {
            r *= k;
            g *= k;
            b *= k;
            a *= k;
            return *this;
        }

         
        static rgba no_color() { return rgba(0,0,0,0); }

         
        static rgba from_wavelength(double wl, double gamma = 1.0);
    
         
        explicit rgba(double wavelen, double gamma=1.0)
        {
            *this = from_wavelength(wavelen, gamma);
        }

    };

    inline rgba operator+(const rgba& a, const rgba& b)
    {
        return rgba(a) += b;
    }

    inline rgba operator*(const rgba& a, double b)
    {
        return rgba(a) *= b;
    }

     
    inline rgba rgba::from_wavelength(double wl, double gamma)
    {
        rgba t(0.0, 0.0, 0.0);

        if (wl >= 380.0 && wl <= 440.0)
        {
            t.r = -1.0 * (wl - 440.0) / (440.0 - 380.0);
            t.b = 1.0;
        }
        else if (wl >= 440.0 && wl <= 490.0)
        {
            t.g = (wl - 440.0) / (490.0 - 440.0);
            t.b = 1.0;
        }
        else if (wl >= 490.0 && wl <= 510.0)
        {
            t.g = 1.0;
            t.b = -1.0 * (wl - 510.0) / (510.0 - 490.0);
        }
        else if (wl >= 510.0 && wl <= 580.0)
        {
            t.r = (wl - 510.0) / (580.0 - 510.0);
            t.g = 1.0;
        }
        else if (wl >= 580.0 && wl <= 645.0)
        {
            t.r = 1.0;
            t.g = -1.0 * (wl - 645.0) / (645.0 - 580.0);
        }
        else if (wl >= 645.0 && wl <= 780.0)
        {
            t.r = 1.0;
        }

        double s = 1.0;
        if (wl > 700.0)       s = 0.3 + 0.7 * (780.0 - wl) / (780.0 - 700.0);
        else if (wl <  420.0) s = 0.3 + 0.7 * (wl - 380.0) / (420.0 - 380.0);

        t.r = std::pow(t.r * s, gamma);
        t.g = std::pow(t.g * s, gamma);
        t.b = std::pow(t.b * s, gamma);
        return t;
    }

    inline rgba rgba_pre(double r, double g, double b, double a)
    {
        return rgba(r, g, b, a).premultiply();
    }

    
     
    template<class Colorspace>
    struct rgba8T
    {
        typedef int8u  value_type;
        typedef int32u calc_type;
        typedef int32  long_type;
        enum base_scale_e
        {
            base_shift = 8,
            base_scale = 1 << base_shift,
            base_mask  = base_scale - 1,
            base_MSB = 1 << (base_shift - 1)
        };
        typedef rgba8T self_type;


        value_type r;
        value_type g;
        value_type b;
        value_type a;

        static void convert(rgba8T<linear>& dst, const rgba8T<sRGB>& src)
        {
            dst.r = sRGB_conv<value_type>::rgb_from_sRGB(src.r);
            dst.g = sRGB_conv<value_type>::rgb_from_sRGB(src.g);
            dst.b = sRGB_conv<value_type>::rgb_from_sRGB(src.b);
            dst.a = src.a;
        }

        static void convert(rgba8T<sRGB>& dst, const rgba8T<linear>& src)
        {
            dst.r = sRGB_conv<value_type>::rgb_to_sRGB(src.r);
            dst.g = sRGB_conv<value_type>::rgb_to_sRGB(src.g);
            dst.b = sRGB_conv<value_type>::rgb_to_sRGB(src.b);
            dst.a = src.a;
        }

        static void convert(rgba8T<linear>& dst, const rgba& src)
        {
            dst.r = value_type(uround(src.r * (rgba::value_type)base_mask));
            dst.g = value_type(uround(src.g * (rgba::value_type)base_mask));
            dst.b = value_type(uround(src.b * (rgba::value_type)base_mask));
            dst.a = value_type(uround(src.a * (rgba::value_type)base_mask));
        }

        static void convert(rgba8T<sRGB>& dst, const rgba& src)
        {
             
            dst.r = sRGB_conv<float>::rgb_to_sRGB(float(src.r));
            dst.g = sRGB_conv<float>::rgb_to_sRGB(float(src.g));
            dst.b = sRGB_conv<float>::rgb_to_sRGB(float(src.b));
            dst.a = sRGB_conv<float>::alpha_to_sRGB(float(src.a));
        }

        static void convert(rgba& dst, const rgba8T<linear>& src)
        {
            dst.r = src.r / 255.0;
            dst.g = src.g / 255.0;
            dst.b = src.b / 255.0;
            dst.a = src.a / 255.0;
        }

        static void convert(rgba& dst, const rgba8T<sRGB>& src)
        {
             
            dst.r = sRGB_conv<float>::rgb_from_sRGB(src.r);
            dst.g = sRGB_conv<float>::rgb_from_sRGB(src.g);
            dst.b = sRGB_conv<float>::rgb_from_sRGB(src.b);
            dst.a = sRGB_conv<float>::alpha_from_sRGB(src.a);
        }

         
        rgba8T() {}

         
        rgba8T(unsigned r_, unsigned g_, unsigned b_, unsigned a_ = base_mask) :
            r(value_type(r_)), 
            g(value_type(g_)), 
            b(value_type(b_)), 
            a(value_type(a_)) {}

         
        rgba8T(const rgba& c)
        {
            convert(*this, c);
        }

         
        rgba8T(const self_type& c, unsigned a_) :
            r(c.r), g(c.g), b(c.b), a(value_type(a_)) {}

         
        template<class T>
        rgba8T(const rgba8T<T>& c)
        {
            convert(*this, c);
        }

         
        operator rgba() const 
        {
            rgba c;
            convert(c, *this);
            return c;
        }

         
        static AGG_INLINE double to_double(value_type a)
        {
            return double(a) / (double)base_mask;
        }

         
        static AGG_INLINE value_type from_double(double a)
        {
            return value_type(uround(a * (double)base_mask));
        }

         
        static AGG_INLINE value_type empty_value()
        {
            return 0;
        }

         
        static AGG_INLINE value_type full_value()
        {
            return base_mask;
        }

         
        AGG_INLINE bool is_transparent() const
        {
            return a == 0;
        }

         
        AGG_INLINE bool is_opaque() const
        {
            return a == base_mask;
        }

         
        static AGG_INLINE value_type invert(value_type x) 
        {
            return base_mask - x;
        }

         
         
        static AGG_INLINE value_type multiply(value_type a, value_type b) 
        {
            calc_type t = a * b + base_MSB;
            return value_type(((t >> base_shift) + t) >> base_shift);
        }
        
         
        static AGG_INLINE value_type demultiply(value_type a, value_type b) 
        {
            if (a * b == 0)
            {
                return 0;
            }
            else if (a >= b)
            {
                return base_mask;
            }
            else return value_type((a * base_mask + (b >> 1)) / b); 
        }

         
        template<typename T>
        static AGG_INLINE T downscale(T a) 
        {
            return a >> base_shift;
        }

         
        template<typename T>
        static AGG_INLINE T downshift(T a, unsigned n) 
        {
            return a >> n;
        }

         
         
         
        static AGG_INLINE value_type mult_cover(value_type a, cover_type b) 
        {
            return multiply(a, b);
        }
        
         
        static AGG_INLINE cover_type scale_cover(cover_type a, value_type b) 
        {
            return multiply(b, a);
        }
        
         
         
        static AGG_INLINE value_type prelerp(value_type p, value_type q, value_type a) 
        {
            return p + q - multiply(p, a);
        }
        
         
         
        static AGG_INLINE value_type lerp(value_type p, value_type q, value_type a) 
        {
            int t = (q - p) * a + base_MSB - (p > q);
            return value_type(p + (((t >> base_shift) + t) >> base_shift));
        }
        
         
        self_type& clear()
        {
            r = g = b = a = 0;
			return *this;
        }
        
         
        self_type& transparent()
        {
            a = 0;
            return *this;
        }

         
        self_type& opacity(double a_)
        {
            if (a_ < 0) a = 0;
            else if (a_ > 1) a = 1;
            else a = (value_type)uround(a_ * double(base_mask));
            return *this;
        }

         
        double opacity() const
        {
            return double(a) / double(base_mask);
        }

         
        AGG_INLINE self_type& premultiply()
        {
            if (a != base_mask)
            {
                if (a == 0)
                {
                    r = g = b = 0;
                }
                else
                {
                    r = multiply(r, a);
                    g = multiply(g, a);
                    b = multiply(b, a);
                }
            }
            return *this;
        }

         
        AGG_INLINE self_type& premultiply(unsigned a_)
        {
            if (a != base_mask || a_ < base_mask)
            {
                if (a == 0 || a_ == 0)
                {
                    r = g = b = a = 0;
                }
                else
                {
                    calc_type r_ = (calc_type(r) * a_) / a;
                    calc_type g_ = (calc_type(g) * a_) / a;
                    calc_type b_ = (calc_type(b) * a_) / a;
                    r = value_type((r_ > a_) ? a_ : r_);
                    g = value_type((g_ > a_) ? a_ : g_);
                    b = value_type((b_ > a_) ? a_ : b_);
                    a = value_type(a_);
                }
            }
            return *this;
        }

         
        AGG_INLINE self_type& demultiply()
        {
            if (a < base_mask)
            {
                if (a == 0)
                {
                    r = g = b = 0;
                }
                else
                {
                    calc_type r_ = (calc_type(r) * base_mask) / a;
                    calc_type g_ = (calc_type(g) * base_mask) / a;
                    calc_type b_ = (calc_type(b) * base_mask) / a;
                    r = value_type((r_ > calc_type(base_mask)) ? calc_type(base_mask) : r_);
                    g = value_type((g_ > calc_type(base_mask)) ? calc_type(base_mask) : g_);
                    b = value_type((b_ > calc_type(base_mask)) ? calc_type(base_mask) : b_);
                }
            }
            return *this;
        }

         
        AGG_INLINE self_type gradient(const self_type& c, double k) const
        {
            self_type ret;
            calc_type ik = uround(k * (double)base_mask);
            ret.r = lerp(r, c.r, ik);
            ret.g = lerp(g, c.g, ik);
            ret.b = lerp(b, c.b, ik);
            ret.a = lerp(a, c.a, ik);
            return ret;
        }

         
        AGG_INLINE void add(const self_type& c, unsigned cover)
        {
            calc_type cr, cg, cb, ca;
            if (cover == cover_mask)
            {
                if (c.a == base_mask) 
                {
                    *this = c;
                    return;
                }
                else
                {
                    cr = r + c.r; 
                    cg = g + c.g; 
                    cb = b + c.b; 
                    ca = a + c.a; 
                }
            }
            else
            {
                cr = r + mult_cover(c.r, cover);
                cg = g + mult_cover(c.g, cover);
                cb = b + mult_cover(c.b, cover);
                ca = a + mult_cover(c.a, cover);
            }
            r = (value_type)((cr > calc_type(base_mask)) ? calc_type(base_mask) : cr);
            g = (value_type)((cg > calc_type(base_mask)) ? calc_type(base_mask) : cg);
            b = (value_type)((cb > calc_type(base_mask)) ? calc_type(base_mask) : cb);
            a = (value_type)((ca > calc_type(base_mask)) ? calc_type(base_mask) : ca);
        }

         
        template<class GammaLUT>
        AGG_INLINE void apply_gamma_dir(const GammaLUT& gamma)
        {
            r = gamma.dir(r);
            g = gamma.dir(g);
            b = gamma.dir(b);
        }

         
        template<class GammaLUT>
        AGG_INLINE void apply_gamma_inv(const GammaLUT& gamma)
        {
            r = gamma.inv(r);
            g = gamma.inv(g);
            b = gamma.inv(b);
        }

         
        static self_type no_color() { return self_type(0,0,0,0); }

         
        static self_type from_wavelength(double wl, double gamma = 1.0)
        {
            return self_type(rgba::from_wavelength(wl, gamma));
        }
    };

    typedef rgba8T<linear> rgba8;
    typedef rgba8T<sRGB> srgba8;


     
    inline rgba8 rgb8_packed(unsigned v)
    {
        return rgba8((v >> 16) & 0xFF, (v >> 8) & 0xFF, v & 0xFF);
    }

     
    inline rgba8 bgr8_packed(unsigned v)
    {
        return rgba8(v & 0xFF, (v >> 8) & 0xFF, (v >> 16) & 0xFF);
    }

     
    inline rgba8 argb8_packed(unsigned v)
    {
        return rgba8((v >> 16) & 0xFF, (v >> 8) & 0xFF, v & 0xFF, v >> 24);
    }

     
    template<class GammaLUT>
    rgba8 rgba8_gamma_dir(rgba8 c, const GammaLUT& gamma)
    {
        return rgba8(gamma.dir(c.r), gamma.dir(c.g), gamma.dir(c.b), c.a);
    }

     
    template<class GammaLUT>
    rgba8 rgba8_gamma_inv(rgba8 c, const GammaLUT& gamma)
    {
        return rgba8(gamma.inv(c.r), gamma.inv(c.g), gamma.inv(c.b), c.a);
    }



     
    struct rgba16
    {
        typedef int16u value_type;
        typedef int32u calc_type;
        typedef int64  long_type;
        enum base_scale_e
        {
            base_shift = 16,
            base_scale = 1 << base_shift,
            base_mask  = base_scale - 1,
            base_MSB = 1 << (base_shift - 1)
        };
        typedef rgba16 self_type;

        value_type r;
        value_type g;
        value_type b;
        value_type a;

         
        rgba16() {}

         
        rgba16(unsigned r_, unsigned g_, unsigned b_, unsigned a_=base_mask) :
            r(value_type(r_)), 
            g(value_type(g_)), 
            b(value_type(b_)), 
            a(value_type(a_)) {}

         
        rgba16(const self_type& c, unsigned a_) :
            r(c.r), g(c.g), b(c.b), a(value_type(a_)) {}

         
        rgba16(const rgba& c) :
            r((value_type)uround(c.r * double(base_mask))), 
            g((value_type)uround(c.g * double(base_mask))), 
            b((value_type)uround(c.b * double(base_mask))), 
            a((value_type)uround(c.a * double(base_mask))) {}

         
        rgba16(const rgba8& c) :
            r(value_type((value_type(c.r) << 8) | c.r)), 
            g(value_type((value_type(c.g) << 8) | c.g)), 
            b(value_type((value_type(c.b) << 8) | c.b)), 
            a(value_type((value_type(c.a) << 8) | c.a)) {}

         
        rgba16(const srgba8& c) :
            r(sRGB_conv<value_type>::rgb_from_sRGB(c.r)), 
            g(sRGB_conv<value_type>::rgb_from_sRGB(c.g)), 
            b(sRGB_conv<value_type>::rgb_from_sRGB(c.b)), 
            a(sRGB_conv<value_type>::alpha_from_sRGB(c.a)) {}

         
        operator rgba() const 
        {
            return rgba(
                r / 65535.0, 
                g / 65535.0, 
                b / 65535.0, 
                a / 65535.0);
        }

         
        operator rgba8() const 
        {
            return rgba8(r >> 8, g >> 8, b >> 8, a >> 8);
        }

         
        operator srgba8() const 
        {
             
            return srgba8(
                sRGB_conv<value_type>::rgb_to_sRGB(r), 
                sRGB_conv<value_type>::rgb_to_sRGB(g), 
                sRGB_conv<value_type>::rgb_to_sRGB(b), 
                sRGB_conv<value_type>::alpha_to_sRGB(a));
        }

         
        static AGG_INLINE double to_double(value_type a)
        {
            return double(a) / (double)base_mask;
        }

         
        static AGG_INLINE value_type from_double(double a)
        {
            return value_type(uround(a * (double)base_mask));
        }

         
        static AGG_INLINE value_type empty_value()
        {
            return 0;
        }

         
        static AGG_INLINE value_type full_value()
        {
            return base_mask;
        }

         
        AGG_INLINE bool is_transparent() const
        {
            return a == 0;
        }

         
        AGG_INLINE bool is_opaque() const
        {
            return a == base_mask;
        }

         
        static AGG_INLINE value_type invert(value_type x) 
        {
            return base_mask - x;
        }

         
         
        static AGG_INLINE value_type multiply(value_type a, value_type b) 
        {
            calc_type t = a * b + base_MSB;
            return value_type(((t >> base_shift) + t) >> base_shift);
        }
        
         
        static AGG_INLINE value_type demultiply(value_type a, value_type b) 
        {
            if (a * b == 0)
            {
                return 0;
            }
            else if (a >= b)
            {
                return base_mask;
            }
            else return value_type((a * base_mask + (b >> 1)) / b); 
        }

         
        template<typename T>
        static AGG_INLINE T downscale(T a) 
        {
            return a >> base_shift;
        }

         
        template<typename T>
        static AGG_INLINE T downshift(T a, unsigned n) 
        {
            return a >> n;
        }

         
         
         
        static AGG_INLINE value_type mult_cover(value_type a, cover_type b) 
        {
            return multiply(a, (b << 8) | b);
        }
        
         
        static AGG_INLINE cover_type scale_cover(cover_type a, value_type b) 
        {
            return multiply((a << 8) | a, b) >> 8;
        }
        
         
         
        static AGG_INLINE value_type prelerp(value_type p, value_type q, value_type a) 
        {
            return p + q - multiply(p, a);
        }
        
         
         
        static AGG_INLINE value_type lerp(value_type p, value_type q, value_type a) 
        {
            int t = (q - p) * a + base_MSB - (p > q);
            return value_type(p + (((t >> base_shift) + t) >> base_shift));
        }
        
         
        self_type& clear()
        {
            r = g = b = a = 0;
			return *this;
        }
        
         
        self_type& transparent()
        {
            a = 0;
            return *this;
        }

         
        AGG_INLINE self_type& opacity(double a_)
        {
            if (a_ < 0) a = 0;
            if (a_ > 1) a = 1;
            a = value_type(uround(a_ * double(base_mask)));
            return *this;
        }

         
        double opacity() const
        {
            return double(a) / double(base_mask);
        }

         
        AGG_INLINE self_type& premultiply()
        {
            if (a != base_mask) 
            {
                if (a == 0)
                {
                    r = g = b = 0;
                }
                else
                {
                    r = multiply(r, a);
                    g = multiply(g, a);
                    b = multiply(b, a);
                }
            }
            return *this;
        }

         
        AGG_INLINE self_type& premultiply(unsigned a_)
        {
            if (a < base_mask || a_ < base_mask)
            {
                if (a == 0 || a_ == 0)
                {
                    r = g = b = a = 0;
                }
                else
                {
                    calc_type r_ = (calc_type(r) * a_) / a;
                    calc_type g_ = (calc_type(g) * a_) / a;
                    calc_type b_ = (calc_type(b) * a_) / a;
                    r = value_type((r_ > a_) ? a_ : r_);
                    g = value_type((g_ > a_) ? a_ : g_);
                    b = value_type((b_ > a_) ? a_ : b_);
                    a = value_type(a_);
                }
            }
            return *this;
        }

         
        AGG_INLINE self_type& demultiply()
        {
            if (a < base_mask)
            {
                if (a == 0)
                {
                    r = g = b = 0;
                }
                else
                {
                    calc_type r_ = (calc_type(r) * base_mask) / a;
                    calc_type g_ = (calc_type(g) * base_mask) / a;
                    calc_type b_ = (calc_type(b) * base_mask) / a;
                    r = value_type((r_ > calc_type(base_mask)) ? calc_type(base_mask) : r_);
                    g = value_type((g_ > calc_type(base_mask)) ? calc_type(base_mask) : g_);
                    b = value_type((b_ > calc_type(base_mask)) ? calc_type(base_mask) : b_);
                }
            }
            return *this;
        }

         
        AGG_INLINE self_type gradient(const self_type& c, double k) const
        {
            self_type ret;
            calc_type ik = uround(k * (double)base_mask);
            ret.r = lerp(r, c.r, ik);
            ret.g = lerp(g, c.g, ik);
            ret.b = lerp(b, c.b, ik);
            ret.a = lerp(a, c.a, ik);
            return ret;
        }

         
        AGG_INLINE void add(const self_type& c, unsigned cover)
        {
            calc_type cr, cg, cb, ca;
            if (cover == cover_mask)
            {
                if (c.a == base_mask) 
                {
                    *this = c;
                    return;
                }
                else
                {
                    cr = r + c.r; 
                    cg = g + c.g; 
                    cb = b + c.b; 
                    ca = a + c.a; 
                }
            }
            else
            {
                cr = r + mult_cover(c.r, cover);
                cg = g + mult_cover(c.g, cover);
                cb = b + mult_cover(c.b, cover);
                ca = a + mult_cover(c.a, cover);
            }
            r = (value_type)((cr > calc_type(base_mask)) ? calc_type(base_mask) : cr);
            g = (value_type)((cg > calc_type(base_mask)) ? calc_type(base_mask) : cg);
            b = (value_type)((cb > calc_type(base_mask)) ? calc_type(base_mask) : cb);
            a = (value_type)((ca > calc_type(base_mask)) ? calc_type(base_mask) : ca);
        }

         
        template<class GammaLUT>
        AGG_INLINE void apply_gamma_dir(const GammaLUT& gamma)
        {
            r = gamma.dir(r);
            g = gamma.dir(g);
            b = gamma.dir(b);
        }

         
        template<class GammaLUT>
        AGG_INLINE void apply_gamma_inv(const GammaLUT& gamma)
        {
            r = gamma.inv(r);
            g = gamma.inv(g);
            b = gamma.inv(b);
        }

         
        static self_type no_color() { return self_type(0,0,0,0); }

         
        static self_type from_wavelength(double wl, double gamma = 1.0)
        {
            return self_type(rgba::from_wavelength(wl, gamma));
        }
    };


     
    template<class GammaLUT>
    rgba16 rgba16_gamma_dir(rgba16 c, const GammaLUT& gamma)
    {
        return rgba16(gamma.dir(c.r), gamma.dir(c.g), gamma.dir(c.b), c.a);
    }

     
    template<class GammaLUT>
    rgba16 rgba16_gamma_inv(rgba16 c, const GammaLUT& gamma)
    {
        return rgba16(gamma.inv(c.r), gamma.inv(c.g), gamma.inv(c.b), c.a);
    }

     
    struct rgba32
    {
        typedef float value_type;
        typedef double calc_type;
        typedef double long_type;
        typedef rgba32 self_type;

        value_type r;
        value_type g;
        value_type b;
        value_type a;

         
        rgba32() {}

         
        rgba32(value_type r_, value_type g_, value_type b_, value_type a_= 1) :
            r(r_), g(g_), b(b_), a(a_) {}

         
        rgba32(const self_type& c, float a_) :
            r(c.r), g(c.g), b(c.b), a(a_) {}

         
        rgba32(const rgba& c) :
            r(value_type(c.r)), g(value_type(c.g)), b(value_type(c.b)), a(value_type(c.a)) {}

         
        rgba32(const rgba8& c) :
            r(value_type(c.r / 255.0)), 
            g(value_type(c.g / 255.0)), 
            b(value_type(c.b / 255.0)), 
            a(value_type(c.a / 255.0)) {}

         
        rgba32(const srgba8& c) :
            r(sRGB_conv<value_type>::rgb_from_sRGB(c.r)), 
            g(sRGB_conv<value_type>::rgb_from_sRGB(c.g)), 
            b(sRGB_conv<value_type>::rgb_from_sRGB(c.b)), 
            a(sRGB_conv<value_type>::alpha_from_sRGB(c.a)) {}

         
        rgba32(const rgba16& c) :
            r(value_type(c.r / 65535.0)), 
            g(value_type(c.g / 65535.0)), 
            b(value_type(c.b / 65535.0)), 
            a(value_type(c.a / 65535.0)) {}

         
        operator rgba() const 
        {
            return rgba(r, g, b, a);
        }

         
        operator rgba8() const 
        {
            return rgba8(
                uround(r * 255.0), 
                uround(g * 255.0), 
                uround(b * 255.0), 
                uround(a * 255.0));
        }

         
        operator srgba8() const 
        {
            return srgba8(
                sRGB_conv<value_type>::rgb_to_sRGB(r), 
                sRGB_conv<value_type>::rgb_to_sRGB(g), 
                sRGB_conv<value_type>::rgb_to_sRGB(b), 
                sRGB_conv<value_type>::alpha_to_sRGB(a));
        }

         
        operator rgba16() const 
        {
            return rgba8(
                uround(r * 65535.0), 
                uround(g * 65535.0), 
                uround(b * 65535.0), 
                uround(a * 65535.0));
        }

         
        static AGG_INLINE double to_double(value_type a)
        {
            return a;
        }

         
        static AGG_INLINE value_type from_double(double a)
        {
            return value_type(a);
        }

         
        static AGG_INLINE value_type empty_value()
        {
            return 0;
        }

         
        static AGG_INLINE value_type full_value()
        {
            return 1;
        }

         
        AGG_INLINE bool is_transparent() const
        {
            return a <= 0;
        }

         
        AGG_INLINE bool is_opaque() const
        {
            return a >= 1;
        }

         
        static AGG_INLINE value_type invert(value_type x) 
        {
            return 1 - x;
        }

         
        static AGG_INLINE value_type multiply(value_type a, value_type b) 
        {
            return value_type(a * b);
        }

         
        static AGG_INLINE value_type demultiply(value_type a, value_type b) 
        {
            return (b == 0) ? 0 : value_type(a / b);
        }

         
        template<typename T>
        static AGG_INLINE T downscale(T a) 
        {
            return a;
        }

         
        template<typename T>
        static AGG_INLINE T downshift(T a, unsigned n) 
        {
            return n > 0 ? a / (1 << n) : a;
        }

         
        static AGG_INLINE value_type mult_cover(value_type a, cover_type b) 
        {
            return value_type(a * b / (value_type)cover_mask);
        }

         
        static AGG_INLINE cover_type scale_cover(cover_type a, value_type b) 
        {
            return cover_type(uround(a * b));
        }
        
         
         
        static AGG_INLINE value_type prelerp(value_type p, value_type q, value_type a) 
        {
            return (1 - a) * p + q;  
        }
        
         
         
        static AGG_INLINE value_type lerp(value_type p, value_type q, value_type a) 
        {
			 
			 
			 
			 
			return (1 - a) * p + a * q;
        }
        
         
        self_type& clear()
        {
            r = g = b = a = 0;
			return *this;
        }
        
         
        self_type& transparent()
        {
            a = 0;
            return *this;
        }

         
        AGG_INLINE self_type& opacity(double a_)
        {
            if (a_ < 0) a = 0;
            else if (a_ > 1) a = 1;
            else a = value_type(a_);
            return *this;
        }

         
        double opacity() const
        {
            return a;
        }

         
        AGG_INLINE self_type& premultiply()
        {
            if (a < 1)
            {
                if (a <= 0)
                {
                    r = g = b = 0;
                }
                else
                {
                    r *= a;
                    g *= a;
                    b *= a;
                }
            }
            return *this;
        }

         
        AGG_INLINE self_type& demultiply()
        {
            if (a < 1)
            {
                if (a <= 0)
                {
                    r = g = b = 0;
                }
                else
                {
                    r /= a;
                    g /= a;
                    b /= a;
                }
            }
            return *this;
        }

         
        AGG_INLINE self_type gradient(const self_type& c, double k) const
        {
            self_type ret;
            ret.r = value_type(r + (c.r - r) * k);
            ret.g = value_type(g + (c.g - g) * k);
            ret.b = value_type(b + (c.b - b) * k);
            ret.a = value_type(a + (c.a - a) * k);
            return ret;
        }

         
        AGG_INLINE void add(const self_type& c, unsigned cover)
        {
            if (cover == cover_mask)
            {
                if (c.is_opaque()) 
                {
                    *this = c;
                    return;
                }
                else
                {
                    r += c.r; 
                    g += c.g; 
                    b += c.b; 
                    a += c.a; 
                }
            }
            else
            {
                r += mult_cover(c.r, cover);
                g += mult_cover(c.g, cover);
                b += mult_cover(c.b, cover);
                a += mult_cover(c.a, cover);
            }
            if (a > 1) a = 1;
            if (r > a) r = a;
            if (g > a) g = a;
            if (b > a) b = a;
        }

         
        template<class GammaLUT>
        AGG_INLINE void apply_gamma_dir(const GammaLUT& gamma)
        {
            r = gamma.dir(r);
            g = gamma.dir(g);
            b = gamma.dir(b);
        }

         
        template<class GammaLUT>
        AGG_INLINE void apply_gamma_inv(const GammaLUT& gamma)
        {
            r = gamma.inv(r);
            g = gamma.inv(g);
            b = gamma.inv(b);
        }

         
        static self_type no_color() { return self_type(0,0,0,0); }

         
        static self_type from_wavelength(double wl, double gamma = 1)
        {
            return self_type(rgba::from_wavelength(wl, gamma));
        }
    };
}



#endif

namespace agg
{

     
    template<class Colorspace>
    struct gray8T
    {
        typedef int8u  value_type;
        typedef int32u calc_type;
        typedef int32  long_type;
        enum base_scale_e
        {
            base_shift = 8,
            base_scale = 1 << base_shift,
            base_mask  = base_scale - 1,
            base_MSB = 1 << (base_shift - 1)
        };
        typedef gray8T self_type;

        value_type v;
        value_type a;

        static value_type luminance(const rgba& c)
        {
             
            return value_type(uround((0.2126 * c.r + 0.7152 * c.g + 0.0722 * c.b) * (double)base_mask));
        }

        static value_type luminance(const rgba8& c)
        {
             
            return value_type((55u * c.r + 184u * c.g + 18u * c.b) >> 8);
        }

        static void convert(gray8T<linear>& dst, const gray8T<sRGB>& src)
        {
            dst.v = sRGB_conv<value_type>::rgb_from_sRGB(src.v);
            dst.a = src.a;
        }

        static void convert(gray8T<sRGB>& dst, const gray8T<linear>& src)
        {
            dst.v = sRGB_conv<value_type>::rgb_to_sRGB(src.v);
            dst.a = src.a;
        }

        static void convert(gray8T<linear>& dst, const rgba8& src)
        {
            dst.v = luminance(src);
            dst.a = src.a;
        }

        static void convert(gray8T<linear>& dst, const srgba8& src)
        {
             
            convert(dst, rgba8(src));
        }

        static void convert(gray8T<sRGB>& dst, const rgba8& src)
        {
            dst.v = sRGB_conv<value_type>::rgb_to_sRGB(luminance(src));
            dst.a = src.a;
        }

        static void convert(gray8T<sRGB>& dst, const srgba8& src)
        {
             
            convert(dst, rgba8(src));
        }

         
        gray8T() {}

         
        explicit gray8T(unsigned v_, unsigned a_ = base_mask) :
            v(int8u(v_)), a(int8u(a_)) {}

         
        gray8T(const self_type& c, unsigned a_) :
            v(c.v), a(value_type(a_)) {}

         
        gray8T(const rgba& c) :
            v(luminance(c)),
            a(value_type(uround(c.a * (double)base_mask))) {}

         
        template<class T>
        gray8T(const gray8T<T>& c)
        {
            convert(*this, c);
        }

         
        template<class T>
        gray8T(const rgba8T<T>& c)
        {
            convert(*this, c);
        }

         
        template<class T> 
        T convert_from_sRGB() const 
        {
            typename T::value_type y = sRGB_conv<typename T::value_type>::rgb_from_sRGB(v);
            return T(y, y, y, sRGB_conv<typename T::value_type>::alpha_from_sRGB(a));
        }

        template<class T> 
        T convert_to_sRGB() const 
        {
            typename T::value_type y = sRGB_conv<typename T::value_type>::rgb_to_sRGB(v);
            return T(y, y, y, sRGB_conv<typename T::value_type>::alpha_to_sRGB(a));
        }

         
        rgba8 make_rgba8(const linear&) const 
        {
            return rgba8(v, v, v, a);
        }

        rgba8 make_rgba8(const sRGB&) const 
        {
            return convert_from_sRGB<srgba8>();
        }

        operator rgba8() const 
        {
            return make_rgba8(Colorspace());
        }

         
        srgba8 make_srgba8(const linear&) const 
        {
            return convert_to_sRGB<rgba8>();
        }

        srgba8 make_srgba8(const sRGB&) const 
        {
            return srgba8(v, v, v, a);
        }

        operator srgba8() const 
        {
            return make_rgba8(Colorspace());
        }

         
        rgba16 make_rgba16(const linear&) const 
        {
            rgba16::value_type rgb = (v << 8) | v;
            return rgba16(rgb, rgb, rgb, (a << 8) | a);
        }

        rgba16 make_rgba16(const sRGB&) const 
        {
            return convert_from_sRGB<rgba16>();
        }

        operator rgba16() const 
        {
            return make_rgba16(Colorspace());
        }

         
        rgba32 make_rgba32(const linear&) const 
        {
            rgba32::value_type v32 = v / 255.0f;
            return rgba32(v32, v32, v32, a / 255.0f);
        }

        rgba32 make_rgba32(const sRGB&) const 
        {
            return convert_from_sRGB<rgba32>();
        }

        operator rgba32() const 
        {
            return make_rgba32(Colorspace());
        }

         
        static AGG_INLINE double to_double(value_type a)
        {
            return double(a) / (double)base_mask;
        }

         
        static AGG_INLINE value_type from_double(double a)
        {
            return value_type(uround(a * (double)base_mask));
        }

         
        static AGG_INLINE value_type empty_value()
        {
            return 0;
        }

         
        static AGG_INLINE value_type full_value()
        {
            return base_mask;
        }

         
        AGG_INLINE bool is_transparent() const
        {
            return a == 0;
        }

         
        AGG_INLINE bool is_opaque() const
        {
            return a == base_mask;
        }

         
         
        static AGG_INLINE value_type multiply(value_type a, value_type b) 
        {
            calc_type t = a * b + base_MSB;
            return value_type(((t >> base_shift) + t) >> base_shift);
        }
        
         
        static AGG_INLINE value_type demultiply(value_type a, value_type b) 
        {
            if (a * b == 0)
            {
                return 0;
            }
            else if (a >= b)
            {
                return base_mask;
            }
            else return value_type((a * base_mask + (b >> 1)) / b); 
        }

         
        template<typename T>
        static AGG_INLINE T downscale(T a) 
        {
            return a >> base_shift;
        }

         
        template<typename T>
        static AGG_INLINE T downshift(T a, unsigned n) 
        {
            return a >> n;
        }

         
         
         
        static AGG_INLINE value_type mult_cover(value_type a, value_type b) 
        {
            return multiply(a, b);
        }
        
         
        static AGG_INLINE cover_type scale_cover(cover_type a, value_type b) 
        {
            return multiply(b, a);
        }
        
         
         
        static AGG_INLINE value_type prelerp(value_type p, value_type q, value_type a) 
        {
            return p + q - multiply(p, a);
        }
        
         
         
        static AGG_INLINE value_type lerp(value_type p, value_type q, value_type a) 
        {
            int t = (q - p) * a + base_MSB - (p > q);
            return value_type(p + (((t >> base_shift) + t) >> base_shift));
        }
        
         
        self_type& clear()
        {
            v = a = 0;
			return *this;
        }

         
        self_type& transparent()
        {
            a = 0;
            return *this;
        }

         
        self_type& opacity(double a_)
        {
            if (a_ < 0) a = 0;
            else if (a_ > 1) a = 1;
            else a = (value_type)uround(a_ * double(base_mask));
			return *this;
        }

         
        double opacity() const
        {
            return double(a) / double(base_mask);
        }
        
         
        self_type& premultiply()
        {
            if (a < base_mask)
            {
                if (a == 0) v = 0;
                else v = multiply(v, a);
            }
            return *this;
        }

         
        self_type& demultiply()
        {
            if (a < base_mask)
            {
                if (a == 0)
                {
                    v = 0;
                }
                else
                {
                    calc_type v_ = (calc_type(v) * base_mask) / a;
                    v = value_type((v_ > base_mask) ? (value_type)base_mask : v_);
                }
            }
            return *this;
        }

         
        self_type gradient(self_type c, double k) const
        {
            self_type ret;
            calc_type ik = uround(k * (double)base_scale);
            ret.v = lerp(v, c.v, ik);
            ret.a = lerp(a, c.a, ik);
            return ret;
        }

         
        AGG_INLINE void add(const self_type& c, unsigned cover)
        {
            calc_type cv, ca;
            if (cover == cover_mask)
            {
                if (c.a == base_mask) 
                {
                    *this = c;
                    return;
                }
                else
                {
                    cv = v + c.v; 
                    ca = a + c.a; 
                }
            }
            else
            {
                cv = v + mult_cover(c.v, cover);
                ca = a + mult_cover(c.a, cover);
            }
            v = (value_type)((cv > calc_type(base_mask)) ? calc_type(base_mask) : cv);
            a = (value_type)((ca > calc_type(base_mask)) ? calc_type(base_mask) : ca);
        }

         
        static self_type no_color() { return self_type(0,0); }
    };

    typedef gray8T<linear> gray8;
    typedef gray8T<sRGB> sgray8;


     
    struct gray16
    {
        typedef int16u value_type;
        typedef int32u calc_type;
        typedef int64  long_type;
        enum base_scale_e
        {
            base_shift = 16,
            base_scale = 1 << base_shift,
            base_mask  = base_scale - 1,
            base_MSB = 1 << (base_shift - 1)
        };
        typedef gray16 self_type;

        value_type v;
        value_type a;

        static value_type luminance(const rgba& c)
        {
             
            return value_type(uround((0.2126 * c.r + 0.7152 * c.g + 0.0722 * c.b) * (double)base_mask));
        }

        static value_type luminance(const rgba16& c)
        {
             
            return value_type((13933u * c.r + 46872u * c.g + 4732u * c.b) >> 16);
        }

        static value_type luminance(const rgba8& c)
        {
            return luminance(rgba16(c));
        }

        static value_type luminance(const srgba8& c)
        {
            return luminance(rgba16(c));
        }

        static value_type luminance(const rgba32& c)
        {
            return luminance(rgba(c));
        }

         
        gray16() {}

         
        explicit gray16(unsigned v_, unsigned a_ = base_mask) :
            v(int16u(v_)), a(int16u(a_)) {}

         
        gray16(const self_type& c, unsigned a_) :
            v(c.v), a(value_type(a_)) {}

         
        gray16(const rgba& c) :
            v(luminance(c)),
            a((value_type)uround(c.a * double(base_mask))) {}

         
        gray16(const rgba8& c) :
            v(luminance(c)),
            a((value_type(c.a) << 8) | c.a) {}

         
        gray16(const srgba8& c) :
            v(luminance(c)),
            a((value_type(c.a) << 8) | c.a) {}

         
        gray16(const rgba16& c) :
            v(luminance(c)),
            a(c.a) {}
        
         
        gray16(const gray8& c) :
            v((value_type(c.v) << 8) | c.v),
            a((value_type(c.a) << 8) | c.a) {}

         
        gray16(const sgray8& c) :
            v(sRGB_conv<value_type>::rgb_from_sRGB(c.v)),
            a(sRGB_conv<value_type>::alpha_from_sRGB(c.a)) {}

         
        operator rgba8() const 
        {
            return rgba8(v >> 8, v >> 8, v >> 8, a >> 8);
        }

         
        operator srgba8() const 
        {
            value_type y = sRGB_conv<value_type>::rgb_to_sRGB(v);
            return srgba8(y, y, y, sRGB_conv<value_type>::alpha_to_sRGB(a));
        }

         
        operator rgba16() const 
        {
            return rgba16(v, v, v, a);
        }

		 
		operator rgba32() const
		{
			rgba32::value_type v32 = v / 65535.0f;
			return rgba32(v32, v32, v32, a / 65535.0f);
		}

		 
        operator gray8() const 
        {
            return gray8(v >> 8, a >> 8);
        }

         
        operator sgray8() const 
        {
            return sgray8(
                sRGB_conv<value_type>::rgb_to_sRGB(v), 
                sRGB_conv<value_type>::alpha_to_sRGB(a));
        }

         
        static AGG_INLINE double to_double(value_type a)
        {
            return double(a) / (double)base_mask;
        }

         
        static AGG_INLINE value_type from_double(double a)
        {
            return value_type(uround(a * (double)base_mask));
        }

         
        static AGG_INLINE value_type empty_value()
        {
            return 0;
        }

         
        static AGG_INLINE value_type full_value()
        {
            return base_mask;
        }

         
        AGG_INLINE bool is_transparent() const
        {
            return a == 0;
        }

         
        AGG_INLINE bool is_opaque() const
        {
            return a == base_mask;
        }

         
         
        static AGG_INLINE value_type multiply(value_type a, value_type b) 
        {
            calc_type t = a * b + base_MSB;
            return value_type(((t >> base_shift) + t) >> base_shift);
        }
        
         
        static AGG_INLINE value_type demultiply(value_type a, value_type b) 
        {
            if (a * b == 0)
            {
                return 0;
            }
            else if (a >= b)
            {
                return base_mask;
            }
            else return value_type((a * base_mask + (b >> 1)) / b); 
        }

         
        template<typename T>
        static AGG_INLINE T downscale(T a) 
        {
            return a >> base_shift;
        }

         
        template<typename T>
        static AGG_INLINE T downshift(T a, unsigned n) 
        {
            return a >> n;
        }

         
         
         
        static AGG_INLINE value_type mult_cover(value_type a, cover_type b) 
        {
            return multiply(a, b << 8 | b);
        }
        
         
        static AGG_INLINE cover_type scale_cover(cover_type a, value_type b) 
        {
            return mult_cover(b, a) >> 8;
        }
        
         
         
        static AGG_INLINE value_type prelerp(value_type p, value_type q, value_type a) 
        {
            return p + q - multiply(p, a);
        }
        
         
         
        static AGG_INLINE value_type lerp(value_type p, value_type q, value_type a) 
        {
            int t = (q - p) * a + base_MSB - (p > q);
            return value_type(p + (((t >> base_shift) + t) >> base_shift));
        }
        
         
        self_type& clear()
        {
            v = a = 0;
			return *this;
        }

         
        self_type& transparent()
        {
            a = 0;
            return *this;
        }

         
        self_type& opacity(double a_)
        {
            if (a_ < 0) a = 0;
            else if(a_ > 1) a = 1;
            else a = (value_type)uround(a_ * double(base_mask));
			return *this;
        }

         
        double opacity() const
        {
            return double(a) / double(base_mask);
        }


         
        self_type& premultiply()
        {
            if (a < base_mask)
            {
                if(a == 0) v = 0;
                else v = multiply(v, a);
            }
            return *this;
        }

         
        self_type& demultiply()
        {
            if (a < base_mask)
            {
                if (a == 0)
                {
                    v = 0;
                }
                else
                {
                    calc_type v_ = (calc_type(v) * base_mask) / a;
					v = (v_ > base_mask) ? value_type(base_mask) : value_type(v_);
                }
            }
            return *this;
        }

         
        self_type gradient(self_type c, double k) const
        {
            self_type ret;
            calc_type ik = uround(k * (double)base_scale);
            ret.v = lerp(v, c.v, ik);
            ret.a = lerp(a, c.a, ik);
            return ret;
        }

         
        AGG_INLINE void add(const self_type& c, unsigned cover)
        {
            calc_type cv, ca;
            if (cover == cover_mask)
            {
                if (c.a == base_mask) 
                {
                    *this = c;
                    return;
                }
                else
                {
                    cv = v + c.v; 
                    ca = a + c.a; 
                }
            }
            else
            {
                cv = v + mult_cover(c.v, cover);
                ca = a + mult_cover(c.a, cover);
            }
            v = (value_type)((cv > calc_type(base_mask)) ? calc_type(base_mask) : cv);
            a = (value_type)((ca > calc_type(base_mask)) ? calc_type(base_mask) : ca);
        }

         
        static self_type no_color() { return self_type(0,0); }
    };


     
    struct gray32
    {
        typedef float value_type;
        typedef double calc_type;
        typedef double long_type;
        typedef gray32 self_type;

        value_type v;
        value_type a;

         
        static value_type luminance(double r, double g, double b)
        {
            return value_type(0.2126 * r + 0.7152 * g + 0.0722 * b);
        }

        static value_type luminance(const rgba& c)
        {
            return luminance(c.r, c.g, c.b);
        }

        static value_type luminance(const rgba32& c)
        {
            return luminance(c.r, c.g, c.b);
        }

        static value_type luminance(const rgba8& c)
        {
            return luminance(c.r / 255.0, c.g / 255.0, c.g / 255.0);
        }

        static value_type luminance(const rgba16& c)
        {
            return luminance(c.r / 65535.0, c.g / 65535.0, c.g / 65535.0);
        }

         
        gray32() {}

         
        explicit gray32(value_type v_, value_type a_ = 1) :
            v(v_), a(a_) {}

         
        gray32(const self_type& c, value_type a_) :
            v(c.v), a(a_) {}

         
        gray32(const rgba& c) :
            v(luminance(c)),
            a(value_type(c.a)) {}

         
        gray32(const rgba8& c) :
            v(luminance(c)),
            a(value_type(c.a / 255.0)) {}

         
        gray32(const srgba8& c) :
            v(luminance(rgba32(c))),
            a(value_type(c.a / 255.0)) {}

         
        gray32(const rgba16& c) :
            v(luminance(c)),
            a(value_type(c.a / 65535.0)) {}

         
        gray32(const rgba32& c) :
            v(luminance(c)),
            a(value_type(c.a)) {}

         
        gray32(const gray8& c) :
            v(value_type(c.v / 255.0)), 
            a(value_type(c.a / 255.0)) {}

         
        gray32(const sgray8& c) :
            v(sRGB_conv<value_type>::rgb_from_sRGB(c.v)), 
            a(sRGB_conv<value_type>::alpha_from_sRGB(c.a)) {}

         
        gray32(const gray16& c) :
            v(value_type(c.v / 65535.0)), 
            a(value_type(c.a / 65535.0)) {}

         
        operator rgba() const 
        {
            return rgba(v, v, v, a);
        }

         
        operator gray8() const 
        {
            return gray8(uround(v * 255.0), uround(a * 255.0));
        }

         
        operator sgray8() const 
        {
             
            return sgray8(
                sRGB_conv<value_type>::rgb_to_sRGB(v), 
                sRGB_conv<value_type>::alpha_to_sRGB(a));
        }

         
        operator gray16() const 
        {
            return gray16(uround(v * 65535.0), uround(a * 65535.0));
        }

         
        operator rgba8() const 
        {
            rgba8::value_type y = uround(v * 255.0);
            return rgba8(y, y, y, uround(a * 255.0));
        }

         
        operator srgba8() const 
        {
            srgba8::value_type y = sRGB_conv<value_type>::rgb_to_sRGB(v);
            return srgba8(y, y, y, sRGB_conv<value_type>::alpha_to_sRGB(a));
        }

		 
		operator rgba16() const
		{
			rgba16::value_type y = uround(v * 65535.0);
			return rgba16(y, y, y, uround(a * 65535.0));
		}

		 
		operator rgba32() const
		{
            return rgba32(v, v, v, a);
		}

		 
        static AGG_INLINE double to_double(value_type a)
        {
            return a;
        }

         
        static AGG_INLINE value_type from_double(double a)
        {
            return value_type(a);
        }

         
        static AGG_INLINE value_type empty_value()
        {
            return 0;
        }

         
        static AGG_INLINE value_type full_value()
        {
            return 1;
        }

         
        AGG_INLINE bool is_transparent() const
        {
            return a <= 0;
        }

         
        AGG_INLINE bool is_opaque() const
        {
            return a >= 1;
        }

         
        static AGG_INLINE value_type invert(value_type x) 
        {
            return 1 - x;
        }

         
        static AGG_INLINE value_type multiply(value_type a, value_type b) 
        {
            return value_type(a * b);
        }

         
        static AGG_INLINE value_type demultiply(value_type a, value_type b) 
        {
            return (b == 0) ? 0 : value_type(a / b);
        }

         
        template<typename T>
        static AGG_INLINE T downscale(T a) 
        {
            return a;
        }

         
        template<typename T>
        static AGG_INLINE T downshift(T a, unsigned n) 
        {
            return n > 0 ? a / (1 << n) : a;
        }

         
        static AGG_INLINE value_type mult_cover(value_type a, cover_type b) 
        {
            return value_type(a * b / (value_type)cover_mask);
        }

         
        static AGG_INLINE cover_type scale_cover(cover_type a, value_type b) 
        {
            return cover_type(uround(a * b));
        }
        
         
         
        static AGG_INLINE value_type prelerp(value_type p, value_type q, value_type a) 
        {
            return (1 - a) * p + q;  
        }
        
         
         
        static AGG_INLINE value_type lerp(value_type p, value_type q, value_type a) 
        {
			 
			 
			 
			 
			return (1 - a) * p + a * q;
        }
        
         
        self_type& clear()
        {
            v = a = 0;
			return *this;
        }

         
        self_type& transparent()
        {
            a = 0;
            return *this;
        }

         
        self_type& opacity(double a_)
        {
            if (a_ < 0) a = 0;
            else if (a_ > 1) a = 1;
            else a = value_type(a_);
			return *this;
        }

         
        double opacity() const
        {
            return a;
        }


         
        self_type& premultiply()
        {
            if (a < 0) v = 0;
            else if(a < 1) v *= a;
            return *this;
        }

         
        self_type& demultiply()
        {
            if (a < 0) v = 0;
            else if (a < 1) v /= a;
            return *this;
        }

         
        self_type gradient(self_type c, double k) const
        {
            return self_type(
                value_type(v + (c.v - v) * k), 
                value_type(a + (c.a - a) * k));
        }

         
        static self_type no_color() { return self_type(0,0); }
    };
}




#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_ADAPTOR_VCGEN_INCLUDED
#define AGG_CONV_ADAPTOR_VCGEN_INCLUDED


namespace agg
{
     
    struct null_markers
    {
        void remove_all() {}
        void add_vertex(double, double, unsigned) {}
        void prepare_src() {}

        void rewind(unsigned) {}
        unsigned vertex(double*, double*) { return path_cmd_stop; }
    };


     
    template<class VertexSource, 
             class Generator, 
             class Markers=null_markers> class conv_adaptor_vcgen
    {
        enum status
        {
            initial,
            accumulate,
            generate
        };

    public:
        explicit conv_adaptor_vcgen(VertexSource& source) :
            m_source(&source), 
            m_status(initial)
        {}
        void attach(VertexSource& source) { m_source = &source; }

        Generator& generator() { return m_generator; }
        const Generator& generator() const { return m_generator; }

        Markers& markers() { return m_markers; }
        const Markers& markers() const { return m_markers; }
        
        void rewind(unsigned path_id) 
        { 
            m_source->rewind(path_id); 
            m_status = initial;
        }

        unsigned vertex(double* x, double* y);

    private:
         
        conv_adaptor_vcgen(const conv_adaptor_vcgen<VertexSource, Generator, Markers>&);
        const conv_adaptor_vcgen<VertexSource, Generator, Markers>& 
            operator = (const conv_adaptor_vcgen<VertexSource, Generator, Markers>&);

        VertexSource* m_source;
        Generator     m_generator;
        Markers       m_markers;
        status        m_status;
        unsigned      m_last_cmd;
        double        m_start_x;
        double        m_start_y;
    };





     
    template<class VertexSource, class Generator, class Markers> 
    unsigned conv_adaptor_vcgen<VertexSource, Generator, Markers>::vertex(double* x, double* y)
    {
        unsigned cmd = path_cmd_stop;
        bool done = false;
        while(!done)
        {
            switch(m_status)
            {
            case initial:
                m_markers.remove_all();
                m_last_cmd = m_source->vertex(&m_start_x, &m_start_y);
                m_status = accumulate;

                AGG_FALLTHROUGH
            case accumulate:
                if(is_stop(m_last_cmd)) return path_cmd_stop;

                m_generator.remove_all();
                m_generator.add_vertex(m_start_x, m_start_y, path_cmd_move_to);
                m_markers.add_vertex(m_start_x, m_start_y, path_cmd_move_to);

                for(;;)
                {
                    cmd = m_source->vertex(x, y);
                    if(is_vertex(cmd))
                    {
                        m_last_cmd = cmd;
                        if(is_move_to(cmd))
                        {
                            m_start_x = *x;
                            m_start_y = *y;
                            break;
                        }
                        m_generator.add_vertex(*x, *y, cmd);
                        m_markers.add_vertex(*x, *y, path_cmd_line_to);
                    }
                    else
                    {
                        if(is_stop(cmd))
                        {
                            m_last_cmd = path_cmd_stop;
                            break;
                        }
                        if(is_end_poly(cmd))
                        {
                            m_generator.add_vertex(*x, *y, cmd);
                            break;
                        }
                    }
                }
                m_generator.rewind(0);
                m_status = generate;

                AGG_FALLTHROUGH
            case generate:
                cmd = m_generator.vertex(x, y);
                if(is_stop(cmd))
                {
                    m_status = accumulate;
                    break;
                }
                done = true;
                break;
            }
        }
        return cmd;
    }

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_ADAPTOR_VPGEN_INCLUDED
#define AGG_CONV_ADAPTOR_VPGEN_INCLUDED


namespace agg
{

     
    template<class VertexSource, class VPGen> class conv_adaptor_vpgen
    {
    public:
        explicit conv_adaptor_vpgen(VertexSource& source) : m_source(&source) {}
        void attach(VertexSource& source) { m_source = &source; }

        VPGen& vpgen() { return m_vpgen; }
        const VPGen& vpgen() const { return m_vpgen; }

        void rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        conv_adaptor_vpgen(const conv_adaptor_vpgen<VertexSource, VPGen>&);
        const conv_adaptor_vpgen<VertexSource, VPGen>& 
            operator = (const conv_adaptor_vpgen<VertexSource, VPGen>&);

        VertexSource* m_source;
        VPGen         m_vpgen;
        double        m_start_x;
        double        m_start_y;
        unsigned      m_poly_flags;
        int           m_vertices;
    };



     
    template<class VertexSource, class VPGen>
    void conv_adaptor_vpgen<VertexSource, VPGen>::rewind(unsigned path_id) 
    { 
        m_source->rewind(path_id);
        m_vpgen.reset();
        m_start_x    = 0;
        m_start_y    = 0;
        m_poly_flags = 0;
        m_vertices   = 0;
    }


     
    template<class VertexSource, class VPGen>
    unsigned conv_adaptor_vpgen<VertexSource, VPGen>::vertex(double* x, double* y)
    {
        unsigned cmd = path_cmd_stop;
        for(;;)
        {
            cmd = m_vpgen.vertex(x, y);
            if(!is_stop(cmd)) break;

            if(m_poly_flags && !m_vpgen.auto_unclose())
            {
                *x = 0.0;
                *y = 0.0;
                cmd = m_poly_flags;
                m_poly_flags = 0;
                break;
            }

            if(m_vertices < 0)
            {
                if(m_vertices < -1) 
                {
                    m_vertices = 0;
                    return path_cmd_stop;
                }
                m_vpgen.move_to(m_start_x, m_start_y);
                m_vertices = 1;
                continue;
            }

            double tx, ty;
            cmd = m_source->vertex(&tx, &ty);
            if(is_vertex(cmd))
            {
                if(is_move_to(cmd)) 
                {
                    if(m_vpgen.auto_close() && m_vertices > 2)
                    {
                        m_vpgen.line_to(m_start_x, m_start_y);
                        m_poly_flags = +path_cmd_end_poly | path_flags_close;
                        m_start_x    = tx;
                        m_start_y    = ty;
                        m_vertices   = -1;
                        continue;
                    }
                    m_vpgen.move_to(tx, ty);
                    m_start_x  = tx;
                    m_start_y  = ty;
                    m_vertices = 1;
                }
                else 
                {
                    m_vpgen.line_to(tx, ty);
                    ++m_vertices;
                }
            }
            else
            {
                if(is_end_poly(cmd))
                {
                    m_poly_flags = cmd;
                    if(is_closed(cmd) || m_vpgen.auto_close())
                    {
                        if(m_vpgen.auto_close()) m_poly_flags |= path_flags_close;
                        if(m_vertices > 2)
                        {
                            m_vpgen.line_to(m_start_x, m_start_y);
                        }
                        m_vertices = 0;
                    }
                }
                else
                {
                     
                    if(m_vpgen.auto_close() && m_vertices > 2)
                    {
                        m_vpgen.line_to(m_start_x, m_start_y);
                        m_poly_flags = +path_cmd_end_poly | path_flags_close;
                        m_vertices   = -2;
                        continue;
                    }
                    break;
                }
            }
        }
        return cmd;
    }


}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CONV_BSPLINE_INCLUDED
#define AGG_CONV_BSPLINE_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_VCGEN_BSPLINE_INCLUDED
#define AGG_VCGEN_BSPLINE_INCLUDED



namespace agg
{

     
    class vcgen_bspline
    {
        enum status_e
        {
            initial,
            ready,
            polygon,
            end_poly,
            stop
        };

    public:
        typedef pod_bvector<point_d, 6> vertex_storage;

        vcgen_bspline();

        void interpolation_step(double v) { m_interpolation_step = v; }
        double interpolation_step() const { return m_interpolation_step; }

         
        void remove_all();
        void add_vertex(double x, double y, unsigned cmd);

         
        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        vcgen_bspline(const vcgen_bspline&);
        const vcgen_bspline& operator = (const vcgen_bspline&);

        vertex_storage m_src_vertices;
        bspline        m_spline_x;
        bspline        m_spline_y;
        double         m_interpolation_step;
        unsigned       m_closed;
        status_e       m_status;
        unsigned       m_src_vertex;
        double         m_cur_abscissa;
        double         m_max_abscissa;
    };

}


#endif



namespace agg
{

     
    template<class VertexSource> 
    struct conv_bspline : public conv_adaptor_vcgen<VertexSource, vcgen_bspline>
    {
        typedef conv_adaptor_vcgen<VertexSource, vcgen_bspline> base_type;

        conv_bspline(VertexSource& vs) : 
            conv_adaptor_vcgen<VertexSource, vcgen_bspline>(vs) {}

        void   interpolation_step(double v) { base_type::generator().interpolation_step(v); }
        double interpolation_step() const { return base_type::generator().interpolation_step(); }

    private:
        conv_bspline(const conv_bspline<VertexSource>&);
        const conv_bspline<VertexSource>& 
            operator = (const conv_bspline<VertexSource>&);
    };

}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CONV_CLIP_POLYGON_INCLUDED
#define AGG_CONV_CLIP_POLYGON_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_VPGEN_CLIP_POLYGON_INCLUDED
#define AGG_VPGEN_CLIP_POLYGON_INCLUDED


namespace agg
{

     
     
     
     
    class vpgen_clip_polygon
    {
    public:
        vpgen_clip_polygon() : 
            m_clip_box(0, 0, 1, 1),
            m_x1(0),
            m_y1(0),
            m_clip_flags(0),
            m_num_vertices(0),
            m_vertex(0),
            m_cmd(path_cmd_move_to)
        {
        }

        void clip_box(double x1, double y1, double x2, double y2)
        {
            m_clip_box.x1 = x1;
            m_clip_box.y1 = y1;
            m_clip_box.x2 = x2;
            m_clip_box.y2 = y2;
            m_clip_box.normalize();
        }


        double x1() const { return m_clip_box.x1; }
        double y1() const { return m_clip_box.y1; }
        double x2() const { return m_clip_box.x2; }
        double y2() const { return m_clip_box.y2; }

        static bool auto_close()   { return true;  }
        static bool auto_unclose() { return false; }

        void     reset();
        void     move_to(double x, double y);
        void     line_to(double x, double y);
        unsigned vertex(double* x, double* y);

    private:
        unsigned clipping_flags(double x, double y);

    private:
        rect_d        m_clip_box;
        double        m_x1;
        double        m_y1;
        unsigned      m_clip_flags;
        double        m_x[4];
        double        m_y[4];
        unsigned      m_num_vertices;
        unsigned      m_vertex;
        unsigned      m_cmd;
    };

}


#endif

namespace agg
{

     
    template<class VertexSource> 
    struct conv_clip_polygon : public conv_adaptor_vpgen<VertexSource, vpgen_clip_polygon>
    {
        typedef conv_adaptor_vpgen<VertexSource, vpgen_clip_polygon> base_type;

        conv_clip_polygon(VertexSource& vs) : 
            conv_adaptor_vpgen<VertexSource, vpgen_clip_polygon>(vs) {}

        void clip_box(double x1, double y1, double x2, double y2)
        {
            base_type::vpgen().clip_box(x1, y1, x2, y2);
        }

        double x1() const { return base_type::vpgen().x1(); }
        double y1() const { return base_type::vpgen().y1(); }
        double x2() const { return base_type::vpgen().x2(); }
        double y2() const { return base_type::vpgen().y2(); }

    private:
        conv_clip_polygon(const conv_clip_polygon<VertexSource>&);
        const conv_clip_polygon<VertexSource>& 
            operator = (const conv_clip_polygon<VertexSource>&);
    };

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CONV_CLIP_polyline_INCLUDED
#define AGG_CONV_CLIP_polyline_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_VPGEN_CLIP_POLYLINE_INCLUDED
#define AGG_VPGEN_CLIP_POLYLINE_INCLUDED


namespace agg
{

     
     
     
     
    class vpgen_clip_polyline
    {
    public:
        vpgen_clip_polyline() : 
            m_clip_box(0, 0, 1, 1),
            m_x1(0),
            m_y1(0),
            m_num_vertices(0),
            m_vertex(0),
            m_move_to(false)
        {
        }

        void clip_box(double x1, double y1, double x2, double y2)
        {
            m_clip_box.x1 = x1;
            m_clip_box.y1 = y1;
            m_clip_box.x2 = x2;
            m_clip_box.y2 = y2;
            m_clip_box.normalize();
        }

        double x1() const { return m_clip_box.x1; }
        double y1() const { return m_clip_box.y1; }
        double x2() const { return m_clip_box.x2; }
        double y2() const { return m_clip_box.y2; }

        static bool auto_close()   { return false; }
        static bool auto_unclose() { return true; }

        void     reset();
        void     move_to(double x, double y);
        void     line_to(double x, double y);
        unsigned vertex(double* x, double* y);

    private:
        rect_d        m_clip_box;
        double        m_x1;
        double        m_y1;
        double        m_x[2];
        double        m_y[2];
        unsigned      m_cmd[2];
        unsigned      m_num_vertices;
        unsigned      m_vertex;
        bool          m_move_to;
    };

}


#endif

namespace agg
{

     
    template<class VertexSource> 
    struct conv_clip_polyline : public conv_adaptor_vpgen<VertexSource, vpgen_clip_polyline>
    {
        typedef conv_adaptor_vpgen<VertexSource, vpgen_clip_polyline> base_type;

        conv_clip_polyline(VertexSource& vs) : 
            conv_adaptor_vpgen<VertexSource, vpgen_clip_polyline>(vs) {}

        void clip_box(double x1, double y1, double x2, double y2)
        {
            base_type::vpgen().clip_box(x1, y1, x2, y2);
        }

        double x1() const { return base_type::vpgen().x1(); }
        double y1() const { return base_type::vpgen().y1(); }
        double x2() const { return base_type::vpgen().x2(); }
        double y2() const { return base_type::vpgen().y2(); }

    private:
        conv_clip_polyline(const conv_clip_polyline<VertexSource>&);
        const conv_clip_polyline<VertexSource>& 
            operator = (const conv_clip_polyline<VertexSource>&);
    };

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_CLOSE_POLYGON_INCLUDED
#define AGG_CONV_CLOSE_POLYGON_INCLUDED


namespace agg
{

     
    template<class VertexSource> class conv_close_polygon
    {
    public:
        explicit conv_close_polygon(VertexSource& vs) : m_source(&vs) {}
        void attach(VertexSource& source) { m_source = &source; }

        void rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        conv_close_polygon(const conv_close_polygon<VertexSource>&);
        const conv_close_polygon<VertexSource>& 
            operator = (const conv_close_polygon<VertexSource>&);

        VertexSource* m_source;
        unsigned      m_cmd[2];
        double        m_x[2];
        double        m_y[2];
        unsigned      m_vertex;
        bool          m_line_to;
    };



     
    template<class VertexSource> 
    void conv_close_polygon<VertexSource>::rewind(unsigned path_id)
    {
        m_source->rewind(path_id);
        m_vertex = 2;
        m_line_to = false;
    }


    
     
    template<class VertexSource> 
    unsigned conv_close_polygon<VertexSource>::vertex(double* x, double* y)
    {
        unsigned cmd = path_cmd_stop;
        for(;;)
        {
            if(m_vertex < 2)
            {
                *x = m_x[m_vertex];
                *y = m_y[m_vertex];
                cmd = m_cmd[m_vertex];
                ++m_vertex;
                break;
            }

            cmd = m_source->vertex(x, y);

            if(is_end_poly(cmd))
            {
                cmd |= path_flags_close;
                break;
            }

            if(is_stop(cmd))
            {
                if(m_line_to)
                {
                    m_cmd[0]  = +path_cmd_end_poly | path_flags_close;
                    m_cmd[1]  = path_cmd_stop;
                    m_vertex  = 0;
                    m_line_to = false;
                    continue;
                }
                break;
            }

            if(is_move_to(cmd))
            {
                if(m_line_to)
                {
                    m_x[0]    = 0.0;
                    m_y[0]    = 0.0;
                    m_cmd[0]  = +path_cmd_end_poly | path_flags_close;
                    m_x[1]    = *x;
                    m_y[1]    = *y;
                    m_cmd[1]  = cmd;
                    m_vertex  = 0;
                    m_line_to = false;
                    continue;
                }
                break;
            }

            if(is_vertex(cmd))
            {
                m_line_to = true;
                break;
            }
        }
        return cmd;
    }

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_CONCAT_INCLUDED
#define AGG_CONV_CONCAT_INCLUDED


namespace agg
{
     
     
     
    template<class VS1, class VS2> class conv_concat
    {
    public:
        conv_concat(VS1& source1, VS2& source2) :
            m_source1(&source1), m_source2(&source2), m_status(2) {}
        void attach1(VS1& source) { m_source1 = &source; }
        void attach2(VS2& source) { m_source2 = &source; }


        void rewind(unsigned path_id)
        { 
            m_source1->rewind(path_id);
            m_source2->rewind(0);
            m_status = 0;
        }

        unsigned vertex(double* x, double* y)
        {
            unsigned cmd;
            if(m_status == 0)
            {
                cmd = m_source1->vertex(x, y);
                if(!is_stop(cmd)) return cmd;
                m_status = 1;
            }
            if(m_status == 1)
            {
                cmd = m_source2->vertex(x, y);
                if(!is_stop(cmd)) return cmd;
                m_status = 2;
            }
            return path_cmd_stop;
        }

    private:
        conv_concat(const conv_concat<VS1, VS2>&);
        const conv_concat<VS1, VS2>& 
            operator = (const conv_concat<VS1, VS2>&);

        VS1* m_source1;
        VS2* m_source2;
        int  m_status;

    };
}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CONV_CONTOUR_INCLUDED
#define AGG_CONV_CONTOUR_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_VCGEN_CONTOUR_INCLUDED
#define AGG_VCGEN_CONTOUR_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_STROKE_MATH_INCLUDED
#define AGG_STROKE_MATH_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_MATH_INCLUDED
#define AGG_MATH_INCLUDED

#include <cmath>

namespace agg
{

     
     
    const double vertex_dist_epsilon = 1e-14;

     
     
    const double intersection_epsilon = 1.0e-30;

     
    AGG_INLINE double cross_product(double x1, double y1, 
                                    double x2, double y2, 
                                    double x,  double y)
    {
        return (x - x2) * (y2 - y1) - (y - y2) * (x2 - x1);
    }

     
    AGG_INLINE bool point_in_triangle(double x1, double y1, 
                                      double x2, double y2, 
                                      double x3, double y3, 
                                      double x,  double y)
    {
        bool cp1 = cross_product(x1, y1, x2, y2, x, y) < 0.0;
        bool cp2 = cross_product(x2, y2, x3, y3, x, y) < 0.0;
        bool cp3 = cross_product(x3, y3, x1, y1, x, y) < 0.0;
        return cp1 == cp2 && cp2 == cp3 && cp3 == cp1;
    }

     
    AGG_INLINE double calc_distance(double x1, double y1, double x2, double y2)
    {
        double dx = x2-x1;
        double dy = y2-y1;
        return std::sqrt(dx * dx + dy * dy);
    }

     
    AGG_INLINE double calc_sq_distance(double x1, double y1, double x2, double y2)
    {
        double dx = x2-x1;
        double dy = y2-y1;
        return dx * dx + dy * dy;
    }

     
    AGG_INLINE double calc_line_point_distance(double x1, double y1, 
                                               double x2, double y2, 
                                               double x,  double y)
    {
        double dx = x2-x1;
        double dy = y2-y1;
        double d = std::sqrt(dx * dx + dy * dy);
        if(d < vertex_dist_epsilon)
        {
            return calc_distance(x1, y1, x, y);
        }
        return ((x - x2) * dy - (y - y2) * dx) / d;
    }

     
    AGG_INLINE double calc_segment_point_u(double x1, double y1, 
                                           double x2, double y2, 
                                           double x,  double y)
    {
        double dx = x2 - x1;
        double dy = y2 - y1;

        if(dx == 0 && dy == 0)
        {
	        return 0;
        }

        double pdx = x - x1;
        double pdy = y - y1;

        return (pdx * dx + pdy * dy) / (dx * dx + dy * dy);
    }

     
    AGG_INLINE double calc_segment_point_sq_distance(double x1, double y1, 
                                                     double x2, double y2, 
                                                     double x,  double y,
                                                     double u)
    {
        if(u <= 0)
        {
	        return calc_sq_distance(x, y, x1, y1);
        }
        else 
        if(u >= 1)
        {
	        return calc_sq_distance(x, y, x2, y2);
        }
        return calc_sq_distance(x, y, x1 + u * (x2 - x1), y1 + u * (y2 - y1));
    }

     
    AGG_INLINE double calc_segment_point_sq_distance(double x1, double y1, 
                                                     double x2, double y2, 
                                                     double x,  double y)
    {
        return 
            calc_segment_point_sq_distance(
                x1, y1, x2, y2, x, y,
                calc_segment_point_u(x1, y1, x2, y2, x, y));
    }

     
    AGG_INLINE bool calc_intersection(double ax, double ay, double bx, double by,
                                      double cx, double cy, double dx, double dy,
                                      double* x, double* y)
    {
        double num = (ay-cy) * (dx-cx) - (ax-cx) * (dy-cy);
        double den = (bx-ax) * (dy-cy) - (by-ay) * (dx-cx);
        if(std::fabs(den) < intersection_epsilon) return false;
        double r = num / den;
        *x = ax + r * (bx-ax);
        *y = ay + r * (by-ay);
        return true;
    }

     
    AGG_INLINE bool intersection_exists(double x1, double y1, double x2, double y2,
                                        double x3, double y3, double x4, double y4)
    {
         
         
        double dx1 = x2 - x1;
        double dy1 = y2 - y1;
        double dx2 = x4 - x3;
        double dy2 = y4 - y3;
        return ((x3 - x2) * dy1 - (y3 - y2) * dx1 < 0.0) != 
               ((x4 - x2) * dy1 - (y4 - y2) * dx1 < 0.0) &&
               ((x1 - x4) * dy2 - (y1 - y4) * dx2 < 0.0) !=
               ((x2 - x4) * dy2 - (y2 - y4) * dx2 < 0.0);

         
         
         
         
         
         
         
         
         
         
    }

     
    AGG_INLINE void calc_orthogonal(double thickness,
                                    double x1, double y1,
                                    double x2, double y2,
                                    double* x, double* y)
    {
        double dx = x2 - x1;
        double dy = y2 - y1;
        double d = std::sqrt(dx*dx + dy*dy); 
        *x =  thickness * dy / d;
        *y = -thickness * dx / d;
    }

     
    AGG_INLINE void dilate_triangle(double x1, double y1,
                                    double x2, double y2,
                                    double x3, double y3,
                                    double *x, double* y,
                                    double d)
    {
        double dx1=0.0;
        double dy1=0.0; 
        double dx2=0.0;
        double dy2=0.0; 
        double dx3=0.0;
        double dy3=0.0; 
        double loc = cross_product(x1, y1, x2, y2, x3, y3);
        if(std::fabs(loc) > intersection_epsilon)
        {
            if(cross_product(x1, y1, x2, y2, x3, y3) > 0.0) 
            {
                d = -d;
            }
            calc_orthogonal(d, x1, y1, x2, y2, &dx1, &dy1);
            calc_orthogonal(d, x2, y2, x3, y3, &dx2, &dy2);
            calc_orthogonal(d, x3, y3, x1, y1, &dx3, &dy3);
        }
        *x++ = x1 + dx1;  *y++ = y1 + dy1;
        *x++ = x2 + dx1;  *y++ = y2 + dy1;
        *x++ = x2 + dx2;  *y++ = y2 + dy2;
        *x++ = x3 + dx2;  *y++ = y3 + dy2;
        *x++ = x3 + dx3;  *y++ = y3 + dy3;
        *x++ = x1 + dx3;  *y++ = y1 + dy3;
    }

     
    AGG_INLINE double calc_triangle_area(double x1, double y1,
                                         double x2, double y2,
                                         double x3, double y3)
    {
        return (x1*y2 - x2*y1 + x2*y3 - x3*y2 + x3*y1 - x1*y3) * 0.5;
    }

     
    template<class Storage> double calc_polygon_area(const Storage& st)
    {
        unsigned i;
        double sum = 0.0;
        double x  = st[0].x;
        double y  = st[0].y;
        double xs = x;
        double ys = y;

        for(i = 1; i < st.size(); i++)
        {
            const typename Storage::value_type& v = st[i];
            sum += x * v.y - y * v.x;
            x = v.x;
            y = v.y;
        }
        return (sum + x * ys - y * xs) * 0.5;
    }

     
     
    extern int16u g_sqrt_table[1024];
    extern int8   g_elder_bit_table[256];


     
     
    #if defined(_MSC_VER)
    #pragma warning(push)
    #pragma warning(disable : 4035)  
    #endif
    AGG_INLINE unsigned fast_sqrt(unsigned val)
    {
    #if defined(_M_IX86) && defined(_MSC_VER) && !defined(AGG_NO_ASM)
         
         
         
         
        __asm
        {
            mov ebx, val
            mov edx, 11
            bsr ecx, ebx
            sub ecx, 9
            jle less_than_9_bits
            shr ecx, 1
            adc ecx, 0
            sub edx, ecx
            shl ecx, 1
            shr ebx, cl
    less_than_9_bits:
            xor eax, eax
            mov  ax, g_sqrt_table[ebx*2]
            mov ecx, edx
            shr eax, cl
        }
    #else

         
         
        unsigned t = val;
        int bit=0;
        unsigned shift = 11;

         
         
         
         
         
        bit = t >> 24;
        if(bit)
        {
            bit = g_elder_bit_table[bit] + 24;
        }
        else
        {
            bit = (t >> 16) & 0xFF;
            if(bit)
            {
                bit = g_elder_bit_table[bit] + 16;
            }
            else
            {
                bit = (t >> 8) & 0xFF;
                if(bit)
                {
                    bit = g_elder_bit_table[bit] + 8;
                }
                else
                {
                    bit = g_elder_bit_table[t];
                }
            }
        }

         
        bit -= 9;
        if(bit > 0)
        {
            bit = (bit >> 1) + (bit & 1);
            shift -= bit;
            val >>= (bit << 1);
        }
        return g_sqrt_table[val] >> shift;
    #endif
    }
    #if defined(_MSC_VER)
    #pragma warning(pop)
    #endif




     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
    inline double besj(double x, int n)
    {
        if(n < 0)
        {
            return 0;
        }
        double d = 1E-6;
        double b = 0;
        if(std::fabs(x) <= d) 
        {
            if(n != 0) return 0;
            return 1;
        }
        double b1 = 0;  
         
        int m1 = (int)std::fabs(x) + 6;
        if(std::fabs(x) > 5) 
        {
            m1 = (int)(std::fabs(1.4 * x + 60 / x));
        }
        int m2 = (int)(n + 2 + std::fabs(x) / 4);
        if (m1 > m2) 
        {
            m2 = m1;
        }
    
         
        for(;;) 
        {
            double c3 = 0;
            double c2 = 1E-30;
            double c4 = 0;
            int m8 = 1;
            if (m2 / 2 * 2 == m2) 
            {
                m8 = -1;
            }
            int imax = m2 - 2;
            for (int i = 1; i <= imax; i++) 
            {
                double c6 = 2 * (m2 - i) * c2 / x - c3;
                c3 = c2;
                c2 = c6;
                if(m2 - i - 1 == n)
                {
                    b = c6;
                }
                m8 = -1 * m8;
                if (m8 > 0)
                {
                    c4 = c4 + 2 * c6;
                }
            }
            double c6 = 2 * c2 / x - c3;
            if(n == 0)
            {
                b = c6;
            }
            c4 += c6;
            b /= c4;
            if(std::fabs(b - b1) < d)
            {
                return b;
            }
            b1 = b;
            m2 += 3;
        }
    }

}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_VERTEX_SEQUENCE_INCLUDED
#define AGG_VERTEX_SEQUENCE_INCLUDED


namespace agg
{

     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
    template<class T, unsigned S=6> 
    class vertex_sequence : public pod_bvector<T, S>
    {
    public:
        typedef pod_bvector<T, S> base_type;

        void add(const T& val);
        void modify_last(const T& val);
        void close(bool remove_flag);
    };



     
    template<class T, unsigned S> 
    void vertex_sequence<T, S>::add(const T& val)
    {
        if(base_type::size() > 1)
        {
            if(!(*this)[base_type::size() - 2]((*this)[base_type::size() - 1])) 
            {
                base_type::remove_last();
            }
        }
        base_type::add(val);
    }


     
    template<class T, unsigned S> 
    void vertex_sequence<T, S>::modify_last(const T& val)
    {
        base_type::remove_last();
        add(val);
    }



     
    template<class T, unsigned S> 
    void vertex_sequence<T, S>::close(bool closed)
    {
        while(base_type::size() > 1)
        {
            if((*this)[base_type::size() - 2]((*this)[base_type::size() - 1])) break;
            T t = (*this)[base_type::size() - 1];
            base_type::remove_last();
            modify_last(t);
        }

        if(closed)
        {
            while(base_type::size() > 1)
            {
                if((*this)[base_type::size() - 1]((*this)[0])) break;
                base_type::remove_last();
            }
        }
    }


     
     
     
     
    struct vertex_dist
    {
        double   x;
        double   y;
        double   dist;

        vertex_dist() {}
        vertex_dist(double x_, double y_) :
            x(x_),
            y(y_),
            dist(0.0)
        {
        }

        bool operator () (const vertex_dist& val)
        {
            bool ret = (dist = calc_distance(x, y, val.x, val.y)) > vertex_dist_epsilon;
            if(!ret) dist = 1.0 / vertex_dist_epsilon;
            return ret;
        }
    };



     
     
    struct vertex_dist_cmd : public vertex_dist
    {
        unsigned cmd;

        vertex_dist_cmd() {}
        vertex_dist_cmd(double x_, double y_, unsigned cmd_) :
            vertex_dist(x_, y_),
            cmd(cmd_)
        {
        }
    };


}

#endif

namespace agg
{
     
    enum line_cap_e
    {
        butt_cap,
        square_cap,
        round_cap
    };

     
    enum line_join_e
    {
        miter_join         = 0,
        miter_join_revert  = 1,
        round_join         = 2,
        bevel_join         = 3,
        miter_join_round   = 4
    };


     
    enum inner_join_e
    {
        inner_bevel,
        inner_miter,
        inner_jag,
        inner_round
    };

     
    template<class VertexConsumer> class math_stroke
    {
    public:
        typedef typename VertexConsumer::value_type coord_type;

        math_stroke();

        void line_cap(line_cap_e lc)     { m_line_cap = lc; }
        void line_join(line_join_e lj)   { m_line_join = lj; }
        void inner_join(inner_join_e ij) { m_inner_join = ij; }

        line_cap_e   line_cap()   const { return m_line_cap; }
        line_join_e  line_join()  const { return m_line_join; }
        inner_join_e inner_join() const { return m_inner_join; }

        void width(double w);
        void miter_limit(double ml) { m_miter_limit = ml; }
        void miter_limit_theta(double t);
        void inner_miter_limit(double ml) { m_inner_miter_limit = ml; }
        void approximation_scale(double as) { m_approx_scale = as; }

        double width() const { return m_width * 2.0; }
        double miter_limit() const { return m_miter_limit; }
        double inner_miter_limit() const { return m_inner_miter_limit; }
        double approximation_scale() const { return m_approx_scale; }

        void calc_cap(VertexConsumer& vc,
                      const vertex_dist& v0, 
                      const vertex_dist& v1, 
                      double len);

        void calc_join(VertexConsumer& vc,
                       const vertex_dist& v0, 
                       const vertex_dist& v1, 
                       const vertex_dist& v2,
                       double len1, 
                       double len2);

    private:
        AGG_INLINE void add_vertex(VertexConsumer& vc, double x, double y)
        {
            vc.add(coord_type(x, y));
        }

        void calc_arc(VertexConsumer& vc,
                      double x,   double y, 
                      double dx1, double dy1, 
                      double dx2, double dy2);

        void calc_miter(VertexConsumer& vc,
                        const vertex_dist& v0, 
                        const vertex_dist& v1, 
                        const vertex_dist& v2,
                        double dx1, double dy1, 
                        double dx2, double dy2,
                        line_join_e lj,
                        double mlimit,
                        double dbevel);

        double       m_width;
        double       m_width_abs;
        double       m_width_eps;
        int          m_width_sign;
        double       m_miter_limit;
        double       m_inner_miter_limit;
        double       m_approx_scale;
        line_cap_e   m_line_cap;
        line_join_e  m_line_join;
        inner_join_e m_inner_join;
    };

     
    template<class VC> math_stroke<VC>::math_stroke() :
        m_width(0.5),
        m_width_abs(0.5),
        m_width_eps(0.5/1024.0),
        m_width_sign(1),
        m_miter_limit(4.0),
        m_inner_miter_limit(1.01),
        m_approx_scale(1.0),
        m_line_cap(butt_cap),
        m_line_join(miter_join),
        m_inner_join(inner_miter)
    {
    }

     
    template<class VC> void math_stroke<VC>::width(double w)
    { 
        m_width = w * 0.5; 
        if(m_width < 0)
        {
            m_width_abs  = -m_width;
            m_width_sign = -1;
        }
        else
        {
            m_width_abs  = m_width;
            m_width_sign = 1;
        }
        m_width_eps = m_width / 1024.0;
    }

     
    template<class VC> void math_stroke<VC>::miter_limit_theta(double t)
    { 
        m_miter_limit = 1.0 / std::sin(t * 0.5) ;
    }

     
    template<class VC> 
    void math_stroke<VC>::calc_arc(VC& vc,
                                   double x,   double y, 
                                   double dx1, double dy1, 
                                   double dx2, double dy2)
    {
        double a1 = std::atan2(dy1 * m_width_sign, dx1 * m_width_sign);
        double a2 = std::atan2(dy2 * m_width_sign, dx2 * m_width_sign);
        double da = std::acos(m_width_abs / (m_width_abs + 0.125 / m_approx_scale)) * 2;
        int i, n;

        add_vertex(vc, x + dx1, y + dy1);
        if(m_width_sign > 0)
        {
            if(a1 > a2) a2 += 2 * pi;
            n = int((a2 - a1) / da);
            da = (a2 - a1) / (n + 1);
            a1 += da;
            for(i = 0; i < n; i++)
            {
                add_vertex(vc, x + std::cos(a1) * m_width, y + std::sin(a1) * m_width);
                a1 += da;
            }
        }
        else
        {
            if(a1 < a2) a2 -= 2 * pi;
            n = int((a1 - a2) / da);
            da = (a1 - a2) / (n + 1);
            a1 -= da;
            for(i = 0; i < n; i++)
            {
                add_vertex(vc, x + std::cos(a1) * m_width, y + std::sin(a1) * m_width);
                a1 -= da;
            }
        }
        add_vertex(vc, x + dx2, y + dy2);
    }

     
    template<class VC> 
    void math_stroke<VC>::calc_miter(VC& vc,
                                     const vertex_dist& v0, 
                                     const vertex_dist& v1, 
                                     const vertex_dist& v2,
                                     double dx1, double dy1, 
                                     double dx2, double dy2,
                                     line_join_e lj,
                                     double mlimit,
                                     double dbevel)
    {
        double xi  = v1.x;
        double yi  = v1.y;
        double di  = 1;
        double lim = m_width_abs * mlimit;
        bool miter_limit_exceeded = true;  
        bool intersection_failed  = true;  

        if(calc_intersection(v0.x + dx1, v0.y - dy1,
                             v1.x + dx1, v1.y - dy1,
                             v1.x + dx2, v1.y - dy2,
                             v2.x + dx2, v2.y - dy2,
                             &xi, &yi))
        {
             
             
            di = calc_distance(v1.x, v1.y, xi, yi);
            if(di <= lim)
            {
                 
                 
                add_vertex(vc, xi, yi);
                miter_limit_exceeded = false;
            }
            intersection_failed = false;
        }
        else
        {
             
             
             
             
             
             
             
             
            double x2 = v1.x + dx1;
            double y2 = v1.y - dy1;
            if((cross_product(v0.x, v0.y, v1.x, v1.y, x2, y2) < 0.0) == 
               (cross_product(v1.x, v1.y, v2.x, v2.y, x2, y2) < 0.0))
            {
                 
                 
                 
                add_vertex(vc, v1.x + dx1, v1.y - dy1);
                miter_limit_exceeded = false;
            }
        }

        if(miter_limit_exceeded)
        {
             
             
            switch(lj)
            {
            case miter_join_revert:
                 
                 
                 
                 
                add_vertex(vc, v1.x + dx1, v1.y - dy1);
                add_vertex(vc, v1.x + dx2, v1.y - dy2);
                break;

            case miter_join_round:
                calc_arc(vc, v1.x, v1.y, dx1, -dy1, dx2, -dy2);
                break;

            default:
                 
                 
                if(intersection_failed)
                {
                    mlimit *= m_width_sign;
                    add_vertex(vc, v1.x + dx1 + dy1 * mlimit, 
                                   v1.y - dy1 + dx1 * mlimit);
                    add_vertex(vc, v1.x + dx2 - dy2 * mlimit, 
                                   v1.y - dy2 - dx2 * mlimit);
                }
                else
                {
                    double x1 = v1.x + dx1;
                    double y1 = v1.y - dy1;
                    double x2 = v1.x + dx2;
                    double y2 = v1.y - dy2;
                    di = (lim - dbevel) / (di - dbevel);
                    add_vertex(vc, x1 + (xi - x1) * di, 
                                   y1 + (yi - y1) * di);
                    add_vertex(vc, x2 + (xi - x2) * di, 
                                   y2 + (yi - y2) * di);
                }
                break;
            }
        }
    }

     
    template<class VC> 
    void math_stroke<VC>::calc_cap(VC& vc,
                                   const vertex_dist& v0, 
                                   const vertex_dist& v1, 
                                   double len)
    {
        vc.remove_all();

        double dx1 = (v1.y - v0.y) / len;
        double dy1 = (v1.x - v0.x) / len;
        double dx2 = 0;
        double dy2 = 0;

        dx1 *= m_width;
        dy1 *= m_width;

        if(m_line_cap != round_cap)
        {
            if(m_line_cap == square_cap)
            {
                dx2 = dy1 * m_width_sign;
                dy2 = dx1 * m_width_sign;
            }
            add_vertex(vc, v0.x - dx1 - dx2, v0.y + dy1 - dy2);
            add_vertex(vc, v0.x + dx1 - dx2, v0.y - dy1 - dy2);
        }
        else
        {
            double da = std::acos(m_width_abs / (m_width_abs + 0.125 / m_approx_scale)) * 2;
            double a1;
            int i;
            int n = int(pi / da);

            da = pi / (n + 1);
            add_vertex(vc, v0.x - dx1, v0.y + dy1);
            if(m_width_sign > 0)
            {
                a1 = std::atan2(dy1, -dx1);
                a1 += da;
                for(i = 0; i < n; i++)
                {
                    add_vertex(vc, v0.x + std::cos(a1) * m_width, 
                                   v0.y + std::sin(a1) * m_width);
                    a1 += da;
                }
            }
            else
            {
                a1 = std::atan2(-dy1, dx1);
                a1 -= da;
                for(i = 0; i < n; i++)
                {
                    add_vertex(vc, v0.x + std::cos(a1) * m_width, 
                                   v0.y + std::sin(a1) * m_width);
                    a1 -= da;
                }
            }
            add_vertex(vc, v0.x + dx1, v0.y - dy1);
        }
    }

     
    template<class VC> 
    void math_stroke<VC>::calc_join(VC& vc,
                                    const vertex_dist& v0, 
                                    const vertex_dist& v1, 
                                    const vertex_dist& v2,
                                    double len1, 
                                    double len2)
    {
        double dx1 = m_width * (v1.y - v0.y) / len1;
        double dy1 = m_width * (v1.x - v0.x) / len1;
        double dx2 = m_width * (v2.y - v1.y) / len2;
        double dy2 = m_width * (v2.x - v1.x) / len2;

        vc.remove_all();

        double cp = cross_product(v0.x, v0.y, v1.x, v1.y, v2.x, v2.y);
        if(cp != 0 && (cp > 0) == (m_width > 0))
        {
             
             
            double limit = ((len1 < len2) ? len1 : len2) / m_width_abs;
            if(limit < m_inner_miter_limit)
            {
                limit = m_inner_miter_limit;
            }

            switch(m_inner_join)
            {
            default:  
                add_vertex(vc, v1.x + dx1, v1.y - dy1);
                add_vertex(vc, v1.x + dx2, v1.y - dy2);
                break;

            case inner_miter:
                calc_miter(vc,
                           v0, v1, v2, dx1, dy1, dx2, dy2, 
                           miter_join_revert, 
                           limit, 0);
                break;

            case inner_jag:
            case inner_round:
                cp = (dx1-dx2) * (dx1-dx2) + (dy1-dy2) * (dy1-dy2);
                if(cp < len1 * len1 && cp < len2 * len2)
                {
                    calc_miter(vc,
                               v0, v1, v2, dx1, dy1, dx2, dy2, 
                               miter_join_revert, 
                               limit, 0);
                }
                else
                {
                    if(m_inner_join == inner_jag)
                    {
                        add_vertex(vc, v1.x + dx1, v1.y - dy1);
                        add_vertex(vc, v1.x,       v1.y      );
                        add_vertex(vc, v1.x + dx2, v1.y - dy2);
                    }
                    else
                    {
                        add_vertex(vc, v1.x + dx1, v1.y - dy1);
                        add_vertex(vc, v1.x,       v1.y      );
                        calc_arc(vc, v1.x, v1.y, dx2, -dy2, dx1, -dy1);
                        add_vertex(vc, v1.x,       v1.y      );
                        add_vertex(vc, v1.x + dx2, v1.y - dy2);
                    }
                }
                break;
            }
        }
        else
        {
             
             

             
             
             
            double dx = (dx1 + dx2) / 2;
            double dy = (dy1 + dy2) / 2;
            double dbevel = std::sqrt(dx * dx + dy * dy);

            if(m_line_join == round_join || m_line_join == bevel_join)
            {
                 
                 
                 
                 
                 
                 
                 
                 
                 
                 
                 
                 
                 
                 
                 
                 
                 
                if(m_approx_scale * (m_width_abs - dbevel) < m_width_eps)
                {
                    if(calc_intersection(v0.x + dx1, v0.y - dy1,
                                         v1.x + dx1, v1.y - dy1,
                                         v1.x + dx2, v1.y - dy2,
                                         v2.x + dx2, v2.y - dy2,
                                         &dx, &dy))
                    {
                        add_vertex(vc, dx, dy);
                    }
                    else
                    {
                        add_vertex(vc, v1.x + dx1, v1.y - dy1);
                    }
                    return;
                }
            }

            switch(m_line_join)
            {
            case miter_join:
            case miter_join_revert:
            case miter_join_round:
                calc_miter(vc, 
                           v0, v1, v2, dx1, dy1, dx2, dy2, 
                           m_line_join, 
                           m_miter_limit,
                           dbevel);
                break;

            case round_join:
                calc_arc(vc, v1.x, v1.y, dx1, -dy1, dx2, -dy2);
                break;

            default:  
                add_vertex(vc, v1.x + dx1, v1.y - dy1);
                add_vertex(vc, v1.x + dx2, v1.y - dy2);
                break;
            }
        }
    }




}

#endif

namespace agg
{

     
     
     
     
    class vcgen_contour
    {
        enum status_e
        {
            initial,
            ready,
            outline,
            out_vertices,
            end_poly,
            stop
        };

    public:
        typedef vertex_sequence<vertex_dist, 6> vertex_storage;
        typedef pod_bvector<point_d, 6>         coord_storage;

        vcgen_contour();

        void line_cap(line_cap_e lc)     { m_stroker.line_cap(lc); }
        void line_join(line_join_e lj)   { m_stroker.line_join(lj); }
        void inner_join(inner_join_e ij) { m_stroker.inner_join(ij); }

        line_cap_e   line_cap()   const { return m_stroker.line_cap(); }
        line_join_e  line_join()  const { return m_stroker.line_join(); }
        inner_join_e inner_join() const { return m_stroker.inner_join(); }

        void width(double w) { m_stroker.width(m_width = w); }
        void miter_limit(double ml) { m_stroker.miter_limit(ml); }
        void miter_limit_theta(double t) { m_stroker.miter_limit_theta(t); }
        void inner_miter_limit(double ml) { m_stroker.inner_miter_limit(ml); }
        void approximation_scale(double as) { m_stroker.approximation_scale(as); }

        double width() const { return m_width; }
        double miter_limit() const { return m_stroker.miter_limit(); }
        double inner_miter_limit() const { return m_stroker.inner_miter_limit(); }
        double approximation_scale() const { return m_stroker.approximation_scale(); }

        void auto_detect_orientation(bool v) { m_auto_detect = v; }
        bool auto_detect_orientation() const { return m_auto_detect; }

         
        void remove_all();
        void add_vertex(double x, double y, unsigned cmd);

         
        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        vcgen_contour(const vcgen_contour&);
        const vcgen_contour& operator = (const vcgen_contour&);

        math_stroke<coord_storage> m_stroker;
        double                     m_width;
        vertex_storage             m_src_vertices;
        coord_storage              m_out_vertices;
        status_e                   m_status;
        unsigned                   m_src_vertex;
        unsigned                   m_out_vertex;
        unsigned                   m_closed;
        unsigned                   m_orientation;
        bool                       m_auto_detect;
    };

}

#endif

namespace agg
{

     
    template<class VertexSource> 
    struct conv_contour : public conv_adaptor_vcgen<VertexSource, vcgen_contour>
    {
        typedef conv_adaptor_vcgen<VertexSource, vcgen_contour> base_type;

        conv_contour(VertexSource& vs) : 
            conv_adaptor_vcgen<VertexSource, vcgen_contour>(vs)
        {
        }

        void line_join(line_join_e lj) { base_type::generator().line_join(lj); }
        void inner_join(inner_join_e ij) { base_type::generator().inner_join(ij); }
        void width(double w) { base_type::generator().width(w); }
        void miter_limit(double ml) { base_type::generator().miter_limit(ml); }
        void miter_limit_theta(double t) { base_type::generator().miter_limit_theta(t); }
        void inner_miter_limit(double ml) { base_type::generator().inner_miter_limit(ml); }
        void approximation_scale(double as) { base_type::generator().approximation_scale(as); }
        void auto_detect_orientation(bool v) { base_type::generator().auto_detect_orientation(v); }

        line_join_e line_join() const { return base_type::generator().line_join(); }
        inner_join_e inner_join() const { return base_type::generator().inner_join(); }
        double width() const { return base_type::generator().width(); }
        double miter_limit() const { return base_type::generator().miter_limit(); }
        double inner_miter_limit() const { return base_type::generator().inner_miter_limit(); }
        double approximation_scale() const { return base_type::generator().approximation_scale(); }
        bool auto_detect_orientation() const { return base_type::generator().auto_detect_orientation(); }

    private:
        conv_contour(const conv_contour<VertexSource>&);
        const conv_contour<VertexSource>& 
            operator = (const conv_contour<VertexSource>&);
    };

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_CURVE_INCLUDED
#define AGG_CONV_CURVE_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CURVES_INCLUDED
#define AGG_CURVES_INCLUDED


namespace agg
{

     

     
    enum curve_approximation_method_e
    {
        curve_inc,
        curve_div
    };
    
     
    class curve3_inc
    {
    public:
        curve3_inc() :
          m_num_steps(0), m_step(0), m_scale(1.0) { }

        curve3_inc(double x1, double y1, 
                   double x2, double y2, 
                   double x3, double y3) :
            m_num_steps(0), m_step(0), m_scale(1.0) 
        { 
            init(x1, y1, x2, y2, x3, y3);
        }

        void reset() { m_num_steps = 0; m_step = -1; }
        void init(double x1, double y1, 
                  double x2, double y2, 
                  double x3, double y3);

        void approximation_method(curve_approximation_method_e) {}
        curve_approximation_method_e approximation_method() const { return curve_inc; }

        void approximation_scale(double s);
        double approximation_scale() const;

        void angle_tolerance(double) {}
        double angle_tolerance() const { return 0.0; }

        void cusp_limit(double) {}
        double cusp_limit() const { return 0.0; }

        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        int      m_num_steps;
        int      m_step;
        double   m_scale;
        double   m_start_x; 
        double   m_start_y;
        double   m_end_x; 
        double   m_end_y;
        double   m_fx; 
        double   m_fy;
        double   m_dfx; 
        double   m_dfy;
        double   m_ddfx; 
        double   m_ddfy;
        double   m_saved_fx; 
        double   m_saved_fy;
        double   m_saved_dfx; 
        double   m_saved_dfy;
    };





     
    class curve3_div
    {
    public:
        curve3_div() : 
            m_approximation_scale(1.0),
            m_distance_tolerance_square(0.0),
            m_angle_tolerance(0.0),
            m_count(0)
        {}

        curve3_div(double x1, double y1, 
                   double x2, double y2, 
                   double x3, double y3) :
            m_approximation_scale(1.0),
            m_angle_tolerance(0.0),
            m_count(0)
        { 
            init(x1, y1, x2, y2, x3, y3);
        }

        void reset() { m_points.remove_all(); m_count = 0; }
        void init(double x1, double y1, 
                  double x2, double y2, 
                  double x3, double y3);

        void approximation_method(curve_approximation_method_e) {}
        curve_approximation_method_e approximation_method() const { return curve_div; }

        void approximation_scale(double s) { m_approximation_scale = s; }
        double approximation_scale() const { return m_approximation_scale;  }

        void angle_tolerance(double a) { m_angle_tolerance = a; }
        double angle_tolerance() const { return m_angle_tolerance;  }

        void cusp_limit(double) {}
        double cusp_limit() const { return 0.0; }

        void rewind(unsigned)
        {
            m_count = 0;
        }

        unsigned vertex(double* x, double* y)
        {
            if(m_count >= m_points.size()) return path_cmd_stop;
            const point_d& p = m_points[m_count++];
            *x = p.x;
            *y = p.y;
            return (m_count == 1) ? path_cmd_move_to : path_cmd_line_to;
        }

    private:
        void bezier(double x1, double y1, 
                    double x2, double y2, 
                    double x3, double y3);
        void recursive_bezier(double x1, double y1, 
                              double x2, double y2, 
                              double x3, double y3,
                              unsigned level);

        double               m_approximation_scale;
        double               m_distance_tolerance_square;
        double               m_angle_tolerance;
        unsigned             m_count;
        pod_bvector<point_d> m_points;
    };







     
    struct curve4_points
    {
        double cp[8];
        curve4_points() {}
        curve4_points(double x1, double y1,
                      double x2, double y2,
                      double x3, double y3,
                      double x4, double y4)
        {
            cp[0] = x1; cp[1] = y1; cp[2] = x2; cp[3] = y2;
            cp[4] = x3; cp[5] = y3; cp[6] = x4; cp[7] = y4;
        }
        void init(double x1, double y1,
                  double x2, double y2,
                  double x3, double y3,
                  double x4, double y4)
        {
            cp[0] = x1; cp[1] = y1; cp[2] = x2; cp[3] = y2;
            cp[4] = x3; cp[5] = y3; cp[6] = x4; cp[7] = y4;
        }
        double  operator [] (unsigned i) const { return cp[i]; }
        double& operator [] (unsigned i)       { return cp[i]; }
    };



     
    class curve4_inc
    {
    public:
        curve4_inc() :
            m_num_steps(0), m_step(0), m_scale(1.0) { }

        curve4_inc(double x1, double y1, 
                   double x2, double y2, 
                   double x3, double y3,
                   double x4, double y4) :
            m_num_steps(0), m_step(0), m_scale(1.0) 
        { 
            init(x1, y1, x2, y2, x3, y3, x4, y4);
        }

        curve4_inc(const curve4_points& cp) :
            m_num_steps(0), m_step(0), m_scale(1.0) 
        { 
            init(cp[0], cp[1], cp[2], cp[3], cp[4], cp[5], cp[6], cp[7]);
        }

        void reset() { m_num_steps = 0; m_step = -1; }
        void init(double x1, double y1, 
                  double x2, double y2, 
                  double x3, double y3,
                  double x4, double y4);

        void init(const curve4_points& cp)
        {
            init(cp[0], cp[1], cp[2], cp[3], cp[4], cp[5], cp[6], cp[7]);
        }

        void approximation_method(curve_approximation_method_e) {}
        curve_approximation_method_e approximation_method() const { return curve_inc; }

        void approximation_scale(double s);
        double approximation_scale() const;

        void angle_tolerance(double) {}
        double angle_tolerance() const { return 0.0; }

        void cusp_limit(double) {}
        double cusp_limit() const { return 0.0; }

        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        int      m_num_steps;
        int      m_step;
        double   m_scale;
        double   m_start_x; 
        double   m_start_y;
        double   m_end_x; 
        double   m_end_y;
        double   m_fx; 
        double   m_fy;
        double   m_dfx; 
        double   m_dfy;
        double   m_ddfx; 
        double   m_ddfy;
        double   m_dddfx; 
        double   m_dddfy;
        double   m_saved_fx; 
        double   m_saved_fy;
        double   m_saved_dfx; 
        double   m_saved_dfy;
        double   m_saved_ddfx; 
        double   m_saved_ddfy;
    };



     
    inline curve4_points catrom_to_bezier(double x1, double y1, 
                                          double x2, double y2, 
                                          double x3, double y3,
                                          double x4, double y4)
    {
         
         
         
         
         
         
         
        return curve4_points(
            x2,
            y2,
            (-x1 + 6*x2 + x3) / 6,
            (-y1 + 6*y2 + y3) / 6,
            ( x2 + 6*x3 - x4) / 6,
            ( y2 + 6*y3 - y4) / 6,
            x3,
            y3);
    }


     
    inline curve4_points
    catrom_to_bezier(const curve4_points& cp)
    {
        return catrom_to_bezier(cp[0], cp[1], cp[2], cp[3], 
                                cp[4], cp[5], cp[6], cp[7]);
    }



     
    inline curve4_points ubspline_to_bezier(double x1, double y1, 
                                            double x2, double y2, 
                                            double x3, double y3,
                                            double x4, double y4)
    {
         
         
         
         
         
         
         
        return curve4_points(
            (x1 + 4*x2 + x3) / 6,
            (y1 + 4*y2 + y3) / 6,
            (4*x2 + 2*x3) / 6,
            (4*y2 + 2*y3) / 6,
            (2*x2 + 4*x3) / 6,
            (2*y2 + 4*y3) / 6,
            (x2 + 4*x3 + x4) / 6,
            (y2 + 4*y3 + y4) / 6);
    }


     
    inline curve4_points 
    ubspline_to_bezier(const curve4_points& cp)
    {
        return ubspline_to_bezier(cp[0], cp[1], cp[2], cp[3], 
                                  cp[4], cp[5], cp[6], cp[7]);
    }




     
    inline curve4_points hermite_to_bezier(double x1, double y1, 
                                           double x2, double y2, 
                                           double x3, double y3,
                                           double x4, double y4)
    {
         
         
         
         
         
         
         
        return curve4_points(
            x1,
            y1,
            (3*x1 + x3) / 3,
            (3*y1 + y3) / 3,
            (3*x2 - x4) / 3,
            (3*y2 - y4) / 3,
            x2,
            y2);
    }



     
    inline curve4_points 
    hermite_to_bezier(const curve4_points& cp)
    {
        return hermite_to_bezier(cp[0], cp[1], cp[2], cp[3], 
                                 cp[4], cp[5], cp[6], cp[7]);
    }


     
    class curve4_div
    {
    public:
        curve4_div() : 
            m_approximation_scale(1.0),
            m_distance_tolerance_square(0.0),
            m_angle_tolerance(0.0),
            m_cusp_limit(0.0),
            m_count(0)
        {}

        curve4_div(double x1, double y1, 
                   double x2, double y2, 
                   double x3, double y3,
                   double x4, double y4) :
            m_approximation_scale(1.0),
            m_angle_tolerance(0.0),
            m_cusp_limit(0.0),
            m_count(0)
        { 
            init(x1, y1, x2, y2, x3, y3, x4, y4);
        }

        curve4_div(const curve4_points& cp) :
            m_approximation_scale(1.0),
            m_angle_tolerance(0.0),
            m_count(0)
        { 
            init(cp[0], cp[1], cp[2], cp[3], cp[4], cp[5], cp[6], cp[7]);
        }

        void reset() { m_points.remove_all(); m_count = 0; }
        void init(double x1, double y1, 
                  double x2, double y2, 
                  double x3, double y3,
                  double x4, double y4);

        void init(const curve4_points& cp)
        {
            init(cp[0], cp[1], cp[2], cp[3], cp[4], cp[5], cp[6], cp[7]);
        }

        void approximation_method(curve_approximation_method_e) {}

        curve_approximation_method_e approximation_method() const 
        { 
            return curve_div; 
        }

        void approximation_scale(double s) { m_approximation_scale = s; }
        double approximation_scale() const { return m_approximation_scale;  }

        void angle_tolerance(double a) { m_angle_tolerance = a; }
        double angle_tolerance() const { return m_angle_tolerance;  }

        void cusp_limit(double v) 
        { 
            m_cusp_limit = (v == 0.0) ? 0.0 : pi - v; 
        }

        double cusp_limit() const 
        { 
            return (m_cusp_limit == 0.0) ? 0.0 : pi - m_cusp_limit; 
        }

        void rewind(unsigned)
        {
            m_count = 0;
        }

        unsigned vertex(double* x, double* y)
        {
            if(m_count >= m_points.size()) return path_cmd_stop;
            const point_d& p = m_points[m_count++];
            *x = p.x;
            *y = p.y;
            return (m_count == 1) ? path_cmd_move_to : path_cmd_line_to;
        }

    private:
        void bezier(double x1, double y1, 
                    double x2, double y2, 
                    double x3, double y3, 
                    double x4, double y4);

        void recursive_bezier(double x1, double y1, 
                              double x2, double y2, 
                              double x3, double y3, 
                              double x4, double y4,
                              unsigned level);

        double               m_approximation_scale;
        double               m_distance_tolerance_square;
        double               m_angle_tolerance;
        double               m_cusp_limit;
        unsigned             m_count;
        pod_bvector<point_d> m_points;
    };


     
    class curve3
    {
    public:
        curve3() : m_approximation_method(curve_div) {}
        curve3(double x1, double y1, 
               double x2, double y2, 
               double x3, double y3) :
            m_approximation_method(curve_div)
        { 
            init(x1, y1, x2, y2, x3, y3);
        }

        void reset() 
        { 
            m_curve_inc.reset();
            m_curve_div.reset();
        }

        void init(double x1, double y1, 
                  double x2, double y2, 
                  double x3, double y3)
        {
            if(m_approximation_method == curve_inc) 
            {
                m_curve_inc.init(x1, y1, x2, y2, x3, y3);
            }
            else
            {
                m_curve_div.init(x1, y1, x2, y2, x3, y3);
            }
        }

        void approximation_method(curve_approximation_method_e v) 
        { 
            m_approximation_method = v; 
        }

        curve_approximation_method_e approximation_method() const 
        { 
            return m_approximation_method; 
        }

        void approximation_scale(double s) 
        { 
            m_curve_inc.approximation_scale(s);
            m_curve_div.approximation_scale(s);
        }

        double approximation_scale() const 
        { 
            return m_curve_inc.approximation_scale(); 
        }

        void angle_tolerance(double a) 
        { 
            m_curve_div.angle_tolerance(a); 
        }

        double angle_tolerance() const 
        { 
            return m_curve_div.angle_tolerance(); 
        }

        void cusp_limit(double v) 
        { 
            m_curve_div.cusp_limit(v); 
        }

        double cusp_limit() const 
        { 
            return m_curve_div.cusp_limit();  
        }

        void rewind(unsigned path_id)
        {
            if(m_approximation_method == curve_inc) 
            {
                m_curve_inc.rewind(path_id);
            }
            else
            {
                m_curve_div.rewind(path_id);
            }
        }

        unsigned vertex(double* x, double* y)
        {
            if(m_approximation_method == curve_inc) 
            {
                return m_curve_inc.vertex(x, y);
            }
            return m_curve_div.vertex(x, y);
        }

    private:
        curve3_inc m_curve_inc;
        curve3_div m_curve_div;
        curve_approximation_method_e m_approximation_method;
    };





     
    class curve4
    {
    public:
        curve4() : m_approximation_method(curve_div) {}
        curve4(double x1, double y1, 
               double x2, double y2, 
               double x3, double y3,
               double x4, double y4) : 
            m_approximation_method(curve_div)
        { 
            init(x1, y1, x2, y2, x3, y3, x4, y4);
        }

        curve4(const curve4_points& cp) :
            m_approximation_method(curve_div)
        { 
            init(cp[0], cp[1], cp[2], cp[3], cp[4], cp[5], cp[6], cp[7]);
        }

        void reset() 
        { 
            m_curve_inc.reset();
            m_curve_div.reset();
        }

        void init(double x1, double y1, 
                  double x2, double y2, 
                  double x3, double y3,
                  double x4, double y4)
        {
            if(m_approximation_method == curve_inc) 
            {
                m_curve_inc.init(x1, y1, x2, y2, x3, y3, x4, y4);
            }
            else
            {
                m_curve_div.init(x1, y1, x2, y2, x3, y3, x4, y4);
            }
        }

        void init(const curve4_points& cp)
        {
            init(cp[0], cp[1], cp[2], cp[3], cp[4], cp[5], cp[6], cp[7]);
        }

        void approximation_method(curve_approximation_method_e v) 
        { 
            m_approximation_method = v; 
        }

        curve_approximation_method_e approximation_method() const 
        { 
            return m_approximation_method; 
        }

        void approximation_scale(double s) 
        { 
            m_curve_inc.approximation_scale(s);
            m_curve_div.approximation_scale(s);
        }
        double approximation_scale() const { return m_curve_inc.approximation_scale(); }

        void angle_tolerance(double v) 
        { 
            m_curve_div.angle_tolerance(v); 
        }

        double angle_tolerance() const 
        { 
            return m_curve_div.angle_tolerance();  
        }

        void cusp_limit(double v) 
        { 
            m_curve_div.cusp_limit(v); 
        }

        double cusp_limit() const 
        { 
            return m_curve_div.cusp_limit();  
        }

        void rewind(unsigned path_id)
        {
            if(m_approximation_method == curve_inc) 
            {
                m_curve_inc.rewind(path_id);
            }
            else
            {
                m_curve_div.rewind(path_id);
            }
        }

        unsigned vertex(double* x, double* y)
        {
            if(m_approximation_method == curve_inc) 
            {
                return m_curve_inc.vertex(x, y);
            }
            return m_curve_div.vertex(x, y);
        }

    private:
        curve4_inc m_curve_inc;
        curve4_div m_curve_div;
        curve_approximation_method_e m_approximation_method;
    };




}

#endif

namespace agg
{


     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
    template<class VertexSource, 
             class Curve3=curve3, 
             class Curve4=curve4> class conv_curve
    {
    public:
        typedef Curve3 curve3_type;
        typedef Curve4 curve4_type;
        typedef conv_curve<VertexSource, Curve3, Curve4> self_type;

        explicit conv_curve(VertexSource& source) :
          m_source(&source), m_last_x(0.0), m_last_y(0.0) {}
        void attach(VertexSource& source) { m_source = &source; }

        void approximation_method(curve_approximation_method_e v) 
        { 
            m_curve3.approximation_method(v);
            m_curve4.approximation_method(v);
        }

        curve_approximation_method_e approximation_method() const 
        { 
            return m_curve4.approximation_method();
        }

        void approximation_scale(double s) 
        { 
            m_curve3.approximation_scale(s); 
            m_curve4.approximation_scale(s); 
        }

        double approximation_scale() const 
        { 
            return m_curve4.approximation_scale();  
        }

        void angle_tolerance(double v) 
        { 
            m_curve3.angle_tolerance(v); 
            m_curve4.angle_tolerance(v); 
        }

        double angle_tolerance() const 
        { 
            return m_curve4.angle_tolerance();  
        }

        void cusp_limit(double v) 
        { 
            m_curve3.cusp_limit(v); 
            m_curve4.cusp_limit(v); 
        }

        double cusp_limit() const 
        { 
            return m_curve4.cusp_limit();  
        }

        void     rewind(unsigned path_id); 
        unsigned vertex(double* x, double* y);

    private:
        conv_curve(const self_type&);
        const self_type& operator = (const self_type&);

        VertexSource* m_source;
        double        m_last_x;
        double        m_last_y;
        curve3_type   m_curve3;
        curve4_type   m_curve4;
    };



     
    template<class VertexSource, class Curve3, class Curve4>
    void conv_curve<VertexSource, Curve3, Curve4>::rewind(unsigned path_id)
    {
        m_source->rewind(path_id);
        m_last_x = 0.0;
        m_last_y = 0.0;
        m_curve3.reset();
        m_curve4.reset();
    }


     
    template<class VertexSource, class Curve3, class Curve4>
    unsigned conv_curve<VertexSource, Curve3, Curve4>::vertex(double* x, double* y)
    {
        if(!is_stop(m_curve3.vertex(x, y)))
        {
            m_last_x = *x;
            m_last_y = *y;
            return path_cmd_line_to;
        }

        if(!is_stop(m_curve4.vertex(x, y)))
        {
            m_last_x = *x;
            m_last_y = *y;
            return path_cmd_line_to;
        }

        double ct2_x = 0;
        double ct2_y = 0;
        double end_x = 0;
        double end_y = 0;

        unsigned cmd = m_source->vertex(x, y);
        switch(cmd)
        {
        case path_cmd_curve3:
            m_source->vertex(&end_x, &end_y);

            m_curve3.init(m_last_x, m_last_y, 
                          *x,       *y, 
                          end_x,     end_y);

            m_curve3.vertex(x, y);     
            m_curve3.vertex(x, y);     
            cmd = path_cmd_line_to;
            break;

        case path_cmd_curve4:
            m_source->vertex(&ct2_x, &ct2_y);
            m_source->vertex(&end_x, &end_y);

            m_curve4.init(m_last_x, m_last_y, 
                          *x,       *y, 
                          ct2_x,    ct2_y, 
                          end_x,    end_y);

            m_curve4.vertex(x, y);     
            m_curve4.vertex(x, y);     
            cmd = path_cmd_line_to;
            break;
        }
        m_last_x = *x;
        m_last_y = *y;
        return cmd;
    }


}



#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CONV_DASH_INCLUDED
#define AGG_CONV_DASH_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_VCGEN_DASH_INCLUDED
#define AGG_VCGEN_DASH_INCLUDED


namespace agg
{

     
     
     
     
    class vcgen_dash
    {
        enum max_dashes_e
        {
            max_dashes = 32
        };

        enum status_e
        {
            initial,
            ready,
            polyline,
            stop
        };

    public:
        typedef vertex_sequence<vertex_dist, 6> vertex_storage;

        vcgen_dash();

        void remove_all_dashes();
        void add_dash(double dash_len, double gap_len);
        void dash_start(double ds);

        void shorten(double s) { m_shorten = s; }
        double shorten() const { return m_shorten; }

         
        void remove_all();
        void add_vertex(double x, double y, unsigned cmd);

         
        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        vcgen_dash(const vcgen_dash&);
        const vcgen_dash& operator = (const vcgen_dash&);

        void calc_dash_start(double ds);

        double             m_dashes[max_dashes];
        double             m_total_dash_len;
        unsigned           m_num_dashes;
        double             m_dash_start;
        double             m_shorten;
        double             m_curr_dash_start;
        unsigned           m_curr_dash;
        double             m_curr_rest;
        const vertex_dist* m_v1;
        const vertex_dist* m_v2;

        vertex_storage m_src_vertices;
        unsigned       m_closed;
        status_e       m_status;
        unsigned       m_src_vertex;
    };


}

#endif

namespace agg
{

     
    template<class VertexSource, class Markers=null_markers> 
    struct conv_dash : public conv_adaptor_vcgen<VertexSource, vcgen_dash, Markers>
    {
        typedef Markers marker_type;
        typedef conv_adaptor_vcgen<VertexSource, vcgen_dash, Markers> base_type;

        conv_dash(VertexSource& vs) : 
            conv_adaptor_vcgen<VertexSource, vcgen_dash, Markers>(vs)
        {
        }

        void remove_all_dashes() 
        { 
            base_type::generator().remove_all_dashes(); 
        }

        void add_dash(double dash_len, double gap_len) 
        { 
            base_type::generator().add_dash(dash_len, gap_len); 
        }

        void dash_start(double ds) 
        { 
            base_type::generator().dash_start(ds); 
        }

        void shorten(double s) { base_type::generator().shorten(s); }
        double shorten() const { return base_type::generator().shorten(); }

    private:
        conv_dash(const conv_dash<VertexSource, Markers>&);
        const conv_dash<VertexSource, Markers>& 
            operator = (const conv_dash<VertexSource, Markers>&);
    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_GPC_INCLUDED
#define AGG_CONV_GPC_INCLUDED

#include <cstring>

extern "C" 
{ 
 

#ifndef __gpc_h
#define __gpc_h

#include <stdio.h>


 

 

#define GPC_EPSILON (DBL_EPSILON)

#define GPC_VERSION "2.33"


 

typedef enum                         
{
  GPC_DIFF,                          
  GPC_INT,                           
  GPC_XOR,                           
  GPC_UNION                          
} gpc_op;

typedef struct                       
{
  double              x;             
  double              y;             
} gpc_vertex;

typedef struct                       
{
  int                 num_vertices;  
  gpc_vertex         *vertex;        
} gpc_vertex_list;

typedef struct                       
{
  int                 num_contours;  
  int                *hole;          
  gpc_vertex_list    *contour;       
} gpc_polygon;

typedef struct                       
{
  int                 num_strips;    
  gpc_vertex_list    *strip;         
} gpc_tristrip;


 

void gpc_read_polygon        (FILE            *infile_ptr, 
                              int              read_hole_flags,
                              gpc_polygon     *polygon);

void gpc_write_polygon       (FILE            *outfile_ptr,
                              int              write_hole_flags,
                              gpc_polygon     *polygon);

void gpc_add_contour         (gpc_polygon     *polygon,
                              gpc_vertex_list *contour,
                              int              hole);

void gpc_polygon_clip        (gpc_op           set_operation,
                              gpc_polygon     *subject_polygon,
                              gpc_polygon     *clip_polygon,
                              gpc_polygon     *result_polygon);

void gpc_tristrip_clip       (gpc_op           set_operation,
                              gpc_polygon     *subject_polygon,
                              gpc_polygon     *clip_polygon,
                              gpc_tristrip    *result_tristrip);

void gpc_polygon_to_tristrip (gpc_polygon     *polygon,
                              gpc_tristrip    *tristrip);

void gpc_free_polygon        (gpc_polygon     *polygon);

void gpc_free_tristrip       (gpc_tristrip    *tristrip);

#endif

 
}

namespace agg
{
    enum gpc_op_e
    {
        gpc_or,
        gpc_and,
        gpc_xor,
        gpc_a_minus_b,
        gpc_b_minus_a
    };


     
    template<class VSA, class VSB> class conv_gpc
    {
        enum status
        {
            status_move_to,
            status_line_to,
            status_stop
        };

        struct contour_header_type
        {
            int num_vertices;
            int hole_flag;
            gpc_vertex* vertices;
        };

        typedef pod_bvector<gpc_vertex, 8>          vertex_array_type;
        typedef pod_bvector<contour_header_type, 6> contour_header_array_type;


    public:
        typedef VSA source_a_type;
        typedef VSB source_b_type;
        typedef conv_gpc<source_a_type, source_b_type> self_type;

        ~conv_gpc()
        {
            free_gpc_data();
        }

        conv_gpc(source_a_type& a, source_b_type& b, gpc_op_e op = gpc_or) :
            m_src_a(&a),
            m_src_b(&b),
            m_status(status_move_to),
            m_vertex(-1),
            m_contour(-1),
            m_operation(op)
        {
            std::memset(&m_poly_a, 0, sizeof(m_poly_a));
            std::memset(&m_poly_b, 0, sizeof(m_poly_b));
            std::memset(&m_result, 0, sizeof(m_result));
        }

        void attach1(VSA& source) { m_src_a = &source; }
        void attach2(VSB& source) { m_src_b = &source; }

        void operation(gpc_op_e v) { m_operation = v; }

         
        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        conv_gpc(const conv_gpc<VSA, VSB>&);
        const conv_gpc<VSA, VSB>& operator = (const conv_gpc<VSA, VSB>&);

         
        void free_polygon(gpc_polygon& p);
        void free_result();
        void free_gpc_data();
        void start_contour();
        void add_vertex(double x, double y);
        void end_contour(unsigned orientation);
        void make_polygon(gpc_polygon& p);
        void start_extracting();
        bool next_contour();
        bool next_vertex(double* x, double* y);


         
        template<class VS> void add(VS& src, gpc_polygon& p)
        {
            unsigned cmd;
            double x, y;
            double start_x = 0.0;
            double start_y = 0.0;
            bool line_to = false;
            unsigned orientation = 0;

            m_contour_accumulator.remove_all();

            while(!is_stop(cmd = src.vertex(&x, &y)))
            {
                if(is_vertex(cmd))
                {
                    if(is_move_to(cmd))
                    {
                        if(line_to)
                        {
                            end_contour(orientation);
                            orientation = 0;
                        }
                        start_contour();
                        start_x = x;
                        start_y = y;
                    }
                    add_vertex(x, y);
                    line_to = true;
                }
                else
                {
                    if(is_end_poly(cmd))
                    {
                        orientation = get_orientation(cmd);
                        if(line_to && is_closed(cmd))
                        {
                            add_vertex(start_x, start_y);
                        }
                    }
                }
            }
            if(line_to)
            {
                end_contour(orientation);
            }
            make_polygon(p);
        }


    private:
         
        source_a_type*             m_src_a;
        source_b_type*             m_src_b;
        status                     m_status;
        int                        m_vertex;
        int                        m_contour;
        gpc_op_e                   m_operation;
        vertex_array_type          m_vertex_accumulator;
        contour_header_array_type  m_contour_accumulator;
        gpc_polygon                m_poly_a;
        gpc_polygon                m_poly_b;
        gpc_polygon                m_result;
    };





     
    template<class VSA, class VSB> 
    void conv_gpc<VSA, VSB>::free_polygon(gpc_polygon& p)
    {
        int i;
        for(i = 0; i < p.num_contours; i++)
        {
            pod_allocator<gpc_vertex>::deallocate(p.contour[i].vertex, 
                                                  p.contour[i].num_vertices);
        }
        pod_allocator<gpc_vertex_list>::deallocate(p.contour, p.num_contours);
        std::memset(&p, 0, sizeof(gpc_polygon));
    }


     
    template<class VSA, class VSB> 
    void conv_gpc<VSA, VSB>::free_result()
    {
        if(m_result.contour)
        {
            gpc_free_polygon(&m_result);
        }
        std::memset(&m_result, 0, sizeof(m_result));
    }


     
    template<class VSA, class VSB> 
    void conv_gpc<VSA, VSB>::free_gpc_data()
    {
        free_polygon(m_poly_a);
        free_polygon(m_poly_b);
        free_result();
    }


     
    template<class VSA, class VSB> 
    void conv_gpc<VSA, VSB>::start_contour()
    {
        contour_header_type h;
        std::memset(&h, 0, sizeof(h));
        m_contour_accumulator.add(h);
        m_vertex_accumulator.remove_all();
    }


     
    template<class VSA, class VSB> 
    inline void conv_gpc<VSA, VSB>::add_vertex(double x, double y)
    {
        gpc_vertex v;
        v.x = x;
        v.y = y;
        m_vertex_accumulator.add(v);
    }


     
    template<class VSA, class VSB> 
    void conv_gpc<VSA, VSB>::end_contour(unsigned  )
    {
        if(m_contour_accumulator.size())
        {
            if(m_vertex_accumulator.size() > 2)
            {
                contour_header_type& h = 
                    m_contour_accumulator[m_contour_accumulator.size() - 1];

                h.num_vertices = m_vertex_accumulator.size();
                h.hole_flag = 0;

                 
                 

                h.vertices = pod_allocator<gpc_vertex>::allocate(h.num_vertices);
                gpc_vertex* d = h.vertices;
                int i;
                for(i = 0; i < h.num_vertices; i++)
                {
                    const gpc_vertex& s = m_vertex_accumulator[i];
                    d->x = s.x;
                    d->y = s.y;
                    ++d;
                }
            }
            else
            {
                m_vertex_accumulator.remove_last();
            }
        }
    }


     
    template<class VSA, class VSB> 
    void conv_gpc<VSA, VSB>::make_polygon(gpc_polygon& p)
    {
        free_polygon(p);
        if(m_contour_accumulator.size())
        {
            p.num_contours = m_contour_accumulator.size();

            p.hole = 0;
            p.contour = pod_allocator<gpc_vertex_list>::allocate(p.num_contours);

            int i;
            gpc_vertex_list* pv = p.contour;
            for(i = 0; i < p.num_contours; i++)
            {
                const contour_header_type& h = m_contour_accumulator[i];
                pv->num_vertices = h.num_vertices;
                pv->vertex = h.vertices;
                ++pv;
            }
        }
    }


     
    template<class VSA, class VSB> 
    void conv_gpc<VSA, VSB>::start_extracting()
    {
        m_status = status_move_to;
        m_contour = -1;
        m_vertex = -1;
    }


     
    template<class VSA, class VSB> 
    bool conv_gpc<VSA, VSB>::next_contour()
    {
        if(++m_contour < m_result.num_contours)
        {
            m_vertex = -1;
            return true;
        }
        return false;
    }


     
    template<class VSA, class VSB> 
    inline bool conv_gpc<VSA, VSB>::next_vertex(double* x, double* y)
    {
        const gpc_vertex_list& vlist = m_result.contour[m_contour];
        if(++m_vertex < vlist.num_vertices)
        {
            const gpc_vertex& v = vlist.vertex[m_vertex];
            *x = v.x;
            *y = v.y;
            return true;
        }
        return false;
    }


     
    template<class VSA, class VSB> 
    void conv_gpc<VSA, VSB>::rewind(unsigned path_id)
    {
        free_result();
        m_src_a->rewind(path_id);
        m_src_b->rewind(path_id);
        add(*m_src_a, m_poly_a);
        add(*m_src_b, m_poly_b);
        switch(m_operation)
        {
           case gpc_or:
                gpc_polygon_clip(GPC_UNION,
                                 &m_poly_a,
                                 &m_poly_b,
                                 &m_result);
               break;

           case gpc_and:
                gpc_polygon_clip(GPC_INT,
                                 &m_poly_a,
                                 &m_poly_b,
                                 &m_result);
               break;

           case gpc_xor:
                gpc_polygon_clip(GPC_XOR,
                                 &m_poly_a,
                                 &m_poly_b,
                                 &m_result);
               break;

           case gpc_a_minus_b:
                gpc_polygon_clip(GPC_DIFF,
                                 &m_poly_a,
                                 &m_poly_b,
                                 &m_result);
               break;

           case gpc_b_minus_a:
                gpc_polygon_clip(GPC_DIFF,
                                 &m_poly_b,
                                 &m_poly_a,
                                 &m_result);
               break;
        }
        start_extracting();
    }


     
    template<class VSA, class VSB> 
    unsigned conv_gpc<VSA, VSB>::vertex(double* x, double* y)
    {
        if(m_status == status_move_to)
        {
            if(next_contour()) 
            {
                if(next_vertex(x, y))
                {
                    m_status = status_line_to;
                    return path_cmd_move_to;
                }
                m_status = status_stop;
                return +path_cmd_end_poly | path_flags_close;
            }
        }
        else
        {
            if(next_vertex(x, y))
            {
                return path_cmd_line_to;
            }
            else
            {
                m_status = status_move_to;
            }
            return +path_cmd_end_poly | path_flags_close;
        }
        return path_cmd_stop;
    }

   
}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_MARKER_ADAPTOR_INCLUDED
#define AGG_CONV_MARKER_ADAPTOR_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_VCGEN_VERTEX_SEQUENCE_INCLUDED
#define AGG_VCGEN_VERTEX_SEQUENCE_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SHORTEN_PATH_INCLUDED
#define AGG_SHORTEN_PATH_INCLUDED


namespace agg
{

     
    template<class VertexSequence> 
    void shorten_path(VertexSequence& vs, double s, unsigned closed = 0)
    {
        typedef typename VertexSequence::value_type vertex_type;

        if(s > 0.0 && vs.size() > 1)
        {
            double d;
            int n = int(vs.size() - 2);
            while(n)
            {
                d = vs[n].dist;
                if(d > s) break;
                vs.remove_last();
                s -= d;
                --n;
            }
            if(vs.size() < 2)
            {
                vs.remove_all();
            }
            else
            {
                n = vs.size() - 1;
                vertex_type& prev = vs[n-1];
                vertex_type& last = vs[n];
                d = (prev.dist - s) / prev.dist;
                double x = prev.x + (last.x - prev.x) * d;
                double y = prev.y + (last.y - prev.y) * d;
                last.x = x;
                last.y = y;
                if(!prev(last)) vs.remove_last();
                vs.close(closed != 0);
            }
        }
    }


}

#endif

namespace agg
{

     
    class vcgen_vertex_sequence
    {
    public:
        typedef vertex_dist_cmd                 vertex_type;
        typedef vertex_sequence<vertex_type, 6> vertex_storage;

        vcgen_vertex_sequence() :
            m_flags(0),
            m_cur_vertex(0),
            m_shorten(0.0),
            m_ready(false)
        {
        }

         
        void remove_all();
        void add_vertex(double x, double y, unsigned cmd);

         
        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

        void shorten(double s) { m_shorten = s; }
        double shorten() const { return m_shorten; }

    private:
        vcgen_vertex_sequence(const vcgen_vertex_sequence&);
        const vcgen_vertex_sequence& operator = (const vcgen_vertex_sequence&);

        vertex_storage m_src_vertices;
        unsigned       m_flags;
        unsigned       m_cur_vertex;
        double         m_shorten;
        bool           m_ready;
    };


     
    inline void vcgen_vertex_sequence::remove_all()
    {
        m_ready = false;
        m_src_vertices.remove_all();
        m_cur_vertex = 0;
        m_flags = 0;
    }

     
    inline void vcgen_vertex_sequence::add_vertex(double x, double y, unsigned cmd)
    {
        m_ready = false;
        if(is_move_to(cmd))
        {
            m_src_vertices.modify_last(vertex_dist_cmd(x, y, cmd));
        }
        else
        {
            if(is_vertex(cmd))
            {
                m_src_vertices.add(vertex_dist_cmd(x, y, cmd));
            }
            else
            {
                m_flags = cmd & path_flags_mask;
            }
        }
    }


     
    inline void vcgen_vertex_sequence::rewind(unsigned) 
    { 
        if(!m_ready)
        {
            m_src_vertices.close(is_closed(m_flags));
            shorten_path(m_src_vertices, m_shorten, get_close_flag(m_flags));
        }
        m_ready = true;
        m_cur_vertex = 0; 
    }

     
    inline unsigned vcgen_vertex_sequence::vertex(double* x, double* y)
    {
        if(!m_ready)
        {
            rewind(0);
        }

        if(m_cur_vertex == m_src_vertices.size())
        {
            ++m_cur_vertex;
            return +path_cmd_end_poly | m_flags;
        }

        if(m_cur_vertex > m_src_vertices.size())
        {
            return path_cmd_stop;
        }

        vertex_type& v = m_src_vertices[m_cur_vertex++];
        *x = v.x;
        *y = v.y;
        return v.cmd;
    }


}

#endif

namespace agg
{

     
    template<class VertexSource, class Markers=null_markers>
    struct conv_marker_adaptor : 
    public conv_adaptor_vcgen<VertexSource, vcgen_vertex_sequence, Markers>
    {
        typedef Markers marker_type;
        typedef conv_adaptor_vcgen<VertexSource, vcgen_vertex_sequence, Markers> base_type;

        conv_marker_adaptor(VertexSource& vs) : 
            conv_adaptor_vcgen<VertexSource, vcgen_vertex_sequence, Markers>(vs)
        {
        }

        void shorten(double s) { base_type::generator().shorten(s); }
        double shorten() const { return base_type::generator().shorten(); }

    private:
        conv_marker_adaptor(const conv_marker_adaptor<VertexSource, Markers>&);
        const conv_marker_adaptor<VertexSource, Markers>& 
            operator = (const conv_marker_adaptor<VertexSource, Markers>&);
    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CONV_MARKER_INCLUDED
#define AGG_CONV_MARKER_INCLUDED

#include <cmath>

namespace agg
{
     
    template<class MarkerLocator, class MarkerShapes>
    class conv_marker
    {
    public:
        conv_marker(MarkerLocator& ml, MarkerShapes& ms);

        trans_affine& transform() { return m_transform; }
        const trans_affine& transform() const { return m_transform; }

        void rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        conv_marker(const conv_marker<MarkerLocator, MarkerShapes>&);
        const conv_marker<MarkerLocator, MarkerShapes>& 
            operator = (const conv_marker<MarkerLocator, MarkerShapes>&);

        enum status_e 
        {
            initial,
            markers,
            polygon,
            stop
        };

        MarkerLocator* m_marker_locator;
        MarkerShapes*  m_marker_shapes;
        trans_affine   m_transform;
        trans_affine   m_mtx;
        status_e       m_status;
        unsigned       m_marker;
        unsigned       m_num_markers;
    };


     
    template<class MarkerLocator, class MarkerShapes> 
    conv_marker<MarkerLocator, MarkerShapes>::conv_marker(MarkerLocator& ml, MarkerShapes& ms) :
        m_marker_locator(&ml),
        m_marker_shapes(&ms),
        m_status(initial),
        m_marker(0),
        m_num_markers(1)
    {
    }


     
    template<class MarkerLocator, class MarkerShapes> 
    void conv_marker<MarkerLocator, MarkerShapes>::rewind(unsigned)
    {
        m_status = initial;
        m_marker = 0;
        m_num_markers = 1;
    }


     
    template<class MarkerLocator, class MarkerShapes> 
    unsigned conv_marker<MarkerLocator, MarkerShapes>::vertex(double* x, double* y)
    {
        unsigned cmd = path_cmd_move_to;
        double x1, y1, x2, y2;

        while(!is_stop(cmd))
        {
            switch(m_status)
            {
            case initial:
                if(m_num_markers == 0)
                {
                   cmd = path_cmd_stop;
                   break;
                }
                m_marker_locator->rewind(m_marker);
                ++m_marker;
                m_num_markers = 0;
                m_status = markers;

            case markers:
                if(is_stop(m_marker_locator->vertex(&x1, &y1)))
                {
                    m_status = initial;
                    break;
                }
                if(is_stop(m_marker_locator->vertex(&x2, &y2)))
                {
                    m_status = initial;
                    break;
                }
                ++m_num_markers;
                m_mtx = m_transform;
                m_mtx *= trans_affine_rotation(std::atan2(y2 - y1, x2 - x1));
                m_mtx *= trans_affine_translation(x1, y1);
                m_marker_shapes->rewind(m_marker - 1);
                m_status = polygon;

            case polygon:
                cmd = m_marker_shapes->vertex(x, y);
                if(is_stop(cmd))
                {
                    cmd = path_cmd_move_to;
                    m_status = markers;
                    break;
                }
                m_mtx.transform(x, y);
                return cmd;

            case stop:
                cmd = path_cmd_stop;
                break;
            }
        }
        return cmd;
    }

}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_SEGMENTATOR_INCLUDED
#define AGG_CONV_SEGMENTATOR_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_VPGEN_SEGMENTATOR_INCLUDED
#define AGG_VPGEN_SEGMENTATOR_INCLUDED


namespace agg
{

     
     
     
     
    class vpgen_segmentator
    {
    public:
        vpgen_segmentator() : m_approximation_scale(1.0) {}

        void approximation_scale(double s) { m_approximation_scale = s;     }
        double approximation_scale() const { return m_approximation_scale;  }

        static bool auto_close()   { return false; }
        static bool auto_unclose() { return false; }

        void reset() { m_cmd = path_cmd_stop; }
        void move_to(double x, double y);
        void line_to(double x, double y);
        unsigned vertex(double* x, double* y);

    private:
        double   m_approximation_scale;
        double   m_x1;
        double   m_y1;
        double   m_dx;
        double   m_dy;
        double   m_dl;
        double   m_ddl;
        unsigned m_cmd;
    };



}

#endif


namespace agg
{

     
    template<class VertexSource> 
    struct conv_segmentator : public conv_adaptor_vpgen<VertexSource, vpgen_segmentator>
    {
        typedef conv_adaptor_vpgen<VertexSource, vpgen_segmentator> base_type;

        conv_segmentator(VertexSource& vs) : 
            conv_adaptor_vpgen<VertexSource, vpgen_segmentator>(vs) {}

        void approximation_scale(double s) { base_type::vpgen().approximation_scale(s);        }
        double approximation_scale() const { return base_type::vpgen().approximation_scale();  }

    private:
        conv_segmentator(const conv_segmentator<VertexSource>&);
        const conv_segmentator<VertexSource>& 
            operator = (const conv_segmentator<VertexSource>&);
    };


}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_SHORTEN_PATH_INCLUDED
#define AGG_CONV_SHORTEN_PATH_INCLUDED


namespace agg
{

     
    template<class VertexSource>  class conv_shorten_path : 
    public conv_adaptor_vcgen<VertexSource, vcgen_vertex_sequence>
    {
    public:
        typedef conv_adaptor_vcgen<VertexSource, vcgen_vertex_sequence> base_type;

        conv_shorten_path(VertexSource& vs) : 
            conv_adaptor_vcgen<VertexSource, vcgen_vertex_sequence>(vs)
        {
        }

        void shorten(double s) { base_type::generator().shorten(s); }
        double shorten() const { return base_type::generator().shorten(); }

    private:
        conv_shorten_path(const conv_shorten_path<VertexSource>&);
        const conv_shorten_path<VertexSource>& 
            operator = (const conv_shorten_path<VertexSource>&);
    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CONV_SMOOTH_POLY1_INCLUDED
#define AGG_CONV_SMOOTH_POLY1_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_VCGEN_SMOOTH_POLY1_INCLUDED
#define AGG_VCGEN_SMOOTH_POLY1_INCLUDED



namespace agg
{

     
     
     
     
     
     
    class vcgen_smooth_poly1
    {
        enum status_e
        {
            initial,
            ready,
            polygon,
            ctrl_b,
            ctrl_e,
            ctrl1,
            ctrl2,
            end_poly,
            stop
        };

    public:
        typedef vertex_sequence<vertex_dist, 6> vertex_storage;

        vcgen_smooth_poly1();

        void   smooth_value(double v) { m_smooth_value = v * 0.5; }
        double smooth_value() const { return m_smooth_value * 2.0; }

         
        void remove_all();
        void add_vertex(double x, double y, unsigned cmd);

         
        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        vcgen_smooth_poly1(const vcgen_smooth_poly1&);
        const vcgen_smooth_poly1& operator = (const vcgen_smooth_poly1&);

        void calculate(const vertex_dist& v0, 
                       const vertex_dist& v1, 
                       const vertex_dist& v2,
                       const vertex_dist& v3);

        vertex_storage m_src_vertices;
        double         m_smooth_value;
        unsigned       m_closed;
        status_e       m_status;
        unsigned       m_src_vertex;
        double         m_ctrl1_x;
        double         m_ctrl1_y;
        double         m_ctrl2_x;
        double         m_ctrl2_y;
    };

}


#endif



namespace agg
{

     
    template<class VertexSource> 
    struct conv_smooth_poly1 : 
    public conv_adaptor_vcgen<VertexSource, vcgen_smooth_poly1>
    {
        typedef conv_adaptor_vcgen<VertexSource, vcgen_smooth_poly1> base_type;

        conv_smooth_poly1(VertexSource& vs) : 
            conv_adaptor_vcgen<VertexSource, vcgen_smooth_poly1>(vs)
        {
        }

        void   smooth_value(double v) { base_type::generator().smooth_value(v); }
        double smooth_value() const { return base_type::generator().smooth_value(); }

    private:
        conv_smooth_poly1(const conv_smooth_poly1<VertexSource>&);
        const conv_smooth_poly1<VertexSource>& 
            operator = (const conv_smooth_poly1<VertexSource>&);
    };



     
    template<class VertexSource> 
    struct conv_smooth_poly1_curve : 
    public conv_curve<conv_smooth_poly1<VertexSource> >
    {
        conv_smooth_poly1_curve(VertexSource& vs) :
            conv_curve<conv_smooth_poly1<VertexSource> >(m_smooth),
            m_smooth(vs)
        {
        }

        void   smooth_value(double v) { m_smooth.generator().smooth_value(v); }
        double smooth_value() const { return m_smooth.generator().smooth_value(); }

    private:
        conv_smooth_poly1_curve(const conv_smooth_poly1_curve<VertexSource>&);
        const conv_smooth_poly1_curve<VertexSource>& 
            operator = (const conv_smooth_poly1_curve<VertexSource>&);

        conv_smooth_poly1<VertexSource> m_smooth;
    };

}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_CONV_STROKE_INCLUDED
#define AGG_CONV_STROKE_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_VCGEN_STROKE_INCLUDED
#define AGG_VCGEN_STROKE_INCLUDED



namespace agg
{

     
     
     
     
     
     
    class vcgen_stroke
    {
        enum status_e
        {
            initial,
            ready,
            cap1,
            cap2,
            outline1,
            close_first,
            outline2,
            out_vertices,
            end_poly1,
            end_poly2,
            stop
        };

    public:
        typedef vertex_sequence<vertex_dist, 6> vertex_storage;
        typedef pod_bvector<point_d, 6>         coord_storage;

        vcgen_stroke();

        void line_cap(line_cap_e lc)     { m_stroker.line_cap(lc); }
        void line_join(line_join_e lj)   { m_stroker.line_join(lj); }
        void inner_join(inner_join_e ij) { m_stroker.inner_join(ij); }

        line_cap_e   line_cap()   const { return m_stroker.line_cap(); }
        line_join_e  line_join()  const { return m_stroker.line_join(); }
        inner_join_e inner_join() const { return m_stroker.inner_join(); }

        void width(double w) { m_stroker.width(w); }
        void miter_limit(double ml) { m_stroker.miter_limit(ml); }
        void miter_limit_theta(double t) { m_stroker.miter_limit_theta(t); }
        void inner_miter_limit(double ml) { m_stroker.inner_miter_limit(ml); }
        void approximation_scale(double as) { m_stroker.approximation_scale(as); }

        double width() const { return m_stroker.width(); }
        double miter_limit() const { return m_stroker.miter_limit(); }
        double inner_miter_limit() const { return m_stroker.inner_miter_limit(); }
        double approximation_scale() const { return m_stroker.approximation_scale(); }

        void shorten(double s) { m_shorten = s; }
        double shorten() const { return m_shorten; }

         
        void remove_all();
        void add_vertex(double x, double y, unsigned cmd);

         
        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        vcgen_stroke(const vcgen_stroke&);
        const vcgen_stroke& operator = (const vcgen_stroke&);

        math_stroke<coord_storage> m_stroker;
        vertex_storage             m_src_vertices;
        coord_storage              m_out_vertices;
        double                     m_shorten;
        unsigned                   m_closed;
        status_e                   m_status;
        status_e                   m_prev_status;
        unsigned                   m_src_vertex;
        unsigned                   m_out_vertex;
    };


}

#endif

namespace agg
{

     
    template<class VertexSource, class Markers=null_markers> 
    struct conv_stroke : 
    public conv_adaptor_vcgen<VertexSource, vcgen_stroke, Markers>
    {
        typedef Markers marker_type;
        typedef conv_adaptor_vcgen<VertexSource, vcgen_stroke, Markers> base_type;

        conv_stroke(VertexSource& vs) : 
            conv_adaptor_vcgen<VertexSource, vcgen_stroke, Markers>(vs)
        {
        }

        void line_cap(line_cap_e lc)     { base_type::generator().line_cap(lc);  }
        void line_join(line_join_e lj)   { base_type::generator().line_join(lj); }
        void inner_join(inner_join_e ij) { base_type::generator().inner_join(ij); }

        line_cap_e   line_cap()   const { return base_type::generator().line_cap();  }
        line_join_e  line_join()  const { return base_type::generator().line_join(); }
        inner_join_e inner_join() const { return base_type::generator().inner_join(); }

        void width(double w) { base_type::generator().width(w); }
        void miter_limit(double ml) { base_type::generator().miter_limit(ml); }
        void miter_limit_theta(double t) { base_type::generator().miter_limit_theta(t); }
        void inner_miter_limit(double ml) { base_type::generator().inner_miter_limit(ml); }
        void approximation_scale(double as) { base_type::generator().approximation_scale(as); }

        double width() const { return base_type::generator().width(); }
        double miter_limit() const { return base_type::generator().miter_limit(); }
        double inner_miter_limit() const { return base_type::generator().inner_miter_limit(); }
        double approximation_scale() const { return base_type::generator().approximation_scale(); }

        void shorten(double s) { base_type::generator().shorten(s); }
        double shorten() const { return base_type::generator().shorten(); }

    private:
       conv_stroke(const conv_stroke<VertexSource, Markers>&);
       const conv_stroke<VertexSource, Markers>& 
           operator = (const conv_stroke<VertexSource, Markers>&);

    };

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_CONV_UNCLOSE_POLYGON_INCLUDED
#define AGG_CONV_UNCLOSE_POLYGON_INCLUDED


namespace agg
{
     
    template<class VertexSource> class conv_unclose_polygon
    {
    public:
        explicit conv_unclose_polygon(VertexSource& vs) : m_source(&vs) {}
        void attach(VertexSource& source) { m_source = &source; }

        void rewind(unsigned path_id)
        {
            m_source->rewind(path_id);
        }

        unsigned vertex(double* x, double* y)
        {
            unsigned cmd = m_source->vertex(x, y);
            if(is_end_poly(cmd)) cmd &= ~path_flags_close;
            return cmd;
        }

    private:
        conv_unclose_polygon(const conv_unclose_polygon<VertexSource>&);
        const conv_unclose_polygon<VertexSource>& 
            operator = (const conv_unclose_polygon<VertexSource>&);

        VertexSource* m_source;
    };

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_DDA_LINE_INCLUDED
#define AGG_DDA_LINE_INCLUDED

#include <cstdlib>

namespace agg
{

     
    template<int FractionShift, int YShift=0> class dda_line_interpolator
    {
    public:
         
        dda_line_interpolator() {}

         
        dda_line_interpolator(int y1, int y2, unsigned count) :
            m_y(y1),
            m_inc(((y2 - y1) << FractionShift) / int(count)),
            m_dy(0)
        {
        }

         
        void operator ++ ()
        {
            m_dy += m_inc;
        }

         
        void operator -- ()
        {
            m_dy -= m_inc;
        }

         
        void operator += (unsigned n)
        {
            m_dy += m_inc * n;
        }

         
        void operator -= (unsigned n)
        {
            m_dy -= m_inc * n;
        }


         
        int y()  const { return m_y + (m_dy >> (FractionShift-YShift)); }
        int dy() const { return m_dy; }


    private:
        int m_y;
        int m_inc;
        int m_dy;
    };





     
    class dda2_line_interpolator
    {
    public:
        typedef int save_data_type;
        enum save_size_e { save_size = 2 };

         
        dda2_line_interpolator() {}

         
        dda2_line_interpolator(int y1, int y2, int count) :
            m_cnt(count <= 0 ? 1 : count),
            m_lft((y2 - y1) / m_cnt),
            m_rem((y2 - y1) % m_cnt),
            m_mod(m_rem),
            m_y(y1)
        {
            if(m_mod <= 0)
            {
                m_mod += count;
                m_rem += count;
                m_lft--;
            }
            m_mod -= count;
        }

         
        dda2_line_interpolator(int y1, int y2, int count, int) :
            m_cnt(count <= 0 ? 1 : count),
            m_lft((y2 - y1) / m_cnt),
            m_rem((y2 - y1) % m_cnt),
            m_mod(m_rem),
            m_y(y1)
        {
            if(m_mod <= 0)
            {
                m_mod += count;
                m_rem += count;
                m_lft--;
            }
        }

         
        dda2_line_interpolator(int y, int count) :
            m_cnt(count <= 0 ? 1 : count),
            m_lft(y / m_cnt),
            m_rem(y % m_cnt),
            m_mod(m_rem),
            m_y(0)
        {
            if(m_mod <= 0)
            {
                m_mod += count;
                m_rem += count;
                m_lft--;
            }
        }


         
        void save(save_data_type* data) const
        {
            data[0] = m_mod;
            data[1] = m_y;
        }

         
        void load(const save_data_type* data)
        {
            m_mod = data[0];
            m_y   = data[1];
        }

         
        void operator++()
        {
            m_mod += m_rem;
            m_y += m_lft;
            if(m_mod > 0)
            {
                m_mod -= m_cnt;
                m_y++;
            }
        }

         
        void operator--()
        {
            if(m_mod <= m_rem)
            {
                m_mod += m_cnt;
                m_y--;
            }
            m_mod -= m_rem;
            m_y -= m_lft;
        }

         
        void adjust_forward()
        {
            m_mod -= m_cnt;
        }

         
        void adjust_backward()
        {
            m_mod += m_cnt;
        }

         
        int mod() const { return m_mod; }
        int rem() const { return m_rem; }
        int lft() const { return m_lft; }

         
        int y() const { return m_y; }

    private:
        int m_cnt;
        int m_lft;
        int m_rem;
        int m_mod;
        int m_y;
    };







     
    class line_bresenham_interpolator
    {
    public:
        enum subpixel_scale_e
        {
            subpixel_shift = 8,
            subpixel_scale = 1 << subpixel_shift,
            subpixel_mask  = subpixel_scale - 1
        };

         
        static int line_lr(int v) { return v >> subpixel_shift; }

         
        line_bresenham_interpolator(int x1, int y1, int x2, int y2) :
            m_x1_lr(line_lr(x1)),
            m_y1_lr(line_lr(y1)),
            m_x2_lr(line_lr(x2)),
            m_y2_lr(line_lr(y2)),
            m_ver(std::abs(m_x2_lr - m_x1_lr) < std::abs(m_y2_lr - m_y1_lr)),
            m_len(m_ver ? std::abs(m_y2_lr - m_y1_lr) : 
                          std::abs(m_x2_lr - m_x1_lr)),
            m_inc(m_ver ? ((y2 > y1) ? 1 : -1) : ((x2 > x1) ? 1 : -1)),
            m_interpolator(m_ver ? x1 : y1, 
                           m_ver ? x2 : y2, 
                           m_len)
        {
        }
    
         
        bool     is_ver() const { return m_ver; }
        unsigned len()    const { return m_len; }
        int      inc()    const { return m_inc; }

         
        void hstep()
        {
            ++m_interpolator;
            m_x1_lr += m_inc;
        }

         
        void vstep()
        {
            ++m_interpolator;
            m_y1_lr += m_inc;
        }

         
        int x1() const { return m_x1_lr; }
        int y1() const { return m_y1_lr; }
        int x2() const { return line_lr(m_interpolator.y()); }
        int y2() const { return line_lr(m_interpolator.y()); }
        int x2_hr() const { return m_interpolator.y(); }
        int y2_hr() const { return m_interpolator.y(); }

    private:
        int                    m_x1_lr;
        int                    m_y1_lr;
        int                    m_x2_lr;
        int                    m_y2_lr;
        bool                   m_ver;
        unsigned               m_len;
        int                    m_inc;
        dda2_line_interpolator m_interpolator;

    };


}



#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_ELLIPSE_BRESENHAM_INCLUDED
#define AGG_ELLIPSE_BRESENHAM_INCLUDED




namespace agg
{

     
    class ellipse_bresenham_interpolator
    {
    public:
        ellipse_bresenham_interpolator(int rx, int ry) :
            m_rx2(rx * rx),
            m_ry2(ry * ry),
            m_two_rx2(m_rx2 << 1),
            m_two_ry2(m_ry2 << 1),
            m_dx(0),
            m_dy(0),
            m_inc_x(0),
            m_inc_y(-ry * m_two_rx2),
            m_cur_f(0)
        {}
        
        int dx() const { return m_dx; }
        int dy() const { return m_dy; }

        void operator++ ()
        {
            int  mx, my, mxy, min_m;
            int  fx, fy, fxy;

            mx = fx = m_cur_f + m_inc_x + m_ry2;
            if(mx < 0) mx = -mx;

            my = fy = m_cur_f + m_inc_y + m_rx2;
            if(my < 0) my = -my;

            mxy = fxy = m_cur_f + m_inc_x + m_ry2 + m_inc_y + m_rx2;
            if(mxy < 0) mxy = -mxy;

            min_m = mx; 
            bool flag = true;

            if(min_m > my)  
            { 
                min_m = my; 
                flag = false; 
            }

            m_dx = m_dy = 0;

            if(min_m > mxy) 
            { 
                m_inc_x += m_two_ry2;
                m_inc_y += m_two_rx2;
                m_cur_f = fxy;
                m_dx = 1; 
                m_dy = 1;
                return;
            }

            if(flag) 
            {
                m_inc_x += m_two_ry2;
                m_cur_f = fx;
                m_dx = 1;
                return;
            }

            m_inc_y += m_two_rx2;
            m_cur_f = fy;
            m_dy = 1;
        }

    private:
        int m_rx2;
        int m_ry2;
        int m_two_rx2;
        int m_two_ry2;
        int m_dx;
        int m_dy;
        int m_inc_x;
        int m_inc_y;
        int m_cur_f;

    };

}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_ELLIPSE_INCLUDED
#define AGG_ELLIPSE_INCLUDED

#include <cmath>

namespace agg
{

     
    class ellipse
    {
    public:
        ellipse() : 
            m_x(0.0), m_y(0.0), m_rx(1.0), m_ry(1.0), m_scale(1.0), 
            m_num(4), m_step(0), m_cw(false) {}

        ellipse(double x, double y, double rx, double ry, 
                unsigned num_steps=0, bool cw=false) :
            m_x(x), m_y(y), m_rx(rx), m_ry(ry), m_scale(1.0), 
            m_num(num_steps), m_step(0), m_cw(cw) 
        {
            if(m_num == 0) calc_num_steps();
        }

        void init(double x, double y, double rx, double ry, 
                  unsigned num_steps=0, bool cw=false);

        void approximation_scale(double scale);
        void rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
        void calc_num_steps();

        double m_x;
        double m_y;
        double m_rx;
        double m_ry;
        double m_scale;
        unsigned m_num;
        unsigned m_step;
        bool m_cw;
    };

     
    inline void ellipse::init(double x, double y, double rx, double ry, 
                              unsigned num_steps, bool cw)
    {
        m_x = x;
        m_y = y;
        m_rx = rx;
        m_ry = ry;
        m_num = num_steps;
        m_step = 0;
        m_cw = cw;
        if(m_num == 0) calc_num_steps();
    }

     
    inline void ellipse::approximation_scale(double scale)
    {   
        m_scale = scale;
        calc_num_steps();
    }

     
    inline void ellipse::calc_num_steps()
    {
        double ra = (std::fabs(m_rx) + std::fabs(m_ry)) / 2;
        double da = std::acos(ra / (ra + 0.125 / m_scale)) * 2;
        m_num = uround(2*pi / da);
    }

     
    inline void ellipse::rewind(unsigned)
    {
        m_step = 0;
    }

     
    inline unsigned ellipse::vertex(double* x, double* y)
    {
        if(m_step == m_num) 
        {
            ++m_step;
            return +path_cmd_end_poly | path_flags_close | path_flags_ccw;
        }
        if(m_step > m_num) return path_cmd_stop;
        double angle = double(m_step) / double(m_num) * 2.0 * pi;
        if(m_cw) angle = 2.0 * pi - angle;
        *x = m_x + std::cos(angle) * m_rx;
        *y = m_y + std::sin(angle) * m_ry;
        m_step++;
        return ((m_step == 1) ? path_cmd_move_to : path_cmd_line_to);
    }

}



#endif


 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_EMBEDDED_RASTER_FONTS_INCLUDED
#define AGG_EMBEDDED_RASTER_FONTS_INCLUDED


namespace agg
{
    extern const int8u gse4x6[];
    extern const int8u gse4x8[];
    extern const int8u gse5x7[];
    extern const int8u gse5x9[];
    extern const int8u gse6x12[];
    extern const int8u gse6x9[];
    extern const int8u gse7x11[];
    extern const int8u gse7x11_bold[];
    extern const int8u gse7x15[];
    extern const int8u gse7x15_bold[];
    extern const int8u gse8x16[];
    extern const int8u gse8x16_bold[];
    extern const int8u mcs11_prop[];
    extern const int8u mcs11_prop_condensed[];
    extern const int8u mcs12_prop[];
    extern const int8u mcs13_prop[];
    extern const int8u mcs5x10_mono[];
    extern const int8u mcs5x11_mono[];
    extern const int8u mcs6x10_mono[];
    extern const int8u mcs6x11_mono[];
    extern const int8u mcs7x12_mono_high[];
    extern const int8u mcs7x12_mono_low[];
    extern const int8u verdana12[];
    extern const int8u verdana12_bold[];
    extern const int8u verdana13[];
    extern const int8u verdana13_bold[];
    extern const int8u verdana14[];
    extern const int8u verdana14_bold[];
    extern const int8u verdana16[];
    extern const int8u verdana16_bold[];
    extern const int8u verdana17[];
    extern const int8u verdana17_bold[];
    extern const int8u verdana18[];
    extern const int8u verdana18_bold[];
}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_FONT_CACHE_MANAGER2_INCLUDED
#define AGG_FONT_CACHE_MANAGER2_INCLUDED

#include <cassert>
#include <exception>
#include <cstring>

namespace agg {

namespace fman {
   
  enum glyph_data_type
  {
    glyph_data_invalid = 0,
    glyph_data_mono    = 1,
    glyph_data_gray8   = 2,
    glyph_data_outline = 3
  };


   
  struct cached_glyph
  {
    void *			cached_font;
    unsigned		glyph_code;
    unsigned        glyph_index;
    int8u*          data;
    unsigned        data_size;
    glyph_data_type data_type;
    rect_i          bounds;
    double          advance_x;
    double          advance_y;
  };


   
  class cached_glyphs
  {
  public:
    enum block_size_e { block_size = 16384-16 };

     
    cached_glyphs()
      : m_allocator(block_size)
    { std::memset(m_glyphs, 0, sizeof(m_glyphs)); }

     
    const cached_glyph* find_glyph(unsigned glyph_code) const
    {
      unsigned msb = (glyph_code >> 8) & 0xFF;
      if(m_glyphs[msb])
      {
        return m_glyphs[msb][glyph_code & 0xFF];
      }
      return 0;
    }

     
    cached_glyph* cache_glyph(
      void *			cached_font,
      unsigned        glyph_code,
      unsigned        glyph_index,
      unsigned        data_size,
      glyph_data_type data_type,
      const rect_i&   bounds,
      double          advance_x,
      double          advance_y)
    {
      unsigned msb = (glyph_code >> 8) & 0xFF;
      if(m_glyphs[msb] == 0)
      {
        m_glyphs[msb] =
          (cached_glyph**)m_allocator.allocate(sizeof(cached_glyph*) * 256,
          sizeof(cached_glyph*));
        std::memset(m_glyphs[msb], 0, sizeof(cached_glyph*) * 256);
      }

      unsigned lsb = glyph_code & 0xFF;
      if(m_glyphs[msb][lsb]) return 0;  

      cached_glyph* glyph =
        (cached_glyph*)m_allocator.allocate(sizeof(cached_glyph),
        sizeof(double));

      glyph->cached_font		  = cached_font;
      glyph->glyph_code		  = glyph_code;
      glyph->glyph_index        = glyph_index;
      glyph->data               = m_allocator.allocate(data_size);
      glyph->data_size          = data_size;
      glyph->data_type          = data_type;
      glyph->bounds             = bounds;
      glyph->advance_x          = advance_x;
      glyph->advance_y          = advance_y;
      return m_glyphs[msb][lsb] = glyph;
    }

  private:
    block_allocator m_allocator;
    cached_glyph**   m_glyphs[256];
  };



   
  enum glyph_rendering
  {
    glyph_ren_native_mono,
    glyph_ren_native_gray8,
    glyph_ren_outline,
    glyph_ren_agg_mono,
    glyph_ren_agg_gray8
  };




   
  template<class FontEngine> class font_cache_manager
  {
  public:
    typedef FontEngine font_engine_type;
    typedef font_cache_manager<FontEngine> self_type;
    typedef typename font_engine_type::path_adaptor_type   path_adaptor_type;
    typedef typename font_engine_type::gray8_adaptor_type  gray8_adaptor_type;
    typedef typename gray8_adaptor_type::embedded_scanline gray8_scanline_type;
    typedef typename font_engine_type::mono_adaptor_type   mono_adaptor_type;
    typedef typename mono_adaptor_type::embedded_scanline  mono_scanline_type;

    struct cached_font
    {
      cached_font(
        font_engine_type& engine,
        typename FontEngine::loaded_face *face,
        double height,
        double width,
        bool hinting,
        glyph_rendering rendering )
        : m_engine( engine )
        , m_face( face )
        , m_height( height )
        , m_width( width )
        , m_hinting( hinting )
        , m_rendering( rendering )
      {
        select_face();
        m_face_height=m_face->height();
        m_face_width=m_face->width();
        m_face_ascent=m_face->ascent();
        m_face_descent=m_face->descent();
        m_face_ascent_b=m_face->ascent_b();
        m_face_descent_b=m_face->descent_b();
      }

      double height() const
      {
        return m_face_height;
      }

      double width() const
      {
        return m_face_width;
      }

      double ascent() const
      {
        return m_face_ascent;
      }

      double descent() const
      {
        return m_face_descent;
      }

      double ascent_b() const
      {
        return m_face_ascent_b;
      }

      double descent_b() const
      {
        return m_face_descent_b;
      }

      bool add_kerning( const cached_glyph *first, const cached_glyph *second, double* x, double* y)
      {
        if( !first || !second )
          return false;
        select_face();
        return m_face->add_kerning(
          first->glyph_index, second->glyph_index, x, y );
      }

      void select_face()
      {
        m_face->select_instance( m_height, m_width, m_hinting, m_rendering );
      }

      const cached_glyph *get_glyph(unsigned cp)
      {
        const cached_glyph *glyph=m_glyphs.find_glyph(cp);
        if( glyph==0 )
        {
          typename FontEngine::prepared_glyph prepared;
          select_face();
          bool success=m_face->prepare_glyph(cp, &prepared);
          if( success )
          {
            glyph=m_glyphs.cache_glyph(
              this,
              prepared.glyph_code,
              prepared.glyph_index,
              prepared.data_size,
              prepared.data_type,
              prepared.bounds,
              prepared.advance_x,
              prepared.advance_y );
            assert( glyph!=0 );
            m_face->write_glyph_to(&prepared,glyph->data);
          }
        }
        return glyph;
      }

      font_engine_type&   m_engine;
      typename FontEngine::loaded_face *m_face;
      double				m_height;
      double				m_width;
      bool				m_hinting;
      glyph_rendering		m_rendering;
      double				m_face_height;
      double				m_face_width;
      double				m_face_ascent;
      double				m_face_descent;
      double				m_face_ascent_b;
      double				m_face_descent_b;
      cached_glyphs		m_glyphs;
    };

     
    font_cache_manager(font_engine_type& engine, unsigned max_fonts=32)
      :m_engine(engine)
    { }

     
    void init_embedded_adaptors(const cached_glyph* gl,
      double x, double y,
      double scale=1.0)
    {
      if(gl)
      {
        switch(gl->data_type)
        {
        default: return;
        case glyph_data_mono:
          m_mono_adaptor.init(gl->data, gl->data_size, x, y);
          break;

        case glyph_data_gray8:
          m_gray8_adaptor.init(gl->data, gl->data_size, x, y);
          break;

        case glyph_data_outline:
          m_path_adaptor.init(gl->data, gl->data_size, x, y, scale);
          break;
        }
      }
    }


     
    path_adaptor_type&   path_adaptor()   { return m_path_adaptor;   }
    gray8_adaptor_type&  gray8_adaptor()  { return m_gray8_adaptor;  }
    gray8_scanline_type& gray8_scanline() { return m_gray8_scanline; }
    mono_adaptor_type&   mono_adaptor()   { return m_mono_adaptor;   }
    mono_scanline_type&  mono_scanline()  { return m_mono_scanline;  }


  private:
     
    font_cache_manager(const self_type&);
    const self_type& operator = (const self_type&);

    font_engine_type&   m_engine;
    path_adaptor_type   m_path_adaptor;
    gray8_adaptor_type  m_gray8_adaptor;
    gray8_scanline_type m_gray8_scanline;
    mono_adaptor_type   m_mono_adaptor;
    mono_scanline_type  m_mono_scanline;
  };

}
}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_FONT_CACHE_MANAGER_INCLUDED
#define AGG_FONT_CACHE_MANAGER_INCLUDED

#include <cstring>

namespace agg
{

     
    enum glyph_data_type
    {
        glyph_data_invalid = 0,
        glyph_data_mono    = 1,
        glyph_data_gray8   = 2,
        glyph_data_outline = 3
    };


     
    struct glyph_cache
    {
        unsigned        glyph_index;
        int8u*          data;
        unsigned        data_size;
        glyph_data_type data_type;
        rect_i          bounds;
        double          advance_x;
        double          advance_y;
    };


     
    class font_cache
    {
    public:
        enum block_size_e { block_size = 16384-16 };

         
        font_cache() : 
            m_allocator(block_size),
            m_font_signature(0)
        {}

         
        void signature(const char* font_signature)
        {
            m_font_signature = (char*)m_allocator.allocate(std::strlen(font_signature) + 1);
            std::strcpy(m_font_signature, font_signature);
            std::memset(m_glyphs, 0, sizeof(m_glyphs));
        }

         
        bool font_is(const char* font_signature) const
        {
            return std::strcmp(font_signature, m_font_signature) == 0;
        }

         
        const glyph_cache* find_glyph(unsigned glyph_code) const
        {
            unsigned msb = (glyph_code >> 8) & 0xFF;
            if(m_glyphs[msb]) 
            {
                return m_glyphs[msb][glyph_code & 0xFF];
            }
            return 0;
        }

         
        glyph_cache* cache_glyph(unsigned        glyph_code, 
                                 unsigned        glyph_index,
                                 unsigned        data_size,
                                 glyph_data_type data_type,
                                 const rect_i&   bounds,
                                 double          advance_x,
                                 double          advance_y)
        {
            unsigned msb = (glyph_code >> 8) & 0xFF;
            if(m_glyphs[msb] == 0)
            {
                m_glyphs[msb] = 
                    (glyph_cache**)m_allocator.allocate(sizeof(glyph_cache*) * 256, 
                                                        sizeof(glyph_cache*));
                std::memset(m_glyphs[msb], 0, sizeof(glyph_cache*) * 256);
            }

            unsigned lsb = glyph_code & 0xFF;
            if(m_glyphs[msb][lsb]) return 0;  

            glyph_cache* glyph = 
                (glyph_cache*)m_allocator.allocate(sizeof(glyph_cache),
                                                   sizeof(double));

            glyph->glyph_index        = glyph_index;
            glyph->data               = m_allocator.allocate(data_size);
            glyph->data_size          = data_size;
            glyph->data_type          = data_type;
            glyph->bounds             = bounds;
            glyph->advance_x          = advance_x;
            glyph->advance_y          = advance_y;
            return m_glyphs[msb][lsb] = glyph;
        }

    private:
        block_allocator m_allocator;
        glyph_cache**   m_glyphs[256];
        char*           m_font_signature;
    };






    
     
    class font_cache_pool
    {
    public:
         
        ~font_cache_pool()
        {
            unsigned i;
            for(i = 0; i < m_num_fonts; ++i)
            {
                obj_allocator<font_cache>::deallocate(m_fonts[i]);
            }
            pod_allocator<font_cache*>::deallocate(m_fonts, m_max_fonts);
        }

         
        font_cache_pool(unsigned max_fonts=32) : 
            m_fonts(pod_allocator<font_cache*>::allocate(max_fonts)),
            m_max_fonts(max_fonts),
            m_num_fonts(0),
            m_cur_font(0)
        {}


         
        void font(const char* font_signature, bool reset_cache = false)
        {
            int idx = find_font(font_signature);
            if(idx >= 0)
            {
                if(reset_cache)
                {
                    obj_allocator<font_cache>::deallocate(m_fonts[idx]);
                    m_fonts[idx] = obj_allocator<font_cache>::allocate();
                    m_fonts[idx]->signature(font_signature);
                }
                m_cur_font = m_fonts[idx];
            }
            else
            {
                if(m_num_fonts >= m_max_fonts)
                {
                    obj_allocator<font_cache>::deallocate(m_fonts[0]);
                    std::memcpy(m_fonts, 
                           m_fonts + 1, 
                           (m_max_fonts - 1) * sizeof(font_cache*));
                    m_num_fonts = m_max_fonts - 1;
                }
                m_fonts[m_num_fonts] = obj_allocator<font_cache>::allocate();
                m_fonts[m_num_fonts]->signature(font_signature);
                m_cur_font = m_fonts[m_num_fonts];
                ++m_num_fonts;
            }
        }

         
        const font_cache* font() const
        {
            return m_cur_font;
        }

         
        const glyph_cache* find_glyph(unsigned glyph_code) const
        {
            if(m_cur_font) return m_cur_font->find_glyph(glyph_code);
            return 0;
        }

         
        glyph_cache* cache_glyph(unsigned        glyph_code, 
                                 unsigned        glyph_index,
                                 unsigned        data_size,
                                 glyph_data_type data_type,
                                 const rect_i&   bounds,
                                 double          advance_x,
                                 double          advance_y)
        {
            if(m_cur_font) 
            {
                return m_cur_font->cache_glyph(glyph_code,
                                               glyph_index,
                                               data_size,
                                               data_type,
                                               bounds,
                                               advance_x,
                                               advance_y);
            }
            return 0;
        }


         
        int find_font(const char* font_signature)
        {
            unsigned i;
            for(i = 0; i < m_num_fonts; i++)
            {
                if(m_fonts[i]->font_is(font_signature)) return int(i);
            }
            return -1;
        }

    private:
        font_cache** m_fonts;
        unsigned     m_max_fonts;
        unsigned     m_num_fonts;
        font_cache*  m_cur_font;
    };




     
    enum glyph_rendering
    {
        glyph_ren_native_mono,
        glyph_ren_native_gray8,
        glyph_ren_outline,
        glyph_ren_agg_mono,
        glyph_ren_agg_gray8
    };




     
    template<class FontEngine> class font_cache_manager
    {
    public:
        typedef FontEngine font_engine_type;
        typedef font_cache_manager<FontEngine> self_type;
        typedef typename font_engine_type::path_adaptor_type   path_adaptor_type;
        typedef typename font_engine_type::gray8_adaptor_type  gray8_adaptor_type;
        typedef typename gray8_adaptor_type::embedded_scanline gray8_scanline_type;
        typedef typename font_engine_type::mono_adaptor_type   mono_adaptor_type;
        typedef typename mono_adaptor_type::embedded_scanline  mono_scanline_type;

         
        font_cache_manager(font_engine_type& engine, unsigned max_fonts=32) :
            m_fonts(max_fonts),
            m_engine(engine),
            m_change_stamp(-1),
            m_prev_glyph(0),
            m_last_glyph(0)
        {}

         
        void reset_last_glyph()
        {
            m_prev_glyph = m_last_glyph = 0;
        }

         
        const glyph_cache* glyph(unsigned glyph_code)
        {
            synchronize();
            const glyph_cache* gl = m_fonts.find_glyph(glyph_code);
            if(gl) 
            {
                m_prev_glyph = m_last_glyph;
                return m_last_glyph = gl;
            }
            else
            {
                if(m_engine.prepare_glyph(glyph_code))
                {
                    m_prev_glyph = m_last_glyph;
                    m_last_glyph = m_fonts.cache_glyph(glyph_code, 
                                                       m_engine.glyph_index(),
                                                       m_engine.data_size(),
                                                       m_engine.data_type(),
                                                       m_engine.bounds(),
                                                       m_engine.advance_x(),
                                                       m_engine.advance_y());
                    m_engine.write_glyph_to(m_last_glyph->data);
                    return m_last_glyph;
                }
            }
            return 0;
        }

         
        void init_embedded_adaptors(const glyph_cache* gl, 
                                    double x, double y, 
                                    double scale=1.0)
        {
            if(gl)
            {
                switch(gl->data_type)
                {
                default: return;
                case glyph_data_mono:
                    m_mono_adaptor.init(gl->data, gl->data_size, x, y);
                    break;

                case glyph_data_gray8:
                    m_gray8_adaptor.init(gl->data, gl->data_size, x, y);
                    break;

                case glyph_data_outline:
                    m_path_adaptor.init(gl->data, gl->data_size, x, y, scale);
                    break;
                }
            }
        }


         
        path_adaptor_type&   path_adaptor()   { return m_path_adaptor;   }
        gray8_adaptor_type&  gray8_adaptor()  { return m_gray8_adaptor;  }
        gray8_scanline_type& gray8_scanline() { return m_gray8_scanline; }
        mono_adaptor_type&   mono_adaptor()   { return m_mono_adaptor;   }
        mono_scanline_type&  mono_scanline()  { return m_mono_scanline;  }

         
        const glyph_cache* perv_glyph() const { return m_prev_glyph; }
        const glyph_cache* last_glyph() const { return m_last_glyph; }

         
        bool add_kerning(double* x, double* y)
        {
            if(m_prev_glyph && m_last_glyph)
            {
                return m_engine.add_kerning(m_prev_glyph->glyph_index, 
                                            m_last_glyph->glyph_index,
                                            x, y);
            }
            return false;
        }

         
        void precache(unsigned from, unsigned to)
        {
            for(; from <= to; ++from) glyph(from);
        }

         
        void reset_cache()
        {
            m_fonts.font(m_engine.font_signature(), true);
            m_change_stamp = m_engine.change_stamp();
            m_prev_glyph = m_last_glyph = 0;
        }

    private:
         
        font_cache_manager(const self_type&);
        const self_type& operator = (const self_type&);

         
        void synchronize()
        {
            if(m_change_stamp != m_engine.change_stamp())
            {
                m_fonts.font(m_engine.font_signature());
                m_change_stamp = m_engine.change_stamp();
                m_prev_glyph = m_last_glyph = 0;
            }
        }

        font_cache_pool     m_fonts;
        font_engine_type&   m_engine;
        int                 m_change_stamp;
        double              m_dx;
        double              m_dy;
        const glyph_cache*  m_prev_glyph;
        const glyph_cache*  m_last_glyph;
        path_adaptor_type   m_path_adaptor;
        gray8_adaptor_type  m_gray8_adaptor;
        gray8_scanline_type m_gray8_scanline;
        mono_adaptor_type   m_mono_adaptor;
        mono_scanline_type  m_mono_scanline;
    };

}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_GLYPH_RASTER_BIN_INCLUDED
#define AGG_GLYPH_RASTER_BIN_INCLUDED

#include <cstring>

namespace agg
{

     
    template<class ColorT> class glyph_raster_bin
    {
    public:
        typedef ColorT color_type;

         
        struct glyph_rect
        {
            int x1,y1,x2,y2;
            double dx, dy;
        };

         
        glyph_raster_bin(const int8u* font) :
            m_font(font),
            m_big_endian(false)
        {
            int t = 1;
            if(*(char*)&t == 0) m_big_endian = true;
            std::memset(m_span, 0, sizeof(m_span));
        }

         
        const int8u* font() const { return m_font; }
        void font(const int8u* f) { m_font = f; }

         
        double height()    const { return m_font[0]; }
        double base_line() const { return m_font[1]; }

         
        template<class CharT>
        double width(const CharT* str) const
        {
            unsigned start_char = m_font[2];
            unsigned num_chars = m_font[3];

            unsigned w = 0;
            while(*str)
            {
                unsigned glyph = *str;
                const int8u* bits = m_font + 4 + num_chars * 2 + 
                                    value(m_font + 4 + (glyph - start_char) * 2);
                w += *bits;
                ++str;
            }
            return w;
        }

         
        void prepare(glyph_rect* r, double x, double y, unsigned glyph, bool flip)
        {
            unsigned start_char = m_font[2];
            unsigned num_chars = m_font[3];

            m_bits = m_font + 4 + num_chars * 2 + 
                     value(m_font + 4 + (glyph - start_char) * 2);

            m_glyph_width = *m_bits++;
            m_glyph_byte_width = (m_glyph_width + 7) >> 3;

            r->x1 = int(x);
            r->x2 = r->x1 + m_glyph_width - 1;
            if(flip)
            {
                r->y1 = int(y) - m_font[0] + m_font[1];
                r->y2 = r->y1 + m_font[0] - 1;
            }
            else
            {
                r->y1 = int(y) - m_font[1] + 1;
                r->y2 = r->y1 + m_font[0] - 1;
            }
            r->dx = m_glyph_width; 
            r->dy = 0;
        }

         
        const cover_type* span(unsigned i)
        {
            i = m_font[0] - i - 1;
            const int8u* bits = m_bits + i * m_glyph_byte_width;
            unsigned j;
            unsigned val = *bits;
            unsigned nb = 0;
            for(j = 0; j < m_glyph_width; ++j)
            {
                m_span[j] = (cover_type)((val & 0x80) ? cover_full : cover_none);
                val <<= 1;
                if(++nb >= 8)
                {
                    val = *++bits;
                    nb = 0;
                }
            }
            return m_span;
        }

    private:
         
        int16u value(const int8u* p) const
        {
            int16u v;
            if(m_big_endian)
            {
                 *(int8u*)&v      = p[1];
                *((int8u*)&v + 1) = p[0];
            }
            else
            {
                 *(int8u*)&v      = p[0];
                *((int8u*)&v + 1) = p[1];
            }
            return v;
        }


         
        const int8u* m_font;
        bool m_big_endian;
        cover_type m_span[32];
        const int8u* m_bits;
        unsigned m_glyph_width;
        unsigned m_glyph_byte_width;
    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_GRADIENT_LUT_INCLUDED
#define AGG_GRADIENT_LUT_INCLUDED


namespace agg
{

     
    template<class ColorT> struct color_interpolator
    {
    public:
        typedef ColorT color_type;

        color_interpolator(const color_type& c1, 
                           const color_type& c2, 
                           unsigned len) :
            m_c1(c1),
            m_c2(c2),
            m_len(len),
            m_count(0)
        {}

        void operator ++ ()
        {
            ++m_count;
        }

        color_type color() const
        {
            return m_c1.gradient(m_c2, double(m_count) / m_len);
        }

    private:
        color_type m_c1;
        color_type m_c2;
        unsigned   m_len;
        unsigned   m_count;
    };

     
     
    template<> struct color_interpolator<rgba8>
    {
    public:
        typedef rgba8 color_type;

        color_interpolator(const color_type& c1, 
                           const color_type& c2, 
                           unsigned len) :
            r(c1.r, c2.r, len),
            g(c1.g, c2.g, len),
            b(c1.b, c2.b, len),
            a(c1.a, c2.a, len)
        {}

        void operator ++ ()
        {
            ++r; ++g; ++b; ++a;
        }

        color_type color() const
        {
            return color_type(r.y(), g.y(), b.y(), a.y());
        }

    private:
        agg::dda_line_interpolator<14> r, g, b, a;
    };

     
     
    template<> struct color_interpolator<gray8>
    {
    public:
        typedef gray8 color_type;

        color_interpolator(const color_type& c1, 
                           const color_type& c2, 
                           unsigned len) :
            v(c1.v, c2.v, len),
            a(c1.a, c2.a, len)
        {}

        void operator ++ ()
        {
            ++v; ++a;
        }

        color_type color() const
        {
            return color_type(v.y(), a.y());
        }

    private:
        agg::dda_line_interpolator<14> v,a;
    };

     
    template<class ColorInterpolator, 
             unsigned ColorLutSize=256> class gradient_lut
    {
    public:
        typedef ColorInterpolator interpolator_type;
        typedef typename interpolator_type::color_type color_type;
        enum { color_lut_size = ColorLutSize };

         
        gradient_lut() : m_color_lut(color_lut_size) {}

         
         
         
         
         
         
         
         
         
        void remove_all();
        void add_color(double offset, const color_type& color);
        void build_lut();

         
         
         
         
        static unsigned size() 
        { 
            return color_lut_size; 
        }
        const color_type& operator [] (unsigned i) const 
        { 
            return m_color_lut[i]; 
        }

    private:
         
        struct color_point
        {
            double     offset;
            color_type color;

            color_point() {}
            color_point(double off, const color_type& c) : 
                offset(off), color(c)
            {
                if(offset < 0.0) offset = 0.0;
                if(offset > 1.0) offset = 1.0;
            }
        };
        typedef agg::pod_bvector<color_point, 4> color_profile_type;
        typedef agg::pod_array<color_type>       color_lut_type;

        static bool offset_less(const color_point& a, const color_point& b)
        {
            return a.offset < b.offset;
        }
        static bool offset_equal(const color_point& a, const color_point& b)
        {
            return a.offset == b.offset;
        }

         
        color_profile_type  m_color_profile;
        color_lut_type      m_color_lut;
    };



     
    template<class T, unsigned S>
    void gradient_lut<T,S>::remove_all()
    { 
        m_color_profile.remove_all(); 
    }

     
    template<class T, unsigned S>
    void gradient_lut<T,S>::add_color(double offset, const color_type& color)
    {
        m_color_profile.add(color_point(offset, color));
    }

     
    template<class T, unsigned S>
    void gradient_lut<T,S>::build_lut()
    {
        quick_sort(m_color_profile, offset_less);
        m_color_profile.cut_at(remove_duplicates(m_color_profile, offset_equal));
        if(m_color_profile.size() >= 2)
        {
            unsigned i;
            unsigned start = uround(m_color_profile[0].offset * (double)color_lut_size);
            unsigned end;
            color_type c = m_color_profile[0].color;
            for(i = 0; i < start; i++) 
            {
                m_color_lut[i] = c;
            }
            for(i = 1; i < m_color_profile.size(); i++)
            {
                end  = uround(m_color_profile[i].offset * (double)color_lut_size);
                interpolator_type ci(m_color_profile[i-1].color, 
                                     m_color_profile[i  ].color, 
                                     end - start + 1);
                while(start < end)
                {
                    m_color_lut[start] = ci.color();
                    ++ci;
                    ++start;
                }
            }
            c = m_color_profile.last().color;
            for(; end < m_color_lut.size(); end++)
            {
                m_color_lut[end] = c;
            }
        }
    }
}




#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_GSV_TEXT_INCLUDED
#define AGG_GSV_TEXT_INCLUDED


namespace agg
{


     
     
     
     
    class gsv_text
    {
        enum status
        {
            initial,
            next_char,
            start_glyph,
            glyph
        };

    public:
        gsv_text();

        void font(const void* font);
        void flip(bool flip_y) { m_flip = flip_y; }
        void load_font(const char* file);
        void size(double height, double width=0.0);
        void space(double space);
        void line_space(double line_space);
        void start_point(double x, double y);
        void text(const char* text);
        
        double text_width();

        void rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

    private:
         
        gsv_text(const gsv_text&);
        const gsv_text& operator = (const gsv_text&);

        int16u value(const int8u* p) const
        {
            int16u v;
            if(m_big_endian)
            {
                 *(int8u*)&v      = p[1];
                *((int8u*)&v + 1) = p[0];
            }
            else
            {
                 *(int8u*)&v      = p[0];
                *((int8u*)&v + 1) = p[1];
            }
            return v;
        }

    private:
        double          m_x;
        double          m_y;
        double          m_start_x;
        double          m_width;
        double          m_height;
        double          m_space;
        double          m_line_space;
        char            m_chr[2];
        char*           m_text;
        pod_array<char> m_text_buf;
        char*           m_cur_chr;
        const void*     m_font;
        pod_array<char> m_loaded_font;
        status          m_status;
        bool            m_big_endian;
        bool            m_flip;
        int8u*          m_indices;
        int8*           m_glyphs;
        int8*           m_bglyph;
        int8*           m_eglyph;
        double          m_w;
        double          m_h;
    };




     
    template<class Transformer = trans_affine> class gsv_text_outline
    {
    public:
        gsv_text_outline(gsv_text& text, Transformer& trans) :
          m_polyline(text),
          m_trans(m_polyline, trans)
        {
        }

        void width(double w) 
        { 
            m_polyline.width(w); 
        }

        void transformer(const Transformer* trans) 
        {
            m_trans->transformer(trans);
        }

        void rewind(unsigned path_id) 
        { 
            m_trans.rewind(path_id); 
            m_polyline.line_join(round_join);
            m_polyline.line_cap(round_cap);
        }

        unsigned vertex(double* x, double* y)
        {
            return m_trans.vertex(x, y);
        }

    private:
        conv_stroke<gsv_text> m_polyline;
        conv_transform<conv_stroke<gsv_text>, Transformer> m_trans;
    };



}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_IMAGE_ACCESSORS_INCLUDED
#define AGG_IMAGE_ACCESSORS_INCLUDED


namespace agg
{

     
    template<class PixFmt> class image_accessor_clip
    {
    public:
        typedef PixFmt   pixfmt_type;
        typedef typename pixfmt_type::color_type color_type;
        typedef typename pixfmt_type::order_type order_type;
        typedef typename pixfmt_type::value_type value_type;
        enum pix_width_e { pix_width = pixfmt_type::pix_width };

        image_accessor_clip() {}
        explicit image_accessor_clip(pixfmt_type& pixf, 
                                     const color_type& bk) : 
            m_pixf(&pixf)
        {
            pixfmt_type::make_pix(m_bk_buf, bk);
        }

        void attach(pixfmt_type& pixf)
        {
            m_pixf = &pixf;
        }

        void background_color(const color_type& bk)
        {
            pixfmt_type::make_pix(m_bk_buf, bk);
        }

    private:
        AGG_INLINE const int8u* pixel() const
        {
            if(m_y >= 0 && m_y < (int)m_pixf->height() &&
               m_x >= 0 && m_x < (int)m_pixf->width())
            {
                return m_pixf->pix_ptr(m_x, m_y);
            }
            return m_bk_buf;
        }

    public:
        AGG_INLINE const int8u* span(int x, int y, unsigned len)
        {
            m_x = m_x0 = x;
            m_y = y;
            if(y >= 0 && y < (int)m_pixf->height() &&
               x >= 0 && x+(int)len <= (int)m_pixf->width())
            {
                return m_pix_ptr = m_pixf->pix_ptr(x, y);
            }
            m_pix_ptr = 0;
            return pixel();
        }

        AGG_INLINE const int8u* next_x()
        {
            if(m_pix_ptr) return m_pix_ptr += pix_width;
            ++m_x;
            return pixel();
        }

        AGG_INLINE const int8u* next_y()
        {
            ++m_y;
            m_x = m_x0;
            if(m_pix_ptr && 
               m_y >= 0 && m_y < (int)m_pixf->height())
            {
                return m_pix_ptr = m_pixf->pix_ptr(m_x, m_y);
            }
            m_pix_ptr = 0;
            return pixel();
        }

    private:
        const pixfmt_type* m_pixf;
        int8u              m_bk_buf[pix_width];
        int                m_x, m_x0, m_y;
        const int8u*       m_pix_ptr;
    };




     
    template<class PixFmt> class image_accessor_no_clip
    {
    public:
        typedef PixFmt   pixfmt_type;
        typedef typename pixfmt_type::color_type color_type;
        typedef typename pixfmt_type::order_type order_type;
        typedef typename pixfmt_type::value_type value_type;
        enum pix_width_e { pix_width = pixfmt_type::pix_width };

        image_accessor_no_clip() {}
        explicit image_accessor_no_clip(pixfmt_type& pixf) : 
            m_pixf(&pixf) 
        {}

        void attach(pixfmt_type& pixf)
        {
            m_pixf = &pixf;
        }

        AGG_INLINE const int8u* span(int x, int y, unsigned)
        {
            m_x = x;
            m_y = y;
            return m_pix_ptr = m_pixf->pix_ptr(x, y);
        }

        AGG_INLINE const int8u* next_x()
        {
            return m_pix_ptr += pix_width;
        }

        AGG_INLINE const int8u* next_y()
        {
            ++m_y;
            return m_pix_ptr = m_pixf->pix_ptr(m_x, m_y);
        }

    private:
        const pixfmt_type* m_pixf;
        int                m_x, m_y;
        const int8u*       m_pix_ptr;
    };




     
    template<class PixFmt> class image_accessor_clone
    {
    public:
        typedef PixFmt   pixfmt_type;
        typedef typename pixfmt_type::color_type color_type;
        typedef typename pixfmt_type::order_type order_type;
        typedef typename pixfmt_type::value_type value_type;
        enum pix_width_e { pix_width = pixfmt_type::pix_width };

        image_accessor_clone() {}
        explicit image_accessor_clone(pixfmt_type& pixf) : 
            m_pixf(&pixf) 
        {}

        void attach(pixfmt_type& pixf)
        {
            m_pixf = &pixf;
        }

    private:
        AGG_INLINE const int8u* pixel() const
        {
            int x = m_x;
            int y = m_y;
            if(x < 0) x = 0;
            if(y < 0) y = 0;
            if(x >= (int)m_pixf->width())  x = m_pixf->width() - 1;
            if(y >= (int)m_pixf->height()) y = m_pixf->height() - 1;
            return m_pixf->pix_ptr(x, y);
        }

    public:
        AGG_INLINE const int8u* span(int x, int y, unsigned len)
        {
            m_x = m_x0 = x;
            m_y = y;
            if(y >= 0 && y < (int)m_pixf->height() &&
               x >= 0 && x + int(len) <= (int)m_pixf->width())
            {
                return m_pix_ptr = m_pixf->pix_ptr(x, y);
            }
            m_pix_ptr = 0;
            return pixel();
        }

        AGG_INLINE const int8u* next_x()
        {
            if(m_pix_ptr) return m_pix_ptr += pix_width;
            ++m_x;
            return pixel();
        }

        AGG_INLINE const int8u* next_y()
        {
            ++m_y;
            m_x = m_x0;
            if(m_pix_ptr && 
               m_y >= 0 && m_y < (int)m_pixf->height())
            {
                return m_pix_ptr = m_pixf->pix_ptr(m_x, m_y);
            }
            m_pix_ptr = 0;
            return pixel();
        }

    private:
        const pixfmt_type* m_pixf;
        int                m_x, m_x0, m_y;
        const int8u*       m_pix_ptr;
    };





     
    template<class PixFmt, class WrapX, class WrapY> class image_accessor_wrap
    {
    public:
        typedef PixFmt   pixfmt_type;
        typedef typename pixfmt_type::color_type color_type;
        typedef typename pixfmt_type::order_type order_type;
        typedef typename pixfmt_type::value_type value_type;
        enum pix_width_e { pix_width = pixfmt_type::pix_width };

        image_accessor_wrap() {}
        explicit image_accessor_wrap(pixfmt_type& pixf) : 
            m_pixf(&pixf), 
            m_wrap_x(pixf.width()), 
            m_wrap_y(pixf.height())
        {}

        void attach(pixfmt_type& pixf)
        {
            m_pixf = &pixf;
        }

        AGG_INLINE const int8u* span(int x, int y, unsigned)
        {
            m_x = x;
            m_row_ptr = m_pixf->pix_ptr(0, m_wrap_y(y));
            return m_row_ptr + m_wrap_x(x) * pix_width;
        }

        AGG_INLINE const int8u* next_x()
        {
            int x = ++m_wrap_x;
            return m_row_ptr + x * pix_width;
        }

        AGG_INLINE const int8u* next_y()
        {
            m_row_ptr = m_pixf->pix_ptr(0, ++m_wrap_y);
            return m_row_ptr + m_wrap_x(m_x) * pix_width;
        }

    private:
        const pixfmt_type* m_pixf;
        const int8u*       m_row_ptr;
        int                m_x;
        WrapX              m_wrap_x;
        WrapY              m_wrap_y;
    };




     
    class wrap_mode_repeat
    {
    public:
        wrap_mode_repeat() {}
        wrap_mode_repeat(unsigned size) : 
            m_size(size), 
            m_add(size * (0x3FFFFFFF / size)),
            m_value(0)
        {}

        AGG_INLINE unsigned operator() (int v)
        { 
            return m_value = (unsigned(v) + m_add) % m_size; 
        }

        AGG_INLINE unsigned operator++ ()
        {
            ++m_value;
            if(m_value >= m_size) m_value = 0;
            return m_value;
        }
    private:
        unsigned m_size;
        unsigned m_add;
        unsigned m_value;
    };


     
    class wrap_mode_repeat_pow2
    {
    public:
        wrap_mode_repeat_pow2() {}
        wrap_mode_repeat_pow2(unsigned size) : m_value(0)
        {
            m_mask = 1;
            while(m_mask < size) m_mask = (m_mask << 1) | 1;
            m_mask >>= 1;
        }
        AGG_INLINE unsigned operator() (int v)
        { 
            return m_value = unsigned(v) & m_mask;
        }
        AGG_INLINE unsigned operator++ ()
        {
            ++m_value;
            if(m_value > m_mask) m_value = 0;
            return m_value;
        }
    private:
        unsigned m_mask;
        unsigned m_value;
    };


     
    class wrap_mode_repeat_auto_pow2
    {
    public:
        wrap_mode_repeat_auto_pow2() {}
        wrap_mode_repeat_auto_pow2(unsigned size) :
            m_size(size),
            m_add(size * (0x3FFFFFFF / size)),
            m_mask((m_size & (m_size-1)) ? 0 : m_size-1),
            m_value(0)
        {}

        AGG_INLINE unsigned operator() (int v) 
        { 
            if(m_mask) return m_value = unsigned(v) & m_mask;
            return m_value = (unsigned(v) + m_add) % m_size;
        }
        AGG_INLINE unsigned operator++ ()
        {
            ++m_value;
            if(m_value >= m_size) m_value = 0;
            return m_value;
        }

    private:
        unsigned m_size;
        unsigned m_add;
        unsigned m_mask;
        unsigned m_value;
    };


     
    class wrap_mode_reflect
    {
    public:
        wrap_mode_reflect() {}
        wrap_mode_reflect(unsigned size) : 
            m_size(size), 
            m_size2(size * 2),
            m_add(m_size2 * (0x3FFFFFFF / m_size2)),
            m_value(0)
        {}

        AGG_INLINE unsigned operator() (int v)
        { 
            m_value = (unsigned(v) + m_add) % m_size2;
            if(m_value >= m_size) return m_size2 - m_value - 1;
            return m_value;
        }

        AGG_INLINE unsigned operator++ ()
        {
            ++m_value;
            if(m_value >= m_size2) m_value = 0;
            if(m_value >= m_size) return m_size2 - m_value - 1;
            return m_value;
        }
    private:
        unsigned m_size;
        unsigned m_size2;
        unsigned m_add;
        unsigned m_value;
    };



     
    class wrap_mode_reflect_pow2
    {
    public:
        wrap_mode_reflect_pow2() {}
        wrap_mode_reflect_pow2(unsigned size) : m_value(0)
        {
            m_mask = 1;
            m_size = 1;
            while(m_mask < size) 
            {
                m_mask = (m_mask << 1) | 1;
                m_size <<= 1;
            }
        }
        AGG_INLINE unsigned operator() (int v)
        { 
            m_value = unsigned(v) & m_mask;
            if(m_value >= m_size) return m_mask - m_value;
            return m_value;
        }
        AGG_INLINE unsigned operator++ ()
        {
            ++m_value;
            m_value &= m_mask;
            if(m_value >= m_size) return m_mask - m_value;
            return m_value;
        }
    private:
        unsigned m_size;
        unsigned m_mask;
        unsigned m_value;
    };



     
    class wrap_mode_reflect_auto_pow2
    {
    public:
        wrap_mode_reflect_auto_pow2() {}
        wrap_mode_reflect_auto_pow2(unsigned size) :
            m_size(size),
            m_size2(size * 2),
            m_add(m_size2 * (0x3FFFFFFF / m_size2)),
            m_mask((m_size2 & (m_size2-1)) ? 0 : m_size2-1),
            m_value(0)
        {}

        AGG_INLINE unsigned operator() (int v) 
        { 
            m_value = m_mask ? unsigned(v) & m_mask : 
                              (unsigned(v) + m_add) % m_size2;
            if(m_value >= m_size) return m_size2 - m_value - 1;
            return m_value;            
        }
        AGG_INLINE unsigned operator++ ()
        {
            ++m_value;
            if(m_value >= m_size2) m_value = 0;
            if(m_value >= m_size) return m_size2 - m_value - 1;
            return m_value;
        }

    private:
        unsigned m_size;
        unsigned m_size2;
        unsigned m_add;
        unsigned m_mask;
        unsigned m_value;
    };


}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_IMAGE_FILTERS_INCLUDED
#define AGG_IMAGE_FILTERS_INCLUDED


namespace agg
{

     

    enum image_filter_scale_e
    {
        image_filter_shift = 14,                       
        image_filter_scale = 1 << image_filter_shift,  
        image_filter_mask  = image_filter_scale - 1    
    };

    enum image_subpixel_scale_e
    {
        image_subpixel_shift = 8,                          
        image_subpixel_scale = 1 << image_subpixel_shift,  
        image_subpixel_mask  = image_subpixel_scale - 1    
    };


     
    class image_filter_lut
    {
    public:
        template<class FilterF> void calculate(const FilterF& filter,
                                               bool normalization=true)
        {
            double r = filter.radius();
            realloc_lut(r);
            unsigned i;
            unsigned pivot = diameter() << (image_subpixel_shift - 1);
            for(i = 0; i < pivot; i++)
            {
                double x = double(i) / double(image_subpixel_scale);
                double y = filter.calc_weight(x);
                m_weight_array[pivot + i] = 
                m_weight_array[pivot - i] = (int16)iround(y * (double)image_filter_scale);
            }
            unsigned end = (diameter() << image_subpixel_shift) - 1;
            m_weight_array[0] = m_weight_array[end];
            if(normalization) 
            {
                normalize();
            }
        }

        image_filter_lut() : m_radius(0), m_diameter(0), m_start(0) {}

        template<class FilterF> image_filter_lut(const FilterF& filter, 
                                                 bool normalization=true)
        {
            calculate(filter, normalization);
        }

        double       radius()       const { return m_radius;   }
        unsigned     diameter()     const { return m_diameter; }
        int          start()        const { return m_start;    }
        const int16* weight_array() const { return &m_weight_array[0]; }
        void         normalize();

    private:
        void realloc_lut(double radius);
        image_filter_lut(const image_filter_lut&);
        const image_filter_lut& operator = (const image_filter_lut&);

        double           m_radius;
        unsigned         m_diameter;
        int              m_start;
        pod_array<int16> m_weight_array;
    };



     
    template<class FilterF> class image_filter : public image_filter_lut
    {
    public:
        image_filter()
        {
            calculate(m_filter_function);
        }
    private:
        FilterF m_filter_function;
    };


     
    struct image_filter_bilinear
    {
        static double radius() { return 1.0; }
        static double calc_weight(double x)
        {
            return 1.0 - x;
        }
    };


     
    struct image_filter_hanning
    {
        static double radius() { return 1.0; }
        static double calc_weight(double x)
        {
            return 0.5 + 0.5 * std::cos(pi * x);
        }
    };


     
    struct image_filter_hamming
    {
        static double radius() { return 1.0; }
        static double calc_weight(double x)
        {
            return 0.54 + 0.46 * std::cos(pi * x);
        }
    };

     
    struct image_filter_hermite
    {
        static double radius() { return 1.0; }
        static double calc_weight(double x)
        {
            return (2.0 * x - 3.0) * x * x + 1.0;
        }
    };
   
     
    struct image_filter_quadric
    {
        static double radius() { return 1.5; }
        static double calc_weight(double x)
        {
            double t;
            if(x <  0.5) return 0.75 - x * x;
            if(x <  1.5) {t = x - 1.5; return 0.5 * t * t;}
            return 0.0;
        }
    };

     
    class image_filter_bicubic
    {
        static double pow3(double x)
        {
            return (x <= 0.0) ? 0.0 : x * x * x;
        }

    public:
        static double radius() { return 2.0; }
        static double calc_weight(double x)
        {
            return
                (1.0/6.0) * 
                (pow3(x + 2) - 4 * pow3(x + 1) + 6 * pow3(x) - 4 * pow3(x - 1));
        }
    };

     
    class image_filter_kaiser
    {
        double a;
        double i0a;
        double epsilon;

    public:
        image_filter_kaiser(double b = 6.33) :
            a(b), epsilon(1e-12)
        {
            i0a = 1.0 / bessel_i0(b);
        }

        static double radius() { return 1.0; }
        double calc_weight(double x) const
        {
            return bessel_i0(a * std::sqrt(1. - x * x)) * i0a;
        }

    private:
        double bessel_i0(double x) const
        {
            int i;
            double sum, y, t;

            sum = 1.;
            y = x * x / 4.;
            t = y;
        
            for(i = 2; t > epsilon; i++)
            {
                sum += t;
                t *= (double)y / (i * i);
            }
            return sum;
        }
    };

     
    struct image_filter_catrom
    {
        static double radius() { return 2.0; }
        static double calc_weight(double x)
        {
            if(x <  1.0) return 0.5 * (2.0 + x * x * (-5.0 + x * 3.0));
            if(x <  2.0) return 0.5 * (4.0 + x * (-8.0 + x * (5.0 - x)));
            return 0.;
        }
    };

     
    class image_filter_mitchell
    {
        double p0, p2, p3;
        double q0, q1, q2, q3;

    public:
        image_filter_mitchell(double b = 1.0/3.0, double c = 1.0/3.0) :
            p0((6.0 - 2.0 * b) / 6.0),
            p2((-18.0 + 12.0 * b + 6.0 * c) / 6.0),
            p3((12.0 - 9.0 * b - 6.0 * c) / 6.0),
            q0((8.0 * b + 24.0 * c) / 6.0),
            q1((-12.0 * b - 48.0 * c) / 6.0),
            q2((6.0 * b + 30.0 * c) / 6.0),
            q3((-b - 6.0 * c) / 6.0)
        {}

        static double radius() { return 2.0; }
        double calc_weight(double x) const
        {
            if(x < 1.0) return p0 + x * x * (p2 + x * p3);
            if(x < 2.0) return q0 + x * (q1 + x * (q2 + x * q3));
            return 0.0;
        }
    };


     
    struct image_filter_spline16
    {
        static double radius() { return 2.0; }
        static double calc_weight(double x)
        {
            if(x < 1.0)
            {
                return ((x - 9.0/5.0 ) * x - 1.0/5.0 ) * x + 1.0;
            }
            return ((-1.0/3.0 * (x-1) + 4.0/5.0) * (x-1) - 7.0/15.0 ) * (x-1);
        }
    };


     
    struct image_filter_spline36
    {
        static double radius() { return 3.0; }
        static double calc_weight(double x)
        {
           if(x < 1.0)
           {
              return ((13.0/11.0 * x - 453.0/209.0) * x - 3.0/209.0) * x + 1.0;
           }
           if(x < 2.0)
           {
              return ((-6.0/11.0 * (x-1) + 270.0/209.0) * (x-1) - 156.0/ 209.0) * (x-1);
           }
           return ((1.0/11.0 * (x-2) - 45.0/209.0) * (x-2) +  26.0/209.0) * (x-2);
        }
    };


     
    struct image_filter_gaussian
    {
        static double radius() { return 2.0; }
        static double calc_weight(double x) 
        {
            return std::exp(-2.0 * x * x) * std::sqrt(2.0 / pi);
        }
    };


     
    struct image_filter_bessel
    {
        static double radius() { return 3.2383; } 
        static double calc_weight(double x)
        {
            return (x == 0.0) ? pi / 4.0 : besj(pi * x, 1) / (2.0 * x);
        }
    };


     
    class image_filter_sinc
    {
    public:
        image_filter_sinc(double r) : m_radius(r < 2.0 ? 2.0 : r) {}
        double radius() const { return m_radius; }
        double calc_weight(double x) const
        {
            if(x == 0.0) return 1.0;
            x *= pi;
            return std::sin(x) / x;
        }
    private:
        double m_radius;
    };


     
    class image_filter_lanczos
    {
    public:
        image_filter_lanczos(double r) : m_radius(r < 2.0 ? 2.0 : r) {}
        double radius() const { return m_radius; }
        double calc_weight(double x) const
        {
           if(x == 0.0) return 1.0;
           if(x > m_radius) return 0.0;
           x *= pi;
           double xr = x / m_radius;
           return (std::sin(x) / x) * (std::sin(xr) / xr);
        }
    private:
        double m_radius;
    };


     
    class image_filter_blackman
    {
    public:
        image_filter_blackman(double r) : m_radius(r < 2.0 ? 2.0 : r) {}
        double radius() const { return m_radius; }
        double calc_weight(double x) const
        {
           if(x == 0.0) return 1.0;
           if(x > m_radius) return 0.0;
           x *= pi;
           double xr = x / m_radius;
           return (std::sin(x) / x) * (0.42 + 0.5*std::cos(xr) + 0.08*std::cos(2*xr));
        }
    private:
        double m_radius;
    };

     
    class image_filter_sinc36 : public image_filter_sinc
    { public: image_filter_sinc36() : image_filter_sinc(3.0){} };

     
    class image_filter_sinc64 : public image_filter_sinc
    { public: image_filter_sinc64() : image_filter_sinc(4.0){} };

     
    class image_filter_sinc100 : public image_filter_sinc
    { public: image_filter_sinc100() : image_filter_sinc(5.0){} };

     
    class image_filter_sinc144 : public image_filter_sinc
    { public: image_filter_sinc144() : image_filter_sinc(6.0){} };

     
    class image_filter_sinc196 : public image_filter_sinc
    { public: image_filter_sinc196() : image_filter_sinc(7.0){} };

     
    class image_filter_sinc256 : public image_filter_sinc
    { public: image_filter_sinc256() : image_filter_sinc(8.0){} };

     
    class image_filter_lanczos36 : public image_filter_lanczos
    { public: image_filter_lanczos36() : image_filter_lanczos(3.0){} };

     
    class image_filter_lanczos64 : public image_filter_lanczos
    { public: image_filter_lanczos64() : image_filter_lanczos(4.0){} };

     
    class image_filter_lanczos100 : public image_filter_lanczos
    { public: image_filter_lanczos100() : image_filter_lanczos(5.0){} };

     
    class image_filter_lanczos144 : public image_filter_lanczos
    { public: image_filter_lanczos144() : image_filter_lanczos(6.0){} };

     
    class image_filter_lanczos196 : public image_filter_lanczos
    { public: image_filter_lanczos196() : image_filter_lanczos(7.0){} };

     
    class image_filter_lanczos256 : public image_filter_lanczos
    { public: image_filter_lanczos256() : image_filter_lanczos(8.0){} };

     
    class image_filter_blackman36 : public image_filter_blackman
    { public: image_filter_blackman36() : image_filter_blackman(3.0){} };

     
    class image_filter_blackman64 : public image_filter_blackman
    { public: image_filter_blackman64() : image_filter_blackman(4.0){} };

     
    class image_filter_blackman100 : public image_filter_blackman
    { public: image_filter_blackman100() : image_filter_blackman(5.0){} };

     
    class image_filter_blackman144 : public image_filter_blackman
    { public: image_filter_blackman144() : image_filter_blackman(6.0){} };

     
    class image_filter_blackman196 : public image_filter_blackman
    { public: image_filter_blackman196() : image_filter_blackman(7.0){} };

     
    class image_filter_blackman256 : public image_filter_blackman
    { public: image_filter_blackman256() : image_filter_blackman(8.0){} };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_LINE_AA_BASICS_INCLUDED
#define AGG_LINE_AA_BASICS_INCLUDED

#include <cstdlib>

namespace agg
{

     

     
    enum line_subpixel_scale_e
    {
        line_subpixel_shift = 8,                           
        line_subpixel_scale  = 1 << line_subpixel_shift,   
        line_subpixel_mask  = line_subpixel_scale - 1,     
        line_max_coord      = (1 << 28) - 1,               
        line_max_length = 1 << (line_subpixel_shift + 10)  
    };

     
    enum line_mr_subpixel_scale_e
    {
        line_mr_subpixel_shift = 4,                            
        line_mr_subpixel_scale = 1 << line_mr_subpixel_shift,  
        line_mr_subpixel_mask  = line_mr_subpixel_scale - 1    
    };

     
    AGG_INLINE int line_mr(int x) 
    { 
        return x >> (+line_subpixel_shift - line_mr_subpixel_shift); 
    }

     
    AGG_INLINE int line_hr(int x) 
    { 
        return x << (+line_subpixel_shift - line_mr_subpixel_shift); 
    }

     
    AGG_INLINE int line_dbl_hr(int x) 
    { 
        return x << line_subpixel_shift;
    }

     
    struct line_coord
    {
        AGG_INLINE static int conv(double x)
        {
            return iround(x * (double)line_subpixel_scale);
        }
    };

     
    struct line_coord_sat
    {
        AGG_INLINE static int conv(double x)
        {
            return saturation<line_max_coord>::iround(x * (double)line_subpixel_scale);
        }
    };

     
    struct line_parameters
    {
         
        line_parameters() {}
        line_parameters(int x1_, int y1_, int x2_, int y2_, int len_) :
            x1(x1_), y1(y1_), x2(x2_), y2(y2_), 
            dx(std::abs(x2_ - x1_)),
            dy(std::abs(y2_ - y1_)),
            sx((x2_ > x1_) ? 1 : -1),
            sy((y2_ > y1_) ? 1 : -1),
            vertical(dy >= dx),
            inc(vertical ? sy : sx),
            len(len_),
            octant((sy & 4) | (sx & 2) | int(vertical))
        {
        }

         
        unsigned orthogonal_quadrant() const { return s_orthogonal_quadrant[octant]; }
        unsigned diagonal_quadrant()   const { return s_diagonal_quadrant[octant];   }

         
        bool same_orthogonal_quadrant(const line_parameters& lp) const
        {
            return s_orthogonal_quadrant[octant] == s_orthogonal_quadrant[lp.octant];
        }

         
        bool same_diagonal_quadrant(const line_parameters& lp) const
        {
            return s_diagonal_quadrant[octant] == s_diagonal_quadrant[lp.octant];
        }

         
        void divide(line_parameters& lp1, line_parameters& lp2) const
        {
            int xmid = (x1 + x2) >> 1;
            int ymid = (y1 + y2) >> 1;
            int len2 = len >> 1;

            lp1 = *this;
            lp2 = *this;

            lp1.x2  = xmid;
            lp1.y2  = ymid;
            lp1.len = len2;
            lp1.dx  = std::abs(lp1.x2 - lp1.x1);
            lp1.dy  = std::abs(lp1.y2 - lp1.y1);

            lp2.x1  = xmid;
            lp2.y1  = ymid;
            lp2.len = len2;
            lp2.dx  = std::abs(lp2.x2 - lp2.x1);
            lp2.dy  = std::abs(lp2.y2 - lp2.y1);
        }
        
         
        int x1, y1, x2, y2, dx, dy, sx, sy;
        bool vertical;
        int inc;
        int len;
        int octant;

         
        static const int8u s_orthogonal_quadrant[8];
        static const int8u s_diagonal_quadrant[8];
    };



     

     
    void bisectrix(const line_parameters& l1, 
                   const line_parameters& l2, 
                   int* x, int* y);


     
    void inline fix_degenerate_bisectrix_start(const line_parameters& lp, 
                                               int* x, int* y)
    {
        int d = iround((double(*x - lp.x2) * double(lp.y2 - lp.y1) - 
                        double(*y - lp.y2) * double(lp.x2 - lp.x1)) / lp.len);
        if(d < line_subpixel_scale/2)
        {
            *x = lp.x1 + (lp.y2 - lp.y1);
            *y = lp.y1 - (lp.x2 - lp.x1);
        }
    }


     
    void inline fix_degenerate_bisectrix_end(const line_parameters& lp, 
                                             int* x, int* y)
    {
        int d = iround((double(*x - lp.x2) * double(lp.y2 - lp.y1) - 
                        double(*y - lp.y2) * double(lp.x2 - lp.x1)) / lp.len);
        if(d < line_subpixel_scale/2)
        {
            *x = lp.x2 + (lp.y2 - lp.y1);
            *y = lp.y2 - (lp.x2 - lp.x1);
        }
    }


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_PATH_LENGTH_INCLUDED
#define AGG_PATH_LENGTH_INCLUDED


namespace agg
{
    template<class VertexSource> 
    double path_length(VertexSource& vs, unsigned path_id = 0)
    {
        double len = 0.0;
        double start_x = 0.0;
        double start_y = 0.0;
        double x1 = 0.0;
        double y1 = 0.0;
        double x2 = 0.0;
        double y2 = 0.0;
        bool first = true;

        unsigned cmd;
        vs.rewind(path_id);
        while(!is_stop(cmd = vs.vertex(&x2, &y2)))
        {
            if(is_vertex(cmd))
            {
                if(first || is_move_to(cmd))
                {
                    start_x = x2;
                    start_y = y2;
                }
                else
                {
                    len += calc_distance(x1, y1, x2, y2);
                }
                x1 = x2;
                y1 = y2;
                first = false;
            }
            else
            {
                if(is_close(cmd) && !first)
                {
                    len += calc_distance(x1, y1, start_x, start_y);
                }
            }
        }
        return len;
    }
}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_PATH_STORAGE_INCLUDED
#define AGG_PATH_STORAGE_INCLUDED

#include <cstring>

namespace agg
{


     
    template<class T, unsigned BlockShift=8, unsigned BlockPool=256>
    class vertex_block_storage
    {
    public:
         
        enum block_scale_e
        {
            block_shift = BlockShift,
            block_size  = 1 << block_shift,
            block_mask  = block_size - 1,
            block_pool  = BlockPool
        };

        typedef T value_type;
        typedef vertex_block_storage<T, BlockShift, BlockPool> self_type;

        ~vertex_block_storage();
        vertex_block_storage();
        vertex_block_storage(const self_type& v);
        const self_type& operator = (const self_type& ps);

        void remove_all();
        void free_all();

        void add_vertex(double x, double y, unsigned cmd);
        void modify_vertex(unsigned idx, double x, double y);
        void modify_vertex(unsigned idx, double x, double y, unsigned cmd);
        void modify_command(unsigned idx, unsigned cmd);
        void swap_vertices(unsigned v1, unsigned v2);

        unsigned last_command() const;
        unsigned last_vertex(double* x, double* y) const;
        unsigned prev_vertex(double* x, double* y) const;

        double last_x() const;
        double last_y() const;

        unsigned total_vertices() const;
        unsigned vertex(unsigned idx, double* x, double* y) const;
        unsigned command(unsigned idx) const;

    private:
        void   allocate_block(unsigned nb);
        int8u* storage_ptrs(T** xy_ptr);

    private:
        unsigned m_total_vertices;
        unsigned m_total_blocks;
        unsigned m_max_blocks;
        T**      m_coord_blocks;
        int8u**  m_cmd_blocks;
    };


     
    template<class T, unsigned S, unsigned P>
    void vertex_block_storage<T,S,P>::free_all()
    {
        if(m_total_blocks)
        {
            T** coord_blk = m_coord_blocks + m_total_blocks - 1;
            while(m_total_blocks--)
            {
                pod_allocator<T>::deallocate(
                    *coord_blk,
                    block_size * 2 + 
                    block_size / (sizeof(T) / sizeof(unsigned char)));
                --coord_blk;
            }
            pod_allocator<T*>::deallocate(m_coord_blocks, m_max_blocks * 2);
            m_total_blocks   = 0;
            m_max_blocks     = 0;
            m_coord_blocks   = 0;
            m_cmd_blocks     = 0;
            m_total_vertices = 0;
        }
    }

     
    template<class T, unsigned S, unsigned P>
    vertex_block_storage<T,S,P>::~vertex_block_storage()
    {
        free_all();
    }

     
    template<class T, unsigned S, unsigned P>
    vertex_block_storage<T,S,P>::vertex_block_storage() :
        m_total_vertices(0),
        m_total_blocks(0),
        m_max_blocks(0),
        m_coord_blocks(0),
        m_cmd_blocks(0)
    {
    }

     
    template<class T, unsigned S, unsigned P>
    vertex_block_storage<T,S,P>::vertex_block_storage(const vertex_block_storage<T,S,P>& v) :
        m_total_vertices(0),
        m_total_blocks(0),
        m_max_blocks(0),
        m_coord_blocks(0),
        m_cmd_blocks(0)
    {
        *this = v;
    }

     
    template<class T, unsigned S, unsigned P>
    const vertex_block_storage<T,S,P>& 
    vertex_block_storage<T,S,P>::operator = (const vertex_block_storage<T,S,P>& v)
    {
        remove_all();
        unsigned i;
        for(i = 0; i < v.total_vertices(); i++)
        {
            double x, y;
            unsigned cmd = v.vertex(i, &x, &y);
            add_vertex(x, y, cmd);
        }
	    return *this;
    }

     
    template<class T, unsigned S, unsigned P>
    inline void vertex_block_storage<T,S,P>::remove_all()
    {
        m_total_vertices = 0;
    }

     
    template<class T, unsigned S, unsigned P>
    inline void vertex_block_storage<T,S,P>::add_vertex(double x, double y, 
                                                        unsigned cmd)
    {
        T* coord_ptr = 0;
        *storage_ptrs(&coord_ptr) = (int8u)cmd;
        coord_ptr[0] = T(x);
        coord_ptr[1] = T(y);
        m_total_vertices++;
    }

     
    template<class T, unsigned S, unsigned P>
    inline void vertex_block_storage<T,S,P>::modify_vertex(unsigned idx, 
                                                           double x, double y)
    {
        T* pv = m_coord_blocks[idx >> block_shift] + ((idx & block_mask) << 1);
        pv[0] = T(x);
        pv[1] = T(y);
    }

     
    template<class T, unsigned S, unsigned P>
    inline void vertex_block_storage<T,S,P>::modify_vertex(unsigned idx, 
                                                           double x, double y, 
                                                           unsigned cmd)
    {
        unsigned block = idx >> block_shift;
        unsigned offset = idx & block_mask;
        T* pv = m_coord_blocks[block] + (offset << 1);
        pv[0] = T(x);
        pv[1] = T(y);
        m_cmd_blocks[block][offset] = (int8u)cmd;
    }

     
    template<class T, unsigned S, unsigned P>
    inline void vertex_block_storage<T,S,P>::modify_command(unsigned idx, 
                                                            unsigned cmd)
    {
        m_cmd_blocks[idx >> block_shift][idx & block_mask] = (int8u)cmd;
    }

     
    template<class T, unsigned S, unsigned P>
    inline void vertex_block_storage<T,S,P>::swap_vertices(unsigned v1, unsigned v2)
    {
        unsigned b1 = v1 >> block_shift;
        unsigned b2 = v2 >> block_shift;
        unsigned o1 = v1 & block_mask;
        unsigned o2 = v2 & block_mask;
        T* pv1 = m_coord_blocks[b1] + (o1 << 1);
        T* pv2 = m_coord_blocks[b2] + (o2 << 1);
        T  val;
        val = pv1[0]; pv1[0] = pv2[0]; pv2[0] = val;
        val = pv1[1]; pv1[1] = pv2[1]; pv2[1] = val;
        int8u cmd = m_cmd_blocks[b1][o1];
        m_cmd_blocks[b1][o1] = m_cmd_blocks[b2][o2];
        m_cmd_blocks[b2][o2] = cmd;
    }

     
    template<class T, unsigned S, unsigned P>
    inline unsigned vertex_block_storage<T,S,P>::last_command() const
    {
        if(m_total_vertices) return command(m_total_vertices - 1);
        return path_cmd_stop;
    }

     
    template<class T, unsigned S, unsigned P>
    inline unsigned vertex_block_storage<T,S,P>::last_vertex(double* x, double* y) const
    {
        if(m_total_vertices) return vertex(m_total_vertices - 1, x, y);
        return path_cmd_stop;
    }

     
    template<class T, unsigned S, unsigned P>
    inline unsigned vertex_block_storage<T,S,P>::prev_vertex(double* x, double* y) const
    {
        if(m_total_vertices > 1) return vertex(m_total_vertices - 2, x, y);
        return path_cmd_stop;
    }

     
    template<class T, unsigned S, unsigned P>
    inline double vertex_block_storage<T,S,P>::last_x() const
    {
        if(m_total_vertices)
        {
            unsigned idx = m_total_vertices - 1;
            return m_coord_blocks[idx >> block_shift][(idx & block_mask) << 1];
        }
        return 0.0;
    }

     
    template<class T, unsigned S, unsigned P>
    inline double vertex_block_storage<T,S,P>::last_y() const
    {
        if(m_total_vertices)
        {
            unsigned idx = m_total_vertices - 1;
            return m_coord_blocks[idx >> block_shift][((idx & block_mask) << 1) + 1];
        }
        return 0.0;
    }

     
    template<class T, unsigned S, unsigned P>
    inline unsigned vertex_block_storage<T,S,P>::total_vertices() const
    {
        return m_total_vertices;
    }

     
    template<class T, unsigned S, unsigned P>
    inline unsigned vertex_block_storage<T,S,P>::vertex(unsigned idx, 
                                                        double* x, double* y) const
    {
        unsigned nb = idx >> block_shift;
        const T* pv = m_coord_blocks[nb] + ((idx & block_mask) << 1);
        *x = pv[0];
        *y = pv[1];
        return m_cmd_blocks[nb][idx & block_mask];
    }

     
    template<class T, unsigned S, unsigned P>
    inline unsigned vertex_block_storage<T,S,P>::command(unsigned idx) const
    {
        return m_cmd_blocks[idx >> block_shift][idx & block_mask];
    }

     
    template<class T, unsigned S, unsigned P>
    void vertex_block_storage<T,S,P>::allocate_block(unsigned nb)
    {
        if(nb >= m_max_blocks) 
        {
            T** new_coords = 
                pod_allocator<T*>::allocate((m_max_blocks + block_pool) * 2);

            unsigned char** new_cmds = 
                (unsigned char**)(new_coords + m_max_blocks + block_pool);

            if(m_coord_blocks)
            {
                std::memcpy(new_coords, 
                       m_coord_blocks, 
                       m_max_blocks * sizeof(T*));

                std::memcpy(new_cmds, 
                       m_cmd_blocks, 
                       m_max_blocks * sizeof(unsigned char*));

                pod_allocator<T*>::deallocate(m_coord_blocks, m_max_blocks * 2);
            }
            m_coord_blocks = new_coords;
            m_cmd_blocks   = new_cmds;
            m_max_blocks  += block_pool;
        }
        m_coord_blocks[nb] = 
            pod_allocator<T>::allocate(block_size * 2 + 
                   block_size / (sizeof(T) / sizeof(unsigned char)));

        m_cmd_blocks[nb]  = 
            (unsigned char*)(m_coord_blocks[nb] + block_size * 2);

        m_total_blocks++;
    }

     
    template<class T, unsigned S, unsigned P>
    int8u* vertex_block_storage<T,S,P>::storage_ptrs(T** xy_ptr)
    {
        unsigned nb = m_total_vertices >> block_shift;
        if(nb >= m_total_blocks)
        {
            allocate_block(nb);
        }
        *xy_ptr = m_coord_blocks[nb] + ((m_total_vertices & block_mask) << 1);
        return m_cmd_blocks[nb] + (m_total_vertices & block_mask);
    }




     
    template<class T> class poly_plain_adaptor
    {
    public:
        typedef T value_type;

        poly_plain_adaptor() : 
            m_data(0), 
            m_ptr(0),
            m_end(0),
            m_closed(false),
            m_stop(false)
        {}

        poly_plain_adaptor(const T* data, unsigned num_points, bool closed) :
            m_data(data), 
            m_ptr(data),
            m_end(data + num_points * 2),
            m_closed(closed),
            m_stop(false)
        {}

        void init(const T* data, unsigned num_points, bool closed)
        {
            m_data = data;
            m_ptr = data;
            m_end = data + num_points * 2;
            m_closed = closed;
            m_stop = false;
        }

        void rewind(unsigned)
        {
            m_ptr = m_data;
            m_stop = false;
        }

        unsigned vertex(double* x, double* y)
        {
            if(m_ptr < m_end)
            {
                bool first = m_ptr == m_data;
                *x = *m_ptr++;
                *y = *m_ptr++;
                return first ? path_cmd_move_to : path_cmd_line_to;
            }
            *x = *y = 0.0;
            if(m_closed && !m_stop)
            {
                m_stop = true;
                return +path_cmd_end_poly | path_flags_close;
            }
            return path_cmd_stop;
        }

    private:
        const T* m_data;
        const T* m_ptr;
        const T* m_end;
        bool     m_closed;
        bool     m_stop;
    };





     
    template<class Container> class poly_container_adaptor
    {
    public:
        typedef typename Container::value_type vertex_type;

        poly_container_adaptor() : 
            m_container(0), 
            m_index(0),
            m_closed(false),
            m_stop(false)
        {}

        poly_container_adaptor(const Container& data, bool closed) :
            m_container(&data), 
            m_index(0),
            m_closed(closed),
            m_stop(false)
        {}

        void init(const Container& data, bool closed)
        {
            m_container = &data;
            m_index = 0;
            m_closed = closed;
            m_stop = false;
        }

        void rewind(unsigned)
        {
            m_index = 0;
            m_stop = false;
        }

        unsigned vertex(double* x, double* y)
        {
            if(m_index < m_container->size())
            {
                bool first = m_index == 0;
                const vertex_type& v = (*m_container)[m_index++];
                *x = v.x;
                *y = v.y;
                return first ? path_cmd_move_to : path_cmd_line_to;
            }
            *x = *y = 0.0;
            if(m_closed && !m_stop)
            {
                m_stop = true;
                return +path_cmd_end_poly | path_flags_close;
            }
            return path_cmd_stop;
        }

    private:
        const Container* m_container;
        unsigned m_index;
        bool     m_closed;
        bool     m_stop;
    };



     
    template<class Container> class poly_container_reverse_adaptor
    {
    public:
        typedef typename Container::value_type vertex_type;

        poly_container_reverse_adaptor() : 
            m_container(0), 
            m_index(-1),
            m_closed(false),
            m_stop(false)
        {}

        poly_container_reverse_adaptor(Container& data, bool closed) :
            m_container(&data), 
            m_index(-1),
            m_closed(closed),
            m_stop(false)
        {}

        void init(Container& data, bool closed)
        {
            m_container = &data;
            m_index = m_container->size() - 1;
            m_closed = closed;
            m_stop = false;
        }

        void rewind(unsigned)
        {
            m_index = m_container->size() - 1;
            m_stop = false;
        }

        unsigned vertex(double* x, double* y)
        {
            if(m_index >= 0)
            {
                bool first = m_index == int(m_container->size() - 1);
                const vertex_type& v = (*m_container)[m_index--];
                *x = v.x;
                *y = v.y;
                return first ? path_cmd_move_to : path_cmd_line_to;
            }
            *x = *y = 0.0;
            if(m_closed && !m_stop)
            {
                m_stop = true;
                return +path_cmd_end_poly | path_flags_close;
            }
            return path_cmd_stop;
        }

    private:
        Container* m_container;
        int  m_index;
        bool m_closed;
        bool m_stop;
    };





     
    class line_adaptor
    {
    public:
        typedef double value_type;

        line_adaptor() : m_line(m_coord, 2, false) {}
        line_adaptor(double x1, double y1, double x2, double y2) :
            m_line(m_coord, 2, false)
        {
            m_coord[0] = x1;
            m_coord[1] = y1;
            m_coord[2] = x2;
            m_coord[3] = y2;
        }
        
        void init(double x1, double y1, double x2, double y2)
        {
            m_coord[0] = x1;
            m_coord[1] = y1;
            m_coord[2] = x2;
            m_coord[3] = y2;
            m_line.rewind(0);
        }

        void rewind(unsigned)
        {
            m_line.rewind(0);
        }

        unsigned vertex(double* x, double* y)
        {
            return m_line.vertex(x, y);
        }

    private:
        double                     m_coord[4];
        poly_plain_adaptor<double> m_line;
    };













     
     
     
     
     
     
     
     
     
     
     
     
    template<class VertexContainer> class path_base
    {
    public:
        typedef VertexContainer            container_type;
        typedef path_base<VertexContainer> self_type;

         
        path_base() : m_vertices(), m_iterator(0) {}
        void remove_all() { m_vertices.remove_all(); m_iterator = 0; }
        void free_all()   { m_vertices.free_all();   m_iterator = 0; }

         
         
        unsigned start_new_path();

        void move_to(double x, double y);
        void move_rel(double dx, double dy);

        void line_to(double x, double y);
        void line_rel(double dx, double dy);

        void hline_to(double x);
        void hline_rel(double dx);

        void vline_to(double y);
        void vline_rel(double dy);

        void arc_to(double rx, double ry,
                    double angle,
                    bool large_arc_flag,
                    bool sweep_flag,
                    double x, double y);

        void arc_rel(double rx, double ry,
                     double angle,
                     bool large_arc_flag,
                     bool sweep_flag,
                     double dx, double dy);

        void curve3(double x_ctrl, double y_ctrl, 
                    double x_to,   double y_to);

        void curve3_rel(double dx_ctrl, double dy_ctrl, 
                        double dx_to,   double dy_to);

        void curve3(double x_to, double y_to);

        void curve3_rel(double dx_to, double dy_to);

        void curve4(double x_ctrl1, double y_ctrl1, 
                    double x_ctrl2, double y_ctrl2, 
                    double x_to,    double y_to);

        void curve4_rel(double dx_ctrl1, double dy_ctrl1, 
                        double dx_ctrl2, double dy_ctrl2, 
                        double dx_to,    double dy_to);

        void curve4(double x_ctrl2, double y_ctrl2, 
                    double x_to,    double y_to);

        void curve4_rel(double x_ctrl2, double y_ctrl2, 
                        double x_to,    double y_to);


        void end_poly(unsigned flags = path_flags_close);
        void close_polygon(unsigned flags = path_flags_none);

         
         
        const container_type& vertices() const { return m_vertices; } 
              container_type& vertices()       { return m_vertices; } 

        unsigned total_vertices() const;

        void rel_to_abs(double* x, double* y) const;

        unsigned last_vertex(double* x, double* y) const;
        unsigned prev_vertex(double* x, double* y) const;

        double last_x() const;
        double last_y() const;

        unsigned vertex(unsigned idx, double* x, double* y) const;
        unsigned command(unsigned idx) const;

        void modify_vertex(unsigned idx, double x, double y);
        void modify_vertex(unsigned idx, double x, double y, unsigned cmd);
        void modify_command(unsigned idx, unsigned cmd);

         
         
        void     rewind(unsigned path_id);
        unsigned vertex(double* x, double* y);

         
         
         
         
         
        unsigned arrange_polygon_orientation(unsigned start, path_flags_e orientation);
        unsigned arrange_orientations(unsigned path_id, path_flags_e orientation);
        void     arrange_orientations_all_paths(path_flags_e orientation);
        void     invert_polygon(unsigned start);

         
         
         
        void flip_x(double x1, double x2);
        void flip_y(double y1, double y2);

         
         
        template<class VertexSource> 
        void concat_path(VertexSource& vs, unsigned path_id = 0)
        {
            double x = 0;
            double y = 0;
            unsigned cmd;
            vs.rewind(path_id);
            while(!is_stop(cmd = vs.vertex(&x, &y)))
            {
                m_vertices.add_vertex(x, y, cmd);
            }
        }

         
         
         
        template<class VertexSource> 
        void join_path(VertexSource& vs, unsigned path_id = 0)
        {
            double x, y;
            unsigned cmd;
            vs.rewind(path_id);
            cmd = vs.vertex(&x, &y);
            if(!is_stop(cmd))
            {
                if(is_vertex(cmd))
                {
                    double x0, y0;
                    unsigned cmd0 = last_vertex(&x0, &y0);
                    if(is_vertex(cmd0))
                    {
                        if(calc_distance(x, y, x0, y0) > vertex_dist_epsilon)
                        {
                            if(is_move_to(cmd)) cmd = path_cmd_line_to;
                            m_vertices.add_vertex(x, y, cmd);
                        }
                    }
                    else
                    {
                        if(is_stop(cmd0))
                        {
                            cmd = path_cmd_move_to;
                        }
                        else
                        {
                            if(is_move_to(cmd)) cmd = path_cmd_line_to;
                        }
                        m_vertices.add_vertex(x, y, cmd);
                    }
                }
                while(!is_stop(cmd = vs.vertex(&x, &y)))
                {
                    m_vertices.add_vertex(x, y, is_move_to(cmd) ? 
                                                    unsigned(path_cmd_line_to) : 
                                                    cmd);
                }
            }
        }

         
         
        template<class T> void concat_poly(const T* data, 
                                           unsigned num_points,
                                           bool closed)
        {
            poly_plain_adaptor<T> poly(data, num_points, closed);
            concat_path(poly);
        }

         
         
        template<class T> void join_poly(const T* data, 
                                         unsigned num_points,
                                         bool closed)
        {
            poly_plain_adaptor<T> poly(data, num_points, closed);
            join_path(poly);
        }

         
        void translate(double dx, double dy, unsigned path_id=0);
        void translate_all_paths(double dx, double dy);

         
        template<class Trans>
        void transform(const Trans& trans, unsigned path_id=0)
        {
            unsigned num_ver = m_vertices.total_vertices();
            for(; path_id < num_ver; path_id++)
            {
                double x, y;
                unsigned cmd = m_vertices.vertex(path_id, &x, &y);
                if(is_stop(cmd)) break;
                if(is_vertex(cmd))
                {
                    trans.transform(&x, &y);
                    m_vertices.modify_vertex(path_id, x, y);
                }
            }
        }

         
        template<class Trans>
        void transform_all_paths(const Trans& trans)
        {
            unsigned idx;
            unsigned num_ver = m_vertices.total_vertices();
            for(idx = 0; idx < num_ver; idx++)
            {
                double x, y;
                if(is_vertex(m_vertices.vertex(idx, &x, &y)))
                {
                    trans.transform(&x, &y);
                    m_vertices.modify_vertex(idx, x, y);
                }
            }
        }


         
         
         
         
        unsigned align_path(unsigned idx = 0)
        {
            if (idx >= total_vertices() || !is_move_to(command(idx))) 
            {
                return total_vertices();
            }

            double start_x, start_y;
            for (; idx < total_vertices() && is_move_to(command(idx)); ++idx)
            {
                vertex(idx, &start_x, &start_y);
            }
            while (idx < total_vertices() && is_drawing(command(idx)))
                ++idx;

            double x, y;
            if (is_drawing(vertex(idx - 1, &x, &y)) &&
                is_equal_eps(x, start_x, 1e-8) &&
                is_equal_eps(y, start_y, 1e-8))
            {
                modify_vertex(idx - 1, start_x, start_y);
            }

            while (idx < total_vertices() && !is_move_to(command(idx)))
                ++idx;
            return idx;
        }

        void align_all_paths()
        {
            for (unsigned i = 0; i < total_vertices(); i = align_path(i));
        }


    private:
        unsigned perceive_polygon_orientation(unsigned start, unsigned end);
        void     invert_polygon(unsigned start, unsigned end);

        VertexContainer m_vertices;
        unsigned        m_iterator;
    };

     
    template<class VC> 
    unsigned path_base<VC>::start_new_path()
    {
        if(!is_stop(m_vertices.last_command()))
        {
            m_vertices.add_vertex(0.0, 0.0, path_cmd_stop);
        }
        return m_vertices.total_vertices();
    }


     
    template<class VC> 
    inline void path_base<VC>::rel_to_abs(double* x, double* y) const
    {
        if(m_vertices.total_vertices())
        {
            double x2;
            double y2;
            if(is_vertex(m_vertices.last_vertex(&x2, &y2)))
            {
                *x += x2;
                *y += y2;
            }
        }
    }

     
    template<class VC> 
    inline void path_base<VC>::move_to(double x, double y)
    {
        m_vertices.add_vertex(x, y, path_cmd_move_to);
    }

     
    template<class VC> 
    inline void path_base<VC>::move_rel(double dx, double dy)
    {
        rel_to_abs(&dx, &dy);
        m_vertices.add_vertex(dx, dy, path_cmd_move_to);
    }

     
    template<class VC> 
    inline void path_base<VC>::line_to(double x, double y)
    {
        m_vertices.add_vertex(x, y, path_cmd_line_to);
    }

     
    template<class VC> 
    inline void path_base<VC>::line_rel(double dx, double dy)
    {
        rel_to_abs(&dx, &dy);
        m_vertices.add_vertex(dx, dy, path_cmd_line_to);
    }

     
    template<class VC> 
    inline void path_base<VC>::hline_to(double x)
    {
        m_vertices.add_vertex(x, last_y(), path_cmd_line_to);
    }

     
    template<class VC> 
    inline void path_base<VC>::hline_rel(double dx)
    {
        double dy = 0;
        rel_to_abs(&dx, &dy);
        m_vertices.add_vertex(dx, dy, path_cmd_line_to);
    }

     
    template<class VC> 
    inline void path_base<VC>::vline_to(double y)
    {
        m_vertices.add_vertex(last_x(), y, path_cmd_line_to);
    }

     
    template<class VC> 
    inline void path_base<VC>::vline_rel(double dy)
    {
        double dx = 0;
        rel_to_abs(&dx, &dy);
        m_vertices.add_vertex(dx, dy, path_cmd_line_to);
    }

     
    template<class VC> 
    void path_base<VC>::arc_to(double rx, double ry,
                               double angle,
                               bool large_arc_flag,
                               bool sweep_flag,
                               double x, double y)
    {
        if(m_vertices.total_vertices() && is_vertex(m_vertices.last_command()))
        {
            const double epsilon = 1e-30;
            double x0 = 0.0;
            double y0 = 0.0;
            m_vertices.last_vertex(&x0, &y0);

            rx = std::fabs(rx);
            ry = std::fabs(ry);

             
             
            if(rx < epsilon || ry < epsilon) 
            {
                line_to(x, y);
                return;
            }

            if(calc_distance(x0, y0, x, y) < epsilon)
            {
                 
                 
                return;
            }
            bezier_arc_svg a(x0, y0, rx, ry, angle, large_arc_flag, sweep_flag, x, y);
            if(a.radii_ok())
            {
                join_path(a);
            }
            else
            {
                line_to(x, y);
            }
        }
        else
        {
            move_to(x, y);
        }
    }

     
    template<class VC> 
    void path_base<VC>::arc_rel(double rx, double ry,
                                double angle,
                                bool large_arc_flag,
                                bool sweep_flag,
                                double dx, double dy)
    {
        rel_to_abs(&dx, &dy);
        arc_to(rx, ry, angle, large_arc_flag, sweep_flag, dx, dy);
    }

     
    template<class VC> 
    void path_base<VC>::curve3(double x_ctrl, double y_ctrl, 
                               double x_to,   double y_to)
    {
        m_vertices.add_vertex(x_ctrl, y_ctrl, path_cmd_curve3);
        m_vertices.add_vertex(x_to,   y_to,   path_cmd_curve3);
    }

     
    template<class VC> 
    void path_base<VC>::curve3_rel(double dx_ctrl, double dy_ctrl, 
                                   double dx_to,   double dy_to)
    {
        rel_to_abs(&dx_ctrl, &dy_ctrl);
        rel_to_abs(&dx_to,   &dy_to);
        m_vertices.add_vertex(dx_ctrl, dy_ctrl, path_cmd_curve3);
        m_vertices.add_vertex(dx_to,   dy_to,   path_cmd_curve3);
    }

     
    template<class VC> 
    void path_base<VC>::curve3(double x_to, double y_to)
    {
        double x0;
        double y0;
        if(is_vertex(m_vertices.last_vertex(&x0, &y0)))
        {
            double x_ctrl;
            double y_ctrl; 
            unsigned cmd = m_vertices.prev_vertex(&x_ctrl, &y_ctrl);
            if(is_curve(cmd))
            {
                x_ctrl = x0 + x0 - x_ctrl;
                y_ctrl = y0 + y0 - y_ctrl;
            }
            else
            {
                x_ctrl = x0;
                y_ctrl = y0;
            }
            curve3(x_ctrl, y_ctrl, x_to, y_to);
        }
    }

     
    template<class VC> 
    void path_base<VC>::curve3_rel(double dx_to, double dy_to)
    {
        rel_to_abs(&dx_to, &dy_to);
        curve3(dx_to, dy_to);
    }

     
    template<class VC> 
    void path_base<VC>::curve4(double x_ctrl1, double y_ctrl1, 
                               double x_ctrl2, double y_ctrl2, 
                               double x_to,    double y_to)
    {
        m_vertices.add_vertex(x_ctrl1, y_ctrl1, path_cmd_curve4);
        m_vertices.add_vertex(x_ctrl2, y_ctrl2, path_cmd_curve4);
        m_vertices.add_vertex(x_to,    y_to,    path_cmd_curve4);
    }

     
    template<class VC> 
    void path_base<VC>::curve4_rel(double dx_ctrl1, double dy_ctrl1, 
                                   double dx_ctrl2, double dy_ctrl2, 
                                   double dx_to,    double dy_to)
    {
        rel_to_abs(&dx_ctrl1, &dy_ctrl1);
        rel_to_abs(&dx_ctrl2, &dy_ctrl2);
        rel_to_abs(&dx_to,    &dy_to);
        m_vertices.add_vertex(dx_ctrl1, dy_ctrl1, path_cmd_curve4);
        m_vertices.add_vertex(dx_ctrl2, dy_ctrl2, path_cmd_curve4);
        m_vertices.add_vertex(dx_to,    dy_to,    path_cmd_curve4);
    }

     
    template<class VC> 
    void path_base<VC>::curve4(double x_ctrl2, double y_ctrl2, 
                               double x_to,    double y_to)
    {
        double x0;
        double y0;
        if(is_vertex(last_vertex(&x0, &y0)))
        {
            double x_ctrl1;
            double y_ctrl1; 
            unsigned cmd = prev_vertex(&x_ctrl1, &y_ctrl1);
            if(is_curve(cmd))
            {
                x_ctrl1 = x0 + x0 - x_ctrl1;
                y_ctrl1 = y0 + y0 - y_ctrl1;
            }
            else
            {
                x_ctrl1 = x0;
                y_ctrl1 = y0;
            }
            curve4(x_ctrl1, y_ctrl1, x_ctrl2, y_ctrl2, x_to, y_to);
        }
    }

     
    template<class VC> 
    void path_base<VC>::curve4_rel(double dx_ctrl2, double dy_ctrl2, 
                                   double dx_to,    double dy_to)
    {
        rel_to_abs(&dx_ctrl2, &dy_ctrl2);
        rel_to_abs(&dx_to,    &dy_to);
        curve4(dx_ctrl2, dy_ctrl2, dx_to, dy_to);
    }

     
    template<class VC> 
    inline void path_base<VC>::end_poly(unsigned flags)
    {
        if(is_vertex(m_vertices.last_command()))
        {
            m_vertices.add_vertex(0.0, 0.0, path_cmd_end_poly | flags);
        }
    }

     
    template<class VC> 
    inline void path_base<VC>::close_polygon(unsigned flags)
    {
        end_poly(path_flags_close | flags);
    }

     
    template<class VC> 
    inline unsigned path_base<VC>::total_vertices() const
    {
        return m_vertices.total_vertices();
    }

     
    template<class VC> 
    inline unsigned path_base<VC>::last_vertex(double* x, double* y) const
    {
        return m_vertices.last_vertex(x, y);
    }

     
    template<class VC> 
    inline unsigned path_base<VC>::prev_vertex(double* x, double* y) const
    {
        return m_vertices.prev_vertex(x, y);
    }

     
    template<class VC> 
    inline double path_base<VC>::last_x() const
    {
        return m_vertices.last_x();
    }

     
    template<class VC> 
    inline double path_base<VC>::last_y() const
    {
        return m_vertices.last_y();
    }

     
    template<class VC> 
    inline unsigned path_base<VC>::vertex(unsigned idx, double* x, double* y) const
    {
        return m_vertices.vertex(idx, x, y);
    }
 
     
    template<class VC> 
    inline unsigned path_base<VC>::command(unsigned idx) const
    {
        return m_vertices.command(idx);
    }

     
    template<class VC> 
    void path_base<VC>::modify_vertex(unsigned idx, double x, double y)
    {
        m_vertices.modify_vertex(idx, x, y);
    }

     
    template<class VC> 
    void path_base<VC>::modify_vertex(unsigned idx, double x, double y, unsigned cmd)
    {
        m_vertices.modify_vertex(idx, x, y, cmd);
    }

     
    template<class VC> 
    void path_base<VC>::modify_command(unsigned idx, unsigned cmd)
    {
        m_vertices.modify_command(idx, cmd);
    }

     
    template<class VC> 
    inline void path_base<VC>::rewind(unsigned path_id)
    {
        m_iterator = path_id;
    }

     
    template<class VC> 
    inline unsigned path_base<VC>::vertex(double* x, double* y)
    {
        if(m_iterator >= m_vertices.total_vertices()) return path_cmd_stop;
        return m_vertices.vertex(m_iterator++, x, y);
    }

     
    template<class VC> 
    unsigned path_base<VC>::perceive_polygon_orientation(unsigned start,
                                                         unsigned end)
    {
         
         
        unsigned np = end - start;
        double area = 0.0;
        unsigned i;
        for(i = 0; i < np; i++)
        {
            double x1, y1, x2, y2;
            m_vertices.vertex(start + i,            &x1, &y1);
            m_vertices.vertex(start + (i + 1) % np, &x2, &y2);
            area += x1 * y2 - y1 * x2;
        }
        return (area < 0.0) ? path_flags_cw : path_flags_ccw;
    }

     
    template<class VC> 
    void path_base<VC>::invert_polygon(unsigned start, unsigned end)
    {
        unsigned i;
        unsigned tmp_cmd = m_vertices.command(start);
        
        --end;  

         
        for(i = start; i < end; i++)
        {
            m_vertices.modify_command(i, m_vertices.command(i + 1));
        }

         
        m_vertices.modify_command(end, tmp_cmd);

         
        while(end > start)
        {
            m_vertices.swap_vertices(start++, end--);
        }
    }

     
    template<class VC> 
    void path_base<VC>::invert_polygon(unsigned start)
    {
         
        while(start < m_vertices.total_vertices() && 
              !is_vertex(m_vertices.command(start))) ++start;

         
        while(start+1 < m_vertices.total_vertices() && 
              is_move_to(m_vertices.command(start)) &&
              is_move_to(m_vertices.command(start+1))) ++start;

         
        unsigned end = start + 1;
        while(end < m_vertices.total_vertices() && 
              !is_next_poly(m_vertices.command(end))) ++end;

        invert_polygon(start, end);
    }

     
    template<class VC> 
    unsigned path_base<VC>::arrange_polygon_orientation(unsigned start, 
                                                        path_flags_e orientation)
    {
        if(orientation == path_flags_none) return start;
        
         
        while(start < m_vertices.total_vertices() && 
              !is_vertex(m_vertices.command(start))) ++start;

         
        while(start+1 < m_vertices.total_vertices() && 
              is_move_to(m_vertices.command(start)) &&
              is_move_to(m_vertices.command(start+1))) ++start;

         
        unsigned end = start + 1;
        while(end < m_vertices.total_vertices() && 
              !is_next_poly(m_vertices.command(end))) ++end;

        if(end - start > 2)
        {
            if(perceive_polygon_orientation(start, end) != unsigned(orientation))
            {
                 
                invert_polygon(start, end);
                unsigned cmd;
                while(end < m_vertices.total_vertices() && 
                      is_end_poly(cmd = m_vertices.command(end)))
                {
                    m_vertices.modify_command(end++, set_orientation(cmd, orientation));
                }
            }
        }
        return end;
    }

     
    template<class VC> 
    unsigned path_base<VC>::arrange_orientations(unsigned start, 
                                                 path_flags_e orientation)
    {
        if(orientation != path_flags_none)
        {
            while(start < m_vertices.total_vertices())
            {
                start = arrange_polygon_orientation(start, orientation);
                if(is_stop(m_vertices.command(start)))
                {
                    ++start;
                    break;
                }
            }
        }
        return start;
    }

     
    template<class VC> 
    void path_base<VC>::arrange_orientations_all_paths(path_flags_e orientation)
    {
        if(orientation != path_flags_none)
        {
            unsigned start = 0;
            while(start < m_vertices.total_vertices())
            {
                start = arrange_orientations(start, orientation);
            }
        }
    }

     
    template<class VC> 
    void path_base<VC>::flip_x(double x1, double x2)
    {
        unsigned i;
        double x, y;
        for(i = 0; i < m_vertices.total_vertices(); i++)
        {
            unsigned cmd = m_vertices.vertex(i, &x, &y);
            if(is_vertex(cmd))
            {
                m_vertices.modify_vertex(i, x2 - x + x1, y);
            }
        }
    }

     
    template<class VC> 
    void path_base<VC>::flip_y(double y1, double y2)
    {
        unsigned i;
        double x, y;
        for(i = 0; i < m_vertices.total_vertices(); i++)
        {
            unsigned cmd = m_vertices.vertex(i, &x, &y);
            if(is_vertex(cmd))
            {
                m_vertices.modify_vertex(i, x, y2 - y + y1);
            }
        }
    }

     
    template<class VC> 
    void path_base<VC>::translate(double dx, double dy, unsigned path_id)
    {
        unsigned num_ver = m_vertices.total_vertices();
        for(; path_id < num_ver; path_id++)
        {
            double x, y;
            unsigned cmd = m_vertices.vertex(path_id, &x, &y);
            if(is_stop(cmd)) break;
            if(is_vertex(cmd))
            {
                x += dx;
                y += dy;
                m_vertices.modify_vertex(path_id, x, y);
            }
        }
    }

     
    template<class VC> 
    void path_base<VC>::translate_all_paths(double dx, double dy)
    {
        unsigned idx;
        unsigned num_ver = m_vertices.total_vertices();
        for(idx = 0; idx < num_ver; idx++)
        {
            double x, y;
            if(is_vertex(m_vertices.vertex(idx, &x, &y)))
            {
                x += dx;
                y += dy;
                m_vertices.modify_vertex(idx, x, y);
            }
        }
    }

     
    template<class Container> class vertex_stl_storage
    {
    public:
        typedef typename Container::value_type vertex_type;
        typedef typename vertex_type::value_type value_type;

        void remove_all() { m_vertices.clear(); }
        void free_all()   { m_vertices.clear(); }

        void add_vertex(double x, double y, unsigned cmd)
        {
            m_vertices.push_back(vertex_type(value_type(x), 
                                             value_type(y), 
                                             int8u(cmd)));
        }

        void modify_vertex(unsigned idx, double x, double y)
        {
            vertex_type& v = m_vertices[idx];
            v.x = value_type(x);
            v.y = value_type(y);
        }

        void modify_vertex(unsigned idx, double x, double y, unsigned cmd)
        {
            vertex_type& v = m_vertices[idx];
            v.x   = value_type(x);
            v.y   = value_type(y);
            v.cmd = int8u(cmd);
        }

        void modify_command(unsigned idx, unsigned cmd)
        {
            m_vertices[idx].cmd = int8u(cmd);
        }

        void swap_vertices(unsigned v1, unsigned v2)
        {
            vertex_type t = m_vertices[v1];
            m_vertices[v1] = m_vertices[v2];
            m_vertices[v2] = t;
        }

        unsigned last_command() const
        {
            return m_vertices.size() ? 
                m_vertices[m_vertices.size() - 1].cmd : 
                path_cmd_stop;
        }

        unsigned last_vertex(double* x, double* y) const
        {
            if(m_vertices.size() == 0)
            {
                *x = *y = 0.0;
                return path_cmd_stop;
            }
            return vertex(m_vertices.size() - 1, x, y);
        }

        unsigned prev_vertex(double* x, double* y) const
        {
            if(m_vertices.size() < 2)
            {
                *x = *y = 0.0;
                return path_cmd_stop;
            }
            return vertex(m_vertices.size() - 2, x, y);
        }

        double last_x() const
        {
            return m_vertices.size() ? m_vertices[m_vertices.size() - 1].x : 0.0;
        }

        double last_y() const
        {
            return m_vertices.size() ? m_vertices[m_vertices.size() - 1].y : 0.0;
        }

        unsigned total_vertices() const
        {
            return m_vertices.size();
        }

        unsigned vertex(unsigned idx, double* x, double* y) const
        {
            const vertex_type& v = m_vertices[idx];
            *x = v.x;
            *y = v.y;
            return v.cmd;
        }

        unsigned command(unsigned idx) const
        {
            return m_vertices[idx].cmd;
        }

    private:
        Container m_vertices;
    };

     
    typedef path_base<vertex_block_storage<double> > path_storage;

     
     
     

}



 
 
 
 
 
 
 




#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_PATH_STORAGE_INTEGER_INCLUDED
#define AGG_PATH_STORAGE_INTEGER_INCLUDED

#include <cstring>

namespace agg
{
     
    template<class T, unsigned CoordShift=6> struct vertex_integer
    {
        enum path_cmd
        {
            cmd_move_to = 0,
            cmd_line_to = 1,
            cmd_curve3  = 2,
            cmd_curve4  = 3
        };

        enum coord_scale_e
        {
            coord_shift = CoordShift,
            coord_scale  = 1 << coord_shift
        };

        T x,y;
        vertex_integer() {}
        vertex_integer(T x_, T y_, unsigned flag) :
            x(((x_ << 1) & ~1) | (flag &  1)),
            y(((y_ << 1) & ~1) | (flag >> 1)) {}

        unsigned vertex(double* x_, double* y_, 
                        double dx=0, double dy=0,
                        double scale=1.0) const
        {
            *x_ = dx + (double(x >> 1) / (double)coord_scale) * scale;
            *y_ = dy + (double(y >> 1) / (double)coord_scale) * scale;
            switch(((y & 1) << 1) | (x & 1))
            {
                case cmd_move_to: return path_cmd_move_to;
                case cmd_line_to: return path_cmd_line_to;
                case cmd_curve3:  return path_cmd_curve3;
                case cmd_curve4:  return path_cmd_curve4;
            }
            return path_cmd_stop;
        }
    };


     
    template<class T, unsigned CoordShift=6> class path_storage_integer
    {
    public:
        typedef T value_type;
        typedef vertex_integer<T, CoordShift> vertex_integer_type;

         
        path_storage_integer() : m_storage(), m_vertex_idx(0), m_closed(true) {}

         
        void remove_all() { m_storage.remove_all(); }

         
        void move_to(T x, T y)
        {
            m_storage.add(vertex_integer_type(x, y, vertex_integer_type::cmd_move_to));
        }

         
        void line_to(T x, T y)
        {
            m_storage.add(vertex_integer_type(x, y, vertex_integer_type::cmd_line_to));
        }

         
        void curve3(T x_ctrl,  T y_ctrl, 
                    T x_to,    T y_to)
        {
            m_storage.add(vertex_integer_type(x_ctrl, y_ctrl, vertex_integer_type::cmd_curve3));
            m_storage.add(vertex_integer_type(x_to,   y_to,   vertex_integer_type::cmd_curve3));
        }

         
        void curve4(T x_ctrl1, T y_ctrl1, 
                    T x_ctrl2, T y_ctrl2, 
                    T x_to,    T y_to)
        {
            m_storage.add(vertex_integer_type(x_ctrl1, y_ctrl1, vertex_integer_type::cmd_curve4));
            m_storage.add(vertex_integer_type(x_ctrl2, y_ctrl2, vertex_integer_type::cmd_curve4));
            m_storage.add(vertex_integer_type(x_to,    y_to,    vertex_integer_type::cmd_curve4));
        }

         
        void close_polygon() {}

         
        unsigned size() const { return m_storage.size(); }
        unsigned vertex(unsigned idx, double* x, double* y) const
        {
            return m_storage[idx].vertex(x, y);
        }

         
        unsigned byte_size() const { return m_storage.size() * sizeof(vertex_integer_type); }
        void serialize(int8u* ptr) const
        {
            unsigned i;
            for(i = 0; i < m_storage.size(); i++)
            {
                std::memcpy(ptr, &m_storage[i], sizeof(vertex_integer_type));
                ptr += sizeof(vertex_integer_type);
            }
        }

         
        void rewind(unsigned) 
        { 
            m_vertex_idx = 0; 
            m_closed = true;
        }

         
        unsigned vertex(double* x, double* y)
        {
            if(m_storage.size() < 2 || m_vertex_idx > m_storage.size()) 
            {
                *x = 0;
                *y = 0;
                return path_cmd_stop;
            }
            if(m_vertex_idx == m_storage.size())
            {
                *x = 0;
                *y = 0;
                ++m_vertex_idx;
                return +path_cmd_end_poly | path_flags_close;
            }
            unsigned cmd = m_storage[m_vertex_idx].vertex(x, y);
            if(is_move_to(cmd) && !m_closed)
            {
                *x = 0;
                *y = 0;
                m_closed = true;
                return +path_cmd_end_poly | path_flags_close;
            }
            m_closed = false;
            ++m_vertex_idx;
            return cmd;
        }

         
        rect_d bounding_rect() const
        {
            rect_d bounds(1e100, 1e100, -1e100, -1e100);
            if(m_storage.size() == 0)
            {
                bounds.x1 = bounds.y1 = bounds.x2 = bounds.y2 = 0.0;
            }
            else
            {
                unsigned i;
                for(i = 0; i < m_storage.size(); i++)
                {
                    double x, y;
                    m_storage[i].vertex(&x, &y);
                    if(x < bounds.x1) bounds.x1 = x;
                    if(y < bounds.y1) bounds.y1 = y;
                    if(x > bounds.x2) bounds.x2 = x;
                    if(y > bounds.y2) bounds.y2 = y;
                }
            }
            return bounds;
        }

    private:
        pod_bvector<vertex_integer_type, 6> m_storage;
        unsigned                            m_vertex_idx;
        bool                                m_closed;
    };




     
    template<class T, unsigned CoordShift=6> class serialized_integer_path_adaptor
    {
    public:
        typedef vertex_integer<T, CoordShift> vertex_integer_type;

         
        serialized_integer_path_adaptor() :
            m_data(0),
            m_end(0),
            m_ptr(0),
            m_dx(0.0),
            m_dy(0.0),
            m_scale(1.0),
            m_vertices(0)
        {}

         
        serialized_integer_path_adaptor(const int8u* data, unsigned size,
                                        double dx, double dy) :
            m_data(data),
            m_end(data + size),
            m_ptr(data),
            m_dx(dx),
            m_dy(dy),
            m_vertices(0)
        {}

         
        void init(const int8u* data, unsigned size, 
                  double dx, double dy, double scale=1.0)
        {
            m_data     = data;
            m_end      = data + size;
            m_ptr      = data;
            m_dx       = dx;
            m_dy       = dy;
            m_scale    = scale;
            m_vertices = 0;
        }


         
        void rewind(unsigned) 
        { 
            m_ptr      = m_data; 
            m_vertices = 0;
        }

         
        unsigned vertex(double* x, double* y)
        {
            if(m_data == 0 || m_ptr > m_end) 
            {
                *x = 0;
                *y = 0;
                return path_cmd_stop;
            }

            if(m_ptr == m_end)
            {
                *x = 0;
                *y = 0;
                m_ptr += sizeof(vertex_integer_type);
                return +path_cmd_end_poly | path_flags_close;
            }

            vertex_integer_type v;
            std::memcpy(&v, m_ptr, sizeof(vertex_integer_type));
            unsigned cmd = v.vertex(x, y, m_dx, m_dy, m_scale);
            if(is_move_to(cmd) && m_vertices > 2)
            {
                *x = 0;
                *y = 0;
                m_vertices = 0;
                return +path_cmd_end_poly | path_flags_close;
            }
            ++m_vertices;
            m_ptr += sizeof(vertex_integer_type);
            return cmd;
        }

    private:
        const int8u* m_data;
        const int8u* m_end;
        const int8u* m_ptr;
        double       m_dx;
        double       m_dy;
        double       m_scale;
        unsigned     m_vertices;
    };

}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_PATTERN_FILTERS_RGBA8_INCLUDED
#define AGG_PATTERN_FILTERS_RGBA8_INCLUDED



namespace agg
{

     
    template<class ColorT> struct pattern_filter_nn
    {
        typedef ColorT color_type;
        static unsigned dilation() { return 0; }

        static void AGG_INLINE pixel_low_res(color_type const* const* buf, 
                                             color_type* p, int x, int y)
        {
            *p = buf[y][x];
        }

        static void AGG_INLINE pixel_high_res(color_type const* const* buf, 
                                              color_type* p, int x, int y)
        {
            *p = buf[y >> line_subpixel_shift]
                    [x >> line_subpixel_shift];
        }
    };

    typedef pattern_filter_nn<rgba8>  pattern_filter_nn_rgba8;
    typedef pattern_filter_nn<rgba16> pattern_filter_nn_rgba16;


     
    template<class ColorT> struct pattern_filter_bilinear_rgba
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;


        static unsigned dilation() { return 1; }

        static AGG_INLINE void pixel_low_res(color_type const* const* buf, 
                                             color_type* p, int x, int y)
        {
            *p = buf[y][x];
        }

        static AGG_INLINE void pixel_high_res(color_type const* const* buf, 
                                              color_type* p, int x, int y)
        {
            calc_type r, g, b, a;
            r = g = b = a = 0;

            calc_type weight;
            int x_lr = x >> line_subpixel_shift;
            int y_lr = y >> line_subpixel_shift;

            x &= line_subpixel_mask;
            y &= line_subpixel_mask;
            const color_type* ptr = buf[y_lr] + x_lr;

            weight = (line_subpixel_scale - x) * 
                     (line_subpixel_scale - y);
            r += weight * ptr->r;
            g += weight * ptr->g;
            b += weight * ptr->b;
            a += weight * ptr->a;

            ++ptr;

            weight = x * (line_subpixel_scale - y);
            r += weight * ptr->r;
            g += weight * ptr->g;
            b += weight * ptr->b;
            a += weight * ptr->a;

            ptr = buf[y_lr + 1] + x_lr;

            weight = (line_subpixel_scale - x) * y;
            r += weight * ptr->r;
            g += weight * ptr->g;
            b += weight * ptr->b;
            a += weight * ptr->a;

            ++ptr;

            weight = x * y;
            r += weight * ptr->r;
            g += weight * ptr->g;
            b += weight * ptr->b;
            a += weight * ptr->a;

            p->r = (value_type)color_type::downshift(r, line_subpixel_shift * 2);
            p->g = (value_type)color_type::downshift(g, line_subpixel_shift * 2);
            p->b = (value_type)color_type::downshift(b, line_subpixel_shift * 2);
            p->a = (value_type)color_type::downshift(a, line_subpixel_shift * 2);
        }
    };

    typedef pattern_filter_bilinear_rgba<rgba8>  pattern_filter_bilinear_rgba8;
    typedef pattern_filter_bilinear_rgba<rgba16> pattern_filter_bilinear_rgba16;
    typedef pattern_filter_bilinear_rgba<rgba32> pattern_filter_bilinear_rgba32;
}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_PIXFMT_AMASK_ADAPTOR_INCLUDED
#define AGG_PIXFMT_AMASK_ADAPTOR_INCLUDED


#include <cstring>


namespace agg
{
     
    template<class PixFmt, class AlphaMask> class pixfmt_amask_adaptor
    {
    public:
        typedef PixFmt pixfmt_type;
        typedef typename pixfmt_type::color_type color_type;
        typedef typename pixfmt_type::row_data row_data;
        typedef AlphaMask amask_type;
        typedef typename amask_type::cover_type cover_type;

    private:
        enum span_extra_tail_e { span_extra_tail = 256 };

        void realloc_span(unsigned len)
        {
            if(len > m_span.size())
            {
                m_span.resize(len + span_extra_tail);
            }
        }

        void init_span(unsigned len)
        {
            realloc_span(len);
            std::memset(&m_span[0], amask_type::cover_full, len * sizeof(cover_type));
        }

        void init_span(unsigned len, const cover_type* covers)
        {
            realloc_span(len);
            std::memcpy(&m_span[0], covers, len * sizeof(cover_type));
        }


    public:
        pixfmt_amask_adaptor(pixfmt_type& pixf, amask_type& mask) :
            m_pixf(&pixf), m_mask(&mask), m_span()
        {}

        void attach_pixfmt(pixfmt_type& pixf) { m_pixf = &pixf; }
        void attach_alpha_mask(amask_type& mask) { m_mask = &mask; }

         
        template<class PixFmt2>
        bool attach_pixfmt(PixFmt2& pixf, int x1, int y1, int x2, int y2)
        {
            return m_pixf->attach(pixf, x1, y1, x2, y2);
        }

         
        unsigned width()  const { return m_pixf->width();  }
        unsigned height() const { return m_pixf->height(); }

         
        color_type pixel(int x, int y)
        {
            return m_pixf->pixel(x, y);
        }

         
        void copy_pixel(int x, int y, const color_type& c)
        {
            m_pixf->blend_pixel(x, y, c, m_mask->pixel(x, y));
        }

         
        void blend_pixel(int x, int y, const color_type& c, cover_type cover)
        {
            m_pixf->blend_pixel(x, y, c, m_mask->combine_pixel(x, y, cover));
        }

         
        void copy_hline(int x, int y, 
                        unsigned len, 
                        const color_type& c)
        {
            realloc_span(len);
            m_mask->fill_hspan(x, y, &m_span[0], len);
            m_pixf->blend_solid_hspan(x, y, len, c, &m_span[0]);
        }

         
        void blend_hline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         cover_type)
        {
            init_span(len);
            m_mask->combine_hspan(x, y, &m_span[0], len);
            m_pixf->blend_solid_hspan(x, y, len, c, &m_span[0]);
        }

         
        void copy_vline(int x, int y,
                        unsigned len, 
                        const color_type& c)
        {
            realloc_span(len);
            m_mask->fill_vspan(x, y, &m_span[0], len);
            m_pixf->blend_solid_vspan(x, y, len, c, &m_span[0]);
        }

         
        void blend_vline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         cover_type)
        {
            init_span(len);
            m_mask->combine_vspan(x, y, &m_span[0], len);
            m_pixf->blend_solid_vspan(x, y, len, c, &m_span[0]);
        }

         
        void copy_from(const rendering_buffer& from, 
                       int xdst, int ydst,
                       int xsrc, int ysrc,
                       unsigned len)
        {
            m_pixf->copy_from(from, xdst, ydst, xsrc, ysrc, len);
        }


         
        void blend_solid_hspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const cover_type* covers)
        {
            init_span(len, covers);
            m_mask->combine_hspan(x, y, &m_span[0], len);
            m_pixf->blend_solid_hspan(x, y, len, c, &m_span[0]);
        }


         
        void blend_solid_vspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const cover_type* covers)
        {
            init_span(len, covers);
            m_mask->combine_vspan(x, y, &m_span[0], len);
            m_pixf->blend_solid_vspan(x, y, len, c, &m_span[0]);
        }


         
        void copy_color_hspan(int x, int y, unsigned len, const color_type* colors)
        {
            realloc_span(len);
            m_mask->fill_hspan(x, y, &m_span[0], len);
            m_pixf->blend_color_hspan(x, y, len, colors, &m_span[0], cover_full);
        }

         
        void copy_color_vspan(int x, int y, unsigned len, const color_type* colors)
        {
            realloc_span(len);
            m_mask->fill_vspan(x, y, &m_span[0], len);
            m_pixf->blend_color_vspan(x, y, len, colors, &m_span[0], cover_full);
        }

         
        void blend_color_hspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const cover_type* covers,
                               cover_type cover = cover_full)
        {
            if(covers) 
            {
                init_span(len, covers);
                m_mask->combine_hspan(x, y, &m_span[0], len);
            }
            else
            {
                realloc_span(len);
                m_mask->fill_hspan(x, y, &m_span[0], len);
            }
            m_pixf->blend_color_hspan(x, y, len, colors, &m_span[0], cover);
        }


         
        void blend_color_vspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const cover_type* covers,
                               cover_type cover = cover_full)
        {
            if(covers) 
            {
                init_span(len, covers);
                m_mask->combine_vspan(x, y, &m_span[0], len);
            }
            else
            {
                realloc_span(len);
                m_mask->fill_vspan(x, y, &m_span[0], len);
            }
            m_pixf->blend_color_vspan(x, y, len, colors, &m_span[0], cover);
        }

    private:
        pixfmt_type*          m_pixf;
        const amask_type*     m_mask;
        pod_array<cover_type> m_span;
    };

}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_PIXFMT_BASE_INCLUDED
#define AGG_PIXFMT_BASE_INCLUDED


namespace agg
{
    struct pixfmt_gray_tag
    {
    };

    struct pixfmt_rgb_tag
    {
    };

    struct pixfmt_rgba_tag
    {
    };

     
    template<class ColorT, class Order = void> 
    struct blender_base
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;

        static rgba get(value_type r, value_type g, value_type b, value_type a, cover_type cover = cover_full)
        {
            if (cover > cover_none)
            {
                rgba c(
                    color_type::to_double(r), 
                    color_type::to_double(g), 
                    color_type::to_double(b), 
                    color_type::to_double(a));

                if (cover < cover_full)
                {
                    double x = double(cover) / (double)cover_full;
                    c.r *= x;
                    c.g *= x;
                    c.b *= x;
                    c.a *= x;
                }

                return c;
            }
            else return rgba::no_color();
        }

        static rgba get(const value_type* p, cover_type cover = cover_full)
        {
            return get(
                p[order_type::R], 
                p[order_type::G], 
                p[order_type::B], 
                p[order_type::A], 
                cover);
        }

        static void set(value_type* p, value_type r, value_type g, value_type b, value_type a)
        {
            p[order_type::R] = r;
            p[order_type::G] = g;
            p[order_type::B] = b;
            p[order_type::A] = a;
        }

        static void set(value_type* p, const rgba& c)
        {
            p[order_type::R] = color_type::from_double(c.r);
            p[order_type::G] = color_type::from_double(c.g);
            p[order_type::B] = color_type::from_double(c.b);
            p[order_type::A] = color_type::from_double(c.a);
        }
    };
}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_PIXFMT_GRAY_INCLUDED
#define AGG_PIXFMT_GRAY_INCLUDED

#include <cstring>

namespace agg
{
 
     
    template<class ColorT> struct blender_gray
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

         
         
         

        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cv, value_type alpha, cover_type cover)
        {
            blend_pix(p, cv, color_type::mult_cover(alpha, cover));
        }

        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cv, value_type alpha)
        {
            *p = color_type::lerp(*p, cv, alpha);
        }
    };


     
    template<class ColorT> struct blender_gray_pre
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

         
         

        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cv, value_type alpha, cover_type cover)
        {
            blend_pix(p, color_type::mult_cover(cv, cover), color_type::mult_cover(alpha, cover));
        }

        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cv, value_type alpha)
        {
            *p = color_type::prelerp(*p, cv, alpha);
        }
    };
    


     
    template<class ColorT, class GammaLut> class apply_gamma_dir_gray
    {
    public:
        typedef typename ColorT::value_type value_type;

        apply_gamma_dir_gray(const GammaLut& gamma) : m_gamma(gamma) {}

        AGG_INLINE void operator () (value_type* p)
        {
            *p = m_gamma.dir(*p);
        }

    private:
        const GammaLut& m_gamma;
    };



     
    template<class ColorT, class GammaLut> class apply_gamma_inv_gray
    {
    public:
        typedef typename ColorT::value_type value_type;

        apply_gamma_inv_gray(const GammaLut& gamma) : m_gamma(gamma) {}

        AGG_INLINE void operator () (value_type* p)
        {
            *p = m_gamma.inv(*p);
        }

    private:
        const GammaLut& m_gamma;
    };



     
    template<class Blender, class RenBuf, unsigned Step = 1, unsigned Offset = 0>
    class pixfmt_alpha_blend_gray
    {
    public:
        typedef pixfmt_gray_tag pixfmt_category;
        typedef RenBuf   rbuf_type;
        typedef typename rbuf_type::row_data row_data;
        typedef Blender  blender_type;
        typedef typename blender_type::color_type color_type;
        typedef int                               order_type;  
        typedef typename color_type::value_type   value_type;
        typedef typename color_type::calc_type    calc_type;
        enum 
        {
            num_components = 1,
            pix_width = sizeof(value_type) * Step,
            pix_step = Step,
            pix_offset = Offset,
        };
        struct pixel_type
        {
            value_type c[num_components];

            void set(value_type v)
            {
                c[0] = v;
            }

            void set(const color_type& color)
            {
                set(color.v);
            }

            void get(value_type& v) const
            {
                v = c[0];
            }

            color_type get() const
            {
                return color_type(c[0]);
            }

            pixel_type* next()
            {
                return (pixel_type*)(c + pix_step);
            }

            const pixel_type* next() const
            {
                return (const pixel_type*)(c + pix_step);
            }

            pixel_type* advance(int n)
            {
                return (pixel_type*)(c + n * pix_step);
            }

            const pixel_type* advance(int n) const
            {
                return (const pixel_type*)(c + n * pix_step);
            }
        };

    private:
         
        AGG_INLINE void blend_pix(pixel_type* p, 
            value_type v, value_type a, 
            unsigned cover)
        {
            blender_type::blend_pix(p->c, v, a, cover);
        }

         
        AGG_INLINE void blend_pix(pixel_type* p, value_type v, value_type a)
        {
            blender_type::blend_pix(p->c, v, a);
        }

         
        AGG_INLINE void blend_pix(pixel_type* p, const color_type& c, unsigned cover)
        {
            blender_type::blend_pix(p->c, c.v, c.a, cover);
        }

         
        AGG_INLINE void blend_pix(pixel_type* p, const color_type& c)
        {
            blender_type::blend_pix(p->c, c.v, c.a);
        }

         
        AGG_INLINE void copy_or_blend_pix(pixel_type* p, const color_type& c, unsigned cover)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque() && cover == cover_mask)
                {
                    p->set(c);
                }
                else
                {
                    blend_pix(p, c, cover);
                }
            }
        }

         
        AGG_INLINE void copy_or_blend_pix(pixel_type* p, const color_type& c)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque())
                {
                    p->set(c);
                }
                else
                {
                    blend_pix(p, c);
                }
            }
        }

    public:
         
        explicit pixfmt_alpha_blend_gray(rbuf_type& rb) :
            m_rbuf(&rb)
        {}
        void attach(rbuf_type& rb) { m_rbuf = &rb; }
         

        template<class PixFmt>
        bool attach(PixFmt& pixf, int x1, int y1, int x2, int y2)
        {
            rect_i r(x1, y1, x2, y2);
            if (r.clip(rect_i(0, 0, pixf.width()-1, pixf.height()-1)))
            {
                int stride = pixf.stride();
                m_rbuf->attach(pixf.pix_ptr(r.x1, stride < 0 ? r.y2 : r.y1), 
                               (r.x2 - r.x1) + 1,
                               (r.y2 - r.y1) + 1,
                               stride);
                return true;
            }
            return false;
        }

         
        AGG_INLINE unsigned width()  const { return m_rbuf->width();  }
        AGG_INLINE unsigned height() const { return m_rbuf->height(); }
        AGG_INLINE int      stride() const { return m_rbuf->stride(); }

         
        int8u* row_ptr(int y)       { return m_rbuf->row_ptr(y); }
        const int8u* row_ptr(int y) const { return m_rbuf->row_ptr(y); }
        row_data     row(int y)     const { return m_rbuf->row(y); }

         
        AGG_INLINE int8u* pix_ptr(int x, int y) 
        { 
            return m_rbuf->row_ptr(y) + sizeof(value_type) * (x * pix_step + pix_offset);
        }

        AGG_INLINE const int8u* pix_ptr(int x, int y) const 
        { 
            return m_rbuf->row_ptr(y) + sizeof(value_type) * (x * pix_step + pix_offset);
        }

         
        AGG_INLINE pixel_type* pix_value_ptr(int x, int y, unsigned len) 
        {
            return (pixel_type*)(m_rbuf->row_ptr(x, y, len) + sizeof(value_type) * (x * pix_step + pix_offset));
        }

         
        AGG_INLINE const pixel_type* pix_value_ptr(int x, int y) const 
        {
            int8u* p = m_rbuf->row_ptr(y);
            return p ? (pixel_type*)(p + sizeof(value_type) * (x * pix_step + pix_offset)) : 0;
        }

         
        AGG_INLINE static pixel_type* pix_value_ptr(void* p) 
        {
            return (pixel_type*)((value_type*)p + pix_offset);
        }

         
        AGG_INLINE static const pixel_type* pix_value_ptr(const void* p) 
        {
            return (const pixel_type*)((const value_type*)p + pix_offset);
        }

         
        AGG_INLINE static void write_plain_color(void* p, color_type c)
        {
             
            c.premultiply();
            pix_value_ptr(p)->set(c);
        }

         
        AGG_INLINE static color_type read_plain_color(const void* p)
        {
            return pix_value_ptr(p)->get();
        }

         
        AGG_INLINE static void make_pix(int8u* p, const color_type& c)
        {
            ((pixel_type*)p)->set(c);
        }

         
        AGG_INLINE color_type pixel(int x, int y) const
        {
            if (const pixel_type* p = pix_value_ptr(x, y))
            {
                return p->get();
            }
            return color_type::no_color();
        }

         
        AGG_INLINE void copy_pixel(int x, int y, const color_type& c)
        {
            pix_value_ptr(x, y, 1)->set(c);
        }

         
        AGG_INLINE void blend_pixel(int x, int y, const color_type& c, int8u cover)
        {
            copy_or_blend_pix(pix_value_ptr(x, y, 1), c, cover);
        }

         
        AGG_INLINE void copy_hline(int x, int y, 
                                   unsigned len, 
                                   const color_type& c)
        {
            pixel_type* p = pix_value_ptr(x, y, len);
            do
            {
                p->set(c);
                p = p->next();
            }
            while(--len);
        }


         
        AGG_INLINE void copy_vline(int x, int y,
                                   unsigned len, 
                                   const color_type& c)
        {
            do
            {
                pix_value_ptr(x, y++, 1)->set(c);
            }
            while (--len);
        }


         
        void blend_hline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         int8u cover)
        {
            if (!c.is_transparent())
            {
                pixel_type* p = pix_value_ptr(x, y, len);

                if (c.is_opaque() && cover == cover_mask)
                {
                    do
                    {
                        p->set(c);
                        p = p->next();
                    }
                    while (--len);
                }
                else
                {
                    do
                    {
                        blend_pix(p, c, cover);
                        p = p->next();
                    }
                    while (--len);
                }
            }
        }


         
        void blend_vline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         int8u cover)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque() && cover == cover_mask)
                {
                    do
                    {
                        pix_value_ptr(x, y++, 1)->set(c);
                    }
                    while (--len);
                }
                else
                {
                    do
                    {
                        blend_pix(pix_value_ptr(x, y++, 1), c, cover);
                    }
                    while (--len);
                }
            }
        }


         
        void blend_solid_hspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const int8u* covers)
        {
            if (!c.is_transparent())
            {
                pixel_type* p = pix_value_ptr(x, y, len);

                do 
                {
                    if (c.is_opaque() && *covers == cover_mask)
                    {
                        p->set(c);
                    }
                    else
                    {
                        blend_pix(p, c, *covers);
                    }
                    p = p->next();
                    ++covers;
                }
                while (--len);
            }
        }


         
        void blend_solid_vspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const int8u* covers)
        {
            if (!c.is_transparent())
            {
                do 
                {
                    pixel_type* p = pix_value_ptr(x, y++, 1);

                    if (c.is_opaque() && *covers == cover_mask)
                    {
                        p->set(c);
                    }
                    else
                    {
                        blend_pix(p, c, *covers);
                    }
                    ++covers;
                }
                while (--len);
            }
        }


         
        void copy_color_hspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            pixel_type* p = pix_value_ptr(x, y, len);

            do 
            {
                p->set(*colors++);
                p = p->next();
            }
            while (--len);
        }


         
        void copy_color_vspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            do 
            {
                pix_value_ptr(x, y++, 1)->set(*colors++);
            }
            while (--len);
        }


         
        void blend_color_hspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const int8u* covers,
                               int8u cover)
        {
            pixel_type* p = pix_value_ptr(x, y, len);

            if (covers)
            {
                do 
                {
                    copy_or_blend_pix(p, *colors++, *covers++);
                    p = p->next();
                }
                while (--len);
            }
            else
            {
                if (cover == cover_mask)
                {
                    do 
                    {
                        copy_or_blend_pix(p, *colors++);
                        p = p->next();
                    }
                    while (--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(p, *colors++, cover);
                        p = p->next();
                    }
                    while (--len);
                }
            }
        }
        

         
        void blend_color_vspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const int8u* covers,
                               int8u cover)
        {
            if (covers)
            {
                do 
                {
                    copy_or_blend_pix(pix_value_ptr(x, y++, 1), *colors++, *covers++);
                }
                while (--len);
            }
            else
            {
                if (cover == cover_mask)
                {
                    do 
                    {
                        copy_or_blend_pix(pix_value_ptr(x, y++, 1), *colors++);
                    }
                    while (--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(pix_value_ptr(x, y++, 1), *colors++, cover);
                    }
                    while (--len);
                }
            }
        }

         
        template<class Function> void for_each_pixel(Function f)
        {
            unsigned y;
            for (y = 0; y < height(); ++y)
            {
                row_data r = m_rbuf->row(y);
                if (r.ptr)
                {
                    unsigned len = r.x2 - r.x1 + 1;
                    pixel_type* p = pix_value_ptr(r.x1, y, len);
                    do
                    {
                        f(p->c);
                        p = p->next();
                    }
                    while (--len);
                }
            }
        }

         
        template<class GammaLut> void apply_gamma_dir(const GammaLut& g)
        {
            for_each_pixel(apply_gamma_dir_gray<color_type, GammaLut>(g));
        }

         
        template<class GammaLut> void apply_gamma_inv(const GammaLut& g)
        {
            for_each_pixel(apply_gamma_inv_gray<color_type, GammaLut>(g));
        }

         
        template<class RenBuf2>
        void copy_from(const RenBuf2& from, 
                       int xdst, int ydst,
                       int xsrc, int ysrc,
                       unsigned len)
        {
            if (const int8u* p = from.row_ptr(ysrc))
            {
                std::memmove(m_rbuf->row_ptr(xdst, ydst, len) + xdst * pix_width, 
                        p + xsrc * pix_width, 
                        len * pix_width);
            }
        }

         
         
        template<class SrcPixelFormatRenderer>
        void blend_from_color(const SrcPixelFormatRenderer& from, 
                              const color_type& color,
                              int xdst, int ydst,
                              int xsrc, int ysrc,
                              unsigned len,
                              int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;
            typedef typename SrcPixelFormatRenderer::color_type src_color_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);

                do 
                {
                    copy_or_blend_pix(pdst, color, src_color_type::scale_cover(cover, psrc->c[0]));
                    psrc = psrc->next();
                    pdst = pdst->next();
                }
                while (--len);
            }
        }

         
         
         
        template<class SrcPixelFormatRenderer>
        void blend_from_lut(const SrcPixelFormatRenderer& from, 
                            const color_type* color_lut,
                            int xdst, int ydst,
                            int xsrc, int ysrc,
                            unsigned len,
                            int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);

                do 
                {
                    copy_or_blend_pix(pdst, color_lut[psrc->c[0]], cover);
                    psrc = psrc->next();
                    pdst = pdst->next();
                }
                while (--len);
            }
        }

    private:
        rbuf_type* m_rbuf;
    };

    typedef blender_gray<gray8> blender_gray8;
    typedef blender_gray<sgray8> blender_sgray8;
    typedef blender_gray<gray16> blender_gray16;
    typedef blender_gray<gray32> blender_gray32;

    typedef blender_gray_pre<gray8> blender_gray8_pre;
    typedef blender_gray_pre<sgray8> blender_sgray8_pre;
    typedef blender_gray_pre<gray16> blender_gray16_pre;
    typedef blender_gray_pre<gray32> blender_gray32_pre;

    typedef pixfmt_alpha_blend_gray<blender_gray8, rendering_buffer> pixfmt_gray8;
    typedef pixfmt_alpha_blend_gray<blender_sgray8, rendering_buffer> pixfmt_sgray8;
    typedef pixfmt_alpha_blend_gray<blender_gray16, rendering_buffer> pixfmt_gray16;
    typedef pixfmt_alpha_blend_gray<blender_gray32, rendering_buffer> pixfmt_gray32;

    typedef pixfmt_alpha_blend_gray<blender_gray8_pre, rendering_buffer> pixfmt_gray8_pre;
    typedef pixfmt_alpha_blend_gray<blender_sgray8_pre, rendering_buffer> pixfmt_sgray8_pre;
    typedef pixfmt_alpha_blend_gray<blender_gray16_pre, rendering_buffer> pixfmt_gray16_pre;
    typedef pixfmt_alpha_blend_gray<blender_gray32_pre, rendering_buffer> pixfmt_gray32_pre;
}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_PIXFMT_RGBA_INCLUDED
#define AGG_PIXFMT_RGBA_INCLUDED

#include <cstring>
#include <cmath>

namespace agg
{
    template<class T> inline T sd_min(T a, T b) { return (a < b) ? a : b; }
    template<class T> inline T sd_max(T a, T b) { return (a > b) ? a : b; }

    inline rgba & clip(rgba & c)
    {
        if (c.a > 1) c.a = 1; else if (c.a < 0) c.a = 0;
        if (c.r > c.a) c.r = c.a; else if (c.r < 0) c.r = 0;
        if (c.g > c.a) c.g = c.a; else if (c.g < 0) c.g = 0;
        if (c.b > c.a) c.b = c.a; else if (c.b < 0) c.b = 0;
        return c;
    }

     
    template<class ColorT, class Order> 
    struct multiplier_rgba
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;

         
        static AGG_INLINE void premultiply(value_type* p)
        {
            value_type a = p[Order::A];
            p[Order::R] = color_type::multiply(p[Order::R], a);
            p[Order::G] = color_type::multiply(p[Order::G], a);
            p[Order::B] = color_type::multiply(p[Order::B], a);
        }


         
        static AGG_INLINE void demultiply(value_type* p)
        {
            value_type a = p[Order::A];
            p[Order::R] = color_type::demultiply(p[Order::R], a);
            p[Order::G] = color_type::demultiply(p[Order::G], a);
            p[Order::B] = color_type::demultiply(p[Order::B], a);
        }
    };

     
    template<class ColorT, class Order, class GammaLut> 
    class apply_gamma_dir_rgba
    {
    public:
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;

        apply_gamma_dir_rgba(const GammaLut& gamma) : m_gamma(gamma) {}

        AGG_INLINE void operator () (value_type* p)
        {
            p[Order::R] = m_gamma.dir(p[Order::R]);
            p[Order::G] = m_gamma.dir(p[Order::G]);
            p[Order::B] = m_gamma.dir(p[Order::B]);
        }

    private:
        const GammaLut& m_gamma;
    };

     
    template<class ColorT, class Order, class GammaLut> class apply_gamma_inv_rgba
    {
    public:
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;

        apply_gamma_inv_rgba(const GammaLut& gamma) : m_gamma(gamma) {}

        AGG_INLINE void operator () (value_type* p)
        {
            p[Order::R] = m_gamma.inv(p[Order::R]);
            p[Order::G] = m_gamma.inv(p[Order::G]);
            p[Order::B] = m_gamma.inv(p[Order::B]);
        }

    private:
        const GammaLut& m_gamma;
    };


    template<class ColorT, class Order> 
    struct conv_rgba_pre
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;

         
        static AGG_INLINE void set_plain_color(value_type* p, color_type c)
        {
            c.premultiply();
            p[Order::R] = c.r;
            p[Order::G] = c.g;
            p[Order::B] = c.b;
            p[Order::A] = c.a;
        }

         
        static AGG_INLINE color_type get_plain_color(const value_type* p)
        {
            return color_type(
                p[Order::R],
                p[Order::G],
                p[Order::B],
                p[Order::A]).demultiply();
        }
    };

    template<class ColorT, class Order> 
    struct conv_rgba_plain
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;

         
        static AGG_INLINE void set_plain_color(value_type* p, color_type c)
        {
            p[Order::R] = c.r;
            p[Order::G] = c.g;
            p[Order::B] = c.b;
            p[Order::A] = c.a;
        }

         
        static AGG_INLINE color_type get_plain_color(const value_type* p)
        {
            return color_type(
                p[Order::R],
                p[Order::G],
                p[Order::B],
                p[Order::A]);
        }
    };

     
     
    template<class ColorT, class Order> 
    struct blender_rgba : conv_rgba_pre<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

         
         
         

         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha, cover_type cover)
        {
            blend_pix(p, cr, cg, cb, color_type::mult_cover(alpha, cover));
        }
        
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha)
        {
            p[Order::R] = color_type::lerp(p[Order::R], cr, alpha);
            p[Order::G] = color_type::lerp(p[Order::G], cg, alpha);
            p[Order::B] = color_type::lerp(p[Order::B], cb, alpha);
            p[Order::A] = color_type::prelerp(p[Order::A], alpha, alpha);
        }
    };


     
     
    template<class ColorT, class Order> 
    struct blender_rgba_pre : conv_rgba_pre<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

         
         

         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha, cover_type cover)
        {
            blend_pix(p, 
                color_type::mult_cover(cr, cover), 
                color_type::mult_cover(cg, cover), 
                color_type::mult_cover(cb, cover), 
                color_type::mult_cover(alpha, cover));
        }
        
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha)
        {
            p[Order::R] = color_type::prelerp(p[Order::R], cr, alpha);
            p[Order::G] = color_type::prelerp(p[Order::G], cg, alpha);
            p[Order::B] = color_type::prelerp(p[Order::B], cb, alpha);
            p[Order::A] = color_type::prelerp(p[Order::A], alpha, alpha);
        }
    };

     
     
    template<class ColorT, class Order> 
    struct blender_rgba_plain : conv_rgba_plain<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

         
         

         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha, cover_type cover)
        {
            blend_pix(p, cr, cg, cb, color_type::mult_cover(alpha, cover));
        }
        
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha)
        {
            if (alpha > color_type::empty_value())
            {
                calc_type a = p[Order::A];
                calc_type r = color_type::multiply(p[Order::R], a);
                calc_type g = color_type::multiply(p[Order::G], a);
                calc_type b = color_type::multiply(p[Order::B], a);
                p[Order::R] = color_type::lerp(r, cr, alpha);
                p[Order::G] = color_type::lerp(g, cg, alpha);
                p[Order::B] = color_type::lerp(b, cb, alpha);
                p[Order::A] = color_type::prelerp(a, alpha, alpha);
                multiplier_rgba<ColorT, Order>::demultiply(p);
            }
        }
    };

     
     

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_clear : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type, value_type, value_type, value_type, cover_type cover)
        {
            if (cover >= cover_full)
            {
                p[0] = p[1] = p[2] = p[3] = color_type::empty_value(); 
            }
            else if (cover > cover_none)
            {
                set(p, get(p, cover_full - cover));
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_src : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            if (cover >= cover_full)
            {
                set(p, r, g, b, a);
            }
            else
            {
                rgba s = get(r, g, b, a, cover);
                rgba d = get(p, cover_full - cover);
                d.r += s.r;
                d.g += s.g;
                d.b += s.b;
                d.a += s.a;
                set(p, d);
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_dst : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;

         
         
        static AGG_INLINE void blend_pix(value_type*, 
            value_type, value_type, value_type, value_type, cover_type)
        {
             
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_src_over : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
#if 1
            blender_rgba_pre<ColorT, Order>::blend_pix(p, r, g, b, a, cover);
#else
            rgba s = get(r, g, b, a, cover);
            rgba d = get(p);
            d.r += s.r - d.r * s.a;
            d.g += s.g - d.g * s.a;
            d.b += s.b - d.b * s.a;
            d.a += s.a - d.a * s.a;
            set(p, d);
#endif
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_dst_over : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            rgba d = get(p);
            double d1a = 1 - d.a;
            d.r += s.r * d1a;
            d.g += s.g * d1a;
            d.b += s.b * d1a;
            d.a += s.a * d1a;
            set(p, d);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_src_in : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            double da = ColorT::to_double(p[Order::A]);
            if (da > 0)
            {
                rgba s = get(r, g, b, a, cover);
                rgba d = get(p, cover_full - cover);
                d.r += s.r * da;
                d.g += s.g * da;
                d.b += s.b * da;
                d.a += s.a * da;
                set(p, d);
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_dst_in : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type, value_type, value_type, value_type a, cover_type cover)
        {
            double sa = ColorT::to_double(a);
            rgba d = get(p, cover_full - cover);
            rgba d2 = get(p, cover);
            d.r += d2.r * sa;
            d.g += d2.g * sa;
            d.b += d2.b * sa;
            d.a += d2.a * sa;
            set(p, d);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_src_out : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            rgba d = get(p, cover_full - cover);
            double d1a = 1 - ColorT::to_double(p[Order::A]);
            d.r += s.r * d1a;
            d.g += s.g * d1a;
            d.b += s.b * d1a;
            d.a += s.a * d1a;
            set(p, d);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_dst_out : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type, value_type, value_type, value_type a, cover_type cover)
        {
            rgba d = get(p, cover_full - cover);
            rgba dc = get(p, cover);
            double s1a = 1 - ColorT::to_double(a);
            d.r += dc.r * s1a;
            d.g += dc.g * s1a;
            d.b += dc.b * s1a;
            d.a += dc.a * s1a;
            set(p, d);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_src_atop : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            rgba d = get(p);
            double s1a = 1 - s.a;
            d.r = s.r * d.a + d.r * s1a;
            d.g = s.g * d.a + d.g * s1a;
            d.b = s.b * d.a + d.g * s1a;
            set(p, d);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_dst_atop : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba sc = get(r, g, b, a, cover);
            rgba dc = get(p, cover);
            rgba d = get(p, cover_full - cover);
            double sa = ColorT::to_double(a);
            double d1a = 1 - ColorT::to_double(p[Order::A]);
            d.r += dc.r * sa + sc.r * d1a;
            d.g += dc.g * sa + sc.g * d1a;
            d.b += dc.b * sa + sc.b * d1a;
            d.a += sc.a;
            set(p, d);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_xor : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            rgba d = get(p);
            double s1a = 1 - s.a;
            double d1a = 1 - ColorT::to_double(p[Order::A]);
            d.r = s.r * d1a + d.r * s1a;
            d.g = s.g * d1a + d.g * s1a;
            d.b = s.b * d1a + d.b * s1a;
            d.a = s.a + d.a - 2 * s.a * d.a;
            set(p, d);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_plus : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                d.a = sd_min(d.a + s.a, 1.0);
                d.r = sd_min(d.r + s.r, d.a);
                d.g = sd_min(d.g + s.g, d.a);
                d.b = sd_min(d.b + s.b, d.a);
                set(p, clip(d));
            }
        }
    };

     
     
    template<class ColorT, class Order> 
    struct comp_op_rgba_minus : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                d.a += s.a - s.a * d.a;
                d.r = sd_max(d.r - s.r, 0.0);
                d.g = sd_max(d.g - s.g, 0.0);
                d.b = sd_max(d.b - s.b, 0.0);
                set(p, clip(d));
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_multiply : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                double s1a = 1 - s.a;
                double d1a = 1 - d.a;
                d.r = s.r * d.r + s.r * d1a + d.r * s1a;
                d.g = s.g * d.g + s.g * d1a + d.g * s1a;
                d.b = s.b * d.b + s.b * d1a + d.b * s1a;
                d.a += s.a - s.a * d.a;
                set(p, clip(d));
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_screen : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                d.r += s.r - s.r * d.r;
                d.g += s.g - s.g * d.g;
                d.b += s.b - s.b * d.b;
                d.a += s.a - s.a * d.a;
                set(p, clip(d));
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_overlay : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
         
         
         
         
        static AGG_INLINE double calc(double dca, double sca, double da, double sa, double sada, double d1a, double s1a)
        {
            return (2 * dca <= da) ? 
                2 * sca * dca + sca * d1a + dca * s1a : 
                sada - 2 * (da - dca) * (sa - sca) + sca * d1a + dca * s1a;
        }

        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                double d1a = 1 - d.a;
                double s1a = 1 - s.a;
                double sada = s.a * d.a;
                d.r = calc(d.r, s.r, d.a, s.a, sada, d1a, s1a);
                d.g = calc(d.g, s.g, d.a, s.a, sada, d1a, s1a);
                d.b = calc(d.b, s.b, d.a, s.a, sada, d1a, s1a);
                d.a += s.a - s.a * d.a;
                set(p, clip(d));
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_darken : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                double d1a = 1 - d.a;
                double s1a = 1 - s.a;
                d.r = sd_min(s.r * d.a, d.r * s.a) + s.r * d1a + d.r * s1a;
                d.g = sd_min(s.g * d.a, d.g * s.a) + s.g * d1a + d.g * s1a;
                d.b = sd_min(s.b * d.a, d.b * s.a) + s.b * d1a + d.b * s1a;
                d.a += s.a - s.a * d.a;
                set(p, clip(d));
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_lighten : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                double d1a = 1 - d.a;
                double s1a = 1 - s.a;
                d.r = sd_max(s.r * d.a, d.r * s.a) + s.r * d1a + d.r * s1a;
                d.g = sd_max(s.g * d.a, d.g * s.a) + s.g * d1a + d.g * s1a;
                d.b = sd_max(s.b * d.a, d.b * s.a) + s.b * d1a + d.b * s1a;
                d.a += s.a - s.a * d.a;
                set(p, clip(d));
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_color_dodge : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
         
         
         
         
         
         
        static AGG_INLINE double calc(double dca, double sca, double da, double sa, double sada, double d1a, double s1a)
        {
            if (sca < sa) return sada * sd_min(1.0, (dca / da) * sa / (sa - sca)) + sca * d1a + dca * s1a;
            if (dca > 0) return sada + sca * d1a + dca * s1a;
            return sca * d1a;
        }

        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                if (d.a > 0)
                {
                    double sada = s.a * d.a;
                    double s1a = 1 - s.a;
                    double d1a = 1 - d.a;
                    d.r = calc(d.r, s.r, d.a, s.a, sada, d1a, s1a);
                    d.g = calc(d.g, s.g, d.a, s.a, sada, d1a, s1a);
                    d.b = calc(d.b, s.b, d.a, s.a, sada, d1a, s1a);
                    d.a += s.a - s.a * d.a;
                    set(p, clip(d));
                }
                else set(p, s);
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_color_burn : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
         
         
         
         
        static AGG_INLINE double calc(double dca, double sca, double da, double sa, double sada, double d1a, double s1a)
        {
            if (sca > 0) return sada * (1 - sd_min(1.0, (1 - dca / da) * sa / sca)) + sca * d1a + dca * s1a;
            if (dca > da) return sada + dca * s1a;
            return dca * s1a;
        }

        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                if (d.a > 0)
                {
                    double sada = s.a * d.a;
                    double s1a = 1 - s.a;
                    double d1a = 1 - d.a;
                    d.r = calc(d.r, s.r, d.a, s.a, sada, d1a, s1a);
                    d.g = calc(d.g, s.g, d.a, s.a, sada, d1a, s1a);
                    d.b = calc(d.b, s.b, d.a, s.a, sada, d1a, s1a);
                    d.a += s.a - sada;
                    set(p, clip(d));
                }
                else set(p, s);
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_hard_light : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
         
         
         
         
        static AGG_INLINE double calc(double dca, double sca, double da, double sa, double sada, double d1a, double s1a)
        {
            return (2 * sca < sa) ? 
                2 * sca * dca + sca * d1a + dca * s1a : 
                sada - 2 * (da - dca) * (sa - sca) + sca * d1a + dca * s1a;
        }

        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                double d1a = 1 - d.a;
                double s1a = 1 - s.a;
                double sada = s.a * d.a;
                d.r = calc(d.r, s.r, d.a, s.a, sada, d1a, s1a);
                d.g = calc(d.g, s.g, d.a, s.a, sada, d1a, s1a);
                d.b = calc(d.b, s.b, d.a, s.a, sada, d1a, s1a);
                d.a += s.a - sada;
                set(p, clip(d));
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_soft_light : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
         
         
         
         
         
         
        static AGG_INLINE double calc(double dca, double sca, double da, double sa, double sada, double d1a, double s1a)
        {
            double dcasa = dca * sa;
            if (2 * sca <= sa) return dcasa - (sada - 2 * sca * da) * dcasa * (sada - dcasa) + sca * d1a + dca * s1a;
            if (4 * dca <= da) return dcasa + (2 * sca * da - sada) * ((((16 * dcasa - 12) * dcasa + 4) * dca * da) - dca * da) + sca * d1a + dca * s1a;
            return dcasa + (2 * sca * da - sada) * (std::sqrt(dcasa) - dcasa) + sca * d1a + dca * s1a;
        }

        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                if (d.a > 0)
                {
                    double sada = s.a * d.a;
                    double s1a = 1 - s.a;
                    double d1a = 1 - d.a;
                    d.r = calc(d.r, s.r, d.a, s.a, sada, d1a, s1a);
                    d.g = calc(d.g, s.g, d.a, s.a, sada, d1a, s1a);
                    d.b = calc(d.b, s.b, d.a, s.a, sada, d1a, s1a);
                    d.a += s.a - sada;
                    set(p, clip(d));
                }
                else set(p, s);
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_difference : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                d.r += s.r - 2 * sd_min(s.r * d.a, d.r * s.a);
                d.g += s.g - 2 * sd_min(s.g * d.a, d.g * s.a);
                d.b += s.b - 2 * sd_min(s.b * d.a, d.b * s.a);
                d.a += s.a - s.a * d.a;
                set(p, clip(d));
            }
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_rgba_exclusion : blender_base<ColorT, Order>
    {
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        using blender_base<ColorT, Order>::get;
        using blender_base<ColorT, Order>::set;

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            rgba s = get(r, g, b, a, cover);
            if (s.a > 0)
            {
                rgba d = get(p);
                double d1a = 1 - d.a;
                double s1a = 1 - s.a;
                d.r = (s.r * d.a + d.r * s.a - 2 * s.r * d.r) + s.r * d1a + d.r * s1a;
                d.g = (s.g * d.a + d.g * s.a - 2 * s.g * d.g) + s.g * d1a + d.g * s1a;
                d.b = (s.b * d.a + d.b * s.a - 2 * s.b * d.b) + s.b * d1a + d.b * s1a;
                d.a += s.a - s.a * d.a;
                set(p, clip(d));
            }
        }
    };

#if 0
     
    template<class ColorT, class Order> struct comp_op_rgba_contrast
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;
        enum base_scale_e
        { 
            base_shift = color_type::base_shift,
            base_mask  = color_type::base_mask
        };


        static AGG_INLINE void blend_pix(value_type* p, 
                                         unsigned sr, unsigned sg, unsigned sb, 
                                         unsigned sa, unsigned cover)
        {
            if (cover < 255)
            {
                sr = (sr * cover + 255) >> 8;
                sg = (sg * cover + 255) >> 8;
                sb = (sb * cover + 255) >> 8;
                sa = (sa * cover + 255) >> 8;
            }
            long_type dr = p[Order::R];
            long_type dg = p[Order::G];
            long_type db = p[Order::B];
            int       da = p[Order::A];
            long_type d2a = da >> 1;
            unsigned s2a = sa >> 1;

            int r = (int)((((dr - d2a) * int((sr - s2a)*2 + base_mask)) >> base_shift) + d2a); 
            int g = (int)((((dg - d2a) * int((sg - s2a)*2 + base_mask)) >> base_shift) + d2a); 
            int b = (int)((((db - d2a) * int((sb - s2a)*2 + base_mask)) >> base_shift) + d2a); 

            r = (r < 0) ? 0 : r;
            g = (g < 0) ? 0 : g;
            b = (b < 0) ? 0 : b;

            p[Order::R] = (value_type)((r > da) ? da : r);
            p[Order::G] = (value_type)((g > da) ? da : g);
            p[Order::B] = (value_type)((b > da) ? da : b);
        }
    };

     
    template<class ColorT, class Order> struct comp_op_rgba_invert
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;
        enum base_scale_e
        { 
            base_shift = color_type::base_shift,
            base_mask  = color_type::base_mask
        };

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
                                         unsigned sr, unsigned sg, unsigned sb, 
                                         unsigned sa, unsigned cover)
        {
            sa = (sa * cover + 255) >> 8;
            if (sa)
            {
                calc_type da = p[Order::A];
                calc_type dr = ((da - p[Order::R]) * sa + base_mask) >> base_shift;
                calc_type dg = ((da - p[Order::G]) * sa + base_mask) >> base_shift;
                calc_type db = ((da - p[Order::B]) * sa + base_mask) >> base_shift;
                calc_type s1a = base_mask - sa;
                p[Order::R] = (value_type)(dr + ((p[Order::R] * s1a + base_mask) >> base_shift));
                p[Order::G] = (value_type)(dg + ((p[Order::G] * s1a + base_mask) >> base_shift));
                p[Order::B] = (value_type)(db + ((p[Order::B] * s1a + base_mask) >> base_shift));
                p[Order::A] = (value_type)(sa + da - ((sa * da + base_mask) >> base_shift));
            }
        }
    };

     
    template<class ColorT, class Order> struct comp_op_rgba_invert_rgb
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;
        enum base_scale_e
        { 
            base_shift = color_type::base_shift,
            base_mask  = color_type::base_mask
        };

         
         
        static AGG_INLINE void blend_pix(value_type* p, 
                                         unsigned sr, unsigned sg, unsigned sb, 
                                         unsigned sa, unsigned cover)
        {
            if (cover < 255)
            {
                sr = (sr * cover + 255) >> 8;
                sg = (sg * cover + 255) >> 8;
                sb = (sb * cover + 255) >> 8;
                sa = (sa * cover + 255) >> 8;
            }
            if (sa)
            {
                calc_type da = p[Order::A];
                calc_type dr = ((da - p[Order::R]) * sr + base_mask) >> base_shift;
                calc_type dg = ((da - p[Order::G]) * sg + base_mask) >> base_shift;
                calc_type db = ((da - p[Order::B]) * sb + base_mask) >> base_shift;
                calc_type s1a = base_mask - sa;
                p[Order::R] = (value_type)(dr + ((p[Order::R] * s1a + base_mask) >> base_shift));
                p[Order::G] = (value_type)(dg + ((p[Order::G] * s1a + base_mask) >> base_shift));
                p[Order::B] = (value_type)(db + ((p[Order::B] * s1a + base_mask) >> base_shift));
                p[Order::A] = (value_type)(sa + da - ((sa * da + base_mask) >> base_shift));
            }
        }
    };
#endif


     
    template<class ColorT, class Order> struct comp_op_table_rgba
    {
        typedef typename ColorT::value_type value_type;
        typedef typename ColorT::calc_type calc_type;
        typedef void (*comp_op_func_type)(value_type* p, 
                                          value_type cr, 
                                          value_type cg, 
                                          value_type cb,
                                          value_type ca,
                                          cover_type cover);
        static comp_op_func_type g_comp_op_func[];
    };

     
    template<class ColorT, class Order> 
    typename comp_op_table_rgba<ColorT, Order>::comp_op_func_type
    comp_op_table_rgba<ColorT, Order>::g_comp_op_func[] = 
    {
        comp_op_rgba_clear      <ColorT,Order>::blend_pix,
        comp_op_rgba_src        <ColorT,Order>::blend_pix,
        comp_op_rgba_dst        <ColorT,Order>::blend_pix,
        comp_op_rgba_src_over   <ColorT,Order>::blend_pix,
        comp_op_rgba_dst_over   <ColorT,Order>::blend_pix,
        comp_op_rgba_src_in     <ColorT,Order>::blend_pix,
        comp_op_rgba_dst_in     <ColorT,Order>::blend_pix,
        comp_op_rgba_src_out    <ColorT,Order>::blend_pix,
        comp_op_rgba_dst_out    <ColorT,Order>::blend_pix,
        comp_op_rgba_src_atop   <ColorT,Order>::blend_pix,
        comp_op_rgba_dst_atop   <ColorT,Order>::blend_pix,
        comp_op_rgba_xor        <ColorT,Order>::blend_pix,
        comp_op_rgba_plus       <ColorT,Order>::blend_pix,
         
        comp_op_rgba_multiply   <ColorT,Order>::blend_pix,
        comp_op_rgba_screen     <ColorT,Order>::blend_pix,
        comp_op_rgba_overlay    <ColorT,Order>::blend_pix,
        comp_op_rgba_darken     <ColorT,Order>::blend_pix,
        comp_op_rgba_lighten    <ColorT,Order>::blend_pix,
        comp_op_rgba_color_dodge<ColorT,Order>::blend_pix,
        comp_op_rgba_color_burn <ColorT,Order>::blend_pix,
        comp_op_rgba_hard_light <ColorT,Order>::blend_pix,
        comp_op_rgba_soft_light <ColorT,Order>::blend_pix,
        comp_op_rgba_difference <ColorT,Order>::blend_pix,
        comp_op_rgba_exclusion  <ColorT,Order>::blend_pix,
         
         
         
        0
    };


     
    enum comp_op_e
    {
        comp_op_clear,          
        comp_op_src,            
        comp_op_dst,            
        comp_op_src_over,       
        comp_op_dst_over,       
        comp_op_src_in,         
        comp_op_dst_in,         
        comp_op_src_out,        
        comp_op_dst_out,        
        comp_op_src_atop,       
        comp_op_dst_atop,       
        comp_op_xor,            
        comp_op_plus,           
         
        comp_op_multiply,       
        comp_op_screen,         
        comp_op_overlay,        
        comp_op_darken,         
        comp_op_lighten,        
        comp_op_color_dodge,    
        comp_op_color_burn,     
        comp_op_hard_light,     
        comp_op_soft_light,     
        comp_op_difference,     
        comp_op_exclusion,      
         
         
         

        end_of_comp_op_e
    };







     
    template<class ColorT, class Order> 
    struct comp_op_adaptor_rgba
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned op, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            comp_op_table_rgba<ColorT, Order>::g_comp_op_func[op](p, 
                color_type::multiply(r, a), 
                color_type::multiply(g, a), 
                color_type::multiply(b, a), 
                a, cover);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_adaptor_clip_to_dst_rgba
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned op, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            r = color_type::multiply(r, a);
            g = color_type::multiply(g, a);
            b = color_type::multiply(b, a);
            value_type da = p[Order::A];
            comp_op_table_rgba<ColorT, Order>::g_comp_op_func[op](p, 
                color_type::multiply(r, da), 
                color_type::multiply(g, da), 
                color_type::multiply(b, da), 
                color_type::multiply(a, da), cover);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_adaptor_rgba_pre
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned op, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            comp_op_table_rgba<ColorT, Order>::g_comp_op_func[op](p, r, g, b, a, cover);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_adaptor_clip_to_dst_rgba_pre
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned op, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            value_type da = p[Order::A];
            comp_op_table_rgba<ColorT, Order>::g_comp_op_func[op](p, 
                color_type::multiply(r, da), 
                color_type::multiply(g, da), 
                color_type::multiply(b, da), 
                color_type::multiply(a, da), cover);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_adaptor_rgba_plain
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned op, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            multiplier_rgba<ColorT, Order>::premultiply(p);
            comp_op_adaptor_rgba<ColorT, Order>::blend_pix(op, p, r, g, b, a, cover);
            multiplier_rgba<ColorT, Order>::demultiply(p);
        }
    };

     
    template<class ColorT, class Order> 
    struct comp_op_adaptor_clip_to_dst_rgba_plain
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned op, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            multiplier_rgba<ColorT, Order>::premultiply(p);
            comp_op_adaptor_clip_to_dst_rgba<ColorT, Order>::blend_pix(op, p, r, g, b, a, cover);
            multiplier_rgba<ColorT, Order>::demultiply(p);
        }
    };

     
    template<class BlenderPre> 
    struct comp_adaptor_rgba
    {
        typedef typename BlenderPre::color_type color_type;
        typedef typename BlenderPre::order_type order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            BlenderPre::blend_pix(p, 
                color_type::multiply(r, a), 
                color_type::multiply(g, a), 
                color_type::multiply(b, a), 
                a, cover);
        }
    };

     
    template<class BlenderPre> 
    struct comp_adaptor_clip_to_dst_rgba
    {
        typedef typename BlenderPre::color_type color_type;
        typedef typename BlenderPre::order_type order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            r = color_type::multiply(r, a);
            g = color_type::multiply(g, a);
            b = color_type::multiply(b, a);
            value_type da = p[order_type::A];
            BlenderPre::blend_pix(p, 
                color_type::multiply(r, da), 
                color_type::multiply(g, da), 
                color_type::multiply(b, da), 
                color_type::multiply(a, da), cover);
        }
    };

     
    template<class BlenderPre> 
    struct comp_adaptor_rgba_pre
    {
        typedef typename BlenderPre::color_type color_type;
        typedef typename BlenderPre::order_type order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            BlenderPre::blend_pix(p, r, g, b, a, cover);
        }
    };

     
    template<class BlenderPre> 
    struct comp_adaptor_clip_to_dst_rgba_pre
    {
        typedef typename BlenderPre::color_type color_type;
        typedef typename BlenderPre::order_type order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            unsigned da = p[order_type::A];
            BlenderPre::blend_pix(p, 
                color_type::multiply(r, da), 
                color_type::multiply(g, da), 
                color_type::multiply(b, da), 
                color_type::multiply(a, da), 
                cover);
        }
    };

     
    template<class BlenderPre> 
    struct comp_adaptor_rgba_plain
    {
        typedef typename BlenderPre::color_type color_type;
        typedef typename BlenderPre::order_type order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned op, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            multiplier_rgba<color_type, order_type>::premultiply(p);
            comp_adaptor_rgba<BlenderPre>::blend_pix(op, p, r, g, b, a, cover);
            multiplier_rgba<color_type, order_type>::demultiply(p);
        }
    };

     
    template<class BlenderPre> 
    struct comp_adaptor_clip_to_dst_rgba_plain
    {
        typedef typename BlenderPre::color_type color_type;
        typedef typename BlenderPre::order_type order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

        static AGG_INLINE void blend_pix(unsigned op, value_type* p, 
            value_type r, value_type g, value_type b, value_type a, cover_type cover)
        {
            multiplier_rgba<color_type, order_type>::premultiply(p);
            comp_adaptor_clip_to_dst_rgba<BlenderPre>::blend_pix(op, p, r, g, b, a, cover);
            multiplier_rgba<color_type, order_type>::demultiply(p);
        }
    };


     
    template<class Blender, class RenBuf> 
    class pixfmt_alpha_blend_rgba
    {
    public:
        typedef pixfmt_rgba_tag pixfmt_category;
        typedef RenBuf   rbuf_type;
        typedef typename rbuf_type::row_data row_data;
        typedef Blender  blender_type;
        typedef typename blender_type::color_type color_type;
        typedef typename blender_type::order_type order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        enum 
        {
            num_components = 4,
            pix_step = 4,
            pix_width = sizeof(value_type) * pix_step,
        };
        struct pixel_type
        {
            value_type c[num_components];

            void set(value_type r, value_type g, value_type b, value_type a)
            {
                c[order_type::R] = r;
                c[order_type::G] = g;
                c[order_type::B] = b;
                c[order_type::A] = a;
            }

            void set(const color_type& color)
            {
                set(color.r, color.g, color.b, color.a);
            }

            void get(value_type& r, value_type& g, value_type& b, value_type& a) const
            {
                r = c[order_type::R];
                g = c[order_type::G];
                b = c[order_type::B];
                a = c[order_type::A];
            }

            color_type get() const
            {
                return color_type(
                    c[order_type::R], 
                    c[order_type::G], 
                    c[order_type::B],
                    c[order_type::A]);
            }

            pixel_type* next()
            {
                return (pixel_type*)(c + pix_step);
            }

            const pixel_type* next() const
            {
                return (const pixel_type*)(c + pix_step);
            }

            pixel_type* advance(int n)
            {
                return (pixel_type*)(c + n * pix_step);
            }

            const pixel_type* advance(int n) const
            {
                return (const pixel_type*)(c + n * pix_step);
            }
        };

    private:
         
        AGG_INLINE void blend_pix(pixel_type* p, const color_type& c, unsigned cover)
        {
            m_blender.blend_pix(p->c, c.r, c.g, c.b, c.a, cover);
        }

         
        AGG_INLINE void blend_pix(pixel_type* p, const color_type& c)
        {
            m_blender.blend_pix(p->c, c.r, c.g, c.b, c.a);
        }

         
        AGG_INLINE void copy_or_blend_pix(pixel_type* p, const color_type& c, unsigned cover)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque() && cover == cover_mask)
                {
                    p->set(c.r, c.g, c.b, c.a);
                }
                else
                {
                    m_blender.blend_pix(p->c, c.r, c.g, c.b, c.a, cover);
                }
            }
        }

         
        AGG_INLINE void copy_or_blend_pix(pixel_type* p, const color_type& c)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque())
                {
                    p->set(c.r, c.g, c.b, c.a);
                }
                else
                {
                    m_blender.blend_pix(p->c, c.r, c.g, c.b, c.a);
                }
            }
        }

    public:
         
        pixfmt_alpha_blend_rgba() : m_rbuf(0) {}
        explicit pixfmt_alpha_blend_rgba(rbuf_type& rb) : m_rbuf(&rb) {}
        void attach(rbuf_type& rb) { m_rbuf = &rb; }

         
        template<class PixFmt>
        bool attach(PixFmt& pixf, int x1, int y1, int x2, int y2)
        {
            rect_i r(x1, y1, x2, y2);
            if (r.clip(rect_i(0, 0, pixf.width()-1, pixf.height()-1)))
            {
                int stride = pixf.stride();
                m_rbuf->attach(pixf.pix_ptr(r.x1, stride < 0 ? r.y2 : r.y1), 
                               (r.x2 - r.x1) + 1,
                               (r.y2 - r.y1) + 1,
                               stride);
                return true;
            }
            return false;
        }

         
        AGG_INLINE unsigned width()  const { return m_rbuf->width();  }
        AGG_INLINE unsigned height() const { return m_rbuf->height(); }
        AGG_INLINE int      stride() const { return m_rbuf->stride(); }

         
        AGG_INLINE       int8u* row_ptr(int y)       { return m_rbuf->row_ptr(y); }
        AGG_INLINE const int8u* row_ptr(int y) const { return m_rbuf->row_ptr(y); }
        AGG_INLINE row_data     row(int y)     const { return m_rbuf->row(y); }

         
        AGG_INLINE int8u* pix_ptr(int x, int y) 
        { 
            return m_rbuf->row_ptr(y) + sizeof(value_type) * (x * pix_step);
        }

        AGG_INLINE const int8u* pix_ptr(int x, int y) const 
        { 
            return m_rbuf->row_ptr(y) + sizeof(value_type) * (x * pix_step);
        }

         
        AGG_INLINE pixel_type* pix_value_ptr(int x, int y, unsigned len) 
        {
            return (pixel_type*)(m_rbuf->row_ptr(x, y, len) + sizeof(value_type) * (x * pix_step));
        }

         
        AGG_INLINE const pixel_type* pix_value_ptr(int x, int y) const 
        {
            int8u* p = m_rbuf->row_ptr(y);
            return p ? (pixel_type*)(p + sizeof(value_type) * (x * pix_step)) : 0;
        }

         
        AGG_INLINE static pixel_type* pix_value_ptr(void* p) 
        {
            return (pixel_type*)p;
        }

         
        AGG_INLINE static const pixel_type* pix_value_ptr(const void* p) 
        {
            return (const pixel_type*)p;
        }

         
        AGG_INLINE static void write_plain_color(void* p, color_type c)
        {
            blender_type::set_plain_color(pix_value_ptr(p)->c, c);
        }

         
        AGG_INLINE static color_type read_plain_color(const void* p)
        {
            return blender_type::get_plain_color(pix_value_ptr(p)->c);
        }

         
        AGG_INLINE static void make_pix(int8u* p, const color_type& c)
        {
            ((pixel_type*)p)->set(c);
        }

         
        AGG_INLINE color_type pixel(int x, int y) const
        {
            if (const pixel_type* p = pix_value_ptr(x, y))
            {
                return p->get();
            }
            return color_type::no_color();
        }

         
        AGG_INLINE void copy_pixel(int x, int y, const color_type& c)
        {
            pix_value_ptr(x, y, 1)->set(c);
        }

         
        AGG_INLINE void blend_pixel(int x, int y, const color_type& c, int8u cover)
        {
            copy_or_blend_pix(pix_value_ptr(x, y, 1), c, cover);
        }

         
        AGG_INLINE void copy_hline(int x, int y, 
                                   unsigned len, 
                                   const color_type& c)
        {
            pixel_type v;
            v.set(c);
            pixel_type* p = pix_value_ptr(x, y, len);
            do
            {
                *p = v;
                p = p->next();
            }
            while (--len);
        }


         
        AGG_INLINE void copy_vline(int x, int y,
                                   unsigned len, 
                                   const color_type& c)
        {
            pixel_type v;
            v.set(c);
            do
            {
                *pix_value_ptr(x, y++, 1) = v;
            }
            while (--len);
        }

         
        void blend_hline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         int8u cover)
        {
            if (!c.is_transparent())
            {
                pixel_type* p = pix_value_ptr(x, y, len);
                if (c.is_opaque() && cover == cover_mask)
                {
                    pixel_type v;
                    v.set(c);
                    do
                    {
                        *p = v;
                        p = p->next();
                    }
                    while (--len);
                }
                else
                {
                    if (cover == cover_mask)
                    {
                        do
                        {
                            blend_pix(p, c);
                            p = p->next();
                        }
                        while (--len);
                    }
                    else
                    {
                        do
                        {
                            blend_pix(p, c, cover);
                            p = p->next();
                        }
                        while (--len);
                    }
                }
            }
        }


         
        void blend_vline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         int8u cover)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque() && cover == cover_mask)
                {
                    pixel_type v;
                    v.set(c);
                    do
                    {
                        *pix_value_ptr(x, y++, 1) = v;
                    }
                    while (--len);
                }
                else
                {
                    if (cover == cover_mask)
                    {
                        do
                        {
                            blend_pix(pix_value_ptr(x, y++, 1), c, c.a);
                        }
                        while (--len);
                    }
                    else
                    {
                        do
                        {
                            blend_pix(pix_value_ptr(x, y++, 1), c, cover);
                        }
                        while (--len);
                    }
                }
            }
        }


         
        void blend_solid_hspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const int8u* covers)
        {
            if (!c.is_transparent())
            {
                pixel_type* p = pix_value_ptr(x, y, len);
                do 
                {
                    if (c.is_opaque() && *covers == cover_mask)
                    {
                        p->set(c);
                    }
                    else
                    {
                        blend_pix(p, c, *covers);
                    }
                    p = p->next();
                    ++covers;
                }
                while (--len);
            }
        }


         
        void blend_solid_vspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const int8u* covers)
        {
            if (!c.is_transparent())
            {
                do 
                {
                    pixel_type* p = pix_value_ptr(x, y++, 1);
                    if (c.is_opaque() && *covers == cover_mask)
                    {
                        p->set(c);
                    }
                    else
                    {
                        blend_pix(p, c, *covers);
                    }
                    ++covers;
                }
                while (--len);
            }
        }

         
        void copy_color_hspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            pixel_type* p = pix_value_ptr(x, y, len);
            do 
            {
                p->set(*colors++);
                p = p->next();
            }
            while (--len);
        }


         
        void copy_color_vspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            do 
            {
                pix_value_ptr(x, y++, 1)->set(*colors++);
            }
            while (--len);
        }

         
        void blend_color_hspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const int8u* covers,
                               int8u cover)
        {
            pixel_type* p = pix_value_ptr(x, y, len);
            if (covers)
            {
                do 
                {
                    copy_or_blend_pix(p, *colors++, *covers++);
                    p = p->next();
                }
                while (--len);
            }
            else
            {
                if (cover == cover_mask)
                {
                    do 
                    {
                        copy_or_blend_pix(p, *colors++);
                        p = p->next();
                    }
                    while (--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(p, *colors++, cover);
                        p = p->next();
                    }
                    while (--len);
                }
            }
        }

         
        void blend_color_vspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const int8u* covers,
                               int8u cover)
        {
            if (covers)
            {
                do 
                {
                    copy_or_blend_pix(pix_value_ptr(x, y++, 1), *colors++, *covers++);
                }
                while (--len);
            }
            else
            {
                if (cover == cover_mask)
                {
                    do 
                    {
                        copy_or_blend_pix(pix_value_ptr(x, y++, 1), *colors++);
                    }
                    while (--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(pix_value_ptr(x, y++, 1), *colors++, cover);
                    }
                    while (--len);
                }
            }
        }

         
        template<class Function> void for_each_pixel(Function f)
        {
            for (unsigned y = 0; y < height(); ++y)
            {
                row_data r = m_rbuf->row(y);
                if (r.ptr)
                {
                    unsigned len = r.x2 - r.x1 + 1;
                    pixel_type* p = pix_value_ptr(r.x1, y, len);
                    do
                    {
                        f(p->c);
                        p = p->next();
                    }
                    while (--len);
                }
            }
        }

         
        void premultiply()
        {
            for_each_pixel(multiplier_rgba<color_type, order_type>::premultiply);
        }

         
        void demultiply()
        {
            for_each_pixel(multiplier_rgba<color_type, order_type>::demultiply);
        }

         
        template<class GammaLut> void apply_gamma_dir(const GammaLut& g)
        {
            for_each_pixel(apply_gamma_dir_rgba<color_type, order_type, GammaLut>(g));
        }

         
        template<class GammaLut> void apply_gamma_inv(const GammaLut& g)
        {
            for_each_pixel(apply_gamma_inv_rgba<color_type, order_type, GammaLut>(g));
        }

         
        template<class RenBuf2> void copy_from(const RenBuf2& from, 
                                               int xdst, int ydst,
                                               int xsrc, int ysrc,
                                               unsigned len)
        {
            if (const int8u* p = from.row_ptr(ysrc))
            {
                std::memmove(m_rbuf->row_ptr(xdst, ydst, len) + xdst * pix_width, 
                        p + xsrc * pix_width, 
                        len * pix_width);
            }
        }

         
         
        template<class SrcPixelFormatRenderer>
        void blend_from(const SrcPixelFormatRenderer& from, 
                        int xdst, int ydst,
                        int xsrc, int ysrc,
                        unsigned len,
                        int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);
                int srcinc = 1;
                int dstinc = 1;

                if (xdst > xsrc)
                {
                    psrc = psrc->advance(len - 1);
                    pdst = pdst->advance(len - 1);
                    srcinc = -1;
                    dstinc = -1;
                }

                if (cover == cover_mask)
                {
                    do 
                    {
                        copy_or_blend_pix(pdst, psrc->get());
                        psrc = psrc->advance(srcinc);
                        pdst = pdst->advance(dstinc);
                    }
                    while (--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(pdst, psrc->get(), cover);
                        psrc = psrc->advance(srcinc);
                        pdst = pdst->advance(dstinc);
                    }
                    while (--len);
                }
            }
        }

         
         
        template<class SrcPixelFormatRenderer>
        void blend_from_color(const SrcPixelFormatRenderer& from, 
                              const color_type& color,
                              int xdst, int ydst,
                              int xsrc, int ysrc,
                              unsigned len,
                              int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;
            typedef typename SrcPixelFormatRenderer::color_type src_color_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);

                do 
                {
                    copy_or_blend_pix(pdst, color, 
                        src_color_type::scale_cover(cover, psrc->c[0]));
                    psrc = psrc->next();
                    pdst = pdst->next();
                }
                while (--len);
            }
        }

         
         
         
        template<class SrcPixelFormatRenderer>
        void blend_from_lut(const SrcPixelFormatRenderer& from, 
                            const color_type* color_lut,
                            int xdst, int ydst,
                            int xsrc, int ysrc,
                            unsigned len,
                            int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);

                if (cover == cover_mask)
                {
                    do 
                    {
                        copy_or_blend_pix(pdst, color_lut[psrc->c[0]]);
                        psrc = psrc->next();
                        pdst = pdst->next();
                    }
                    while (--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(pdst, color_lut[psrc->c[0]], cover);
                        psrc = psrc->next();
                        pdst = pdst->next();
                    }
                    while (--len);
                }
            }
        }

    private:
        rbuf_type* m_rbuf;
        Blender    m_blender;
    };

     
    template<class Blender, class RenBuf> class pixfmt_custom_blend_rgba
    {
    public:
        typedef pixfmt_rgba_tag pixfmt_category;
        typedef RenBuf   rbuf_type;
        typedef typename rbuf_type::row_data row_data;
        typedef Blender  blender_type;
        typedef typename blender_type::color_type color_type;
        typedef typename blender_type::order_type order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        enum 
        {
            num_components = 4,
            pix_step = 4,
            pix_width  = sizeof(value_type) * pix_step,
        };
        struct pixel_type
        {
            value_type c[num_components];

            void set(value_type r, value_type g, value_type b, value_type a)
            {
                c[order_type::R] = r;
                c[order_type::G] = g;
                c[order_type::B] = b;
                c[order_type::A] = a;
            }

            void set(const color_type& color)
            {
                set(color.r, color.g, color.b, color.a);
            }

            void get(value_type& r, value_type& g, value_type& b, value_type& a) const
            {
                r = c[order_type::R];
                g = c[order_type::G];
                b = c[order_type::B];
                a = c[order_type::A];
            }

            color_type get() const
            {
                return color_type(
                    c[order_type::R], 
                    c[order_type::G], 
                    c[order_type::B],
                    c[order_type::A]);
            }

            pixel_type* next()
            {
                return (pixel_type*)(c + pix_step);
            }

            const pixel_type* next() const
            {
                return (const pixel_type*)(c + pix_step);
            }

            pixel_type* advance(int n)
            {
                return (pixel_type*)(c + n * pix_step);
            }

            const pixel_type* advance(int n) const
            {
                return (const pixel_type*)(c + n * pix_step);
            }
        };


    private:
         
        AGG_INLINE void blend_pix(pixel_type* p, const color_type& c, unsigned cover = cover_full)
        {
            m_blender.blend_pix(m_comp_op, p->c, c.r, c.g, c.b, c.a, cover);
        }

         
        AGG_INLINE void copy_or_blend_pix(pixel_type* p, const color_type& c, unsigned cover = cover_full)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque() && cover == cover_mask)
                {
                    p->set(c.r, c.g, c.b, c.a);
                }
                else
                {
                    blend_pix(p, c, cover);
                }
            }
        }

    public:
         
        pixfmt_custom_blend_rgba() : m_rbuf(0), m_comp_op(3) {}
        explicit pixfmt_custom_blend_rgba(rbuf_type& rb, unsigned comp_op=3) : 
            m_rbuf(&rb),
            m_comp_op(comp_op)
        {}
        void attach(rbuf_type& rb) { m_rbuf = &rb; }

         
        template<class PixFmt>
        bool attach(PixFmt& pixf, int x1, int y1, int x2, int y2)
        {
            rect_i r(x1, y1, x2, y2);
            if (r.clip(rect_i(0, 0, pixf.width()-1, pixf.height()-1)))
            {
                int stride = pixf.stride();
                m_rbuf->attach(pixf.pix_ptr(r.x1, stride < 0 ? r.y2 : r.y1), 
                               (r.x2 - r.x1) + 1,
                               (r.y2 - r.y1) + 1,
                               stride);
                return true;
            }
            return false;
        }

         
        void comp_op(unsigned op) { m_comp_op = op; }
        unsigned comp_op() const  { return m_comp_op; }

         
        AGG_INLINE unsigned width()  const { return m_rbuf->width();  }
        AGG_INLINE unsigned height() const { return m_rbuf->height(); }
        AGG_INLINE int      stride() const { return m_rbuf->stride(); }

         
        AGG_INLINE       int8u* row_ptr(int y)       { return m_rbuf->row_ptr(y); }
        AGG_INLINE const int8u* row_ptr(int y) const { return m_rbuf->row_ptr(y); }
        AGG_INLINE row_data     row(int y)     const { return m_rbuf->row(y); }

         
        AGG_INLINE int8u* pix_ptr(int x, int y) 
        { 
            return m_rbuf->row_ptr(y) + sizeof(value_type) * (x * pix_step);
        }

        AGG_INLINE const int8u* pix_ptr(int x, int y) const 
        { 
            return m_rbuf->row_ptr(y) + sizeof(value_type) * (x * pix_step);
        }

         
        AGG_INLINE pixel_type* pix_value_ptr(int x, int y, unsigned len) 
        {
            return (pixel_type*)(m_rbuf->row_ptr(x, y, len) + sizeof(value_type) * (x * pix_step));
        }

         
        AGG_INLINE const pixel_type* pix_value_ptr(int x, int y) const 
        {
            int8u* p = m_rbuf->row_ptr(y);
            return p ? (pixel_type*)(p + sizeof(value_type) * (x * pix_step)) : 0;
        }

         
        AGG_INLINE static pixel_type* pix_value_ptr(void* p) 
        {
            return (pixel_type*)p;
        }

         
        AGG_INLINE static const pixel_type* pix_value_ptr(const void* p) 
        {
            return (const pixel_type*)p;
        }

         
        AGG_INLINE static void make_pix(int8u* p, const color_type& c)
        {
            ((pixel_type*)p)->set(c);
        }

         
        AGG_INLINE color_type pixel(int x, int y) const
        {
            if (const pixel_type* p = pix_value_ptr(x, y))
            {
                return p->get();
            }
            return color_type::no_color();
        }

         
        AGG_INLINE void copy_pixel(int x, int y, const color_type& c)
        {
            pix_value_ptr(x, y, 1)->set(c);
        }

         
        AGG_INLINE void blend_pixel(int x, int y, const color_type& c, int8u cover)
        {
            blend_pix(pix_value_ptr(x, y, 1), c, cover);
        }

         
        AGG_INLINE void copy_hline(int x, int y, 
                                   unsigned len, 
                                   const color_type& c)
        {
            pixel_type v;
            v.set(c);
            pixel_type* p = pix_value_ptr(x, y, len);
            do
            {
                *p = v;
                p = p->next();
            }
            while (--len);
        }


         
        AGG_INLINE void copy_vline(int x, int y,
                                   unsigned len, 
                                   const color_type& c)
        {
            pixel_type v;
            v.set(c);
            do
            {
                *pix_value_ptr(x, y++, 1) = v;
            }
            while (--len);
        }

         
        void blend_hline(int x, int y, unsigned len, 
                         const color_type& c, int8u cover)
        {

            pixel_type* p = pix_value_ptr(x, y, len);
            do
            {
                blend_pix(p, c, cover);
                p = p->next();
            }
            while (--len);
        }

         
        void blend_vline(int x, int y, unsigned len, 
                         const color_type& c, int8u cover)
        {
            do
            {
                blend_pix(pix_value_ptr(x, y++, 1), c, cover);
            }
            while (--len);
        }

         
        void blend_solid_hspan(int x, int y, unsigned len, 
                               const color_type& c, const int8u* covers)
        {
            pixel_type* p = pix_value_ptr(x, y, len);

            do 
            {
                blend_pix(p, c, *covers++);
                p = p->next();
            }
            while (--len);
        }

         
        void blend_solid_vspan(int x, int y, unsigned len, 
                               const color_type& c, const int8u* covers)
        {
            do 
            {
                blend_pix(pix_value_ptr(x, y++, 1), c, *covers++);
            }
            while (--len);
        }

         
        void copy_color_hspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            pixel_type* p = pix_value_ptr(x, y, len);

            do 
            {
                p->set(*colors++);
                p = p->next();
            }
            while (--len);
        }

         
        void copy_color_vspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            do 
            {
                pix_value_ptr(x, y++, 1)->set(*colors++);
            }
            while (--len);
        }

         
        void blend_color_hspan(int x, int y, unsigned len, 
                               const color_type* colors, 
                               const int8u* covers,
                               int8u cover)
        {
            pixel_type* p = pix_value_ptr(x, y, len);

            do 
            {
                blend_pix(p, *colors++, covers ? *covers++ : cover);
                p = p->next();
            }
            while (--len);
        }

         
        void blend_color_vspan(int x, int y, unsigned len, 
                               const color_type* colors, 
                               const int8u* covers,
                               int8u cover)
        {
            do 
            {
                blend_pix(pix_value_ptr(x, y++, 1), *colors++, covers ? *covers++ : cover);
            }
            while (--len);

        }

         
        template<class Function> void for_each_pixel(Function f)
        {
            unsigned y;
            for (y = 0; y < height(); ++y)
            {
                row_data r = m_rbuf->row(y);
                if (r.ptr)
                {
                    unsigned len = r.x2 - r.x1 + 1;
                    pixel_type* p = pix_value_ptr(r.x1, y, len);
                    do
                    {
                        f(p->c);
                        p = p->next();
                    }
                    while (--len);
                }
            }
        }

         
        void premultiply()
        {
            for_each_pixel(multiplier_rgba<color_type, order_type>::premultiply);
        }

         
        void demultiply()
        {
            for_each_pixel(multiplier_rgba<color_type, order_type>::demultiply);
        }

         
        template<class GammaLut> void apply_gamma_dir(const GammaLut& g)
        {
            for_each_pixel(apply_gamma_dir_rgba<color_type, order_type, GammaLut>(g));
        }

         
        template<class GammaLut> void apply_gamma_inv(const GammaLut& g)
        {
            for_each_pixel(apply_gamma_inv_rgba<color_type, order_type, GammaLut>(g));
        }

         
        template<class RenBuf2> void copy_from(const RenBuf2& from, 
                                               int xdst, int ydst,
                                               int xsrc, int ysrc,
                                               unsigned len)
        {
            if (const int8u* p = from.row_ptr(ysrc))
            {
                std::memmove(m_rbuf->row_ptr(xdst, ydst, len) + xdst * pix_width, 
                        p + xsrc * pix_width, 
                        len * pix_width);
            }
        }

         
         
        template<class SrcPixelFormatRenderer> 
        void blend_from(const SrcPixelFormatRenderer& from, 
                        int xdst, int ydst,
                        int xsrc, int ysrc,
                        unsigned len,
                        int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);
                int srcinc = 1;
                int dstinc = 1;

                if (xdst > xsrc)
                {
                    psrc = psrc->advance(len - 1);
                    pdst = pdst->advance(len - 1);
                    srcinc = -1;
                    dstinc = -1;
                }

                do 
                {
                    blend_pix(pdst, psrc->get(), cover);
                    psrc = psrc->advance(srcinc);
                    pdst = pdst->advance(dstinc);
                }
                while (--len);
            }
        }

         
         
        template<class SrcPixelFormatRenderer>
        void blend_from_color(const SrcPixelFormatRenderer& from, 
                              const color_type& color,
                              int xdst, int ydst,
                              int xsrc, int ysrc,
                              unsigned len,
                              int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;
            typedef typename SrcPixelFormatRenderer::color_type src_color_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);

                do 
                {
                    blend_pix(pdst, color,
                        src_color_type::scale_cover(cover, psrc->c[0]));
                    psrc = psrc->next();
                    pdst = pdst->next();
                }
                while (--len);
            }
        }

         
         
         
        template<class SrcPixelFormatRenderer>
        void blend_from_lut(const SrcPixelFormatRenderer& from, 
                            const color_type* color_lut,
                            int xdst, int ydst,
                            int xsrc, int ysrc,
                            unsigned len,
                            int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);

                do 
                {
                    blend_pix(pdst, color_lut[psrc->c[0]], cover);
                    psrc = psrc->next();
                    pdst = pdst->next();
                }
                while (--len);
            }
        }

    private:
        rbuf_type* m_rbuf;
        Blender m_blender;
        unsigned m_comp_op;
    };


     
    typedef blender_rgba<rgba8, order_rgba> blender_rgba32;
    typedef blender_rgba<rgba8, order_argb> blender_argb32;
    typedef blender_rgba<rgba8, order_abgr> blender_abgr32;
    typedef blender_rgba<rgba8, order_bgra> blender_bgra32;

    typedef blender_rgba<srgba8, order_rgba> blender_srgba32;
    typedef blender_rgba<srgba8, order_argb> blender_sargb32;
    typedef blender_rgba<srgba8, order_abgr> blender_sabgr32;
    typedef blender_rgba<srgba8, order_bgra> blender_sbgra32;

    typedef blender_rgba_pre<rgba8, order_rgba> blender_rgba32_pre;
    typedef blender_rgba_pre<rgba8, order_argb> blender_argb32_pre;
    typedef blender_rgba_pre<rgba8, order_abgr> blender_abgr32_pre;
    typedef blender_rgba_pre<rgba8, order_bgra> blender_bgra32_pre;

    typedef blender_rgba_pre<srgba8, order_rgba> blender_srgba32_pre;
    typedef blender_rgba_pre<srgba8, order_argb> blender_sargb32_pre;
    typedef blender_rgba_pre<srgba8, order_abgr> blender_sabgr32_pre;
    typedef blender_rgba_pre<srgba8, order_bgra> blender_sbgra32_pre;

    typedef blender_rgba_plain<rgba8, order_rgba> blender_rgba32_plain;
    typedef blender_rgba_plain<rgba8, order_argb> blender_argb32_plain;
    typedef blender_rgba_plain<rgba8, order_abgr> blender_abgr32_plain;
    typedef blender_rgba_plain<rgba8, order_bgra> blender_bgra32_plain;

    typedef blender_rgba_plain<srgba8, order_rgba> blender_srgba32_plain;
    typedef blender_rgba_plain<srgba8, order_argb> blender_sargb32_plain;
    typedef blender_rgba_plain<srgba8, order_abgr> blender_sabgr32_plain;
    typedef blender_rgba_plain<srgba8, order_bgra> blender_sbgra32_plain;

    typedef blender_rgba<rgba16, order_rgba> blender_rgba64;
    typedef blender_rgba<rgba16, order_argb> blender_argb64;
    typedef blender_rgba<rgba16, order_abgr> blender_abgr64;
    typedef blender_rgba<rgba16, order_bgra> blender_bgra64;

    typedef blender_rgba_pre<rgba16, order_rgba> blender_rgba64_pre;
    typedef blender_rgba_pre<rgba16, order_argb> blender_argb64_pre;
    typedef blender_rgba_pre<rgba16, order_abgr> blender_abgr64_pre;
    typedef blender_rgba_pre<rgba16, order_bgra> blender_bgra64_pre;

	typedef blender_rgba_plain<rgba16, order_rgba> blender_rgba64_plain;
	typedef blender_rgba_plain<rgba16, order_argb> blender_argb64_plain;
	typedef blender_rgba_plain<rgba16, order_abgr> blender_abgr64_plain;
	typedef blender_rgba_plain<rgba16, order_bgra> blender_bgra64_plain;

	typedef blender_rgba<rgba32, order_rgba> blender_rgba128;
    typedef blender_rgba<rgba32, order_argb> blender_argb128;
    typedef blender_rgba<rgba32, order_abgr> blender_abgr128;
    typedef blender_rgba<rgba32, order_bgra> blender_bgra128;

    typedef blender_rgba_pre<rgba32, order_rgba> blender_rgba128_pre;
    typedef blender_rgba_pre<rgba32, order_argb> blender_argb128_pre;
    typedef blender_rgba_pre<rgba32, order_abgr> blender_abgr128_pre;
    typedef blender_rgba_pre<rgba32, order_bgra> blender_bgra128_pre;

    typedef blender_rgba_plain<rgba32, order_rgba> blender_rgba128_plain;
    typedef blender_rgba_plain<rgba32, order_argb> blender_argb128_plain;
    typedef blender_rgba_plain<rgba32, order_abgr> blender_abgr128_plain;
    typedef blender_rgba_plain<rgba32, order_bgra> blender_bgra128_plain;


     
    typedef pixfmt_alpha_blend_rgba<blender_rgba32, rendering_buffer> pixfmt_rgba32;
    typedef pixfmt_alpha_blend_rgba<blender_argb32, rendering_buffer> pixfmt_argb32;
    typedef pixfmt_alpha_blend_rgba<blender_abgr32, rendering_buffer> pixfmt_abgr32;
    typedef pixfmt_alpha_blend_rgba<blender_bgra32, rendering_buffer> pixfmt_bgra32;

    typedef pixfmt_alpha_blend_rgba<blender_srgba32, rendering_buffer> pixfmt_srgba32;
    typedef pixfmt_alpha_blend_rgba<blender_sargb32, rendering_buffer> pixfmt_sargb32;
    typedef pixfmt_alpha_blend_rgba<blender_sabgr32, rendering_buffer> pixfmt_sabgr32;
    typedef pixfmt_alpha_blend_rgba<blender_sbgra32, rendering_buffer> pixfmt_sbgra32;

    typedef pixfmt_alpha_blend_rgba<blender_rgba32_pre, rendering_buffer> pixfmt_rgba32_pre;
    typedef pixfmt_alpha_blend_rgba<blender_argb32_pre, rendering_buffer> pixfmt_argb32_pre;
    typedef pixfmt_alpha_blend_rgba<blender_abgr32_pre, rendering_buffer> pixfmt_abgr32_pre;
    typedef pixfmt_alpha_blend_rgba<blender_bgra32_pre, rendering_buffer> pixfmt_bgra32_pre;

    typedef pixfmt_alpha_blend_rgba<blender_srgba32_pre, rendering_buffer> pixfmt_srgba32_pre;
    typedef pixfmt_alpha_blend_rgba<blender_sargb32_pre, rendering_buffer> pixfmt_sargb32_pre;
    typedef pixfmt_alpha_blend_rgba<blender_sabgr32_pre, rendering_buffer> pixfmt_sabgr32_pre;
    typedef pixfmt_alpha_blend_rgba<blender_sbgra32_pre, rendering_buffer> pixfmt_sbgra32_pre;

    typedef pixfmt_alpha_blend_rgba<blender_rgba32_plain, rendering_buffer> pixfmt_rgba32_plain;
    typedef pixfmt_alpha_blend_rgba<blender_argb32_plain, rendering_buffer> pixfmt_argb32_plain;
    typedef pixfmt_alpha_blend_rgba<blender_abgr32_plain, rendering_buffer> pixfmt_abgr32_plain;
    typedef pixfmt_alpha_blend_rgba<blender_bgra32_plain, rendering_buffer> pixfmt_bgra32_plain;

    typedef pixfmt_alpha_blend_rgba<blender_srgba32_plain, rendering_buffer> pixfmt_srgba32_plain;
    typedef pixfmt_alpha_blend_rgba<blender_sargb32_plain, rendering_buffer> pixfmt_sargb32_plain;
    typedef pixfmt_alpha_blend_rgba<blender_sabgr32_plain, rendering_buffer> pixfmt_sabgr32_plain;
    typedef pixfmt_alpha_blend_rgba<blender_sbgra32_plain, rendering_buffer> pixfmt_sbgra32_plain;

    typedef pixfmt_alpha_blend_rgba<blender_rgba64, rendering_buffer> pixfmt_rgba64;
    typedef pixfmt_alpha_blend_rgba<blender_argb64, rendering_buffer> pixfmt_argb64;
    typedef pixfmt_alpha_blend_rgba<blender_abgr64, rendering_buffer> pixfmt_abgr64;
    typedef pixfmt_alpha_blend_rgba<blender_bgra64, rendering_buffer> pixfmt_bgra64;

    typedef pixfmt_alpha_blend_rgba<blender_rgba64_pre, rendering_buffer> pixfmt_rgba64_pre;
    typedef pixfmt_alpha_blend_rgba<blender_argb64_pre, rendering_buffer> pixfmt_argb64_pre;
    typedef pixfmt_alpha_blend_rgba<blender_abgr64_pre, rendering_buffer> pixfmt_abgr64_pre;
    typedef pixfmt_alpha_blend_rgba<blender_bgra64_pre, rendering_buffer> pixfmt_bgra64_pre;

	typedef pixfmt_alpha_blend_rgba<blender_rgba64_plain, rendering_buffer> pixfmt_rgba64_plain;
	typedef pixfmt_alpha_blend_rgba<blender_argb64_plain, rendering_buffer> pixfmt_argb64_plain;
	typedef pixfmt_alpha_blend_rgba<blender_abgr64_plain, rendering_buffer> pixfmt_abgr64_plain;
	typedef pixfmt_alpha_blend_rgba<blender_bgra64_plain, rendering_buffer> pixfmt_bgra64_plain;

	typedef pixfmt_alpha_blend_rgba<blender_rgba128, rendering_buffer> pixfmt_rgba128;
    typedef pixfmt_alpha_blend_rgba<blender_argb128, rendering_buffer> pixfmt_argb128;
    typedef pixfmt_alpha_blend_rgba<blender_abgr128, rendering_buffer> pixfmt_abgr128;
    typedef pixfmt_alpha_blend_rgba<blender_bgra128, rendering_buffer> pixfmt_bgra128;

    typedef pixfmt_alpha_blend_rgba<blender_rgba128_pre, rendering_buffer> pixfmt_rgba128_pre;
    typedef pixfmt_alpha_blend_rgba<blender_argb128_pre, rendering_buffer> pixfmt_argb128_pre;
    typedef pixfmt_alpha_blend_rgba<blender_abgr128_pre, rendering_buffer> pixfmt_abgr128_pre;
    typedef pixfmt_alpha_blend_rgba<blender_bgra128_pre, rendering_buffer> pixfmt_bgra128_pre;

    typedef pixfmt_alpha_blend_rgba<blender_rgba128_plain, rendering_buffer> pixfmt_rgba128_plain;
    typedef pixfmt_alpha_blend_rgba<blender_argb128_plain, rendering_buffer> pixfmt_argb128_plain;
    typedef pixfmt_alpha_blend_rgba<blender_abgr128_plain, rendering_buffer> pixfmt_abgr128_plain;
    typedef pixfmt_alpha_blend_rgba<blender_bgra128_plain, rendering_buffer> pixfmt_bgra128_plain;

}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_PIXFMT_RGB_INCLUDED
#define AGG_PIXFMT_RGB_INCLUDED

#include <cstring>

namespace agg
{

     
    template<class ColorT, class Order, class GammaLut> class apply_gamma_dir_rgb
    {
    public:
        typedef typename ColorT::value_type value_type;

        apply_gamma_dir_rgb(const GammaLut& gamma) : m_gamma(gamma) {}

        AGG_INLINE void operator () (value_type* p)
        {
            p[Order::R] = m_gamma.dir(p[Order::R]);
            p[Order::G] = m_gamma.dir(p[Order::G]);
            p[Order::B] = m_gamma.dir(p[Order::B]);
        }

    private:
        const GammaLut& m_gamma;
    };



     
    template<class ColorT, class Order, class GammaLut> class apply_gamma_inv_rgb
    {
    public:
        typedef typename ColorT::value_type value_type;

        apply_gamma_inv_rgb(const GammaLut& gamma) : m_gamma(gamma) {}

        AGG_INLINE void operator () (value_type* p)
        {
            p[Order::R] = m_gamma.inv(p[Order::R]);
            p[Order::G] = m_gamma.inv(p[Order::G]);
            p[Order::B] = m_gamma.inv(p[Order::B]);
        }

    private:
        const GammaLut& m_gamma;
    };


     
    template<class ColorT, class Order> 
    struct blender_rgb
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

         
         
         

         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha, cover_type cover)
        {
            blend_pix(p, cr, cg, cb, color_type::mult_cover(alpha, cover));
        }
        
         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha)
        {
            p[Order::R] = color_type::lerp(p[Order::R], cr, alpha);
            p[Order::G] = color_type::lerp(p[Order::G], cg, alpha);
            p[Order::B] = color_type::lerp(p[Order::B], cb, alpha);
        }
    };

     
    template<class ColorT, class Order> 
    struct blender_rgb_pre
    {
        typedef ColorT color_type;
        typedef Order order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

         
         

         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha, cover_type cover)
        {
            blend_pix(p, 
                color_type::mult_cover(cr, cover), 
                color_type::mult_cover(cg, cover), 
                color_type::mult_cover(cb, cover), 
                color_type::mult_cover(alpha, cover));
        }

         
        static AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha)
        {
            p[Order::R] = color_type::prelerp(p[Order::R], cr, alpha);
            p[Order::G] = color_type::prelerp(p[Order::G], cg, alpha);
            p[Order::B] = color_type::prelerp(p[Order::B], cb, alpha);
        }
    };

     
    template<class ColorT, class Order, class Gamma> 
    class blender_rgb_gamma : public blender_base<ColorT, Order>
    {
    public:
        typedef ColorT color_type;
        typedef Order order_type;
        typedef Gamma gamma_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        typedef typename color_type::long_type long_type;

         
        blender_rgb_gamma() : m_gamma(0) {}
        void gamma(const gamma_type& g) { m_gamma = &g; }

         
        AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha, cover_type cover)
        {
            blend_pix(p, cr, cg, cb, color_type::mult_cover(alpha, cover));
        }
        
         
        AGG_INLINE void blend_pix(value_type* p, 
            value_type cr, value_type cg, value_type cb, value_type alpha)
        {
            calc_type r = m_gamma->dir(p[Order::R]);
            calc_type g = m_gamma->dir(p[Order::G]);
            calc_type b = m_gamma->dir(p[Order::B]);
            p[Order::R] = m_gamma->inv(color_type::downscale((m_gamma->dir(cr) - r) * alpha) + r);
            p[Order::G] = m_gamma->inv(color_type::downscale((m_gamma->dir(cg) - g) * alpha) + g);
            p[Order::B] = m_gamma->inv(color_type::downscale((m_gamma->dir(cb) - b) * alpha) + b);
        }
        
    private:
        const gamma_type* m_gamma;
    };


     
    template<class Blender, class RenBuf, unsigned Step, unsigned Offset = 0> 
    class pixfmt_alpha_blend_rgb
    {
    public:
        typedef pixfmt_rgb_tag pixfmt_category;
        typedef RenBuf   rbuf_type;
        typedef Blender  blender_type;
        typedef typename rbuf_type::row_data row_data;
        typedef typename blender_type::color_type color_type;
        typedef typename blender_type::order_type order_type;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;
        enum 
        {
            num_components = 3,
            pix_step = Step,
            pix_offset = Offset,
            pix_width = sizeof(value_type) * pix_step
        };
        struct pixel_type
        {
            value_type c[num_components];

            void set(value_type r, value_type g, value_type b)
            {
                c[order_type::R] = r;
                c[order_type::G] = g;
                c[order_type::B] = b;
            }

            void set(const color_type& color)
            {
                set(color.r, color.g, color.b);
            }

            void get(value_type& r, value_type& g, value_type& b) const
            {
                r = c[order_type::R];
                g = c[order_type::G];
                b = c[order_type::B];
            }

            color_type get() const
            {
                return color_type(
                    c[order_type::R], 
                    c[order_type::G], 
                    c[order_type::B]);
            }

            pixel_type* next()
            {
                return (pixel_type*)(c + pix_step);
            }

            const pixel_type* next() const
            {
                return (const pixel_type*)(c + pix_step);
            }

            pixel_type* advance(int n)
            {
                return (pixel_type*)(c + n * pix_step);
            }

            const pixel_type* advance(int n) const
            {
                return (const pixel_type*)(c + n * pix_step);
            }
        };

    private:
         
        AGG_INLINE void blend_pix(pixel_type* p, 
            value_type r, value_type g, value_type b, value_type a, 
            unsigned cover)
        {
            m_blender.blend_pix(p->c, r, g, b, a, cover);
        }

         
        AGG_INLINE void blend_pix(pixel_type* p, 
            value_type r, value_type g, value_type b, value_type a)
        {
            m_blender.blend_pix(p->c, r, g, b, a);
        }

         
        AGG_INLINE void blend_pix(pixel_type* p, const color_type& c, unsigned cover)
        {
            m_blender.blend_pix(p->c, c.r, c.g, c.b, c.a, cover);
        }

         
        AGG_INLINE void blend_pix(pixel_type* p, const color_type& c)
        {
            m_blender.blend_pix(p->c, c.r, c.g, c.b, c.a);
        }

         
        AGG_INLINE void copy_or_blend_pix(pixel_type* p, const color_type& c, unsigned cover)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque() && cover == cover_mask)
                {
                    p->set(c);
                }
                else
                {
                    blend_pix(p, c, cover);
                }
            }
        }

         
        AGG_INLINE void copy_or_blend_pix(pixel_type* p, const color_type& c)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque())
                {
                    p->set(c);
                }
                else
                {
                    blend_pix(p, c);
                }
            }
        }

    public:
         
        explicit pixfmt_alpha_blend_rgb(rbuf_type& rb) :
            m_rbuf(&rb)
        {}
        void attach(rbuf_type& rb) { m_rbuf = &rb; }

         
        template<class PixFmt>
        bool attach(PixFmt& pixf, int x1, int y1, int x2, int y2)
        {
            rect_i r(x1, y1, x2, y2);
            if (r.clip(rect_i(0, 0, pixf.width()-1, pixf.height()-1)))
            {
                int stride = pixf.stride();
                m_rbuf->attach(pixf.pix_ptr(r.x1, stride < 0 ? r.y2 : r.y1), 
                               (r.x2 - r.x1) + 1,
                               (r.y2 - r.y1) + 1,
                               stride);
                return true;
            }
            return false;
        }

         
        Blender& blender() { return m_blender; }

         
        AGG_INLINE unsigned width()  const { return m_rbuf->width();  }
        AGG_INLINE unsigned height() const { return m_rbuf->height(); }
        AGG_INLINE int      stride() const { return m_rbuf->stride(); }

         
        AGG_INLINE       int8u* row_ptr(int y)       { return m_rbuf->row_ptr(y); }
        AGG_INLINE const int8u* row_ptr(int y) const { return m_rbuf->row_ptr(y); }
        AGG_INLINE row_data     row(int y)     const { return m_rbuf->row(y); }

         
        AGG_INLINE int8u* pix_ptr(int x, int y) 
        { 
            return m_rbuf->row_ptr(y) + sizeof(value_type) * (x * pix_step + pix_offset);
        }

        AGG_INLINE const int8u* pix_ptr(int x, int y) const 
        { 
            return m_rbuf->row_ptr(y) + sizeof(value_type) * (x * pix_step + pix_offset);
        }

         
        AGG_INLINE pixel_type* pix_value_ptr(int x, int y, unsigned len) 
        {
            return (pixel_type*)(m_rbuf->row_ptr(x, y, len) + sizeof(value_type) * (x * pix_step + pix_offset));
        }

         
        AGG_INLINE const pixel_type* pix_value_ptr(int x, int y) const 
        {
            int8u* p = m_rbuf->row_ptr(y);
            return p ? (pixel_type*)(p + sizeof(value_type) * (x * pix_step + pix_offset)) : 0;
        }

         
        AGG_INLINE static pixel_type* pix_value_ptr(void* p) 
        {
            return (pixel_type*)((value_type*)p + pix_offset);
        }

         
        AGG_INLINE static const pixel_type* pix_value_ptr(const void* p) 
        {
            return (const pixel_type*)((const value_type*)p + pix_offset);
        }

         
        AGG_INLINE static void write_plain_color(void* p, color_type c)
        {
             
            c.premultiply();
            pix_value_ptr(p)->set(c);
        }

         
        AGG_INLINE static color_type read_plain_color(const void* p)
        {
            return pix_value_ptr(p)->get();
        }

         
        AGG_INLINE static void make_pix(int8u* p, const color_type& c)
        {
            ((pixel_type*)p)->set(c);
        }

         
        AGG_INLINE color_type pixel(int x, int y) const
        {
            if (const pixel_type* p = pix_value_ptr(x, y))
            {
                return p->get();
            }
            return color_type::no_color();
        }

         
        AGG_INLINE void copy_pixel(int x, int y, const color_type& c)
        {
            pix_value_ptr(x, y, 1)->set(c);
        }

         
        AGG_INLINE void blend_pixel(int x, int y, const color_type& c, int8u cover)
        {
            copy_or_blend_pix(pix_value_ptr(x, y, 1), c, cover);
        }

         
        AGG_INLINE void copy_hline(int x, int y, 
                                   unsigned len, 
                                   const color_type& c)
        {
            pixel_type* p = pix_value_ptr(x, y, len);
            do
            {
                p->set(c);
                p = p->next();
            }
            while(--len);
        }


         
        AGG_INLINE void copy_vline(int x, int y,
                                   unsigned len, 
                                   const color_type& c)
        {
            do
            {
                pix_value_ptr(x, y++, 1)->set(c);
            }
            while (--len);
        }

         
        void blend_hline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         int8u cover)
        {
            if (!c.is_transparent())
            {
                pixel_type* p = pix_value_ptr(x, y, len);

                if (c.is_opaque() && cover == cover_mask)
                {
                    do
                    {
                        p->set(c);
                        p = p->next();
                    }
                    while (--len);
                }
                else
                {
                    do
                    {
                        blend_pix(p, c, cover);
                        p = p->next();
                    }
                    while (--len);
                }
            }
        }


         
        void blend_vline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         int8u cover)
        {
            if (!c.is_transparent())
            {
                if (c.is_opaque() && cover == cover_mask)
                {
                    do
                    {
                        pix_value_ptr(x, y++, 1)->set(c);
                    }
                    while (--len);
                }
                else
                {
                    do
                    {
                        blend_pix(pix_value_ptr(x, y++, 1), c, cover);
                    }
                    while (--len);
                }
            }
        }

         
        void blend_solid_hspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const int8u* covers)
        {
            if (!c.is_transparent())
            {
                pixel_type* p = pix_value_ptr(x, y, len);

                do 
                {
                    if (c.is_opaque() && *covers == cover_mask)
                    {
                        p->set(c);
                    }
                    else
                    {
                        blend_pix(p, c, *covers);
                    }
                    p = p->next();
                    ++covers;
                }
                while (--len);
            }
        }


         
        void blend_solid_vspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const int8u* covers)
        {
            if (!c.is_transparent())
            {
                do 
                {
                    pixel_type* p = pix_value_ptr(x, y++, 1);

                    if (c.is_opaque() && *covers == cover_mask)
                    {
                        p->set(c);
                    }
                    else
                    {
                        blend_pix(p, c, *covers);
                    }
                    ++covers;
                }
                while (--len);
            }
        }

         
        void copy_color_hspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            pixel_type* p = pix_value_ptr(x, y, len);

            do 
            {
                p->set(*colors++);
                p = p->next();
            }
            while (--len);
        }


         
        void copy_color_vspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            do 
            {
                pix_value_ptr(x, y++, 1)->set(*colors++);
            }
            while (--len);
        }

         
        void blend_color_hspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const int8u* covers,
                               int8u cover)
        {
            pixel_type* p = pix_value_ptr(x, y, len);

            if (covers)
            {
                do 
                {
                    copy_or_blend_pix(p, *colors++, *covers++);
                    p = p->next();
                }
                while (--len);
            }
            else
            {
                if (cover == cover_mask)
                {
                    do 
                    {
                        copy_or_blend_pix(p, *colors++);
                        p = p->next();
                    }
                    while (--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(p, *colors++, cover);
                        p = p->next();
                    }
                    while (--len);
                }
            }
        }

         
        void blend_color_vspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const int8u* covers,
                               int8u cover)
        {
            if (covers)
            {
                do 
                {
                    copy_or_blend_pix(pix_value_ptr(x, y++, 1), *colors++, *covers++);
                }
                while (--len);
            }
            else
            {
                if (cover == cover_mask)
                {
                    do 
                    {
                        copy_or_blend_pix(pix_value_ptr(x, y++, 1), *colors++);
                    }
                    while (--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(pix_value_ptr(x, y++, 1), *colors++, cover);
                    }
                    while (--len);
                }
            }
        }

         
        template<class Function> void for_each_pixel(Function f)
        {
            for (unsigned y = 0; y < height(); ++y)
            {
                row_data r = m_rbuf->row(y);
                if (r.ptr)
                {
                    unsigned len = r.x2 - r.x1 + 1;
                    pixel_type* p = pix_value_ptr(r.x1, y, len);
                    do
                    {
                        f(p->c);
                        p = p->next();
                    }
                    while (--len);
                }
            }
        }

         
        template<class GammaLut> void apply_gamma_dir(const GammaLut& g)
        {
            for_each_pixel(apply_gamma_dir_rgb<color_type, order_type, GammaLut>(g));
        }

         
        template<class GammaLut> void apply_gamma_inv(const GammaLut& g)
        {
            for_each_pixel(apply_gamma_inv_rgb<color_type, order_type, GammaLut>(g));
        }

         
        template<class RenBuf2>
        void copy_from(const RenBuf2& from, 
                       int xdst, int ydst,
                       int xsrc, int ysrc,
                       unsigned len)
        {
            if (const int8u* p = from.row_ptr(ysrc))
            {
                std::memmove(m_rbuf->row_ptr(xdst, ydst, len) + xdst * pix_width, 
                        p + xsrc * pix_width, 
                        len * pix_width);
            }
        }

         
         
        template<class SrcPixelFormatRenderer>
        void blend_from(const SrcPixelFormatRenderer& from, 
                        int xdst, int ydst,
                        int xsrc, int ysrc,
                        unsigned len,
                        int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;
            typedef typename SrcPixelFormatRenderer::order_type src_order;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);

                if (cover == cover_mask)
                {
                    do 
                    {
                        value_type alpha = psrc->c[src_order::A];
                        if (alpha <= color_type::empty_value())
                        {
                            if (alpha >= color_type::full_value())
                            {
                                pdst->c[order_type::R] = psrc->c[src_order::R];
                                pdst->c[order_type::G] = psrc->c[src_order::G];
                                pdst->c[order_type::B] = psrc->c[src_order::B];
                            }
                            else
                            {
                                blend_pix(pdst, 
                                    psrc->c[src_order::R],
                                    psrc->c[src_order::G],
                                    psrc->c[src_order::B],
                                    alpha);
                            }
                        }
                        psrc = psrc->next();
                        pdst = pdst->next();
                    }
                    while(--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(pdst, psrc->get(), cover);
                        psrc = psrc->next();
                        pdst = pdst->next();
                    }
                    while (--len);
                }
            }
        }

         
         
        template<class SrcPixelFormatRenderer>
        void blend_from_color(const SrcPixelFormatRenderer& from, 
                              const color_type& color,
                              int xdst, int ydst,
                              int xsrc, int ysrc,
                              unsigned len,
                              int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;
            typedef typename SrcPixelFormatRenderer::color_type src_color_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);

                do 
                {
                    copy_or_blend_pix(pdst, color, src_color_type::scale_cover(cover, psrc->c[0]));
                    psrc = psrc->next();
                    pdst = pdst->next();
                }
                while (--len);
            }
        }

         
         
         
        template<class SrcPixelFormatRenderer>
        void blend_from_lut(const SrcPixelFormatRenderer& from, 
                            const color_type* color_lut,
                            int xdst, int ydst,
                            int xsrc, int ysrc,
                            unsigned len,
                            int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::pixel_type src_pixel_type;

            if (const src_pixel_type* psrc = from.pix_value_ptr(xsrc, ysrc))
            {
                pixel_type* pdst = pix_value_ptr(xdst, ydst, len);

                if (cover == cover_mask)
                {
                    do 
                    {
                        const color_type& color = color_lut[psrc->c[0]];
                        blend_pix(pdst, color);
                        psrc = psrc->next();
                        pdst = pdst->next();
                    }
                    while(--len);
                }
                else
                {
                    do 
                    {
                        copy_or_blend_pix(pdst, color_lut[psrc->c[0]], cover);
                        psrc = psrc->next();
                        pdst = pdst->next();
                    }
                    while(--len);
                }
            }
        }

    private:
        rbuf_type* m_rbuf;
        Blender    m_blender;
    };
    
     
    typedef blender_rgb<rgba8, order_rgb> blender_rgb24;
    typedef blender_rgb<rgba8, order_bgr> blender_bgr24;
    typedef blender_rgb<srgba8, order_rgb> blender_srgb24;
    typedef blender_rgb<srgba8, order_bgr> blender_sbgr24;
    typedef blender_rgb<rgba16, order_rgb> blender_rgb48;
    typedef blender_rgb<rgba16, order_bgr> blender_bgr48;
    typedef blender_rgb<rgba32, order_rgb> blender_rgb96;
    typedef blender_rgb<rgba32, order_bgr> blender_bgr96;

    typedef blender_rgb_pre<rgba8, order_rgb> blender_rgb24_pre;
    typedef blender_rgb_pre<rgba8, order_bgr> blender_bgr24_pre;
    typedef blender_rgb_pre<srgba8, order_rgb> blender_srgb24_pre;
    typedef blender_rgb_pre<srgba8, order_bgr> blender_sbgr24_pre;
    typedef blender_rgb_pre<rgba16, order_rgb> blender_rgb48_pre;
    typedef blender_rgb_pre<rgba16, order_bgr> blender_bgr48_pre;
    typedef blender_rgb_pre<rgba32, order_rgb> blender_rgb96_pre;
    typedef blender_rgb_pre<rgba32, order_bgr> blender_bgr96_pre;

    typedef pixfmt_alpha_blend_rgb<blender_rgb24, rendering_buffer, 3> pixfmt_rgb24;
    typedef pixfmt_alpha_blend_rgb<blender_bgr24, rendering_buffer, 3> pixfmt_bgr24;
    typedef pixfmt_alpha_blend_rgb<blender_srgb24, rendering_buffer, 3> pixfmt_srgb24;
    typedef pixfmt_alpha_blend_rgb<blender_sbgr24, rendering_buffer, 3> pixfmt_sbgr24;
    typedef pixfmt_alpha_blend_rgb<blender_rgb48, rendering_buffer, 3> pixfmt_rgb48;
    typedef pixfmt_alpha_blend_rgb<blender_bgr48, rendering_buffer, 3> pixfmt_bgr48;
    typedef pixfmt_alpha_blend_rgb<blender_rgb96, rendering_buffer, 3> pixfmt_rgb96;
    typedef pixfmt_alpha_blend_rgb<blender_bgr96, rendering_buffer, 3> pixfmt_bgr96;

    typedef pixfmt_alpha_blend_rgb<blender_rgb24_pre, rendering_buffer, 3> pixfmt_rgb24_pre;
    typedef pixfmt_alpha_blend_rgb<blender_bgr24_pre, rendering_buffer, 3> pixfmt_bgr24_pre;
    typedef pixfmt_alpha_blend_rgb<blender_srgb24_pre, rendering_buffer, 3> pixfmt_srgb24_pre;
    typedef pixfmt_alpha_blend_rgb<blender_sbgr24_pre, rendering_buffer, 3> pixfmt_sbgr24_pre;
    typedef pixfmt_alpha_blend_rgb<blender_rgb48_pre, rendering_buffer, 3> pixfmt_rgb48_pre;
    typedef pixfmt_alpha_blend_rgb<blender_bgr48_pre, rendering_buffer, 3> pixfmt_bgr48_pre;
    typedef pixfmt_alpha_blend_rgb<blender_rgb96_pre, rendering_buffer, 3> pixfmt_rgb96_pre;
    typedef pixfmt_alpha_blend_rgb<blender_bgr96_pre, rendering_buffer, 3> pixfmt_bgr96_pre;

    typedef pixfmt_alpha_blend_rgb<blender_rgb24, rendering_buffer, 4, 0> pixfmt_rgbx32;
    typedef pixfmt_alpha_blend_rgb<blender_rgb24, rendering_buffer, 4, 1> pixfmt_xrgb32;
    typedef pixfmt_alpha_blend_rgb<blender_bgr24, rendering_buffer, 4, 1> pixfmt_xbgr32;
    typedef pixfmt_alpha_blend_rgb<blender_bgr24, rendering_buffer, 4, 0> pixfmt_bgrx32;
    typedef pixfmt_alpha_blend_rgb<blender_srgb24, rendering_buffer, 4, 0> pixfmt_srgbx32;
    typedef pixfmt_alpha_blend_rgb<blender_srgb24, rendering_buffer, 4, 1> pixfmt_sxrgb32;
    typedef pixfmt_alpha_blend_rgb<blender_sbgr24, rendering_buffer, 4, 1> pixfmt_sxbgr32;
    typedef pixfmt_alpha_blend_rgb<blender_sbgr24, rendering_buffer, 4, 0> pixfmt_sbgrx32;
    typedef pixfmt_alpha_blend_rgb<blender_rgb48, rendering_buffer, 4, 0> pixfmt_rgbx64;
    typedef pixfmt_alpha_blend_rgb<blender_rgb48, rendering_buffer, 4, 1> pixfmt_xrgb64;
    typedef pixfmt_alpha_blend_rgb<blender_bgr48, rendering_buffer, 4, 1> pixfmt_xbgr64;
    typedef pixfmt_alpha_blend_rgb<blender_bgr48, rendering_buffer, 4, 0> pixfmt_bgrx64;
    typedef pixfmt_alpha_blend_rgb<blender_rgb96, rendering_buffer, 4, 0> pixfmt_rgbx128;
    typedef pixfmt_alpha_blend_rgb<blender_rgb96, rendering_buffer, 4, 1> pixfmt_xrgb128;
    typedef pixfmt_alpha_blend_rgb<blender_bgr96, rendering_buffer, 4, 1> pixfmt_xbgr128;
    typedef pixfmt_alpha_blend_rgb<blender_bgr96, rendering_buffer, 4, 0> pixfmt_bgrx128;

    typedef pixfmt_alpha_blend_rgb<blender_rgb24_pre, rendering_buffer, 4, 0> pixfmt_rgbx32_pre;
    typedef pixfmt_alpha_blend_rgb<blender_rgb24_pre, rendering_buffer, 4, 1> pixfmt_xrgb32_pre;
    typedef pixfmt_alpha_blend_rgb<blender_bgr24_pre, rendering_buffer, 4, 1> pixfmt_xbgr32_pre;
    typedef pixfmt_alpha_blend_rgb<blender_bgr24_pre, rendering_buffer, 4, 0> pixfmt_bgrx32_pre;
    typedef pixfmt_alpha_blend_rgb<blender_srgb24_pre, rendering_buffer, 4, 0> pixfmt_srgbx32_pre;
    typedef pixfmt_alpha_blend_rgb<blender_srgb24_pre, rendering_buffer, 4, 1> pixfmt_sxrgb32_pre;
    typedef pixfmt_alpha_blend_rgb<blender_sbgr24_pre, rendering_buffer, 4, 1> pixfmt_sxbgr32_pre;
    typedef pixfmt_alpha_blend_rgb<blender_sbgr24_pre, rendering_buffer, 4, 0> pixfmt_sbgrx32_pre;
    typedef pixfmt_alpha_blend_rgb<blender_rgb48_pre, rendering_buffer, 4, 0> pixfmt_rgbx64_pre;
    typedef pixfmt_alpha_blend_rgb<blender_rgb48_pre, rendering_buffer, 4, 1> pixfmt_xrgb64_pre;
    typedef pixfmt_alpha_blend_rgb<blender_bgr48_pre, rendering_buffer, 4, 1> pixfmt_xbgr64_pre;
    typedef pixfmt_alpha_blend_rgb<blender_bgr48_pre, rendering_buffer, 4, 0> pixfmt_bgrx64_pre;
    typedef pixfmt_alpha_blend_rgb<blender_rgb96_pre, rendering_buffer, 4, 0> pixfmt_rgbx128_pre;
    typedef pixfmt_alpha_blend_rgb<blender_rgb96_pre, rendering_buffer, 4, 1> pixfmt_xrgb128_pre;
    typedef pixfmt_alpha_blend_rgb<blender_bgr96_pre, rendering_buffer, 4, 1> pixfmt_xbgr128_pre;
    typedef pixfmt_alpha_blend_rgb<blender_bgr96_pre, rendering_buffer, 4, 0> pixfmt_bgrx128_pre;
    

     
    template<class Gamma> class pixfmt_rgb24_gamma : 
    public pixfmt_alpha_blend_rgb<blender_rgb_gamma<rgba8, order_rgb, Gamma>, rendering_buffer, 3>
    {
    public:
        pixfmt_rgb24_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb<blender_rgb_gamma<rgba8, order_rgb, Gamma>, rendering_buffer, 3>(rb) 
        {
            this->blender().gamma(g);
        }
    };
        
     
    template<class Gamma> class pixfmt_srgb24_gamma : 
    public pixfmt_alpha_blend_rgb<blender_rgb_gamma<srgba8, order_rgb, Gamma>, rendering_buffer, 3>
    {
    public:
        pixfmt_srgb24_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb<blender_rgb_gamma<srgba8, order_rgb, Gamma>, rendering_buffer, 3>(rb) 
        {
            this->blender().gamma(g);
        }
    };
        
     
    template<class Gamma> class pixfmt_bgr24_gamma : 
    public pixfmt_alpha_blend_rgb<blender_rgb_gamma<rgba8, order_bgr, Gamma>, rendering_buffer, 3>
    {
    public:
        pixfmt_bgr24_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb<blender_rgb_gamma<rgba8, order_bgr, Gamma>, rendering_buffer, 3>(rb) 
        {
            this->blender().gamma(g);
        }
    };

     
    template<class Gamma> class pixfmt_sbgr24_gamma : 
    public pixfmt_alpha_blend_rgb<blender_rgb_gamma<srgba8, order_bgr, Gamma>, rendering_buffer, 3>
    {
    public:
        pixfmt_sbgr24_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb<blender_rgb_gamma<srgba8, order_bgr, Gamma>, rendering_buffer, 3>(rb) 
        {
            this->blender().gamma(g);
        }
    };

     
    template<class Gamma> class pixfmt_rgb48_gamma : 
    public pixfmt_alpha_blend_rgb<blender_rgb_gamma<rgba16, order_rgb, Gamma>, rendering_buffer, 3>
    {
    public:
        pixfmt_rgb48_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb<blender_rgb_gamma<rgba16, order_rgb, Gamma>, rendering_buffer, 3>(rb) 
        {
            this->blender().gamma(g);
        }
    };
        
     
    template<class Gamma> class pixfmt_bgr48_gamma : 
    public pixfmt_alpha_blend_rgb<blender_rgb_gamma<rgba16, order_bgr, Gamma>, rendering_buffer, 3>
    {
    public:
        pixfmt_bgr48_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb<blender_rgb_gamma<rgba16, order_bgr, Gamma>, rendering_buffer, 3>(rb) 
        {
            this->blender().gamma(g);
        }
    };
    
}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_PIXFMT_RGB_PACKED_INCLUDED
#define AGG_PIXFMT_RGB_PACKED_INCLUDED

#include <cstring>

namespace agg
{
     
    struct blender_rgb555
    {
        typedef rgba8 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int16u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned)
        {
            pixel_type rgb = *p;
            calc_type r = (rgb >> 7) & 0xF8;
            calc_type g = (rgb >> 2) & 0xF8;
            calc_type b = (rgb << 3) & 0xF8;
            *p = (pixel_type)
               (((((cr - r) * alpha + (r << 8)) >> 1)  & 0x7C00) |
                ((((cg - g) * alpha + (g << 8)) >> 6)  & 0x03E0) |
                 (((cb - b) * alpha + (b << 8)) >> 11) | 0x8000);
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xF8) << 7) | 
                                ((g & 0xF8) << 2) | 
                                 (b >> 3) | 0x8000);
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 7) & 0xF8, 
                              (p >> 2) & 0xF8, 
                              (p << 3) & 0xF8);
        }
    };


     
    struct blender_rgb555_pre
    {
        typedef rgba8 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int16u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned cover)
        {
            alpha = color_type::base_mask - alpha;
            pixel_type rgb = *p;
            calc_type r = (rgb >> 7) & 0xF8;
            calc_type g = (rgb >> 2) & 0xF8;
            calc_type b = (rgb << 3) & 0xF8;
            *p = (pixel_type)
               ((((r * alpha + cr * cover) >> 1)  & 0x7C00) |
                (((g * alpha + cg * cover) >> 6)  & 0x03E0) |
                 ((b * alpha + cb * cover) >> 11) | 0x8000);
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xF8) << 7) | 
                                ((g & 0xF8) << 2) | 
                                 (b >> 3) | 0x8000);
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 7) & 0xF8, 
                              (p >> 2) & 0xF8, 
                              (p << 3) & 0xF8);
        }
    };




     
    template<class Gamma> class blender_rgb555_gamma
    {
    public:
        typedef rgba8 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int16u pixel_type;
        typedef Gamma gamma_type;

        blender_rgb555_gamma() : m_gamma(0) {}
        void gamma(const gamma_type& g) { m_gamma = &g; }

        AGG_INLINE void blend_pix(pixel_type* p, 
                                  unsigned cr, unsigned cg, unsigned cb,
                                  unsigned alpha, 
                                  unsigned)
        {
            pixel_type rgb = *p;
            calc_type r = m_gamma->dir((rgb >> 7) & 0xF8);
            calc_type g = m_gamma->dir((rgb >> 2) & 0xF8);
            calc_type b = m_gamma->dir((rgb << 3) & 0xF8);
            *p = (pixel_type)
               (((m_gamma->inv(((m_gamma->dir(cr) - r) * alpha + (r << 8)) >> 8) << 7) & 0x7C00) |
                ((m_gamma->inv(((m_gamma->dir(cg) - g) * alpha + (g << 8)) >> 8) << 2) & 0x03E0) |
                 (m_gamma->inv(((m_gamma->dir(cb) - b) * alpha + (b << 8)) >> 8) >> 3) | 0x8000);
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xF8) << 7) | 
                                ((g & 0xF8) << 2) | 
                                 (b >> 3) | 0x8000);
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 7) & 0xF8, 
                              (p >> 2) & 0xF8, 
                              (p << 3) & 0xF8);
        }

    private:
        const Gamma* m_gamma;
    };





     
    struct blender_rgb565
    {
        typedef rgba8 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int16u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned)
        {
            pixel_type rgb = *p;
            calc_type r = (rgb >> 8) & 0xF8;
            calc_type g = (rgb >> 3) & 0xFC;
            calc_type b = (rgb << 3) & 0xF8;
            *p = (pixel_type)
               (((((cr - r) * alpha + (r << 8))     ) & 0xF800) |
                ((((cg - g) * alpha + (g << 8)) >> 5) & 0x07E0) |
                 (((cb - b) * alpha + (b << 8)) >> 11));
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3));
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 8) & 0xF8, 
                              (p >> 3) & 0xFC, 
                              (p << 3) & 0xF8);
        }
    };



     
    struct blender_rgb565_pre
    {
        typedef rgba8 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int16u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned cover)
        {
            alpha = color_type::base_mask - alpha;
            pixel_type rgb = *p;
            calc_type r = (rgb >> 8) & 0xF8;
            calc_type g = (rgb >> 3) & 0xFC;
            calc_type b = (rgb << 3) & 0xF8;
            *p = (pixel_type)
               ((((r * alpha + cr * cover)      ) & 0xF800) |
                (((g * alpha + cg * cover) >> 5 ) & 0x07E0) |
                 ((b * alpha + cb * cover) >> 11));
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3));
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 8) & 0xF8, 
                              (p >> 3) & 0xFC, 
                              (p << 3) & 0xF8);
        }
    };



     
    template<class Gamma> class blender_rgb565_gamma
    {
    public:
        typedef rgba8 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int16u pixel_type;
        typedef Gamma gamma_type;

        blender_rgb565_gamma() : m_gamma(0) {}
        void gamma(const gamma_type& g) { m_gamma = &g; }

        AGG_INLINE void blend_pix(pixel_type* p, 
                                  unsigned cr, unsigned cg, unsigned cb,
                                  unsigned alpha, 
                                  unsigned)
        {
            pixel_type rgb = *p;
            calc_type r = m_gamma->dir((rgb >> 8) & 0xF8);
            calc_type g = m_gamma->dir((rgb >> 3) & 0xFC);
            calc_type b = m_gamma->dir((rgb << 3) & 0xF8);
            *p = (pixel_type)
               (((m_gamma->inv(((m_gamma->dir(cr) - r) * alpha + (r << 8)) >> 8) << 8) & 0xF800) |
                ((m_gamma->inv(((m_gamma->dir(cg) - g) * alpha + (g << 8)) >> 8) << 3) & 0x07E0) |
                 (m_gamma->inv(((m_gamma->dir(cb) - b) * alpha + (b << 8)) >> 8) >> 3));
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3));
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 8) & 0xF8, 
                              (p >> 3) & 0xFC, 
                              (p << 3) & 0xF8);
        }

    private:
        const Gamma* m_gamma;
    };



     
    struct blender_rgbAAA
    {
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned)
        {
            pixel_type rgb = *p;
            calc_type r = (rgb >> 14) & 0xFFC0;
            calc_type g = (rgb >> 4)  & 0xFFC0;
            calc_type b = (rgb << 6)  & 0xFFC0;
            *p = (pixel_type)
               (((((cr - r) * alpha + (r << 16)) >> 2)  & 0x3FF00000) |
                ((((cg - g) * alpha + (g << 16)) >> 12) & 0x000FFC00) |
                 (((cb - b) * alpha + (b << 16)) >> 22) | 0xC0000000);
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xFFC0) << 14) | 
                                ((g & 0xFFC0) << 4) | 
                                 (b >> 6) | 0xC0000000);
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 14) & 0xFFC0, 
                              (p >> 4)  & 0xFFC0, 
                              (p << 6)  & 0xFFC0);
        }
    };



     
    struct blender_rgbAAA_pre
    {
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned cover)
        {
            alpha = color_type::base_mask - alpha;
            cover = (cover + 1) << (color_type::base_shift - 8);
            pixel_type rgb = *p;
            calc_type r = (rgb >> 14) & 0xFFC0;
            calc_type g = (rgb >> 4)  & 0xFFC0;
            calc_type b = (rgb << 6)  & 0xFFC0;
            *p = (pixel_type)
               ((((r * alpha + cr * cover) >> 2)  & 0x3FF00000) |
                (((g * alpha + cg * cover) >> 12) & 0x000FFC00) |
                 ((b * alpha + cb * cover) >> 22) | 0xC0000000);
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xFFC0) << 14) | 
                                ((g & 0xFFC0) << 4) | 
                                 (b >> 6) | 0xC0000000);
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 14) & 0xFFC0, 
                              (p >> 4)  & 0xFFC0, 
                              (p << 6)  & 0xFFC0);
        }
    };



     
    template<class Gamma> class blender_rgbAAA_gamma
    {
    public:
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;
        typedef Gamma gamma_type;

        blender_rgbAAA_gamma() : m_gamma(0) {}
        void gamma(const gamma_type& g) { m_gamma = &g; }

        AGG_INLINE void blend_pix(pixel_type* p, 
                                  unsigned cr, unsigned cg, unsigned cb,
                                  unsigned alpha, 
                                  unsigned)
        {
            pixel_type rgb = *p;
            calc_type r = m_gamma->dir((rgb >> 14) & 0xFFC0);
            calc_type g = m_gamma->dir((rgb >> 4)  & 0xFFC0);
            calc_type b = m_gamma->dir((rgb << 6)  & 0xFFC0);
            *p = (pixel_type)
               (((m_gamma->inv(((m_gamma->dir(cr) - r) * alpha + (r << 16)) >> 16) << 14) & 0x3FF00000) |
                ((m_gamma->inv(((m_gamma->dir(cg) - g) * alpha + (g << 16)) >> 16) << 4 ) & 0x000FFC00) |
                 (m_gamma->inv(((m_gamma->dir(cb) - b) * alpha + (b << 16)) >> 16) >> 6 ) | 0xC0000000);
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xFFC0) << 14) | 
                                ((g & 0xFFC0) << 4) | 
                                 (b >> 6) | 0xC0000000);
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 14) & 0xFFC0, 
                              (p >> 4)  & 0xFFC0, 
                              (p << 6)  & 0xFFC0);
        }
    private:
        const Gamma* m_gamma;
    };


     
    struct blender_bgrAAA
    {
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned)
        {
            pixel_type bgr = *p;
            calc_type b = (bgr >> 14) & 0xFFC0;
            calc_type g = (bgr >> 4)  & 0xFFC0;
            calc_type r = (bgr << 6)  & 0xFFC0;
            *p = (pixel_type)
               (((((cb - b) * alpha + (b << 16)) >> 2)  & 0x3FF00000) |
                ((((cg - g) * alpha + (g << 16)) >> 12) & 0x000FFC00) |
                 (((cr - r) * alpha + (r << 16)) >> 22) | 0xC0000000);
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((b & 0xFFC0) << 14) | 
                                ((g & 0xFFC0) << 4) | 
                                 (r >> 6) | 0xC0000000);
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p << 6)  & 0xFFC0, 
                              (p >> 4)  & 0xFFC0, 
                              (p >> 14) & 0xFFC0);
        }
    };



     
    struct blender_bgrAAA_pre
    {
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned cover)
        {
            alpha = color_type::base_mask - alpha;
            cover = (cover + 1) << (color_type::base_shift - 8);
            pixel_type bgr = *p;
            calc_type b = (bgr >> 14) & 0xFFC0;
            calc_type g = (bgr >> 4)  & 0xFFC0;
            calc_type r = (bgr << 6)  & 0xFFC0;
            *p = (pixel_type)
               ((((b * alpha + cb * cover) >> 2)  & 0x3FF00000) |
                (((g * alpha + cg * cover) >> 12) & 0x000FFC00) |
                 ((r * alpha + cr * cover) >> 22) | 0xC0000000);
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((b & 0xFFC0) << 14) | 
                                ((g & 0xFFC0) << 4) | 
                                 (r >> 6) | 0xC0000000);
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p << 6)  & 0xFFC0, 
                              (p >> 4)  & 0xFFC0, 
                              (p >> 14) & 0xFFC0);
        }
    };



     
    template<class Gamma> class blender_bgrAAA_gamma
    {
    public:
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;
        typedef Gamma gamma_type;

        blender_bgrAAA_gamma() : m_gamma(0) {}
        void gamma(const gamma_type& g) { m_gamma = &g; }

        AGG_INLINE void blend_pix(pixel_type* p, 
                                  unsigned cr, unsigned cg, unsigned cb,
                                  unsigned alpha, 
                                  unsigned)
        {
            pixel_type bgr = *p;
            calc_type b = m_gamma->dir((bgr >> 14) & 0xFFC0);
            calc_type g = m_gamma->dir((bgr >> 4)  & 0xFFC0);
            calc_type r = m_gamma->dir((bgr << 6)  & 0xFFC0);
            *p = (pixel_type)
               (((m_gamma->inv(((m_gamma->dir(cb) - b) * alpha + (b << 16)) >> 16) << 14) & 0x3FF00000) |
                ((m_gamma->inv(((m_gamma->dir(cg) - g) * alpha + (g << 16)) >> 16) << 4 ) & 0x000FFC00) |
                 (m_gamma->inv(((m_gamma->dir(cr) - r) * alpha + (r << 16)) >> 16) >> 6 ) | 0xC0000000);
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((b & 0xFFC0) << 14) | 
                                ((g & 0xFFC0) << 4) | 
                                 (r >> 6) | 0xC0000000);
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p << 6)  & 0xFFC0, 
                              (p >> 4)  & 0xFFC0, 
                              (p >> 14) & 0xFFC0);
        }

    private:
        const Gamma* m_gamma;
    };



     
    struct blender_rgbBBA
    {
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned)
        {
            pixel_type rgb = *p;
            calc_type r = (rgb >> 16) & 0xFFE0;
            calc_type g = (rgb >> 5)  & 0xFFE0;
            calc_type b = (rgb << 6)  & 0xFFC0;
            *p = (pixel_type)
               (((((cr - r) * alpha + (r << 16))      ) & 0xFFE00000) |
                ((((cg - g) * alpha + (g << 16)) >> 11) & 0x001FFC00) |
                 (((cb - b) * alpha + (b << 16)) >> 22));
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xFFE0) << 16) | ((g & 0xFFE0) << 5) | (b >> 6));
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 16) & 0xFFE0, 
                              (p >> 5)  & 0xFFE0, 
                              (p << 6)  & 0xFFC0);
        }
    };


     
    struct blender_rgbBBA_pre
    {
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned cover)
        {
            alpha = color_type::base_mask - alpha;
            cover = (cover + 1) << (color_type::base_shift - 8);
            pixel_type rgb = *p;
            calc_type r = (rgb >> 16) & 0xFFE0;
            calc_type g = (rgb >> 5)  & 0xFFE0;
            calc_type b = (rgb << 6)  & 0xFFC0;
            *p = (pixel_type)
               ((((r * alpha + cr * cover)      ) & 0xFFE00000) |
                (((g * alpha + cg * cover) >> 11) & 0x001FFC00) |
                 ((b * alpha + cb * cover) >> 22));
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xFFE0) << 16) | ((g & 0xFFE0) << 5) | (b >> 6));
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 16) & 0xFFE0, 
                              (p >> 5)  & 0xFFE0, 
                              (p << 6)  & 0xFFC0);
        }
    };



     
    template<class Gamma> class blender_rgbBBA_gamma
    {
    public:
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;
        typedef Gamma gamma_type;

        blender_rgbBBA_gamma() : m_gamma(0) {}
        void gamma(const gamma_type& g) { m_gamma = &g; }

        AGG_INLINE void blend_pix(pixel_type* p, 
                                  unsigned cr, unsigned cg, unsigned cb,
                                  unsigned alpha, 
                                  unsigned)
        {
            pixel_type rgb = *p;
            calc_type r = m_gamma->dir((rgb >> 16) & 0xFFE0);
            calc_type g = m_gamma->dir((rgb >> 5)  & 0xFFE0);
            calc_type b = m_gamma->dir((rgb << 6)  & 0xFFC0);
            *p = (pixel_type)
               (((m_gamma->inv(((m_gamma->dir(cr) - r) * alpha + (r << 16)) >> 16) << 16) & 0xFFE00000) |
                ((m_gamma->inv(((m_gamma->dir(cg) - g) * alpha + (g << 16)) >> 16) << 5 ) & 0x001FFC00) |
                 (m_gamma->inv(((m_gamma->dir(cb) - b) * alpha + (b << 16)) >> 16) >> 6 ));
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((r & 0xFFE0) << 16) | ((g & 0xFFE0) << 5) | (b >> 6));
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p >> 16) & 0xFFE0, 
                              (p >> 5)  & 0xFFE0, 
                              (p << 6)  & 0xFFC0);
        }

    private:
        const Gamma* m_gamma;
    };


     
    struct blender_bgrABB
    {
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned)
        {
            pixel_type bgr = *p;
            calc_type b = (bgr >> 16) & 0xFFC0;
            calc_type g = (bgr >> 6)  & 0xFFE0;
            calc_type r = (bgr << 5)  & 0xFFE0;
            *p = (pixel_type)
               (((((cb - b) * alpha + (b << 16))      ) & 0xFFC00000) |
                ((((cg - g) * alpha + (g << 16)) >> 10) & 0x003FF800) |
                 (((cr - r) * alpha + (r << 16)) >> 21));
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((b & 0xFFC0) << 16) | ((g & 0xFFE0) << 6) | (r >> 5));
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p << 5)  & 0xFFE0,
                              (p >> 6)  & 0xFFE0, 
                              (p >> 16) & 0xFFC0);
        }
    };


     
    struct blender_bgrABB_pre
    {
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;

        static AGG_INLINE void blend_pix(pixel_type* p, 
                                         unsigned cr, unsigned cg, unsigned cb,
                                         unsigned alpha, 
                                         unsigned cover)
        {
            alpha = color_type::base_mask - alpha;
            cover = (cover + 1) << (color_type::base_shift - 8);
            pixel_type bgr = *p;
            calc_type b = (bgr >> 16) & 0xFFC0;
            calc_type g = (bgr >> 6)  & 0xFFE0;
            calc_type r = (bgr << 5)  & 0xFFE0;
            *p = (pixel_type)
               ((((b * alpha + cb * cover)      ) & 0xFFC00000) |
                (((g * alpha + cg * cover) >> 10) & 0x003FF800) |
                 ((r * alpha + cr * cover) >> 21));
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((b & 0xFFC0) << 16) | ((g & 0xFFE0) << 6) | (r >> 5));
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p << 5)  & 0xFFE0,
                              (p >> 6)  & 0xFFE0, 
                              (p >> 16) & 0xFFC0);
        }
    };



     
    template<class Gamma> class blender_bgrABB_gamma
    {
    public:
        typedef rgba16 color_type;
        typedef color_type::value_type value_type;
        typedef color_type::calc_type calc_type;
        typedef int32u pixel_type;
        typedef Gamma gamma_type;

        blender_bgrABB_gamma() : m_gamma(0) {}
        void gamma(const gamma_type& g) { m_gamma = &g; }

        AGG_INLINE void blend_pix(pixel_type* p, 
                                  unsigned cr, unsigned cg, unsigned cb,
                                  unsigned alpha, 
                                  unsigned)
        {
            pixel_type bgr = *p;
            calc_type b = m_gamma->dir((bgr >> 16) & 0xFFC0);
            calc_type g = m_gamma->dir((bgr >> 6)  & 0xFFE0);
            calc_type r = m_gamma->dir((bgr << 5)  & 0xFFE0);
            *p = (pixel_type)
               (((m_gamma->inv(((m_gamma->dir(cb) - b) * alpha + (b << 16)) >> 16) << 16) & 0xFFC00000) |
                ((m_gamma->inv(((m_gamma->dir(cg) - g) * alpha + (g << 16)) >> 16) << 6 ) & 0x003FF800) |
                 (m_gamma->inv(((m_gamma->dir(cr) - r) * alpha + (r << 16)) >> 16) >> 5 ));
        }

        static AGG_INLINE pixel_type make_pix(unsigned r, unsigned g, unsigned b)
        {
            return (pixel_type)(((b & 0xFFC0) << 16) | ((g & 0xFFE0) << 6) | (r >> 5));
        }

        static AGG_INLINE color_type make_color(pixel_type p)
        {
            return color_type((p << 5)  & 0xFFE0,
                              (p >> 6)  & 0xFFE0, 
                              (p >> 16) & 0xFFC0);
        }

    private:
        const Gamma* m_gamma;
    };


    
     
    template<class Blender,  class RenBuf> class pixfmt_alpha_blend_rgb_packed
    {
    public:
        typedef RenBuf   rbuf_type;
        typedef typename rbuf_type::row_data row_data;
        typedef Blender  blender_type;
        typedef typename blender_type::color_type color_type;
        typedef typename blender_type::pixel_type pixel_type;
        typedef int                               order_type;  
        typedef typename color_type::value_type   value_type;
        typedef typename color_type::calc_type    calc_type;
        enum base_scale_e 
        {
            base_shift = color_type::base_shift,
            base_scale = color_type::base_scale,
            base_mask  = color_type::base_mask,
            pix_width  = sizeof(pixel_type),
        };

    private:
         
        AGG_INLINE void copy_or_blend_pix(pixel_type* p, const color_type& c, unsigned cover)
        {
            if (c.a)
            {
                calc_type alpha = (calc_type(c.a) * (cover + 1)) >> 8;
                if(alpha == base_mask)
                {
                    *p = m_blender.make_pix(c.r, c.g, c.b);
                }
                else
                {
                    m_blender.blend_pix(p, c.r, c.g, c.b, alpha, cover);
                }
            }
        }

    public:
         
        explicit pixfmt_alpha_blend_rgb_packed(rbuf_type& rb) : m_rbuf(&rb) {}
        void attach(rbuf_type& rb) { m_rbuf = &rb; }

         
        template<class PixFmt>
        bool attach(PixFmt& pixf, int x1, int y1, int x2, int y2)
        {
            rect_i r(x1, y1, x2, y2);
            if(r.clip(rect_i(0, 0, pixf.width()-1, pixf.height()-1)))
            {
                int stride = pixf.stride();
                m_rbuf->attach(pixf.pix_ptr(r.x1, stride < 0 ? r.y2 : r.y1), 
                               (r.x2 - r.x1) + 1,
                               (r.y2 - r.y1) + 1,
                               stride);
                return true;
            }
            return false;
        }

        Blender& blender() { return m_blender; }

         
        AGG_INLINE unsigned width()  const { return m_rbuf->width();  }
        AGG_INLINE unsigned height() const { return m_rbuf->height(); }
        AGG_INLINE int      stride() const { return m_rbuf->stride(); }

         
        AGG_INLINE       int8u* row_ptr(int y)       { return m_rbuf->row_ptr(y); }
        AGG_INLINE const int8u* row_ptr(int y) const { return m_rbuf->row_ptr(y); }
        AGG_INLINE row_data     row(int y)     const { return m_rbuf->row(y); }

         
        AGG_INLINE int8u* pix_ptr(int x, int y)
        {
            return m_rbuf->row_ptr(y) + x * pix_width;
        }

        AGG_INLINE const int8u* pix_ptr(int x, int y) const
        {
            return m_rbuf->row_ptr(y) + x * pix_width;
        }

         
        AGG_INLINE void make_pix(int8u* p, const color_type& c)
        {
            *(pixel_type*)p = m_blender.make_pix(c.r, c.g, c.b);
        }

         
        AGG_INLINE color_type pixel(int x, int y) const
        {
            return m_blender.make_color(((pixel_type*)m_rbuf->row_ptr(y))[x]);
        }

         
        AGG_INLINE void copy_pixel(int x, int y, const color_type& c)
        {
            ((pixel_type*)
                m_rbuf->row_ptr(x, y, 1))[x] = 
                    m_blender.make_pix(c.r, c.g, c.b);
        }

         
        AGG_INLINE void blend_pixel(int x, int y, const color_type& c, int8u cover)
        {
            copy_or_blend_pix((pixel_type*)m_rbuf->row_ptr(x, y, 1) + x, c, cover);
        }

         
        AGG_INLINE void copy_hline(int x, int y, 
                                   unsigned len, 
                                   const color_type& c)
        {
            pixel_type* p = (pixel_type*)m_rbuf->row_ptr(x, y, len) + x;
            pixel_type v = m_blender.make_pix(c.r, c.g, c.b);
            do
            {
                *p++ = v;
            }
            while(--len);
        }

         
        AGG_INLINE void copy_vline(int x, int y,
                                   unsigned len, 
                                   const color_type& c)
        {
            pixel_type v = m_blender.make_pix(c.r, c.g, c.b);
            do
            {
                pixel_type* p = (pixel_type*)m_rbuf->row_ptr(x, y++, 1) + x;
                *p = v;
            }
            while(--len);
        }

         
        void blend_hline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         int8u cover)
        {
            if (c.a)
            {
                pixel_type* p = (pixel_type*)m_rbuf->row_ptr(x, y, len) + x;
                calc_type alpha = (calc_type(c.a) * (cover + 1)) >> 8;
                if(alpha == base_mask)
                {
                    pixel_type v = m_blender.make_pix(c.r, c.g, c.b);
                    do
                    {
                        *p++ = v;
                    }
                    while(--len);
                }
                else
                {
                    do
                    {
                        m_blender.blend_pix(p, c.r, c.g, c.b, alpha, cover);
                        ++p;
                    }
                    while(--len);
                }
            }
        }

         
        void blend_vline(int x, int y,
                         unsigned len, 
                         const color_type& c,
                         int8u cover)
        {
            if (c.a)
            {
                calc_type alpha = (calc_type(c.a) * (cover + 1)) >> 8;
                if(alpha == base_mask)
                {
                    pixel_type v = m_blender.make_pix(c.r, c.g, c.b);
                    do
                    {
                        ((pixel_type*)m_rbuf->row_ptr(x, y++, 1))[x] = v;
                    }
                    while(--len);
                }
                else
                {
                    do
                    {
                        m_blender.blend_pix(
                            (pixel_type*)m_rbuf->row_ptr(x, y++, 1), 
                            c.r, c.g, c.b, alpha, cover);
                    }
                    while(--len);
                }
            }
        }

         
        void blend_solid_hspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const int8u* covers)
        {
            pixel_type* p = (pixel_type*)m_rbuf->row_ptr(x, y, len) + x;
            do 
            {
                copy_or_blend_pix(p, c, *covers++);
                ++p;
            }
            while(--len);
        }

         
        void blend_solid_vspan(int x, int y,
                               unsigned len, 
                               const color_type& c,
                               const int8u* covers)
        {
            do 
            {
                copy_or_blend_pix((pixel_type*)m_rbuf->row_ptr(x, y++, 1) + x, 
                                  c, *covers++);
            }
            while(--len);
        }

         
        void copy_color_hspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            pixel_type* p = (pixel_type*)m_rbuf->row_ptr(x, y, len) + x;
            do 
            {
                *p++ = m_blender.make_pix(colors->r, colors->g, colors->b);
                ++colors;
            }
            while(--len);
        }

         
        void copy_color_vspan(int x, int y,
                              unsigned len, 
                              const color_type* colors)
        {
            do 
            {
                pixel_type* p = (pixel_type*)m_rbuf->row_ptr(x, y++, 1) + x;
                *p = m_blender.make_pix(colors->r, colors->g, colors->b);
                ++colors;
            }
            while(--len);
        }

         
        void blend_color_hspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const int8u* covers,
                               int8u cover)
        {
            pixel_type* p = (pixel_type*)m_rbuf->row_ptr(x, y, len) + x;
            do 
            {
                copy_or_blend_pix(p++, *colors++, covers ? *covers++ : cover);
            }
            while(--len);
        }

         
        void blend_color_vspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const int8u* covers,
                               int8u cover)
        {
            do 
            {
                copy_or_blend_pix((pixel_type*)m_rbuf->row_ptr(x, y++, 1) + x, 
                                  *colors++, covers ? *covers++ : cover);
            }
            while(--len);
        }
        
         
        template<class RenBuf2>
        void copy_from(const RenBuf2& from, 
                       int xdst, int ydst,
                       int xsrc, int ysrc,
                       unsigned len)
        {
            const int8u* p = from.row_ptr(ysrc);
            if(p)
            {
                std::memmove(m_rbuf->row_ptr(xdst, ydst, len) + xdst * pix_width, 
                        p + xsrc * pix_width, 
                        len * pix_width);
            }
        }

         
        template<class SrcPixelFormatRenderer>
        void blend_from(const SrcPixelFormatRenderer& from, 
                        int xdst, int ydst,
                        int xsrc, int ysrc,
                        unsigned len,
                        int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::order_type src_order;

            const value_type* psrc = (const value_type*)from.row_ptr(ysrc);
            if(psrc)
            {
                psrc += xsrc * 4;
                pixel_type* pdst = 
                    (pixel_type*)m_rbuf->row_ptr(xdst, ydst, len) + xdst;
                do 
                {
                    value_type alpha = psrc[src_order::A];
                    if(alpha)
                    {
                        if(alpha == base_mask && cover == 255)
                        {
                            *pdst = m_blender.make_pix(psrc[src_order::R], 
                                                       psrc[src_order::G],
                                                       psrc[src_order::B]);
                        }
                        else
                        {
                            m_blender.blend_pix(pdst, 
                                                psrc[src_order::R],
                                                psrc[src_order::G],
                                                psrc[src_order::B],
                                                alpha,
                                                cover);
                        }
                    }
                    psrc += 4;
                    ++pdst;
                }
                while(--len);
            }
        }

         
        template<class SrcPixelFormatRenderer>
        void blend_from_color(const SrcPixelFormatRenderer& from, 
                              const color_type& color,
                              int xdst, int ydst,
                              int xsrc, int ysrc,
                              unsigned len,
                              int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::value_type src_value_type;
            typedef typename SrcPixelFormatRenderer::color_type src_color_type;
            const src_value_type* psrc = (src_value_type*)from.row_ptr(ysrc);
            if(psrc)
            {
                psrc += xsrc * SrcPixelFormatRenderer::pix_step + SrcPixelFormatRenderer::pix_offset;
                pixel_type* pdst = 
                    (pixel_type*)m_rbuf->row_ptr(xdst, ydst, len) + xdst;

                do 
                {
                    m_blender.blend_pix(pdst, 
                                        color.r, color.g, color.b, color.a,
                                        cover);
                    psrc += SrcPixelFormatRenderer::pix_step;
                    ++pdst;
                }
                while(--len);
            }
        }

         
        template<class SrcPixelFormatRenderer>
        void blend_from_lut(const SrcPixelFormatRenderer& from, 
                            const color_type* color_lut,
                            int xdst, int ydst,
                            int xsrc, int ysrc,
                            unsigned len,
                            int8u cover)
        {
            typedef typename SrcPixelFormatRenderer::value_type src_value_type;
            const src_value_type* psrc = (src_value_type*)from.row_ptr(ysrc);
            if(psrc)
            {
                psrc += xsrc * SrcPixelFormatRenderer::pix_step + SrcPixelFormatRenderer::pix_offset;
                pixel_type* pdst = 
                    (pixel_type*)m_rbuf->row_ptr(xdst, ydst, len) + xdst;

                do 
                {
                    const color_type& color = color_lut[*psrc];
                    m_blender.blend_pix(pdst, 
                                        color.r, color.g, color.b, color.a,
                                        cover);
                    psrc += SrcPixelFormatRenderer::pix_step;
                    ++pdst;
                }
                while(--len);
            }
        }



    private:
        rbuf_type* m_rbuf;
        Blender    m_blender;
    };

    typedef pixfmt_alpha_blend_rgb_packed<blender_rgb555, rendering_buffer> pixfmt_rgb555;  
    typedef pixfmt_alpha_blend_rgb_packed<blender_rgb565, rendering_buffer> pixfmt_rgb565;  

    typedef pixfmt_alpha_blend_rgb_packed<blender_rgb555_pre, rendering_buffer> pixfmt_rgb555_pre;  
    typedef pixfmt_alpha_blend_rgb_packed<blender_rgb565_pre, rendering_buffer> pixfmt_rgb565_pre;  

    typedef pixfmt_alpha_blend_rgb_packed<blender_rgbAAA, rendering_buffer> pixfmt_rgbAAA;  
    typedef pixfmt_alpha_blend_rgb_packed<blender_bgrAAA, rendering_buffer> pixfmt_bgrAAA;  
    typedef pixfmt_alpha_blend_rgb_packed<blender_rgbBBA, rendering_buffer> pixfmt_rgbBBA;  
    typedef pixfmt_alpha_blend_rgb_packed<blender_bgrABB, rendering_buffer> pixfmt_bgrABB;  

    typedef pixfmt_alpha_blend_rgb_packed<blender_rgbAAA_pre, rendering_buffer> pixfmt_rgbAAA_pre;  
    typedef pixfmt_alpha_blend_rgb_packed<blender_bgrAAA_pre, rendering_buffer> pixfmt_bgrAAA_pre;  
    typedef pixfmt_alpha_blend_rgb_packed<blender_rgbBBA_pre, rendering_buffer> pixfmt_rgbBBA_pre;  
    typedef pixfmt_alpha_blend_rgb_packed<blender_bgrABB_pre, rendering_buffer> pixfmt_bgrABB_pre;  


     
    template<class Gamma> class pixfmt_rgb555_gamma : 
    public pixfmt_alpha_blend_rgb_packed<blender_rgb555_gamma<Gamma>, 
                                         rendering_buffer>
    {
    public:
        pixfmt_rgb555_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb_packed<blender_rgb555_gamma<Gamma>, 
                                          rendering_buffer>(rb) 
        {
            this->blender().gamma(g);
        }
    };


     
    template<class Gamma> class pixfmt_rgb565_gamma : 
    public pixfmt_alpha_blend_rgb_packed<blender_rgb565_gamma<Gamma>, rendering_buffer>
    {
    public:
        pixfmt_rgb565_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb_packed<blender_rgb565_gamma<Gamma>, rendering_buffer>(rb) 
        {
            this->blender().gamma(g);
        }
    };


     
    template<class Gamma> class pixfmt_rgbAAA_gamma : 
    public pixfmt_alpha_blend_rgb_packed<blender_rgbAAA_gamma<Gamma>, 
                                         rendering_buffer>
    {
    public:
        pixfmt_rgbAAA_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb_packed<blender_rgbAAA_gamma<Gamma>, 
                                          rendering_buffer>(rb) 
        {
            this->blender().gamma(g);
        }
    };


     
    template<class Gamma> class pixfmt_bgrAAA_gamma : 
    public pixfmt_alpha_blend_rgb_packed<blender_bgrAAA_gamma<Gamma>, 
                                         rendering_buffer>
    {
    public:
        pixfmt_bgrAAA_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb_packed<blender_bgrAAA_gamma<Gamma>, 
                                          rendering_buffer>(rb) 
        {
            this->blender().gamma(g);
        }
    };


     
    template<class Gamma> class pixfmt_rgbBBA_gamma : 
    public pixfmt_alpha_blend_rgb_packed<blender_rgbBBA_gamma<Gamma>, 
                                         rendering_buffer>
    {
    public:
        pixfmt_rgbBBA_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb_packed<blender_rgbBBA_gamma<Gamma>, 
                                          rendering_buffer>(rb) 
        {
            this->blender().gamma(g);
        }
    };


     
    template<class Gamma> class pixfmt_bgrABB_gamma : 
    public pixfmt_alpha_blend_rgb_packed<blender_bgrABB_gamma<Gamma>, 
                                         rendering_buffer>
    {
    public:
        pixfmt_bgrABB_gamma(rendering_buffer& rb, const Gamma& g) :
            pixfmt_alpha_blend_rgb_packed<blender_bgrABB_gamma<Gamma>, 
                                          rendering_buffer>(rb) 
        {
            this->blender().gamma(g);
        }
    };


}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_PIXFMT_TRANSPOSER_INCLUDED
#define AGG_PIXFMT_TRANSPOSER_INCLUDED


namespace agg
{
     
    template<class PixFmt> class pixfmt_transposer
    {
    public:
        typedef PixFmt pixfmt_type;
        typedef typename pixfmt_type::color_type color_type;
        typedef typename pixfmt_type::row_data row_data;
        typedef typename color_type::value_type value_type;
        typedef typename color_type::calc_type calc_type;

         
        pixfmt_transposer() : m_pixf(0) {}
        explicit pixfmt_transposer(pixfmt_type& pixf) : m_pixf(&pixf) {}
        void attach(pixfmt_type& pixf) { m_pixf = &pixf; }

         
        AGG_INLINE unsigned width()  const { return m_pixf->height();  }
        AGG_INLINE unsigned height() const { return m_pixf->width(); }

         
        AGG_INLINE color_type pixel(int x, int y) const
        {
            return m_pixf->pixel(y, x);
        }

         
        AGG_INLINE void copy_pixel(int x, int y, const color_type& c)
        {
            m_pixf->copy_pixel(y, x, c);
        }

         
        AGG_INLINE void blend_pixel(int x, int y, 
                                    const color_type& c, 
                                    int8u cover)
        {
            m_pixf->blend_pixel(y, x, c, cover);
        }

         
        AGG_INLINE void copy_hline(int x, int y, 
                                   unsigned len, 
                                   const color_type& c)
        {
            m_pixf->copy_vline(y, x, len, c);
        }

         
        AGG_INLINE void copy_vline(int x, int y,
                                   unsigned len, 
                                   const color_type& c)
        {
            m_pixf->copy_hline(y, x, len, c);
        }

         
        AGG_INLINE void blend_hline(int x, int y,
                                    unsigned len, 
                                    const color_type& c,
                                    int8u cover)
        {
            m_pixf->blend_vline(y, x, len, c, cover);
        }

         
        AGG_INLINE void blend_vline(int x, int y,
                                    unsigned len, 
                                    const color_type& c,
                                    int8u cover)
        {
            m_pixf->blend_hline(y, x, len, c, cover);
        }

         
        AGG_INLINE void blend_solid_hspan(int x, int y,
                                          unsigned len, 
                                          const color_type& c,
                                          const int8u* covers)
        {
            m_pixf->blend_solid_vspan(y, x, len, c, covers);
        }

         
        AGG_INLINE void blend_solid_vspan(int x, int y,
                                          unsigned len, 
                                          const color_type& c,
                                          const int8u* covers)
        {
            m_pixf->blend_solid_hspan(y, x, len, c, covers);
        }

         
        AGG_INLINE void copy_color_hspan(int x, int y,
                                         unsigned len, 
                                         const color_type* colors)
        {
            m_pixf->copy_color_vspan(y, x, len, colors);
        }

         
        AGG_INLINE void copy_color_vspan(int x, int y,
                                         unsigned len, 
                                         const color_type* colors)
        {
            m_pixf->copy_color_hspan(y, x, len, colors);
        }

         
        AGG_INLINE void blend_color_hspan(int x, int y,
                                          unsigned len, 
                                          const color_type* colors,
                                          const int8u* covers,
                                          int8u cover)
        {
            m_pixf->blend_color_vspan(y, x, len, colors, covers, cover);
        }

         
        AGG_INLINE void blend_color_vspan(int x, int y,
                               unsigned len, 
                               const color_type* colors,
                               const int8u* covers,
                               int8u cover)
        {
            m_pixf->blend_color_hspan(y, x, len, colors, covers, cover);
        }

    private:
        pixfmt_type* m_pixf;
    };
}

#endif


 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_RASTERIZER_CELLS_AA_INCLUDED
#define AGG_RASTERIZER_CELLS_AA_INCLUDED

#include <cstring>
#include <cstdlib>
#include <limits>


namespace agg
{

     
     
     
    template<class Cell> class rasterizer_cells_aa
    {
        enum cell_block_scale_e
        {
            cell_block_shift = 12,
            cell_block_size  = 1 << cell_block_shift,
            cell_block_mask  = cell_block_size - 1,
            cell_block_pool  = 256
        };

        struct sorted_y
        {
            unsigned start;
            unsigned num;
        };

    public:
        typedef Cell cell_type;
        typedef rasterizer_cells_aa<Cell> self_type;

        ~rasterizer_cells_aa();
        rasterizer_cells_aa(unsigned cell_block_limit=1024);

        void reset();
        void style(const cell_type& style_cell);
        void line(int x1, int y1, int x2, int y2);

        int min_x() const { return m_min_x; }
        int min_y() const { return m_min_y; }
        int max_x() const { return m_max_x; }
        int max_y() const { return m_max_y; }

        void sort_cells();

        unsigned total_cells() const 
        {
            return m_num_cells;
        }

        unsigned scanline_num_cells(unsigned y) const 
        { 
            return m_sorted_y[y - m_min_y].num; 
        }

        const cell_type* const* scanline_cells(unsigned y) const
        { 
            return m_sorted_cells.data() + m_sorted_y[y - m_min_y].start; 
        }

        bool sorted() const { return m_sorted; }

    private:
        rasterizer_cells_aa(const self_type&);
        const self_type& operator = (const self_type&);

        void set_curr_cell(int x, int y);
        void add_curr_cell();
        void render_hline(int ey, int x1, int y1, int x2, int y2);
        void allocate_block();
        
    private:
        unsigned                m_num_blocks;
        unsigned                m_max_blocks;
        unsigned                m_curr_block;
        unsigned                m_num_cells;
	unsigned                m_cell_block_limit;
        cell_type**             m_cells;
        cell_type*              m_curr_cell_ptr;
        pod_vector<cell_type*>  m_sorted_cells;
        pod_vector<sorted_y>    m_sorted_y;
        cell_type               m_curr_cell;
        cell_type               m_style_cell;
        int                     m_min_x;
        int                     m_min_y;
        int                     m_max_x;
        int                     m_max_y;
        bool                    m_sorted;
    };




     
    template<class Cell> 
    rasterizer_cells_aa<Cell>::~rasterizer_cells_aa()
    {
        if(m_num_blocks)
        {
            cell_type** ptr = m_cells + m_num_blocks - 1;
            while(m_num_blocks--)
            {
                pod_allocator<cell_type>::deallocate(*ptr, cell_block_size);
                ptr--;
            }
            pod_allocator<cell_type*>::deallocate(m_cells, m_max_blocks);
        }
    }

     
    template<class Cell> 
    rasterizer_cells_aa<Cell>::rasterizer_cells_aa(unsigned cell_block_limit) :
        m_num_blocks(0),
        m_max_blocks(0),
        m_curr_block(0),
        m_num_cells(0),
	m_cell_block_limit(cell_block_limit),
        m_cells(0),
        m_curr_cell_ptr(0),
        m_sorted_cells(),
        m_sorted_y(),
        m_min_x(std::numeric_limits<int>::max()),
        m_min_y(std::numeric_limits<int>::max()),
        m_max_x(std::numeric_limits<int>::min()),
        m_max_y(std::numeric_limits<int>::min()),
        m_sorted(false)
    {
        m_style_cell.initial();
        m_curr_cell.initial();
    }

     
    template<class Cell> 
    void rasterizer_cells_aa<Cell>::reset()
    {
        m_num_cells = 0; 
        m_curr_block = 0;
        m_curr_cell.initial();
        m_style_cell.initial();
        m_sorted = false;
        m_min_x = std::numeric_limits<int>::max();
        m_min_y = std::numeric_limits<int>::max();
        m_max_x = std::numeric_limits<int>::min();
        m_max_y = std::numeric_limits<int>::min();
    }

     
    template<class Cell> 
    AGG_INLINE void rasterizer_cells_aa<Cell>::add_curr_cell()
    {
        if(m_curr_cell.area | m_curr_cell.cover)
        {
            if((m_num_cells & cell_block_mask) == 0)
            {
                if(m_num_blocks >= m_cell_block_limit) return;
                allocate_block();
            }
            *m_curr_cell_ptr++ = m_curr_cell;
            ++m_num_cells;
        }
    }

     
    template<class Cell> 
    AGG_INLINE void rasterizer_cells_aa<Cell>::set_curr_cell(int x, int y)
    {
        if(m_curr_cell.not_equal(x, y, m_style_cell))
        {
            add_curr_cell();
            m_curr_cell.style(m_style_cell);
            m_curr_cell.x     = x;
            m_curr_cell.y     = y;
            m_curr_cell.cover = 0;
            m_curr_cell.area  = 0;
        }
    }

     
    template<class Cell> 
    AGG_INLINE void rasterizer_cells_aa<Cell>::render_hline(int ey, 
                                                            int x1, int y1, 
                                                            int x2, int y2)
    {
        int ex1 = x1 >> poly_subpixel_shift;
        int ex2 = x2 >> poly_subpixel_shift;
        int fx1 = x1 & poly_subpixel_mask;
        int fx2 = x2 & poly_subpixel_mask;

        int delta, p, first;
        long long dx;
        int incr, lift, mod, rem;

         
        if(y1 == y2)
        {
            set_curr_cell(ex2, ey);
            return;
        }

         
        if(ex1 == ex2)
        {
            delta = y2 - y1;
            m_curr_cell.cover += delta;
            m_curr_cell.area  += (fx1 + fx2) * delta;
            return;
        }

         
         
        p     = (poly_subpixel_scale - fx1) * (y2 - y1);
        first = poly_subpixel_scale;
        incr  = 1;

        dx = (long long)x2 - (long long)x1;

        if(dx < 0)
        {
            p     = fx1 * (y2 - y1);
            first = 0;
            incr  = -1;
            dx    = -dx;
        }

        delta = (int)(p / dx);
        mod   = (int)(p % dx);

        if(mod < 0)
        {
            delta--;
            mod += dx;
        }

        m_curr_cell.cover += delta;
        m_curr_cell.area  += (fx1 + first) * delta;

        ex1 += incr;
        set_curr_cell(ex1, ey);
        y1  += delta;

        if(ex1 != ex2)
        {
            p     = poly_subpixel_scale * (y2 - y1 + delta);
            lift  = (int)(p / dx);
            rem   = (int)(p % dx);

            if (rem < 0)
            {
                lift--;
                rem += dx;
            }

            mod -= dx;

            while (ex1 != ex2)
            {
                delta = lift;
                mod  += rem;
                if(mod >= 0)
                {
                    mod -= dx;
                    delta++;
                }

                m_curr_cell.cover += delta;
                m_curr_cell.area  += poly_subpixel_scale * delta;
                y1  += delta;
                ex1 += incr;
                set_curr_cell(ex1, ey);
            }
        }
        delta = y2 - y1;
        m_curr_cell.cover += delta;
        m_curr_cell.area  += (fx2 + poly_subpixel_scale - first) * delta;
    }

     
    template<class Cell> 
    AGG_INLINE void rasterizer_cells_aa<Cell>::style(const cell_type& style_cell)
    { 
        m_style_cell.style(style_cell); 
    }

     
    template<class Cell> 
    void rasterizer_cells_aa<Cell>::line(int x1, int y1, int x2, int y2)
    {
        enum dx_limit_e { dx_limit = 16384 << poly_subpixel_shift };

        long long dx = (long long)x2 - (long long)x1;

        if(dx >= dx_limit || dx <= -dx_limit)
        {
            int cx = (int)(((long long)x1 + (long long)x2) >> 1);
            int cy = (int)(((long long)y1 + (long long)y2) >> 1);
            line(x1, y1, cx, cy);
            line(cx, cy, x2, y2);
        }

        long long dy = (long long)y2 - (long long)y1;
        int ex1 = x1 >> poly_subpixel_shift;
        int ex2 = x2 >> poly_subpixel_shift;
        int ey1 = y1 >> poly_subpixel_shift;
        int ey2 = y2 >> poly_subpixel_shift;
        int fy1 = y1 & poly_subpixel_mask;
        int fy2 = y2 & poly_subpixel_mask;

        int x_from, x_to;
        int rem, mod, lift, delta, first, incr;
        long long p;

        if(ex1 < m_min_x) m_min_x = ex1;
        if(ex1 > m_max_x) m_max_x = ex1;
        if(ey1 < m_min_y) m_min_y = ey1;
        if(ey1 > m_max_y) m_max_y = ey1;
        if(ex2 < m_min_x) m_min_x = ex2;
        if(ex2 > m_max_x) m_max_x = ex2;
        if(ey2 < m_min_y) m_min_y = ey2;
        if(ey2 > m_max_y) m_max_y = ey2;

        set_curr_cell(ex1, ey1);

         
        if(ey1 == ey2)
        {
            render_hline(ey1, x1, fy1, x2, fy2);
            return;
        }

         
         
         
         
        incr  = 1;
        if(dx == 0)
        {
            int ex = x1 >> poly_subpixel_shift;
            int two_fx = (x1 - (ex << poly_subpixel_shift)) << 1;
            int area;

            first = poly_subpixel_scale;
            if(dy < 0)
            {
                first = 0;
                incr  = -1;
            }

            x_from = x1;

             
            delta = first - fy1;
            m_curr_cell.cover += delta;
            m_curr_cell.area  += two_fx * delta;

            ey1 += incr;
            set_curr_cell(ex, ey1);

            delta = first + first - poly_subpixel_scale;
            area = two_fx * delta;
            while(ey1 != ey2)
            {
                 
                m_curr_cell.cover = delta;
                m_curr_cell.area  = area;
                ey1 += incr;
                set_curr_cell(ex, ey1);
            }
             
            delta = fy2 - poly_subpixel_scale + first;
            m_curr_cell.cover += delta;
            m_curr_cell.area  += two_fx * delta;
            return;
        }

         
        p     = (poly_subpixel_scale - fy1) * dx;
        first = poly_subpixel_scale;

        if(dy < 0)
        {
            p     = fy1 * dx;
            first = 0;
            incr  = -1;
            dy    = -dy;
        }

        delta = (int)(p / dy);
        mod   = (int)(p % dy);

        if(mod < 0)
        {
            delta--;
            mod += dy;
        }

        x_from = x1 + delta;
        render_hline(ey1, x1, fy1, x_from, first);

        ey1 += incr;
        set_curr_cell(x_from >> poly_subpixel_shift, ey1);

        if(ey1 != ey2)
        {
            p     = poly_subpixel_scale * dx;
            lift  = (int)(p / dy);
            rem   = (int)(p % dy);

            if(rem < 0)
            {
                lift--;
                rem += dy;
            }
            mod -= dy;

            while(ey1 != ey2)
            {
                delta = lift;
                mod  += rem;
                if (mod >= 0)
                {
                    mod -= dy;
                    delta++;
                }

                x_to = x_from + delta;
                render_hline(ey1, x_from, poly_subpixel_scale - first, x_to, first);
                x_from = x_to;

                ey1 += incr;
                set_curr_cell(x_from >> poly_subpixel_shift, ey1);
            }
        }
        render_hline(ey1, x_from, poly_subpixel_scale - first, x2, fy2);
    }

     
    template<class Cell> 
    void rasterizer_cells_aa<Cell>::allocate_block()
    {
        if(m_curr_block >= m_num_blocks)
        {
            if(m_num_blocks >= m_max_blocks)
            {
                cell_type** new_cells = 
                    pod_allocator<cell_type*>::allocate(m_max_blocks + 
                                                        cell_block_pool);

                if(m_cells)
                {
                    std::memcpy(new_cells, m_cells, m_max_blocks * sizeof(cell_type*));
                    pod_allocator<cell_type*>::deallocate(m_cells, m_max_blocks);
                }
                m_cells = new_cells;
                m_max_blocks += cell_block_pool;
            }

            m_cells[m_num_blocks++] = 
                pod_allocator<cell_type>::allocate(cell_block_size);

        }
        m_curr_cell_ptr = m_cells[m_curr_block++];
    }



     
    template <class T> static AGG_INLINE void swap_cells(T* a, T* b)
    {
        T temp = *a;
        *a = *b;
        *b = temp;
    }


     
    enum
    {
        qsort_threshold = 9
    };


     
    template<class Cell>
    void qsort_cells(Cell** start, unsigned num)
    {
        Cell**  stack[80];
        Cell*** top; 
        Cell**  limit;
        Cell**  base;

        limit = start + num;
        base  = start;
        top   = stack;

        for (;;)
        {
            int len = int(limit - base);

            Cell** i;
            Cell** j;
            Cell** pivot;

            if(len > qsort_threshold)
            {
                 
                pivot = base + len / 2;
                swap_cells(base, pivot);

                i = base + 1;
                j = limit - 1;

                 
                if((*j)->x < (*i)->x)
                {
                    swap_cells(i, j);
                }

                if((*base)->x < (*i)->x)
                {
                    swap_cells(base, i);
                }

                if((*j)->x < (*base)->x)
                {
                    swap_cells(base, j);
                }

                for(;;)
                {
                    int x = (*base)->x;
                    do i++; while( (*i)->x < x );
                    do j--; while( x < (*j)->x );

                    if(i > j)
                    {
                        break;
                    }

                    swap_cells(i, j);
                }

                swap_cells(base, j);

                 
                if(j - base > limit - i)
                {
                    top[0] = base;
                    top[1] = j;
                    base   = i;
                }
                else
                {
                    top[0] = i;
                    top[1] = limit;
                    limit  = j;
                }
                top += 2;
            }
            else
            {
                 
                j = base;
                i = j + 1;

                for(; i < limit; j = i, i++)
                {
                    for(; j[1]->x < (*j)->x; j--)
                    {
                        swap_cells(j + 1, j);
                        if (j == base)
                        {
                            break;
                        }
                    }
                }

                if(top > stack)
                {
                    top  -= 2;
                    base  = top[0];
                    limit = top[1];
                }
                else
                {
                    break;
                }
            }
        }
    }


     
    template<class Cell> 
    void rasterizer_cells_aa<Cell>::sort_cells()
    {
        if(m_sorted) return;  

        add_curr_cell();
        m_curr_cell.x     = std::numeric_limits<int>::max();
        m_curr_cell.y     = std::numeric_limits<int>::max();
        m_curr_cell.cover = 0;
        m_curr_cell.area  = 0;

        if(m_num_cells == 0) return;

 
 
 
 
 
 
 
 
 
 
 
 
         
        m_sorted_cells.allocate(m_num_cells, 16);

         
        m_sorted_y.allocate(m_max_y - m_min_y + 1, 16);
        m_sorted_y.zero();

         
        cell_type** block_ptr = m_cells;
        cell_type*  cell_ptr;
        unsigned nb = m_num_cells;
        unsigned i;
        while(nb)
        {
            cell_ptr = *block_ptr++;
            i = (nb > cell_block_size) ? unsigned(cell_block_size) : nb;
            nb -= i;
            while(i--) 
            {
                m_sorted_y[cell_ptr->y - m_min_y].start++;
                ++cell_ptr;
            }
        }

         
        unsigned start = 0;
        for(i = 0; i < m_sorted_y.size(); i++)
        {
            unsigned v = m_sorted_y[i].start;
            m_sorted_y[i].start = start;
            start += v;
        }

         
        block_ptr = m_cells;
        nb = m_num_cells;
        while(nb)
        {
            cell_ptr = *block_ptr++;
            i = (nb > cell_block_size) ? unsigned(cell_block_size) : nb;
            nb -= i;
            while(i--)
            {
                sorted_y& curr_y = m_sorted_y[cell_ptr->y - m_min_y];
                m_sorted_cells[curr_y.start + curr_y.num] = cell_ptr;
                ++curr_y.num;
                ++cell_ptr;
            }
        }
        
         
        for(i = 0; i < m_sorted_y.size(); i++)
        {
            const sorted_y& curr_y = m_sorted_y[i];
            if(curr_y.num)
            {
                qsort_cells(m_sorted_cells.data() + curr_y.start, curr_y.num);
            }
        }
        m_sorted = true;
    }



     
    class scanline_hit_test
    {
    public:
        scanline_hit_test(int x) : m_x(x), m_hit(false) {}

        void reset_spans() {}
        void finalize(int) {}
        void add_cell(int x, int)
        {
            if(m_x == x) m_hit = true;
        }
        void add_span(int x, int len, int)
        {
            if(m_x >= x && m_x < x+len) m_hit = true;
        }
        unsigned num_spans() const { return 1; }
        bool hit() const { return m_hit; }

    private:
        int  m_x;
        bool m_hit;
    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_RASTERIZER_COMPOUND_AA_INCLUDED
#define AGG_RASTERIZER_COMPOUND_AA_INCLUDED

#include <limits>
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_RASTERIZER_SL_CLIP_INCLUDED
#define AGG_RASTERIZER_SL_CLIP_INCLUDED


namespace agg
{
     
    enum poly_max_coord_e
    {
        poly_max_coord = (1 << 30) - 1  
    };
    
     
    struct ras_conv_int
    {
        typedef int coord_type;
        static AGG_INLINE int mul_div(double a, double b, double c)
        {
            return iround(a * b / c);
        }
        static int xi(int v) { return v; }
        static int yi(int v) { return v; }
        static int upscale(double v) { return iround(v * (double)poly_subpixel_scale); }
        static int downscale(int v)  { return v; }
    };

     
    struct ras_conv_int_sat
    {
        typedef int coord_type;
        static AGG_INLINE int mul_div(double a, double b, double c)
        {
            return saturation<poly_max_coord>::iround(a * b / c);
        }
        static int xi(int v) { return v; }
        static int yi(int v) { return v; }
        static int upscale(double v) 
        { 
            return saturation<poly_max_coord>::iround(v * (double)poly_subpixel_scale); 
        }
        static int downscale(int v) { return v; }
    };

     
    struct ras_conv_int_3x
    {
        typedef int coord_type;
        static AGG_INLINE int mul_div(double a, double b, double c)
        {
            return iround(a * b / c);
        }
        static int xi(int v) { return v * 3; }
        static int yi(int v) { return v; }
        static int upscale(double v) { return iround(v * (double)poly_subpixel_scale); }
        static int downscale(int v)  { return v; }
    };

     
    struct ras_conv_dbl
    {
        typedef double coord_type;
        static AGG_INLINE double mul_div(double a, double b, double c)
        {
            return a * b / c;
        }
        static int xi(double v) { return iround(v * (double)poly_subpixel_scale); }
        static int yi(double v) { return iround(v * (double)poly_subpixel_scale); }
        static double upscale(double v) { return v; }
        static double downscale(int v)  { return v / double(poly_subpixel_scale); }
    };

     
    struct ras_conv_dbl_3x
    {
        typedef double coord_type;
        static AGG_INLINE double mul_div(double a, double b, double c)
        {
            return a * b / c;
        }
        static int xi(double v) { return iround(v * (double)poly_subpixel_scale * 3.0); }
        static int yi(double v) { return iround(v * (double)poly_subpixel_scale); }
        static double upscale(double v) { return v; }
        static double downscale(int v)  { return v / double(poly_subpixel_scale); }
    };





     
    template<class Conv> class rasterizer_sl_clip
    {
    public:
        typedef Conv                      conv_type;
        typedef typename Conv::coord_type coord_type;
        typedef rect_base<coord_type>     rect_type;

         
        rasterizer_sl_clip() :  
            m_clip_box(0,0,0,0),
            m_x1(0),
            m_y1(0),
            m_f1(0),
            m_clipping(false) 
        {}

         
        void reset_clipping()
        {
            m_clipping = false;
        }

         
        void clip_box(coord_type x1, coord_type y1, coord_type x2, coord_type y2)
        {
            m_clip_box = rect_type(x1, y1, x2, y2);
            m_clip_box.normalize();
            m_clipping = true;
        }

         
        void move_to(coord_type x1, coord_type y1)
        {
            m_x1 = x1;
            m_y1 = y1;
            if(m_clipping) m_f1 = clipping_flags(x1, y1, m_clip_box);
        }

    private:
         
        template<class Rasterizer>
        AGG_INLINE void line_clip_y(Rasterizer& ras,
                                    coord_type x1, coord_type y1, 
                                    coord_type x2, coord_type y2, 
                                    unsigned   f1, unsigned   f2) const
        {
            f1 &= 10;
            f2 &= 10;
            if((f1 | f2) == 0)
            {
                 
                ras.line(Conv::xi(x1), Conv::yi(y1), Conv::xi(x2), Conv::yi(y2)); 
            }
            else
            {
                if(f1 == f2)
                {
                     
                    return;
                }

                coord_type tx1 = x1;
                coord_type ty1 = y1;
                coord_type tx2 = x2;
                coord_type ty2 = y2;

                if(f1 & 8)  
                {
                    tx1 = x1 + Conv::mul_div(m_clip_box.y1-y1, x2-x1, y2-y1);
                    ty1 = m_clip_box.y1;
                }

                if(f1 & 2)  
                {
                    tx1 = x1 + Conv::mul_div(m_clip_box.y2-y1, x2-x1, y2-y1);
                    ty1 = m_clip_box.y2;
                }

                if(f2 & 8)  
                {
                    tx2 = x1 + Conv::mul_div(m_clip_box.y1-y1, x2-x1, y2-y1);
                    ty2 = m_clip_box.y1;
                }

                if(f2 & 2)  
                {
                    tx2 = x1 + Conv::mul_div(m_clip_box.y2-y1, x2-x1, y2-y1);
                    ty2 = m_clip_box.y2;
                }
                ras.line(Conv::xi(tx1), Conv::yi(ty1), 
                         Conv::xi(tx2), Conv::yi(ty2)); 
            }
        }


    public:
         
        template<class Rasterizer>
        void line_to(Rasterizer& ras, coord_type x2, coord_type y2)
        {
            if(m_clipping)
            {
                unsigned f2 = clipping_flags(x2, y2, m_clip_box);

                if((m_f1 & 10) == (f2 & 10) && (m_f1 & 10) != 0)
                {
                     
                    m_x1 = x2;
                    m_y1 = y2;
                    m_f1 = f2;
                    return;
                }

                coord_type x1 = m_x1;
                coord_type y1 = m_y1;
                unsigned   f1 = m_f1;
                coord_type y3, y4;
                unsigned   f3, f4;

                switch(((f1 & 5) << 1) | (f2 & 5))
                {
                case 0:  
                    line_clip_y(ras, x1, y1, x2, y2, f1, f2);
                    break;

                case 1:  
                    y3 = y1 + Conv::mul_div(m_clip_box.x2-x1, y2-y1, x2-x1);
                    f3 = clipping_flags_y(y3, m_clip_box);
                    line_clip_y(ras, x1, y1, m_clip_box.x2, y3, f1, f3);
                    line_clip_y(ras, m_clip_box.x2, y3, m_clip_box.x2, y2, f3, f2);
                    break;

                case 2:  
                    y3 = y1 + Conv::mul_div(m_clip_box.x2-x1, y2-y1, x2-x1);
                    f3 = clipping_flags_y(y3, m_clip_box);
                    line_clip_y(ras, m_clip_box.x2, y1, m_clip_box.x2, y3, f1, f3);
                    line_clip_y(ras, m_clip_box.x2, y3, x2, y2, f3, f2);
                    break;

                case 3:  
                    line_clip_y(ras, m_clip_box.x2, y1, m_clip_box.x2, y2, f1, f2);
                    break;

                case 4:  
                    y3 = y1 + Conv::mul_div(m_clip_box.x1-x1, y2-y1, x2-x1);
                    f3 = clipping_flags_y(y3, m_clip_box);
                    line_clip_y(ras, x1, y1, m_clip_box.x1, y3, f1, f3);
                    line_clip_y(ras, m_clip_box.x1, y3, m_clip_box.x1, y2, f3, f2);
                    break;

                case 6:  
                    y3 = y1 + Conv::mul_div(m_clip_box.x2-x1, y2-y1, x2-x1);
                    y4 = y1 + Conv::mul_div(m_clip_box.x1-x1, y2-y1, x2-x1);
                    f3 = clipping_flags_y(y3, m_clip_box);
                    f4 = clipping_flags_y(y4, m_clip_box);
                    line_clip_y(ras, m_clip_box.x2, y1, m_clip_box.x2, y3, f1, f3);
                    line_clip_y(ras, m_clip_box.x2, y3, m_clip_box.x1, y4, f3, f4);
                    line_clip_y(ras, m_clip_box.x1, y4, m_clip_box.x1, y2, f4, f2);
                    break;

                case 8:  
                    y3 = y1 + Conv::mul_div(m_clip_box.x1-x1, y2-y1, x2-x1);
                    f3 = clipping_flags_y(y3, m_clip_box);
                    line_clip_y(ras, m_clip_box.x1, y1, m_clip_box.x1, y3, f1, f3);
                    line_clip_y(ras, m_clip_box.x1, y3, x2, y2, f3, f2);
                    break;

                case 9:   
                    y3 = y1 + Conv::mul_div(m_clip_box.x1-x1, y2-y1, x2-x1);
                    y4 = y1 + Conv::mul_div(m_clip_box.x2-x1, y2-y1, x2-x1);
                    f3 = clipping_flags_y(y3, m_clip_box);
                    f4 = clipping_flags_y(y4, m_clip_box);
                    line_clip_y(ras, m_clip_box.x1, y1, m_clip_box.x1, y3, f1, f3);
                    line_clip_y(ras, m_clip_box.x1, y3, m_clip_box.x2, y4, f3, f4);
                    line_clip_y(ras, m_clip_box.x2, y4, m_clip_box.x2, y2, f4, f2);
                    break;

                case 12:  
                    line_clip_y(ras, m_clip_box.x1, y1, m_clip_box.x1, y2, f1, f2);
                    break;
                }
                m_f1 = f2;
            }
            else
            {
                ras.line(Conv::xi(m_x1), Conv::yi(m_y1), 
                         Conv::xi(x2),   Conv::yi(y2)); 
            }
            m_x1 = x2;
            m_y1 = y2;
        }


    private:
        rect_type        m_clip_box;
        coord_type       m_x1;
        coord_type       m_y1;
        unsigned         m_f1;
        bool             m_clipping;
    };




     
    class rasterizer_sl_no_clip
    {
    public:
        typedef ras_conv_int conv_type;
        typedef int          coord_type;

        rasterizer_sl_no_clip() : m_x1(0), m_y1(0) {}

        void reset_clipping() {}
        void clip_box(coord_type, coord_type, coord_type, coord_type) {}
        void move_to(coord_type x1, coord_type y1) { m_x1 = x1; m_y1 = y1; }

        template<class Rasterizer>
        void line_to(Rasterizer& ras, coord_type x2, coord_type y2) 
        { 
            ras.line(m_x1, m_y1, x2, y2); 
            m_x1 = x2; 
            m_y1 = y2;
        }

    private:
        int m_x1, m_y1;
    };


     
     
     
     
     
     
    typedef rasterizer_sl_clip<ras_conv_int>     rasterizer_sl_clip_int;
    typedef rasterizer_sl_clip<ras_conv_int_sat> rasterizer_sl_clip_int_sat;
    typedef rasterizer_sl_clip<ras_conv_int_3x>  rasterizer_sl_clip_int_3x;
    typedef rasterizer_sl_clip<ras_conv_dbl>     rasterizer_sl_clip_dbl;
    typedef rasterizer_sl_clip<ras_conv_dbl_3x>  rasterizer_sl_clip_dbl_3x;


}

#endif

namespace agg
{

     
     
     
     
    struct cell_style_aa
    {
        int   x;
        int   y;
        int   cover;
        int   area;
        int16 left, right;

        void initial()
        {
            x     = std::numeric_limits<int>::max();
            y     = std::numeric_limits<int>::max();
            cover = 0;
            area  = 0;
            left  = -1;
            right = -1;
        }

        void style(const cell_style_aa& c)
        {
            left  = c.left;
            right = c.right;
        }

        int not_equal(int ex, int ey, const cell_style_aa& c) const
        {
            return ((unsigned)ex - (unsigned)x) | ((unsigned)ey - (unsigned)y) | 
                   ((unsigned)left - (unsigned)c.left) | ((unsigned)right - (unsigned)c.right);
        }
    };


     
    enum layer_order_e
    {
        layer_unsorted,  
        layer_direct,    
        layer_inverse    
    };


     
    template<class Clip=rasterizer_sl_clip_int> class rasterizer_compound_aa
    {
        struct style_info 
        { 
            unsigned start_cell;
            unsigned num_cells;
            int      last_x;
        };

        struct cell_info
        {
            int x, area, cover; 
        };

    public:
        typedef Clip                      clip_type;
        typedef typename Clip::conv_type  conv_type;
        typedef typename Clip::coord_type coord_type;

        enum aa_scale_e
        {
            aa_shift  = 8,
            aa_scale  = 1 << aa_shift,
            aa_mask   = aa_scale - 1,
            aa_scale2 = aa_scale * 2,
            aa_mask2  = aa_scale2 - 1
        };

         
        rasterizer_compound_aa() : 
            m_outline(),
            m_clipper(),
            m_filling_rule(fill_non_zero),
            m_layer_order(layer_direct),
            m_styles(),   
            m_ast(),      
            m_asm(),      
            m_cells(),
            m_cover_buf(),
            m_min_style(std::numeric_limits<int>::max()),
            m_max_style(std::numeric_limits<int>::min()),
            m_start_x(0),
            m_start_y(0),
            m_scan_y(std::numeric_limits<int>::max()),
            m_sl_start(0),
            m_sl_len(0)
        {}

         
        void reset(); 
        void reset_clipping();
        void clip_box(double x1, double y1, double x2, double y2);
        void filling_rule(filling_rule_e filling_rule);
        void layer_order(layer_order_e order);

         
        void styles(int left, int right);
        void move_to(int x, int y);
        void line_to(int x, int y);
        void move_to_d(double x, double y);
        void line_to_d(double x, double y);
        void add_vertex(double x, double y, unsigned cmd);

        void edge(int x1, int y1, int x2, int y2);
        void edge_d(double x1, double y1, double x2, double y2);

         
        template<class VertexSource>
        void add_path(VertexSource& vs, unsigned path_id=0)
        {
            double x = 0;
            double y = 0;

            unsigned cmd;
            vs.rewind(path_id);
            if(m_outline.sorted()) reset();
            while(!is_stop(cmd = vs.vertex(&x, &y)))
            {
                add_vertex(x, y, cmd);
            }
        }

        
         
        int min_x()     const { return m_outline.min_x(); }
        int min_y()     const { return m_outline.min_y(); }
        int max_x()     const { return m_outline.max_x(); }
        int max_y()     const { return m_outline.max_y(); }
        int min_style() const { return m_min_style; }
        int max_style() const { return m_max_style; }

         
        void sort();
        bool rewind_scanlines();
        unsigned sweep_styles();
        int      scanline_start()  const { return m_sl_start; }
        unsigned scanline_length() const { return m_sl_len;   }
        unsigned style(unsigned style_idx) const;

        cover_type* allocate_cover_buffer(unsigned len);

         
        bool navigate_scanline(int y); 
        bool hit_test(int tx, int ty);

         
        AGG_INLINE unsigned calculate_alpha(int area) const
        {
            int cover = area >> (poly_subpixel_shift*2 + 1 - aa_shift);
            if(cover < 0) cover = -cover;
            if(m_filling_rule == fill_even_odd)
            {
                cover &= aa_mask2;
                if(cover > aa_scale)
                {
                    cover = aa_scale2 - cover;
                }
            }
            if(cover > aa_mask) cover = aa_mask;
            return cover;
        }

         
         
         
        template<class Scanline> bool sweep_scanline(Scanline& sl, int style_idx)
        {
            int scan_y = m_scan_y - 1;
            if(scan_y > m_outline.max_y()) return false;

            sl.reset_spans();

            if(style_idx < 0) 
            {
                style_idx = 0;
            }
            else 
            {
                style_idx++;
            }

            const style_info& st = m_styles[m_ast[style_idx]];

            unsigned num_cells = st.num_cells;
            cell_info* cell = &m_cells[st.start_cell];

            int cover = 0;
            while(num_cells--)
            {
                unsigned alpha;
                int x = cell->x;
                int area = cell->area;

                cover += cell->cover;

                ++cell;

                if(area)
                {
                    alpha = calculate_alpha((cover << (poly_subpixel_shift + 1)) - area);
                    sl.add_cell(x, alpha);
                    x++;
                }

                if(num_cells && cell->x > x)
                {
                    alpha = calculate_alpha(cover << (poly_subpixel_shift + 1));
                    if(alpha)
                    {
                        sl.add_span(x, cell->x - x, alpha);
                    }
                }
            }

            if(sl.num_spans() == 0) return false;
            sl.finalize(scan_y);
            return true;
        }

    private:
        void add_style(int style_id);

         
         
        rasterizer_compound_aa(const rasterizer_compound_aa<Clip>&);
        const rasterizer_compound_aa<Clip>& 
        operator = (const rasterizer_compound_aa<Clip>&);

    private:
        rasterizer_cells_aa<cell_style_aa> m_outline;
        clip_type              m_clipper;
        filling_rule_e         m_filling_rule;
        layer_order_e          m_layer_order;
        pod_vector<style_info> m_styles;   
        pod_vector<unsigned>   m_ast;      
        pod_vector<int8u>      m_asm;      
        pod_vector<cell_info>  m_cells;
        pod_vector<cover_type> m_cover_buf;

        int        m_min_style;
        int        m_max_style;
        coord_type m_start_x;
        coord_type m_start_y;
        int        m_scan_y;
        int        m_sl_start;
        unsigned   m_sl_len;
    };










     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::reset() 
    { 
        m_outline.reset(); 
        m_min_style = std::numeric_limits<int>::max();
        m_max_style = std::numeric_limits<int>::min();
        m_scan_y    = std::numeric_limits<int>::max();
        m_sl_start  =  0;
        m_sl_len    = 0;
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::filling_rule(filling_rule_e filling_rule) 
    { 
        m_filling_rule = filling_rule; 
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::layer_order(layer_order_e order)
    {
        m_layer_order = order;
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::clip_box(double x1, double y1, 
                                                double x2, double y2)
    {
        reset();
        m_clipper.clip_box(conv_type::upscale(x1), conv_type::upscale(y1), 
                           conv_type::upscale(x2), conv_type::upscale(y2));
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::reset_clipping()
    {
        reset();
        m_clipper.reset_clipping();
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::styles(int left, int right)
    {
        cell_style_aa cell;
        cell.initial();
        cell.left = (int16)left;
        cell.right = (int16)right;
        m_outline.style(cell);
        if(left  >= 0 && left  < m_min_style) m_min_style = left;
        if(left  >= 0 && left  > m_max_style) m_max_style = left;
        if(right >= 0 && right < m_min_style) m_min_style = right;
        if(right >= 0 && right > m_max_style) m_max_style = right;
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::move_to(int x, int y)
    {
        if(m_outline.sorted()) reset();
        m_clipper.move_to(m_start_x = conv_type::downscale(x), 
                          m_start_y = conv_type::downscale(y));
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::line_to(int x, int y)
    {
        m_clipper.line_to(m_outline, 
                          conv_type::downscale(x), 
                          conv_type::downscale(y));
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::move_to_d(double x, double y) 
    { 
        if(m_outline.sorted()) reset();
        m_clipper.move_to(m_start_x = conv_type::upscale(x), 
                          m_start_y = conv_type::upscale(y)); 
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::line_to_d(double x, double y) 
    { 
        m_clipper.line_to(m_outline, 
                          conv_type::upscale(x), 
                          conv_type::upscale(y)); 
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::add_vertex(double x, double y, unsigned cmd)
    {
        if(is_move_to(cmd)) 
        {
            move_to_d(x, y);
        }
        else 
        if(is_vertex(cmd))
        {
            line_to_d(x, y);
        }
        else
        if(is_close(cmd))
        {
            m_clipper.line_to(m_outline, m_start_x, m_start_y);
        }
    }

     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::edge(int x1, int y1, int x2, int y2)
    {
        if(m_outline.sorted()) reset();
        m_clipper.move_to(conv_type::downscale(x1), conv_type::downscale(y1));
        m_clipper.line_to(m_outline, 
                          conv_type::downscale(x2), 
                          conv_type::downscale(y2));
    }
    
     
    template<class Clip> 
    void rasterizer_compound_aa<Clip>::edge_d(double x1, double y1, 
                                              double x2, double y2)
    {
        if(m_outline.sorted()) reset();
        m_clipper.move_to(conv_type::upscale(x1), conv_type::upscale(y1)); 
        m_clipper.line_to(m_outline, 
                          conv_type::upscale(x2), 
                          conv_type::upscale(y2)); 
    }

     
    template<class Clip> 
    AGG_INLINE void rasterizer_compound_aa<Clip>::sort()
    {
        m_outline.sort_cells();
    }

     
    template<class Clip> 
    AGG_INLINE bool rasterizer_compound_aa<Clip>::rewind_scanlines()
    {
        m_outline.sort_cells();
        if(m_outline.total_cells() == 0) 
        {
            return false;
        }
        if(m_max_style < m_min_style)
        {
            return false;
        }
        m_scan_y = m_outline.min_y();
        m_styles.allocate(m_max_style - m_min_style + 2, 128);
        return true;
    }

     
    template<class Clip> 
    AGG_INLINE void rasterizer_compound_aa<Clip>::add_style(int style_id)
    {
        if(style_id < 0) style_id  = 0;
        else             style_id -= m_min_style - 1;

        unsigned nbyte = style_id >> 3;
        unsigned mask = 1 << (style_id & 7);

        style_info* style = &m_styles[style_id];
        if((m_asm[nbyte] & mask) == 0)
        {
            m_ast.add(style_id);
            m_asm[nbyte] |= mask;
            style->start_cell = 0;
            style->num_cells = 0;
            style->last_x = std::numeric_limits<int>::min();
        }
        ++style->start_cell;
    }

     
     
    template<class Clip> 
    unsigned rasterizer_compound_aa<Clip>::sweep_styles()
    {
        for(;;)
        {
            if(m_scan_y > m_outline.max_y()) return 0;
            unsigned num_cells = m_outline.scanline_num_cells(m_scan_y);
            const cell_style_aa* const* cells = m_outline.scanline_cells(m_scan_y);
            unsigned num_styles = m_max_style - m_min_style + 2;
            const cell_style_aa* curr_cell;
            unsigned style_id;
            style_info* style;
            cell_info* cell;

            m_cells.allocate(num_cells * 2, 256);  
            m_ast.capacity(num_styles, 64);
            m_asm.allocate((num_styles + 7) >> 3, 8);
            m_asm.zero();

            if(num_cells)
            {
                 
                 
                m_asm[0] |= 1; 
                m_ast.add(0);
                style = &m_styles[0];
                style->start_cell = 0;
                style->num_cells = 0;
                style->last_x = std::numeric_limits<int>::min();

                m_sl_start = cells[0]->x;
                m_sl_len   = cells[num_cells-1]->x - m_sl_start + 1;
                while(num_cells--)
                {
                    curr_cell = *cells++;
                    add_style(curr_cell->left);
                    add_style(curr_cell->right);
                }

                 
                unsigned i;
                unsigned start_cell = 0;
                for(i = 0; i < m_ast.size(); i++)
                {
                    style_info& st = m_styles[m_ast[i]];
                    unsigned v = st.start_cell;
                    st.start_cell = start_cell;
                    start_cell += v;
                }

                cells = m_outline.scanline_cells(m_scan_y);
                num_cells = m_outline.scanline_num_cells(m_scan_y);

                while(num_cells--)
                {
                    curr_cell = *cells++;
                    style_id = (curr_cell->left < 0) ? 0 : 
                                curr_cell->left - m_min_style + 1;

                    style = &m_styles[style_id];
                    if(curr_cell->x == style->last_x)
                    {
                        cell = &m_cells[style->start_cell + style->num_cells - 1];
                        cell->area  += curr_cell->area;
                        cell->cover += curr_cell->cover;
                    }
                    else
                    {
                        cell = &m_cells[style->start_cell + style->num_cells];
                        cell->x       = curr_cell->x;
                        cell->area    = curr_cell->area;
                        cell->cover   = curr_cell->cover;
                        style->last_x = curr_cell->x;
                        style->num_cells++;
                    }

                    style_id = (curr_cell->right < 0) ? 0 : 
                                curr_cell->right - m_min_style + 1;

                    style = &m_styles[style_id];
                    if(curr_cell->x == style->last_x)
                    {
                        cell = &m_cells[style->start_cell + style->num_cells - 1];
                        cell->area  -= curr_cell->area;
                        cell->cover -= curr_cell->cover;
                    }
                    else
                    {
                        cell = &m_cells[style->start_cell + style->num_cells];
                        cell->x       =  curr_cell->x;
                        cell->area    = -curr_cell->area;
                        cell->cover   = -curr_cell->cover;
                        style->last_x =  curr_cell->x;
                        style->num_cells++;
                    }
                }
            }
            if(m_ast.size() > 1) break;
            ++m_scan_y;
        }
        ++m_scan_y;

        if(m_layer_order != layer_unsorted)
        {
            range_adaptor<pod_vector<unsigned> > ra(m_ast, 1, m_ast.size() - 1);
            if(m_layer_order == layer_direct) quick_sort(ra, unsigned_greater);
            else                              quick_sort(ra, unsigned_less);
        }

        return m_ast.size() - 1;
    }

     
     
    template<class Clip> 
    AGG_INLINE 
    unsigned rasterizer_compound_aa<Clip>::style(unsigned style_idx) const
    {
        return m_ast[style_idx + 1] + m_min_style - 1;
    }

     
    template<class Clip> 
    AGG_INLINE bool rasterizer_compound_aa<Clip>::navigate_scanline(int y)
    {
        m_outline.sort_cells();
        if(m_outline.total_cells() == 0) 
        {
            return false;
        }
        if(m_max_style < m_min_style)
        {
            return false;
        }
        if(y < m_outline.min_y() || y > m_outline.max_y()) 
        {
            return false;
        }
        m_scan_y = y;
        m_styles.allocate(m_max_style - m_min_style + 2, 128);
        return true;
    }
    
     
    template<class Clip> 
    bool rasterizer_compound_aa<Clip>::hit_test(int tx, int ty)
    {
        if(!navigate_scanline(ty)) 
        {
            return false;
        }

        unsigned num_styles = sweep_styles(); 
        if(num_styles <= 0)
        {
            return false;
        }

        scanline_hit_test sl(tx);
        sweep_scanline(sl, -1);
        return sl.hit();
    }

     
    template<class Clip> 
    cover_type* rasterizer_compound_aa<Clip>::allocate_cover_buffer(unsigned len)
    {
        m_cover_buf.allocate(len, 256);
        return &m_cover_buf[0];
    }

}



#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_RASTERIZER_OUTLINE_AA_INCLUDED
#define AGG_RASTERIZER_OUTLINE_AA_INCLUDED

#include <cmath>

namespace agg
{

     
    inline bool cmp_dist_start(int d) { return d > 0;  }
    inline bool cmp_dist_end(int d)   { return d <= 0; }



     
     
     
    struct line_aa_vertex
    {
        int x;
        int y;
        int len;

        line_aa_vertex() {}
        line_aa_vertex(int x_, int y_) :
            x(x_),
            y(y_),
            len(0)
        {
        }

        bool operator () (const line_aa_vertex& val)
        {
            double dx = val.x - x;
            double dy = val.y - y;
            return (len = uround(std::sqrt(dx * dx + dy * dy))) > 
                   (line_subpixel_scale + line_subpixel_scale / 2);
        }
    };


     
    enum outline_aa_join_e
    {
        outline_no_join,              
        outline_miter_join,           
        outline_round_join,           
        outline_miter_accurate_join   
    };

     
    template<class Renderer, class Coord=line_coord> class rasterizer_outline_aa
    {
    private:
         
        struct draw_vars
        {
            unsigned idx;
            int x1, y1, x2, y2;
            line_parameters curr, next;
            int lcurr, lnext;
            int xb1, yb1, xb2, yb2;
            unsigned flags;
        };

        void draw(draw_vars& dv, unsigned start, unsigned end);

    public:
        typedef line_aa_vertex                  vertex_type;
        typedef vertex_sequence<vertex_type, 6> vertex_storage_type;

        explicit rasterizer_outline_aa(Renderer& ren) : 
            m_ren(&ren), 
            m_line_join(ren.accurate_join_only() ? 
                            outline_miter_accurate_join : 
                            outline_round_join),
            m_round_cap(false),
            m_start_x(0),
            m_start_y(0)
        {}
        void attach(Renderer& ren) { m_ren = &ren; }

         
        void line_join(outline_aa_join_e join) 
        { 
            m_line_join = m_ren->accurate_join_only() ? 
                outline_miter_accurate_join : 
                join; 
        }
        bool line_join() const { return m_line_join; }

         
        void round_cap(bool v) { m_round_cap = v; }
        bool round_cap() const { return m_round_cap; }

         
        void move_to(int x, int y)
        {
            m_src_vertices.modify_last(vertex_type(m_start_x = x, m_start_y = y));
        }

         
        void line_to(int x, int y)
        {
            m_src_vertices.add(vertex_type(x, y));
        }

         
        void move_to_d(double x, double y)
        {
            move_to(Coord::conv(x), Coord::conv(y));
        }

         
        void line_to_d(double x, double y)
        {
            line_to(Coord::conv(x), Coord::conv(y));
        }

         
        void render(bool close_polygon);

         
        void add_vertex(double x, double y, unsigned cmd)
        {
            if(is_move_to(cmd)) 
            {
                render(false);
                move_to_d(x, y);
            }
            else 
            {
                if(is_end_poly(cmd))
                {
                    render(is_closed(cmd));
                    if(is_closed(cmd)) 
                    {
                        move_to(m_start_x, m_start_y);
                    }
                }
                else
                {
                    line_to_d(x, y);
                }
            }
        }

         
        template<class VertexSource>
        void add_path(VertexSource& vs, unsigned path_id=0)
        {
            double x;
            double y;

            unsigned cmd;
            vs.rewind(path_id);
            while(!is_stop(cmd = vs.vertex(&x, &y)))
            {
                add_vertex(x, y, cmd);
            }
            render(false);
        }


         
        template<class VertexSource, class ColorStorage, class PathId>
        void render_all_paths(VertexSource& vs, 
                              const ColorStorage& colors, 
                              const PathId& path_id,
                              unsigned num_paths)
        {
            for(unsigned i = 0; i < num_paths; i++)
            {
                m_ren->color(colors[i]);
                add_path(vs, path_id[i]);
            }
        }


         
        template<class Ctrl> void render_ctrl(Ctrl& c)
        {
            unsigned i;
            for(i = 0; i < c.num_paths(); i++)
            {
                m_ren->color(c.color(i));
                add_path(c, i);
            }
        }

    private:
        rasterizer_outline_aa(const rasterizer_outline_aa<Renderer, Coord>&);
        const rasterizer_outline_aa<Renderer, Coord>& operator = 
            (const rasterizer_outline_aa<Renderer, Coord>&);

        Renderer*           m_ren;
        vertex_storage_type m_src_vertices;
        outline_aa_join_e   m_line_join;
        bool                m_round_cap;
        int                 m_start_x;
        int                 m_start_y;
    };








     
    template<class Renderer, class Coord> 
    void rasterizer_outline_aa<Renderer, Coord>::draw(draw_vars& dv, 
                                                      unsigned start, 
                                                      unsigned end)
    {
        unsigned i;
        const vertex_storage_type::value_type* v;

        for(i = start; i < end; i++)
        {
            if(m_line_join == outline_round_join)
            {
                dv.xb1 = dv.curr.x1 + (dv.curr.y2 - dv.curr.y1); 
                dv.yb1 = dv.curr.y1 - (dv.curr.x2 - dv.curr.x1); 
                dv.xb2 = dv.curr.x2 + (dv.curr.y2 - dv.curr.y1); 
                dv.yb2 = dv.curr.y2 - (dv.curr.x2 - dv.curr.x1);
            }

            switch(dv.flags)
            {
            case 0: m_ren->line3(dv.curr, dv.xb1, dv.yb1, dv.xb2, dv.yb2); break;
            case 1: m_ren->line2(dv.curr, dv.xb2, dv.yb2); break;
            case 2: m_ren->line1(dv.curr, dv.xb1, dv.yb1); break;
            case 3: m_ren->line0(dv.curr); break;
            }

            if(m_line_join == outline_round_join && (dv.flags & 2) == 0)
            {
                m_ren->pie(dv.curr.x2, dv.curr.y2, 
                           dv.curr.x2 + (dv.curr.y2 - dv.curr.y1),
                           dv.curr.y2 - (dv.curr.x2 - dv.curr.x1),
                           dv.curr.x2 + (dv.next.y2 - dv.next.y1),
                           dv.curr.y2 - (dv.next.x2 - dv.next.x1));
            }

            dv.x1 = dv.x2;
            dv.y1 = dv.y2;
            dv.lcurr = dv.lnext;
            dv.lnext = m_src_vertices[dv.idx].len;

            ++dv.idx;
            if(dv.idx >= m_src_vertices.size()) dv.idx = 0; 

            v = &m_src_vertices[dv.idx];
            dv.x2 = v->x;
            dv.y2 = v->y;

            dv.curr = dv.next;
            dv.next = line_parameters(dv.x1, dv.y1, dv.x2, dv.y2, dv.lnext);
            dv.xb1 = dv.xb2;
            dv.yb1 = dv.yb2;

            switch(m_line_join)
            {
            case outline_no_join:
                dv.flags = 3;
                break;

            case outline_miter_join:
                dv.flags >>= 1;
                dv.flags |= ((dv.curr.diagonal_quadrant() == 
                              dv.next.diagonal_quadrant()) << 1);
                if((dv.flags & 2) == 0)
                {
                    bisectrix(dv.curr, dv.next, &dv.xb2, &dv.yb2);
                }
                break;

            case outline_round_join:
                dv.flags >>= 1;
                dv.flags |= ((dv.curr.diagonal_quadrant() == 
                              dv.next.diagonal_quadrant()) << 1);
                break;

            case outline_miter_accurate_join:
                dv.flags = 0;
                bisectrix(dv.curr, dv.next, &dv.xb2, &dv.yb2);
                break;
            }
        }
    }




     
    template<class Renderer, class Coord> 
    void rasterizer_outline_aa<Renderer, Coord>::render(bool close_polygon)
    {
        m_src_vertices.close(close_polygon);
        draw_vars dv;
        const vertex_storage_type::value_type* v;
        int x1;
        int y1;
        int x2;
        int y2;
        int lprev;

        if(close_polygon)
        {
            if(m_src_vertices.size() >= 3)
            {
                dv.idx = 2;

                v     = &m_src_vertices[m_src_vertices.size() - 1];
                x1    = v->x;
                y1    = v->y;
                lprev = v->len;

                v  = &m_src_vertices[0];
                x2 = v->x;
                y2 = v->y;
                dv.lcurr = v->len;
                line_parameters prev(x1, y1, x2, y2, lprev);

                v = &m_src_vertices[1];
                dv.x1    = v->x;
                dv.y1    = v->y;
                dv.lnext = v->len;
                dv.curr = line_parameters(x2, y2, dv.x1, dv.y1, dv.lcurr);

                v = &m_src_vertices[dv.idx];
                dv.x2 = v->x;
                dv.y2 = v->y;
                dv.next = line_parameters(dv.x1, dv.y1, dv.x2, dv.y2, dv.lnext);

                dv.xb1 = 0;
                dv.yb1 = 0;
                dv.xb2 = 0;
                dv.yb2 = 0;

                switch(m_line_join)
                {
                case outline_no_join:
                    dv.flags = 3;
                    break;

                case outline_miter_join:
                case outline_round_join:
                    dv.flags = 
                            (prev.diagonal_quadrant() == dv.curr.diagonal_quadrant()) |
                        ((dv.curr.diagonal_quadrant() == dv.next.diagonal_quadrant()) << 1);
                    break;

                case outline_miter_accurate_join:
                    dv.flags = 0;
                    break;
                }

                if((dv.flags & 1) == 0 && m_line_join != outline_round_join)
                {
                    bisectrix(prev, dv.curr, &dv.xb1, &dv.yb1);
                }

                if((dv.flags & 2) == 0 && m_line_join != outline_round_join)
                {
                    bisectrix(dv.curr, dv.next, &dv.xb2, &dv.yb2);
                }
                draw(dv, 0, m_src_vertices.size());
            }
        }
        else
        {
            switch(m_src_vertices.size())
            {
                case 0:
                case 1:
                    break;

                case 2:
                {
                    v     = &m_src_vertices[0];
                    x1    = v->x;
                    y1    = v->y;
                    lprev = v->len;
                    v     = &m_src_vertices[1];
                    x2    = v->x;
                    y2    = v->y;
                    line_parameters lp(x1, y1, x2, y2, lprev);
                    if(m_round_cap) 
                    {
                        m_ren->semidot(cmp_dist_start, x1, y1, x1 + (y2 - y1), y1 - (x2 - x1));
                    }
                    m_ren->line3(lp, 
                                 x1 + (y2 - y1), 
                                 y1 - (x2 - x1),
                                 x2 + (y2 - y1), 
                                 y2 - (x2 - x1));
                    if(m_round_cap) 
                    {
                        m_ren->semidot(cmp_dist_end, x2, y2, x2 + (y2 - y1), y2 - (x2 - x1));
                    }
                }
                break;

                case 3:
                {
                    int x3, y3;
                    int lnext;
                    v     = &m_src_vertices[0];
                    x1    = v->x;
                    y1    = v->y;
                    lprev = v->len;
                    v     = &m_src_vertices[1];
                    x2    = v->x;
                    y2    = v->y;
                    lnext = v->len;
                    v     = &m_src_vertices[2];
                    x3    = v->x;
                    y3    = v->y;
                    line_parameters lp1(x1, y1, x2, y2, lprev);
                    line_parameters lp2(x2, y2, x3, y3, lnext);

                    if(m_round_cap) 
                    {
                        m_ren->semidot(cmp_dist_start, x1, y1, x1 + (y2 - y1), y1 - (x2 - x1));
                    }

                    if(m_line_join == outline_round_join)
                    {
                        m_ren->line3(lp1, x1 + (y2 - y1), y1 - (x2 - x1), 
                                          x2 + (y2 - y1), y2 - (x2 - x1));

                        m_ren->pie(x2, y2, x2 + (y2 - y1), y2 - (x2 - x1),
                                           x2 + (y3 - y2), y2 - (x3 - x2));

                        m_ren->line3(lp2, x2 + (y3 - y2), y2 - (x3 - x2),
                                          x3 + (y3 - y2), y3 - (x3 - x2));
                    }
                    else
                    {
                        bisectrix(lp1, lp2, &dv.xb1, &dv.yb1);
                        m_ren->line3(lp1, x1 + (y2 - y1), y1 - (x2 - x1),
                                          dv.xb1,         dv.yb1);

                        m_ren->line3(lp2, dv.xb1,         dv.yb1,
                                          x3 + (y3 - y2), y3 - (x3 - x2));
                    }
                    if(m_round_cap) 
                    {
                        m_ren->semidot(cmp_dist_end, x3, y3, x3 + (y3 - y2), y3 - (x3 - x2));
                    }
                }
                break;

                default:
                {
                    dv.idx = 3;

                    v     = &m_src_vertices[0];
                    x1    = v->x;
                    y1    = v->y;
                    lprev = v->len;

                    v  = &m_src_vertices[1];
                    x2 = v->x;
                    y2 = v->y;
                    dv.lcurr = v->len;
                    line_parameters prev(x1, y1, x2, y2, lprev);

                    v = &m_src_vertices[2];
                    dv.x1    = v->x;
                    dv.y1    = v->y;
                    dv.lnext = v->len;
                    dv.curr = line_parameters(x2, y2, dv.x1, dv.y1, dv.lcurr);

                    v = &m_src_vertices[dv.idx];
                    dv.x2 = v->x;
                    dv.y2 = v->y;
                    dv.next = line_parameters(dv.x1, dv.y1, dv.x2, dv.y2, dv.lnext);

                    dv.xb1 = 0;
                    dv.yb1 = 0;
                    dv.xb2 = 0;
                    dv.yb2 = 0;

                    switch(m_line_join)
                    {
                    case outline_no_join:
                        dv.flags = 3;
                        break;

                    case outline_miter_join:
                    case outline_round_join:
                        dv.flags = 
                                (prev.diagonal_quadrant() == dv.curr.diagonal_quadrant()) |
                            ((dv.curr.diagonal_quadrant() == dv.next.diagonal_quadrant()) << 1);
                        break;

                    case outline_miter_accurate_join:
                        dv.flags = 0;
                        break;
                    }

                    if(m_round_cap) 
                    {
                        m_ren->semidot(cmp_dist_start, x1, y1, x1 + (y2 - y1), y1 - (x2 - x1));
                    }
                    if((dv.flags & 1) == 0)
                    {
                        if(m_line_join == outline_round_join)
                        {
                            m_ren->line3(prev, x1 + (y2 - y1), y1 - (x2 - x1),
                                               x2 + (y2 - y1), y2 - (x2 - x1));
                            m_ren->pie(prev.x2, prev.y2, 
                                       x2 + (y2 - y1), y2 - (x2 - x1),
                                       dv.curr.x1 + (dv.curr.y2 - dv.curr.y1), 
                                       dv.curr.y1 - (dv.curr.x2 - dv.curr.x1));
                        }
                        else
                        {
                            bisectrix(prev, dv.curr, &dv.xb1, &dv.yb1);
                            m_ren->line3(prev, x1 + (y2 - y1), y1 - (x2 - x1),
                                               dv.xb1,         dv.yb1);
                        }
                    }
                    else
                    {
                        m_ren->line1(prev, 
                                     x1 + (y2 - y1), 
                                     y1 - (x2 - x1));
                    }
                    if((dv.flags & 2) == 0 && m_line_join != outline_round_join)
                    {
                        bisectrix(dv.curr, dv.next, &dv.xb2, &dv.yb2);
                    }

                    draw(dv, 1, m_src_vertices.size() - 2);

                    if((dv.flags & 1) == 0)
                    {
                        if(m_line_join == outline_round_join)
                        {
                            m_ren->line3(dv.curr, 
                                         dv.curr.x1 + (dv.curr.y2 - dv.curr.y1), 
                                         dv.curr.y1 - (dv.curr.x2 - dv.curr.x1),
                                         dv.curr.x2 + (dv.curr.y2 - dv.curr.y1), 
                                         dv.curr.y2 - (dv.curr.x2 - dv.curr.x1));
                        }
                        else
                        {
                            m_ren->line3(dv.curr, dv.xb1, dv.yb1,
                                         dv.curr.x2 + (dv.curr.y2 - dv.curr.y1), 
                                         dv.curr.y2 - (dv.curr.x2 - dv.curr.x1));
                        }
                    }
                    else
                    {
                        m_ren->line2(dv.curr, 
                                     dv.curr.x2 + (dv.curr.y2 - dv.curr.y1), 
                                     dv.curr.y2 - (dv.curr.x2 - dv.curr.x1));
                    }
                    if(m_round_cap) 
                    {
                        m_ren->semidot(cmp_dist_end, dv.curr.x2, dv.curr.y2,
                                       dv.curr.x2 + (dv.curr.y2 - dv.curr.y1),
                                       dv.curr.y2 - (dv.curr.x2 - dv.curr.x1));
                    }

                }
                break;
            }
        }
        m_src_vertices.remove_all();
    }


}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_RASTERIZER_OUTLINE_INCLUDED
#define AGG_RASTERIZER_OUTLINE_INCLUDED


namespace agg
{
     
    template<class Renderer> class rasterizer_outline
    {
    public:
        explicit rasterizer_outline(Renderer& ren) : 
            m_ren(&ren), 
            m_start_x(0), 
            m_start_y(0), 
            m_vertices(0)
        {}
        void attach(Renderer& ren) { m_ren = &ren; }


         
        void move_to(int x, int y)
        {
            m_vertices = 1;
            m_ren->move_to(m_start_x = x, m_start_y = y);
        }

         
        void line_to(int x, int y)
        {
            ++m_vertices;
            m_ren->line_to(x, y);
        }

         
        void move_to_d(double x, double y)
        {
            move_to(m_ren->coord(x), m_ren->coord(y));
        }

         
        void line_to_d(double x, double y)
        {
            line_to(m_ren->coord(x), m_ren->coord(y));
        }

         
        void close()
        {
            if(m_vertices > 2)
            {
                line_to(m_start_x, m_start_y);
            }
            m_vertices = 0;
        }

         
        void add_vertex(double x, double y, unsigned cmd)
        {
            if(is_move_to(cmd)) 
            {
                move_to_d(x, y);
            }
            else 
            {
                if(is_end_poly(cmd))
                {
                    if(is_closed(cmd)) close();
                }
                else
                {
                    line_to_d(x, y);
                }
            }
        }


         
        template<class VertexSource>
        void add_path(VertexSource& vs, unsigned path_id=0)
        {
            double x;
            double y;

            unsigned cmd;
            vs.rewind(path_id);
            while(!is_stop(cmd = vs.vertex(&x, &y)))
            {
                add_vertex(x, y, cmd);
            }
        }


         
        template<class VertexSource, class ColorStorage, class PathId>
        void render_all_paths(VertexSource& vs, 
                              const ColorStorage& colors, 
                              const PathId& path_id,
                              unsigned num_paths)
        {
            for(unsigned i = 0; i < num_paths; i++)
            {
                m_ren->line_color(colors[i]);
                add_path(vs, path_id[i]);
            }
        }


         
        template<class Ctrl> void render_ctrl(Ctrl& c)
        {
            unsigned i;
            for(i = 0; i < c.num_paths(); i++)
            {
                m_ren->line_color(c.color(i));
                add_path(c, i);
            }
        }


    private:
        Renderer* m_ren;
        int       m_start_x;
        int       m_start_y;
        unsigned  m_vertices;
    };


}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_RASTERIZER_SCANLINE_AA_INCLUDED
#define AGG_RASTERIZER_SCANLINE_AA_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_RASTERIZER_SCANLINE_AA_NOGAMMA_INCLUDED
#define AGG_RASTERIZER_SCANLINE_AA_NOGAMMA_INCLUDED

#include <limits>


namespace agg
{


     
     
     
     
    struct cell_aa
    {
        int x;
        int y;
        int cover;
        int area;

        void initial()
        {
            x = std::numeric_limits<int>::max();
            y = std::numeric_limits<int>::max();
            cover = 0;
            area  = 0;
        }

        void style(const cell_aa&) {}

        int not_equal(int ex, int ey, const cell_aa&) const
        {
            return ((unsigned)ex - (unsigned)x) | ((unsigned)ey - (unsigned)y);
        }
    };


     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
    template<class Clip=rasterizer_sl_clip_int> class rasterizer_scanline_aa_nogamma
    {
        enum status
        {
            status_initial,
            status_move_to,
            status_line_to,
            status_closed
        };

    public:
        typedef Clip                      clip_type;
        typedef typename Clip::conv_type  conv_type;
        typedef typename Clip::coord_type coord_type;

        enum aa_scale_e
        {
            aa_shift  = 8,
            aa_scale  = 1 << aa_shift,
            aa_mask   = aa_scale - 1,
            aa_scale2 = aa_scale * 2,
            aa_mask2  = aa_scale2 - 1
        };

         
        rasterizer_scanline_aa_nogamma(unsigned cell_block_limit=1024) : 
            m_outline(cell_block_limit),
            m_clipper(),
            m_filling_rule(fill_non_zero),
            m_auto_close(true),
            m_start_x(0),
            m_start_y(0),
            m_status(status_initial)
        {
        }

         
        void reset(); 
        void reset_clipping();
        void clip_box(double x1, double y1, double x2, double y2);
        void filling_rule(filling_rule_e filling_rule);
        void auto_close(bool flag) { m_auto_close = flag; }

         
        unsigned apply_gamma(unsigned cover) const 
        { 
            return cover;
        }

         
        void move_to(int x, int y);
        void line_to(int x, int y);
        void move_to_d(double x, double y);
        void line_to_d(double x, double y);
        void close_polygon();
        void add_vertex(double x, double y, unsigned cmd);

        void edge(int x1, int y1, int x2, int y2);
        void edge_d(double x1, double y1, double x2, double y2);

         
        template<class VertexSource>
        void add_path(VertexSource& vs, unsigned path_id=0)
        {
            double x;
            double y;

            unsigned cmd;
            vs.rewind(path_id);
            if(m_outline.sorted()) reset();
            while(!is_stop(cmd = vs.vertex(&x, &y)))
            {
                add_vertex(x, y, cmd);
            }
        }
        
         
        int min_x() const { return m_outline.min_x(); }
        int min_y() const { return m_outline.min_y(); }
        int max_x() const { return m_outline.max_x(); }
        int max_y() const { return m_outline.max_y(); }

         
        void sort();
        bool rewind_scanlines();
        bool navigate_scanline(int y);

         
        AGG_INLINE unsigned calculate_alpha(int area) const
        {
            int cover = area >> (poly_subpixel_shift*2 + 1 - aa_shift);

            if(cover < 0) cover = -cover;
            if(m_filling_rule == fill_even_odd)
            {
                cover &= aa_mask2;
                if(cover > aa_scale)
                {
                    cover = aa_scale2 - cover;
                }
            }
            if(cover > aa_mask) cover = aa_mask;
            return cover;
        }

         
        template<class Scanline> bool sweep_scanline(Scanline& sl)
        {
            for(;;)
            {
                if(m_scan_y > m_outline.max_y()) return false;
                sl.reset_spans();
                unsigned num_cells = m_outline.scanline_num_cells(m_scan_y);
                const cell_aa* const* cells = m_outline.scanline_cells(m_scan_y);
                int cover = 0;

                while(num_cells)
                {
                    const cell_aa* cur_cell = *cells;
                    int x    = cur_cell->x;
                    int area = cur_cell->area;
                    unsigned alpha;

                    cover += cur_cell->cover;

                     
                    while(--num_cells)
                    {
                        cur_cell = *++cells;
                        if(cur_cell->x != x) break;
                        area  += cur_cell->area;
                        cover += cur_cell->cover;
                    }

                    if(area)
                    {
                        alpha = calculate_alpha((cover << (poly_subpixel_shift + 1)) - area);
                        if(alpha)
                        {
                            sl.add_cell(x, alpha);
                        }
                        x++;
                    }

                    if(num_cells && cur_cell->x > x)
                    {
                        alpha = calculate_alpha(cover << (poly_subpixel_shift + 1));
                        if(alpha)
                        {
                            sl.add_span(x, cur_cell->x - x, alpha);
                        }
                    }
                }
        
                if(sl.num_spans()) break;
                ++m_scan_y;
            }

            sl.finalize(m_scan_y);
            ++m_scan_y;
            return true;
        }

         
        bool hit_test(int tx, int ty);


    private:
         
         
        rasterizer_scanline_aa_nogamma(const rasterizer_scanline_aa_nogamma<Clip>&);
        const rasterizer_scanline_aa_nogamma<Clip>& 
        operator = (const rasterizer_scanline_aa_nogamma<Clip>&);

    private:
        rasterizer_cells_aa<cell_aa> m_outline;
        clip_type      m_clipper;
        filling_rule_e m_filling_rule;
        bool           m_auto_close;
        coord_type     m_start_x;
        coord_type     m_start_y;
        unsigned       m_status;
        int            m_scan_y;
    };












     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::reset() 
    { 
        m_outline.reset(); 
        m_status = status_initial;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::filling_rule(filling_rule_e filling_rule) 
    { 
        m_filling_rule = filling_rule; 
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::clip_box(double x1, double y1, 
                                                double x2, double y2)
    {
        reset();
        m_clipper.clip_box(conv_type::upscale(x1), conv_type::upscale(y1), 
                           conv_type::upscale(x2), conv_type::upscale(y2));
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::reset_clipping()
    {
        reset();
        m_clipper.reset_clipping();
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::close_polygon()
    {
        if(m_status == status_line_to)
        {
            m_clipper.line_to(m_outline, m_start_x, m_start_y);
            m_status = status_closed;
        }
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::move_to(int x, int y)
    {
        if(m_outline.sorted()) reset();
        if(m_auto_close) close_polygon();
        m_clipper.move_to(m_start_x = conv_type::downscale(x), 
                          m_start_y = conv_type::downscale(y));
        m_status = status_move_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::line_to(int x, int y)
    {
        m_clipper.line_to(m_outline, 
                          conv_type::downscale(x), 
                          conv_type::downscale(y));
        m_status = status_line_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::move_to_d(double x, double y) 
    { 
        if(m_outline.sorted()) reset();
        if(m_auto_close) close_polygon();
        m_clipper.move_to(m_start_x = conv_type::upscale(x), 
                          m_start_y = conv_type::upscale(y)); 
        m_status = status_move_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::line_to_d(double x, double y) 
    { 
        m_clipper.line_to(m_outline, 
                          conv_type::upscale(x), 
                          conv_type::upscale(y)); 
        m_status = status_line_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::add_vertex(double x, double y, unsigned cmd)
    {
        if(is_move_to(cmd)) 
        {
            move_to_d(x, y);
        }
        else 
        if(is_vertex(cmd))
        {
            line_to_d(x, y);
        }
        else
        if(is_close(cmd))
        {
            close_polygon();
        }
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::edge(int x1, int y1, int x2, int y2)
    {
        if(m_outline.sorted()) reset();
        m_clipper.move_to(conv_type::downscale(x1), conv_type::downscale(y1));
        m_clipper.line_to(m_outline, 
                          conv_type::downscale(x2), 
                          conv_type::downscale(y2));
        m_status = status_move_to;
    }
    
     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::edge_d(double x1, double y1, 
                                              double x2, double y2)
    {
        if(m_outline.sorted()) reset();
        m_clipper.move_to(conv_type::upscale(x1), conv_type::upscale(y1)); 
        m_clipper.line_to(m_outline, 
                          conv_type::upscale(x2), 
                          conv_type::upscale(y2)); 
        m_status = status_move_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa_nogamma<Clip>::sort()
    {
        if(m_auto_close) close_polygon();
        m_outline.sort_cells();
    }

     
    template<class Clip> 
    AGG_INLINE bool rasterizer_scanline_aa_nogamma<Clip>::rewind_scanlines()
    {
        if(m_auto_close) close_polygon();
        m_outline.sort_cells();
        if(m_outline.total_cells() == 0) 
        {
            return false;
        }
        m_scan_y = m_outline.min_y();
        return true;
    }


     
    template<class Clip> 
    AGG_INLINE bool rasterizer_scanline_aa_nogamma<Clip>::navigate_scanline(int y)
    {
        if(m_auto_close) close_polygon();
        m_outline.sort_cells();
        if(m_outline.total_cells() == 0 || 
           y < m_outline.min_y() || 
           y > m_outline.max_y()) 
        {
            return false;
        }
        m_scan_y = y;
        return true;
    }

     
    template<class Clip> 
    bool rasterizer_scanline_aa_nogamma<Clip>::hit_test(int tx, int ty)
    {
        if(!navigate_scanline(ty)) return false;
        scanline_hit_test sl(tx);
        sweep_scanline(sl);
        return sl.hit();
    }



}



#endif



namespace agg
{
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
    template<class Clip=rasterizer_sl_clip_int> class rasterizer_scanline_aa
    {
        enum status
        {
            status_initial,
            status_move_to,
            status_line_to,
            status_closed
        };

    public:
        typedef Clip                      clip_type;
        typedef typename Clip::conv_type  conv_type;
        typedef typename Clip::coord_type coord_type;

        enum aa_scale_e
        {
            aa_shift  = 8,
            aa_scale  = 1 << aa_shift,
            aa_mask   = aa_scale - 1,
            aa_scale2 = aa_scale * 2,
            aa_mask2  = aa_scale2 - 1
        };

         
        rasterizer_scanline_aa(unsigned cell_block_limit=1024) : 
            m_outline(cell_block_limit),
            m_clipper(),
            m_filling_rule(fill_non_zero),
            m_auto_close(true),
            m_start_x(0),
            m_start_y(0),
            m_status(status_initial)
        {
            int i;
            for(i = 0; i < aa_scale; i++) m_gamma[i] = i;
        }

         
        template<class GammaF> 
        rasterizer_scanline_aa(const GammaF& gamma_function, unsigned cell_block_limit) : 
            m_outline(cell_block_limit),
            m_clipper(m_outline),
            m_filling_rule(fill_non_zero),
            m_auto_close(true),
            m_start_x(0),
            m_start_y(0),
            m_status(status_initial)
        {
            gamma(gamma_function);
        }

         
        void reset(); 
        void reset_clipping();
        void clip_box(double x1, double y1, double x2, double y2);
        void filling_rule(filling_rule_e filling_rule);
        void auto_close(bool flag) { m_auto_close = flag; }

         
        template<class GammaF> void gamma(const GammaF& gamma_function)
        { 
            int i;
            for(i = 0; i < aa_scale; i++)
            {
                m_gamma[i] = uround(gamma_function(double(i) / (double)aa_mask) * (double)aa_mask);
            }
        }

         
        unsigned apply_gamma(unsigned cover) const 
        { 
            return m_gamma[cover]; 
        }

         
        void move_to(int x, int y);
        void line_to(int x, int y);
        void move_to_d(double x, double y);
        void line_to_d(double x, double y);
        void close_polygon();
        void add_vertex(double x, double y, unsigned cmd);

        void edge(int x1, int y1, int x2, int y2);
        void edge_d(double x1, double y1, double x2, double y2);

         
        template<class VertexSource>
        void add_path(VertexSource& vs, unsigned path_id=0)
        {
            double x = 0;
            double y = 0;

            unsigned cmd;
            vs.rewind(path_id);
            if(m_outline.sorted()) reset();
            while(!is_stop(cmd = vs.vertex(&x, &y)))
            {
                add_vertex(x, y, cmd);
            }
        }
        
         
        int min_x() const { return m_outline.min_x(); }
        int min_y() const { return m_outline.min_y(); }
        int max_x() const { return m_outline.max_x(); }
        int max_y() const { return m_outline.max_y(); }

         
        void sort();
        bool rewind_scanlines();
        bool navigate_scanline(int y);

         
        AGG_INLINE unsigned calculate_alpha(int area) const
        {
            int cover = area >> (poly_subpixel_shift*2 + 1 - aa_shift);

            if(cover < 0) cover = -cover;
            if(m_filling_rule == fill_even_odd)
            {
                cover &= aa_mask2;
                if(cover > aa_scale)
                {
                    cover = aa_scale2 - cover;
                }
            }
            if(cover > aa_mask) cover = aa_mask;
            return m_gamma[cover];
        }

         
        template<class Scanline> bool sweep_scanline(Scanline& sl)
        {
            for(;;)
            {
                if(m_scan_y > m_outline.max_y()) return false;
                sl.reset_spans();
                unsigned num_cells = m_outline.scanline_num_cells(m_scan_y);
                const cell_aa* const* cells = m_outline.scanline_cells(m_scan_y);
                int cover = 0;

                while(num_cells)
                {
                    const cell_aa* cur_cell = *cells;
                    int x    = cur_cell->x;
                    int area = cur_cell->area;
                    unsigned alpha;

                    cover += cur_cell->cover;

                     
                    while(--num_cells)
                    {
                        cur_cell = *++cells;
                        if(cur_cell->x != x) break;
                        area  += cur_cell->area;
                        cover += cur_cell->cover;
                    }

                    if(area)
                    {
                        alpha = calculate_alpha((cover << (poly_subpixel_shift + 1)) - area);
                        if(alpha)
                        {
                            sl.add_cell(x, alpha);
                        }
                        x++;
                    }

                    if(num_cells && cur_cell->x > x)
                    {
                        alpha = calculate_alpha(cover << (poly_subpixel_shift + 1));
                        if(alpha)
                        {
                            sl.add_span(x, cur_cell->x - x, alpha);
                        }
                    }
                }
        
                if(sl.num_spans()) break;
                ++m_scan_y;
            }

            sl.finalize(m_scan_y);
            ++m_scan_y;
            return true;
        }

         
        bool hit_test(int tx, int ty);


    private:
         
         
        rasterizer_scanline_aa(const rasterizer_scanline_aa<Clip>&);
        const rasterizer_scanline_aa<Clip>& 
        operator = (const rasterizer_scanline_aa<Clip>&);

    private:
        rasterizer_cells_aa<cell_aa> m_outline;
        clip_type      m_clipper;
        int            m_gamma[aa_scale];
        filling_rule_e m_filling_rule;
        bool           m_auto_close;
        coord_type     m_start_x;
        coord_type     m_start_y;
        unsigned       m_status;
        int            m_scan_y;
    };












     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::reset() 
    { 
        m_outline.reset(); 
        m_status = status_initial;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::filling_rule(filling_rule_e filling_rule) 
    { 
        m_filling_rule = filling_rule; 
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::clip_box(double x1, double y1, 
                                                double x2, double y2)
    {
        reset();
        m_clipper.clip_box(conv_type::upscale(x1), conv_type::upscale(y1), 
                           conv_type::upscale(x2), conv_type::upscale(y2));
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::reset_clipping()
    {
        reset();
        m_clipper.reset_clipping();
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::close_polygon()
    {
        if(m_status == status_line_to)
        {
            m_clipper.line_to(m_outline, m_start_x, m_start_y);
            m_status = status_closed;
        }
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::move_to(int x, int y)
    {
        if(m_outline.sorted()) reset();
        if(m_auto_close) close_polygon();
        m_clipper.move_to(m_start_x = conv_type::downscale(x), 
                          m_start_y = conv_type::downscale(y));
        m_status = status_move_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::line_to(int x, int y)
    {
        m_clipper.line_to(m_outline, 
                          conv_type::downscale(x), 
                          conv_type::downscale(y));
        m_status = status_line_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::move_to_d(double x, double y) 
    { 
        if(m_outline.sorted()) reset();
        if(m_auto_close) close_polygon();
        m_clipper.move_to(m_start_x = conv_type::upscale(x), 
                          m_start_y = conv_type::upscale(y)); 
        m_status = status_move_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::line_to_d(double x, double y) 
    { 
        m_clipper.line_to(m_outline, 
                          conv_type::upscale(x), 
                          conv_type::upscale(y)); 
        m_status = status_line_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::add_vertex(double x, double y, unsigned cmd)
    {
        if(is_move_to(cmd)) 
        {
            move_to_d(x, y);
        }
        else 
        if(is_vertex(cmd))
        {
            line_to_d(x, y);
        }
        else
        if(is_close(cmd))
        {
            close_polygon();
        }
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::edge(int x1, int y1, int x2, int y2)
    {
        if(m_outline.sorted()) reset();
        m_clipper.move_to(conv_type::downscale(x1), conv_type::downscale(y1));
        m_clipper.line_to(m_outline, 
                          conv_type::downscale(x2), 
                          conv_type::downscale(y2));
        m_status = status_move_to;
    }
    
     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::edge_d(double x1, double y1, 
                                              double x2, double y2)
    {
        if(m_outline.sorted()) reset();
        m_clipper.move_to(conv_type::upscale(x1), conv_type::upscale(y1)); 
        m_clipper.line_to(m_outline, 
                          conv_type::upscale(x2), 
                          conv_type::upscale(y2)); 
        m_status = status_move_to;
    }

     
    template<class Clip> 
    void rasterizer_scanline_aa<Clip>::sort()
    {
        if(m_auto_close) close_polygon();
        m_outline.sort_cells();
    }

     
    template<class Clip> 
    AGG_INLINE bool rasterizer_scanline_aa<Clip>::rewind_scanlines()
    {
        if(m_auto_close) close_polygon();
        m_outline.sort_cells();
        if(m_outline.total_cells() == 0) 
        {
            return false;
        }
        m_scan_y = m_outline.min_y();
        return true;
    }


     
    template<class Clip> 
    AGG_INLINE bool rasterizer_scanline_aa<Clip>::navigate_scanline(int y)
    {
        if(m_auto_close) close_polygon();
        m_outline.sort_cells();
        if(m_outline.total_cells() == 0 || 
           y < m_outline.min_y() || 
           y > m_outline.max_y()) 
        {
            return false;
        }
        m_scan_y = y;
        return true;
    }

     
    template<class Clip> 
    bool rasterizer_scanline_aa<Clip>::hit_test(int tx, int ty)
    {
        if(!navigate_scanline(ty)) return false;
        scanline_hit_test sl(tx);
        sweep_scanline(sl);
        return sl.hit();
    }



}



#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_RENDERER_BASE_INCLUDED
#define AGG_RENDERER_BASE_INCLUDED


namespace agg
{

     
    template<class PixelFormat> class renderer_base
    {
    public:
        typedef PixelFormat pixfmt_type;
        typedef typename pixfmt_type::color_type color_type;
        typedef typename pixfmt_type::row_data row_data;

         
        renderer_base() : m_ren(0), m_clip_box(1, 1, 0, 0) {}
        explicit renderer_base(pixfmt_type& ren) :
            m_ren(&ren),
            m_clip_box(0, 0, ren.width() - 1, ren.height() - 1)
        {}
        void attach(pixfmt_type& ren)
        {
            m_ren = &ren;
            m_clip_box = rect_i(0, 0, ren.width() - 1, ren.height() - 1);
        }

         
        const pixfmt_type& ren() const { return *m_ren;  }
        pixfmt_type& ren() { return *m_ren;  }
          
         
        unsigned width()  const { return m_ren->width();  }
        unsigned height() const { return m_ren->height(); }

         
        bool clip_box(int x1, int y1, int x2, int y2)
        {
            rect_i cb(x1, y1, x2, y2);
            cb.normalize();
            if(cb.clip(rect_i(0, 0, width() - 1, height() - 1)))
            {
                m_clip_box = cb;
                return true;
            }
            m_clip_box.x1 = 1;
            m_clip_box.y1 = 1;
            m_clip_box.x2 = 0;
            m_clip_box.y2 = 0;
            return false;
        }

         
        void reset_clipping(bool visibility)
        {
            if(visibility)
            {
                m_clip_box.x1 = 0;
                m_clip_box.y1 = 0;
                m_clip_box.x2 = width() - 1;
                m_clip_box.y2 = height() - 1;
            }
            else
            {
                m_clip_box.x1 = 1;
                m_clip_box.y1 = 1;
                m_clip_box.x2 = 0;
                m_clip_box.y2 = 0;
            }
        }

         
        void clip_box_naked(int x1, int y1, int x2, int y2)
        {
            m_clip_box.x1 = x1;
            m_clip_box.y1 = y1;
            m_clip_box.x2 = x2;
            m_clip_box.y2 = y2;
        }

         
        bool inbox(int x, int y) const
        {
            return x >= m_clip_box.x1 && y >= m_clip_box.y1 &&
                   x <= m_clip_box.x2 && y <= m_clip_box.y2;
        }

         
        const rect_i& clip_box() const { return m_clip_box;    }
        int           xmin()     const { return m_clip_box.x1; }
        int           ymin()     const { return m_clip_box.y1; }
        int           xmax()     const { return m_clip_box.x2; }
        int           ymax()     const { return m_clip_box.y2; }

         
        const rect_i& bounding_clip_box() const { return m_clip_box;    }
        int           bounding_xmin()     const { return m_clip_box.x1; }
        int           bounding_ymin()     const { return m_clip_box.y1; }
        int           bounding_xmax()     const { return m_clip_box.x2; }
        int           bounding_ymax()     const { return m_clip_box.y2; }

         
        void clear(const color_type& c)
        {
            unsigned y;
            if(width())
            {
                for(y = 0; y < height(); y++)
                {
                    m_ren->copy_hline(0, y, width(), c);
                }
            }
        }
          

         
        void fill(const color_type& c)
        {
            unsigned y;
            if(width())
            {
                for(y = 0; y < height(); y++)
                {
                    m_ren->blend_hline(0, y, width(), c, cover_mask);
                }
            }
        }
        
         
        void copy_pixel(int x, int y, const color_type& c)
        {
            if(inbox(x, y))
            {
                m_ren->copy_pixel(x, y, c);
            }
        }

         
        void blend_pixel(int x, int y, const color_type& c, cover_type cover)
        {
            if(inbox(x, y))
            {
                m_ren->blend_pixel(x, y, c, cover);
            }
        }

         
        color_type pixel(int x, int y) const
        {
            return inbox(x, y) ? 
                   m_ren->pixel(x, y) :
                   color_type::no_color();
        }

         
        void copy_hline(int x1, int y, int x2, const color_type& c)
        {
            if(x1 > x2) { int t = x2; x2 = x1; x1 = t; }
            if(y  > ymax()) return;
            if(y  < ymin()) return;
            if(x1 > xmax()) return;
            if(x2 < xmin()) return;

            if(x1 < xmin()) x1 = xmin();
            if(x2 > xmax()) x2 = xmax();

            m_ren->copy_hline(x1, y, x2 - x1 + 1, c);
        }

         
        void copy_vline(int x, int y1, int y2, const color_type& c)
        {
            if(y1 > y2) { int t = y2; y2 = y1; y1 = t; }
            if(x  > xmax()) return;
            if(x  < xmin()) return;
            if(y1 > ymax()) return;
            if(y2 < ymin()) return;

            if(y1 < ymin()) y1 = ymin();
            if(y2 > ymax()) y2 = ymax();

            m_ren->copy_vline(x, y1, y2 - y1 + 1, c);
        }

         
        void blend_hline(int x1, int y, int x2, 
                         const color_type& c, cover_type cover)
        {
            if(x1 > x2) { int t = x2; x2 = x1; x1 = t; }
            if(y  > ymax()) return;
            if(y  < ymin()) return;
            if(x1 > xmax()) return;
            if(x2 < xmin()) return;

            if(x1 < xmin()) x1 = xmin();
            if(x2 > xmax()) x2 = xmax();

            m_ren->blend_hline(x1, y, x2 - x1 + 1, c, cover);
        }

         
        void blend_vline(int x, int y1, int y2, 
                         const color_type& c, cover_type cover)
        {
            if(y1 > y2) { int t = y2; y2 = y1; y1 = t; }
            if(x  > xmax()) return;
            if(x  < xmin()) return;
            if(y1 > ymax()) return;
            if(y2 < ymin()) return;

            if(y1 < ymin()) y1 = ymin();
            if(y2 > ymax()) y2 = ymax();

            m_ren->blend_vline(x, y1, y2 - y1 + 1, c, cover);
        }


         
        void copy_bar(int x1, int y1, int x2, int y2, const color_type& c)
        {
            rect_i rc(x1, y1, x2, y2);
            rc.normalize();
            if(rc.clip(clip_box()))
            {
                int y;
                for(y = rc.y1; y <= rc.y2; y++)
                {
                    m_ren->copy_hline(rc.x1, y, unsigned(rc.x2 - rc.x1 + 1), c);
                }
            }
        }

         
        void blend_bar(int x1, int y1, int x2, int y2, 
                       const color_type& c, cover_type cover)
        {
            rect_i rc(x1, y1, x2, y2);
            rc.normalize();
            if(rc.clip(clip_box()))
            {
                int y;
                for(y = rc.y1; y <= rc.y2; y++)
                {
                    m_ren->blend_hline(rc.x1,
                                       y,
                                       unsigned(rc.x2 - rc.x1 + 1), 
                                       c, 
                                       cover);
                }
            }
        }

         
        void blend_solid_hspan(int x, int y, int len, 
                               const color_type& c, 
                               const cover_type* covers)
        {
            if(y > ymax()) return;
            if(y < ymin()) return;

            if(x < xmin())
            {
                len -= xmin() - x;
                if(len <= 0) return;
                covers += xmin() - x;
                x = xmin();
            }
            if(x + len > xmax())
            {
                len = xmax() - x + 1;
                if(len <= 0) return;
            }
            m_ren->blend_solid_hspan(x, y, len, c, covers);
        }

         
        void blend_solid_vspan(int x, int y, int len, 
                               const color_type& c, 
                               const cover_type* covers)
        {
            if(x > xmax()) return;
            if(x < xmin()) return;

            if(y < ymin())
            {
                len -= ymin() - y;
                if(len <= 0) return;
                covers += ymin() - y;
                y = ymin();
            }
            if(y + len > ymax())
            {
                len = ymax() - y + 1;
                if(len <= 0) return;
            }
            m_ren->blend_solid_vspan(x, y, len, c, covers);
        }


         
        void copy_color_hspan(int x, int y, int len, const color_type* colors)
        {
            if(y > ymax()) return;
            if(y < ymin()) return;

            if(x < xmin())
            {
                int d = xmin() - x;
                len -= d;
                if(len <= 0) return;
                colors += d;
                x = xmin();
            }
            if(x + len > xmax())
            {
                len = xmax() - x + 1;
                if(len <= 0) return;
            }
            m_ren->copy_color_hspan(x, y, len, colors);
        }


         
        void copy_color_vspan(int x, int y, int len, const color_type* colors)
        {
            if(x > xmax()) return;
            if(x < xmin()) return;

            if(y < ymin())
            {
                int d = ymin() - y;
                len -= d;
                if(len <= 0) return;
                colors += d;
                y = ymin();
            }
            if(y + len > ymax())
            {
                len = ymax() - y + 1;
                if(len <= 0) return;
            }
            m_ren->copy_color_vspan(x, y, len, colors);
        }


         
        void blend_color_hspan(int x, int y, int len, 
                               const color_type* colors, 
                               const cover_type* covers,
                               cover_type cover = agg::cover_full)
        {
            if(y > ymax()) return;
            if(y < ymin()) return;

            if(x < xmin())
            {
                int d = xmin() - x;
                len -= d;
                if(len <= 0) return;
                if(covers) covers += d;
                colors += d;
                x = xmin();
            }
            if(x + len > xmax())
            {
                len = xmax() - x + 1;
                if(len <= 0) return;
            }
            m_ren->blend_color_hspan(x, y, len, colors, covers, cover);
        }

         
        void blend_color_vspan(int x, int y, int len, 
                               const color_type* colors, 
                               const cover_type* covers,
                               cover_type cover = agg::cover_full)
        {
            if(x > xmax()) return;
            if(x < xmin()) return;

            if(y < ymin())
            {
                int d = ymin() - y;
                len -= d;
                if(len <= 0) return;
                if(covers) covers += d;
                colors += d;
                y = ymin();
            }
            if(y + len > ymax())
            {
                len = ymax() - y + 1;
                if(len <= 0) return;
            }
            m_ren->blend_color_vspan(x, y, len, colors, covers, cover);
        }

         
        rect_i clip_rect_area(rect_i& dst, rect_i& src, int wsrc, int hsrc) const
        {
            rect_i rc(0,0,0,0);
            rect_i cb = clip_box();
            ++cb.x2;
            ++cb.y2;

            if(src.x1 < 0)
            {
                dst.x1 -= src.x1;
                src.x1 = 0;
            }
            if(src.y1 < 0)
            {
                dst.y1 -= src.y1;
                src.y1 = 0;
            }

            if(src.x2 > wsrc) src.x2 = wsrc;
            if(src.y2 > hsrc) src.y2 = hsrc;

            if(dst.x1 < cb.x1)
            {
                src.x1 += cb.x1 - dst.x1;
                dst.x1 = cb.x1;
            }
            if(dst.y1 < cb.y1)
            {
                src.y1 += cb.y1 - dst.y1;
                dst.y1 = cb.y1;
            }

            if(dst.x2 > cb.x2) dst.x2 = cb.x2;
            if(dst.y2 > cb.y2) dst.y2 = cb.y2;

            rc.x2 = dst.x2 - dst.x1;
            rc.y2 = dst.y2 - dst.y1;

            if(rc.x2 > src.x2 - src.x1) rc.x2 = src.x2 - src.x1;
            if(rc.y2 > src.y2 - src.y1) rc.y2 = src.y2 - src.y1;
            return rc;
        }

         
        template<class RenBuf>
        void copy_from(const RenBuf& src, 
                       const rect_i* rect_src_ptr = 0, 
                       int dx = 0, 
                       int dy = 0)
        {
            rect_i rsrc(0, 0, src.width(), src.height());
            if(rect_src_ptr)
            {
                rsrc.x1 = rect_src_ptr->x1; 
                rsrc.y1 = rect_src_ptr->y1;
                rsrc.x2 = rect_src_ptr->x2 + 1;
                rsrc.y2 = rect_src_ptr->y2 + 1;
            }

             
             

             
            rect_i rdst(rsrc.x1 + dx, rsrc.y1 + dy, rsrc.x2 + dx, rsrc.y2 + dy);

            rect_i rc = clip_rect_area(rdst, rsrc, src.width(), src.height());

            if(rc.x2 > 0)
            {
                int incy = 1;
                if(rdst.y1 > rsrc.y1)
                {
                    rsrc.y1 += rc.y2 - 1;
                    rdst.y1 += rc.y2 - 1;
                    incy = -1;
                }
                while(rc.y2 > 0)
                {
                    m_ren->copy_from(src, 
                                     rdst.x1, rdst.y1,
                                     rsrc.x1, rsrc.y1,
                                     rc.x2);
                    rdst.y1 += incy;
                    rsrc.y1 += incy;
                    --rc.y2;
                }
            }
        }

         
        template<class SrcPixelFormatRenderer>
        void blend_from(const SrcPixelFormatRenderer& src, 
                        const rect_i* rect_src_ptr = 0, 
                        int dx = 0, 
                        int dy = 0,
                        cover_type cover = agg::cover_full)
        {
            rect_i rsrc(0, 0, src.width(), src.height());
            if(rect_src_ptr)
            {
                rsrc.x1 = rect_src_ptr->x1; 
                rsrc.y1 = rect_src_ptr->y1;
                rsrc.x2 = rect_src_ptr->x2 + 1;
                rsrc.y2 = rect_src_ptr->y2 + 1;
            }

             
             

             
            rect_i rdst(rsrc.x1 + dx, rsrc.y1 + dy, rsrc.x2 + dx, rsrc.y2 + dy);
            rect_i rc = clip_rect_area(rdst, rsrc, src.width(), src.height());

            if(rc.x2 > 0)
            {
                int incy = 1;
                if(rdst.y1 > rsrc.y1)
                {
                    rsrc.y1 += rc.y2 - 1;
                    rdst.y1 += rc.y2 - 1;
                    incy = -1;
                }
                while(rc.y2 > 0)
                {
                    typename SrcPixelFormatRenderer::row_data rw = src.row(rsrc.y1);
                    if(rw.ptr)
                    {
                        int x1src = rsrc.x1;
                        int x1dst = rdst.x1;
                        int len   = rc.x2;
                        if(rw.x1 > x1src)
                        {
                            x1dst += rw.x1 - x1src;
                            len   -= rw.x1 - x1src;
                            x1src  = rw.x1;
                        }
                        if(len > 0)
                        {
                            if(x1src + len-1 > rw.x2)
                            {
                                len -= x1src + len - rw.x2 - 1;
                            }
                            if(len > 0)
                            {
                                m_ren->blend_from(src,
                                                  x1dst, rdst.y1,
                                                  x1src, rsrc.y1,
                                                  len,
                                                  cover);
                            }
                        }
                    }
                    rdst.y1 += incy;
                    rsrc.y1 += incy;
                    --rc.y2;
                }
            }
        }

         
        template<class SrcPixelFormatRenderer>
        void blend_from_color(const SrcPixelFormatRenderer& src, 
                              const color_type& color,
                              const rect_i* rect_src_ptr = 0, 
                              int dx = 0, 
                              int dy = 0,
                              cover_type cover = agg::cover_full)
        {
            rect_i rsrc(0, 0, src.width(), src.height());
            if(rect_src_ptr)
            {
                rsrc.x1 = rect_src_ptr->x1; 
                rsrc.y1 = rect_src_ptr->y1;
                rsrc.x2 = rect_src_ptr->x2 + 1;
                rsrc.y2 = rect_src_ptr->y2 + 1;
            }

             
             

             
            rect_i rdst(rsrc.x1 + dx, rsrc.y1 + dy, rsrc.x2 + dx, rsrc.y2 + dy);
            rect_i rc = clip_rect_area(rdst, rsrc, src.width(), src.height());

            if(rc.x2 > 0)
            {
                int incy = 1;
                if(rdst.y1 > rsrc.y1)
                {
                    rsrc.y1 += rc.y2 - 1;
                    rdst.y1 += rc.y2 - 1;
                    incy = -1;
                }
                while(rc.y2 > 0)
                {
                    typename SrcPixelFormatRenderer::row_data rw = src.row(rsrc.y1);
                    if(rw.ptr)
                    {
                        int x1src = rsrc.x1;
                        int x1dst = rdst.x1;
                        int len   = rc.x2;
                        if(rw.x1 > x1src)
                        {
                            x1dst += rw.x1 - x1src;
                            len   -= rw.x1 - x1src;
                            x1src  = rw.x1;
                        }
                        if(len > 0)
                        {
                            if(x1src + len-1 > rw.x2)
                            {
                                len -= x1src + len - rw.x2 - 1;
                            }
                            if(len > 0)
                            {
                                m_ren->blend_from_color(src,
                                                        color,
                                                        x1dst, rdst.y1,
                                                        x1src, rsrc.y1,
                                                        len,
                                                        cover);
                            }
                        }
                    }
                    rdst.y1 += incy;
                    rsrc.y1 += incy;
                    --rc.y2;
                }
            }
        }

         
        template<class SrcPixelFormatRenderer>
        void blend_from_lut(const SrcPixelFormatRenderer& src, 
                            const color_type* color_lut,
                            const rect_i* rect_src_ptr = 0, 
                            int dx = 0, 
                            int dy = 0,
                            cover_type cover = agg::cover_full)
        {
            rect_i rsrc(0, 0, src.width(), src.height());
            if(rect_src_ptr)
            {
                rsrc.x1 = rect_src_ptr->x1; 
                rsrc.y1 = rect_src_ptr->y1;
                rsrc.x2 = rect_src_ptr->x2 + 1;
                rsrc.y2 = rect_src_ptr->y2 + 1;
            }

             
             

             
            rect_i rdst(rsrc.x1 + dx, rsrc.y1 + dy, rsrc.x2 + dx, rsrc.y2 + dy);
            rect_i rc = clip_rect_area(rdst, rsrc, src.width(), src.height());

            if(rc.x2 > 0)
            {
                int incy = 1;
                if(rdst.y1 > rsrc.y1)
                {
                    rsrc.y1 += rc.y2 - 1;
                    rdst.y1 += rc.y2 - 1;
                    incy = -1;
                }
                while(rc.y2 > 0)
                {
                    typename SrcPixelFormatRenderer::row_data rw = src.row(rsrc.y1);
                    if(rw.ptr)
                    {
                        int x1src = rsrc.x1;
                        int x1dst = rdst.x1;
                        int len   = rc.x2;
                        if(rw.x1 > x1src)
                        {
                            x1dst += rw.x1 - x1src;
                            len   -= rw.x1 - x1src;
                            x1src  = rw.x1;
                        }
                        if(len > 0)
                        {
                            if(x1src + len-1 > rw.x2)
                            {
                                len -= x1src + len - rw.x2 - 1;
                            }
                            if(len > 0)
                            {
                                m_ren->blend_from_lut(src,
                                                      color_lut,
                                                      x1dst, rdst.y1,
                                                      x1src, rsrc.y1,
                                                      len,
                                                      cover);
                            }
                        }
                    }
                    rdst.y1 += incy;
                    rsrc.y1 += incy;
                    --rc.y2;
                }
            }
        }

    private:
        pixfmt_type* m_ren;
        rect_i       m_clip_box;
    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_RENDERER_MARKERS_INCLUDED
#define AGG_RENDERER_MARKERS_INCLUDED

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_RENDERER_PRIMITIVES_INCLUDED
#define AGG_RENDERER_PRIMITIVES_INCLUDED


namespace agg
{
     
    template<class BaseRenderer> class renderer_primitives
    {
    public:
        typedef BaseRenderer base_ren_type;
        typedef typename base_ren_type::color_type color_type;

         
        explicit renderer_primitives(base_ren_type& ren) :
            m_ren(&ren),
            m_fill_color(),
            m_line_color(),
            m_curr_x(0),
            m_curr_y(0)
        {}
        void attach(base_ren_type& ren) { m_ren = &ren; }

         
        static int coord(double c) 
        { 
            return iround(c * (double)line_bresenham_interpolator::subpixel_scale); 
        }

         
        void fill_color(const color_type& c) { m_fill_color = c; }
        void line_color(const color_type& c) { m_line_color = c; }
        const color_type& fill_color() const { return m_fill_color; }
        const color_type& line_color() const { return m_line_color; }

         
        void rectangle(int x1, int y1, int x2, int y2)
        {
            m_ren->blend_hline(x1,   y1,   x2-1, m_line_color, cover_full);
            m_ren->blend_vline(x2,   y1,   y2-1, m_line_color, cover_full);
            m_ren->blend_hline(x1+1, y2,   x2,   m_line_color, cover_full);
            m_ren->blend_vline(x1,   y1+1, y2,   m_line_color, cover_full);
        }

         
        void solid_rectangle(int x1, int y1, int x2, int y2)
        {
            m_ren->blend_bar(x1, y1, x2, y2, m_fill_color, cover_full);
        }

         
        void outlined_rectangle(int x1, int y1, int x2, int y2) 
        {
            rectangle(x1, y1, x2, y2);
            m_ren->blend_bar(x1+1, y1+1, x2-1, y2-1, m_fill_color, cover_full);
        }

         
        void ellipse(int x, int y, int rx, int ry)
        {
            ellipse_bresenham_interpolator ei(rx, ry);
            int dx = 0;
            int dy = -ry;
            do
            {
                dx += ei.dx();
                dy += ei.dy();
                m_ren->blend_pixel(x + dx, y + dy, m_line_color, cover_full);
                m_ren->blend_pixel(x + dx, y - dy, m_line_color, cover_full);
                m_ren->blend_pixel(x - dx, y - dy, m_line_color, cover_full);
                m_ren->blend_pixel(x - dx, y + dy, m_line_color, cover_full);
                ++ei;
            }
            while(dy < 0);
        }

         
        void solid_ellipse(int x, int y, int rx, int ry)
        {
            ellipse_bresenham_interpolator ei(rx, ry);
            int dx = 0;
            int dy = -ry;
            int dy0 = dy;
            int dx0 = dx;

            do
            {
                dx += ei.dx();
                dy += ei.dy();

                if(dy != dy0)
                {
                    m_ren->blend_hline(x-dx0, y+dy0, x+dx0, m_fill_color, cover_full);
                    m_ren->blend_hline(x-dx0, y-dy0, x+dx0, m_fill_color, cover_full);
                }
                dx0 = dx;
                dy0 = dy;
                ++ei;
            }
            while(dy < 0);
            m_ren->blend_hline(x-dx0, y+dy0, x+dx0, m_fill_color, cover_full);
        }

         
        void outlined_ellipse(int x, int y, int rx, int ry)
        {
            ellipse_bresenham_interpolator ei(rx, ry);
            int dx = 0;
            int dy = -ry;

            do
            {
                dx += ei.dx();
                dy += ei.dy();

                m_ren->blend_pixel(x + dx, y + dy, m_line_color, cover_full);
                m_ren->blend_pixel(x + dx, y - dy, m_line_color, cover_full);
                m_ren->blend_pixel(x - dx, y - dy, m_line_color, cover_full);
                m_ren->blend_pixel(x - dx, y + dy, m_line_color, cover_full);

                if(ei.dy() && dx)
                {
                   m_ren->blend_hline(x-dx+1, y+dy, x+dx-1, m_fill_color, cover_full);
                   m_ren->blend_hline(x-dx+1, y-dy, x+dx-1, m_fill_color, cover_full);
                }
                ++ei;
            }
            while(dy < 0);
        }

         
        void line(int x1, int y1, int x2, int y2, bool last=false)
        {
            line_bresenham_interpolator li(x1, y1, x2, y2);

            unsigned len = li.len();
            if(len == 0)
            {
                if(last)
                {
                    m_ren->blend_pixel(li.line_lr(x1), li.line_lr(y1), m_line_color, cover_full);
                }
                return;
            }

            if(last) ++len;

            if(li.is_ver())
            {
                do
                {
                    m_ren->blend_pixel(li.x2(), li.y1(), m_line_color, cover_full);
                    li.vstep();
                }
                while(--len);
            }
            else
            {
                do
                {
                    m_ren->blend_pixel(li.x1(), li.y2(), m_line_color, cover_full);
                    li.hstep();
                }
                while(--len);
            }
        }

         
        void move_to(int x, int y)
        {
            m_curr_x = x;
            m_curr_y = y;
        }

         
        void line_to(int x, int y, bool last=false)
        {
            line(m_curr_x, m_curr_y, x, y, last);
            m_curr_x = x;
            m_curr_y = y;
        }

         
        const base_ren_type& ren() const { return *m_ren; }        
        base_ren_type& ren() { return *m_ren; }        

         
        const rendering_buffer& rbuf() const { return m_ren->rbuf(); }
        rendering_buffer& rbuf() { return m_ren->rbuf(); }

    private:
        base_ren_type* m_ren;
        color_type m_fill_color;
        color_type m_line_color;
        int m_curr_x;
        int m_curr_y;
    };

}

#endif

namespace agg
{

     
    enum marker_e
    {
        marker_square,
        marker_diamond,
        marker_circle,
        marker_crossed_circle,
        marker_semiellipse_left,
        marker_semiellipse_right,
        marker_semiellipse_up,
        marker_semiellipse_down,
        marker_triangle_left,
        marker_triangle_right,
        marker_triangle_up,
        marker_triangle_down,
        marker_four_rays,
        marker_cross,
        marker_x,
        marker_dash,
        marker_dot,
        marker_pixel,
        
        end_of_markers
    };



     
    template<class BaseRenderer> class renderer_markers :
    public renderer_primitives<BaseRenderer>
    {
    public:
        typedef renderer_primitives<BaseRenderer> base_type;
        typedef BaseRenderer base_ren_type;
        typedef typename base_ren_type::color_type color_type;

         
        renderer_markers(base_ren_type& rbuf) :
            base_type(rbuf)
        {}

         
        bool visible(int x, int y, int r) const
        {
            rect_i rc(x-r, y-r, x+y, y+r);
            return rc.clip(base_type::ren().bounding_clip_box());  
        }

         
        void square(int x, int y, int r)
        {
            if(visible(x, y, r)) 
            {  
                if(r) base_type::outlined_rectangle(x-r, y-r, x+r, y+r);
                else  base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
            }
        }

         
        void diamond(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int dy = -r;
                    int dx = 0;
                    do
                    {
                        base_type::ren().blend_pixel(x - dx, y + dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dx, y + dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dx, y - dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dx, y - dy, base_type::line_color(), cover_full);
                        
                        if(dx)
                        {
                            base_type::ren().blend_hline(x-dx+1, y+dy, x+dx-1, base_type::fill_color(), cover_full);
                            base_type::ren().blend_hline(x-dx+1, y-dy, x+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++dy;
                        ++dx;
                    }
                    while(dy <= 0);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }

         
        void circle(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r) base_type::outlined_ellipse(x, y, r, r);
                else  base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
            }
        }



         
        void crossed_circle(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    base_type::outlined_ellipse(x, y, r, r);
                    int r6 = r + (r >> 1);
                    if(r <= 2) r6++;
                    r >>= 1;
                    base_type::ren().blend_hline(x-r6, y, x-r,  base_type::line_color(), cover_full);
                    base_type::ren().blend_hline(x+r,  y, x+r6, base_type::line_color(), cover_full);
                    base_type::ren().blend_vline(x, y-r6, y-r,  base_type::line_color(), cover_full);
                    base_type::ren().blend_vline(x, y+r,  y+r6, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void semiellipse_left(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int r8 = r * 4 / 5;
                    int dy = -r;
                    int dx = 0;
                    ellipse_bresenham_interpolator ei(r * 3 / 5, r+r8);
                    do
                    {
                        dx += ei.dx();
                        dy += ei.dy();
                        
                        base_type::ren().blend_pixel(x + dy, y + dx, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dy, y - dx, base_type::line_color(), cover_full);
                        
                        if(ei.dy() && dx)
                        {
                            base_type::ren().blend_vline(x+dy, y-dx+1, y+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++ei;
                    }
                    while(dy < r8);
                    base_type::ren().blend_vline(x+dy, y-dx, y+dx, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void semiellipse_right(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int r8 = r * 4 / 5;
                    int dy = -r;
                    int dx = 0;
                    ellipse_bresenham_interpolator ei(r * 3 / 5, r+r8);
                    do
                    {
                        dx += ei.dx();
                        dy += ei.dy();
                        
                        base_type::ren().blend_pixel(x - dy, y + dx, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dy, y - dx, base_type::line_color(), cover_full);
                        
                        if(ei.dy() && dx)
                        {
                            base_type::ren().blend_vline(x-dy, y-dx+1, y+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++ei;
                    }
                    while(dy < r8);
                    base_type::ren().blend_vline(x-dy, y-dx, y+dx, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void semiellipse_up(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int r8 = r * 4 / 5;
                    int dy = -r;
                    int dx = 0;
                    ellipse_bresenham_interpolator ei(r * 3 / 5, r+r8);
                    do
                    {
                        dx += ei.dx();
                        dy += ei.dy();
                        
                        base_type::ren().blend_pixel(x + dx, y - dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dx, y - dy, base_type::line_color(), cover_full);
                        
                        if(ei.dy() && dx)
                        {
                            base_type::ren().blend_hline(x-dx+1, y-dy, x+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++ei;
                    }
                    while(dy < r8);
                    base_type::ren().blend_hline(x-dx, y-dy-1, x+dx, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void semiellipse_down(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int r8 = r * 4 / 5;
                    int dy = -r;
                    int dx = 0;
                    ellipse_bresenham_interpolator ei(r * 3 / 5, r+r8);
                    do
                    {
                        dx += ei.dx();
                        dy += ei.dy();
                        
                        base_type::ren().blend_pixel(x + dx, y + dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dx, y + dy, base_type::line_color(), cover_full);
                        
                        if(ei.dy() && dx)
                        {
                            base_type::ren().blend_hline(x-dx+1, y+dy, x+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++ei;
                    }
                    while(dy < r8);
                    base_type::ren().blend_hline(x-dx, y+dy+1, x+dx, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void triangle_left(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int dy = -r;
                    int dx = 0;
                    int flip = 0;
                    int r6 = r * 3 / 5;
                    do
                    {
                        base_type::ren().blend_pixel(x + dy, y - dx, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dy, y + dx, base_type::line_color(), cover_full);
                        
                        if(dx)
                        {
                            base_type::ren().blend_vline(x+dy, y-dx+1, y+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++dy;
                        dx += flip;
                        flip ^= 1;
                    }
                    while(dy < r6);
                    base_type::ren().blend_vline(x+dy, y-dx, y+dx, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void triangle_right(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int dy = -r;
                    int dx = 0;
                    int flip = 0;
                    int r6 = r * 3 / 5;
                    do
                    {
                        base_type::ren().blend_pixel(x - dy, y - dx, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dy, y + dx, base_type::line_color(), cover_full);
                        
                        if(dx)
                        {
                            base_type::ren().blend_vline(x-dy, y-dx+1, y+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++dy;
                        dx += flip;
                        flip ^= 1;
                    }
                    while(dy < r6);
                    base_type::ren().blend_vline(x-dy, y-dx, y+dx, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void triangle_up(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int dy = -r;
                    int dx = 0;
                    int flip = 0;
                    int r6 = r * 3 / 5;
                    do
                    {
                        base_type::ren().blend_pixel(x - dx, y - dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dx, y - dy, base_type::line_color(), cover_full);
                        
                        if(dx)
                        {
                            base_type::ren().blend_hline(x-dx+1, y-dy, x+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++dy;
                        dx += flip;
                        flip ^= 1;
                    }
                    while(dy < r6);
                    base_type::ren().blend_hline(x-dx, y-dy, x+dx, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void triangle_down(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int dy = -r;
                    int dx = 0;
                    int flip = 0;
                    int r6 = r * 3 / 5;
                    do
                    {
                        base_type::ren().blend_pixel(x - dx, y + dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dx, y + dy, base_type::line_color(), cover_full);
                        
                        if(dx)
                        {
                            base_type::ren().blend_hline(x-dx+1, y+dy, x+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++dy;
                        dx += flip;
                        flip ^= 1;
                    }
                    while(dy < r6);
                    base_type::ren().blend_hline(x-dx, y+dy, x+dx, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void four_rays(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int dy = -r;
                    int dx = 0;
                    int flip = 0;
                    int r3 = -(r / 3);
                    do
                    {
                        base_type::ren().blend_pixel(x - dx, y + dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dx, y + dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dx, y - dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dx, y - dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dy, y - dx, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dy, y + dx, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dy, y - dx, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dy, y + dx, base_type::line_color(), cover_full);
                        
                        if(dx)
                        {
                            base_type::ren().blend_hline(x-dx+1, y+dy,   x+dx-1, base_type::fill_color(), cover_full);
                            base_type::ren().blend_hline(x-dx+1, y-dy,   x+dx-1, base_type::fill_color(), cover_full);
                            base_type::ren().blend_vline(x+dy,   y-dx+1, y+dx-1, base_type::fill_color(), cover_full);
                            base_type::ren().blend_vline(x-dy,   y-dx+1, y+dx-1, base_type::fill_color(), cover_full);
                        }
                        ++dy;
                        dx += flip;
                        flip ^= 1;
                    }
                    while(dy <= r3);
                    base_type::solid_rectangle(x+r3+1, y+r3+1, x-r3-1, y-r3-1);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }


         
        void cross(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    base_type::ren().blend_vline(x, y-r, y+r, base_type::line_color(), cover_full);
                    base_type::ren().blend_hline(x-r, y, x+r, base_type::line_color(), cover_full);
                }
                else
                {
                    base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
                }
            }
        }
        
        
         
        void xing(int x, int y, int r)
        {
            if(visible(x, y, r))
            {
                if(r)
                {
                    int dy = -r * 7 / 10;
                    do
                    {
                        base_type::ren().blend_pixel(x + dy, y + dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dy, y + dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x + dy, y - dy, base_type::line_color(), cover_full);
                        base_type::ren().blend_pixel(x - dy, y - dy, base_type::line_color(), cover_full);
                        ++dy;
                    }
                    while(dy < 0);
                }
                base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
            }
        }
        
        
         
        void dash(int x, int y, int r)
        {
            if(visible(x, y, r)) 
            {
                if(r) base_type::ren().blend_hline(x-r, y, x+r, base_type::line_color(), cover_full);
                else  base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
            }
        }
        
        
         
        void dot(int x, int y, int r)
        {
            if(visible(x, y, r)) 
            {
                if(r) base_type::solid_ellipse(x, y, r, r);
                else  base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
            }
        }
        
         
        void pixel(int x, int y, int)
        {
            base_type::ren().blend_pixel(x, y, base_type::fill_color(), cover_full);
        }
        
         
        void marker(int x, int y, int r, marker_e type)
        {
            switch(type)
            {
                case marker_square:            square(x, y, r);            break;
                case marker_diamond:           diamond(x, y, r);           break;
                case marker_circle:            circle(x, y, r);            break;
                case marker_crossed_circle:    crossed_circle(x, y, r);    break;
                case marker_semiellipse_left:  semiellipse_left(x, y, r);  break;
                case marker_semiellipse_right: semiellipse_right(x, y, r); break;
                case marker_semiellipse_up:    semiellipse_up(x, y, r);    break;
                case marker_semiellipse_down:  semiellipse_down(x, y, r);  break;
                case marker_triangle_left:     triangle_left(x, y, r);     break;
                case marker_triangle_right:    triangle_right(x, y, r);    break;
                case marker_triangle_up:       triangle_up(x, y, r);       break;
                case marker_triangle_down:     triangle_down(x, y, r);     break;
                case marker_four_rays:         four_rays(x, y, r);         break;
                case marker_cross:             cross(x, y, r);             break;
                case marker_x:                 xing(x, y, r);              break;
                case marker_dash:              dash(x, y, r);              break;
                case marker_dot:               dot(x, y, r);               break;
                case marker_pixel:             pixel(x, y, r);             break;
                case end_of_markers: break;
            }
        }


         
        template<class T>
        void markers(int n, const T* x, const T* y, T r, marker_e type)
        {
            if(n <= 0) return;
            if(r == 0)
            {
                do
                {
                    base_type::ren().blend_pixel(int(*x), int(*y), base_type::fill_color(), cover_full);
                    ++x;
                    ++y;
                }
                while(--n);
                return;
            }
            
            switch(type)
            {
                case marker_square:            do { square           (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_diamond:           do { diamond          (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_circle:            do { circle           (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_crossed_circle:    do { crossed_circle   (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_semiellipse_left:  do { semiellipse_left (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_semiellipse_right: do { semiellipse_right(int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_semiellipse_up:    do { semiellipse_up   (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_semiellipse_down:  do { semiellipse_down (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_triangle_left:     do { triangle_left    (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_triangle_right:    do { triangle_right   (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_triangle_up:       do { triangle_up      (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_triangle_down:     do { triangle_down    (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_four_rays:         do { four_rays        (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_cross:             do { cross            (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_x:                 do { xing             (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_dash:              do { dash             (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_dot:               do { dot              (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case marker_pixel:             do { pixel            (int(*x), int(*y), int(r)); ++x; ++y; } while(--n); break;
                case end_of_markers: break;
            }                                                                                  
        }
        
         
        template<class T>
        void markers(int n, const T* x, const T* y, const T* r, marker_e type)
        {
            if(n <= 0) return;
            switch(type)
            {
                case marker_square:            do { square           (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_diamond:           do { diamond          (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_circle:            do { circle           (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_crossed_circle:    do { crossed_circle   (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_semiellipse_left:  do { semiellipse_left (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_semiellipse_right: do { semiellipse_right(int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_semiellipse_up:    do { semiellipse_up   (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_semiellipse_down:  do { semiellipse_down (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_triangle_left:     do { triangle_left    (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_triangle_right:    do { triangle_right   (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_triangle_up:       do { triangle_up      (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_triangle_down:     do { triangle_down    (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_four_rays:         do { four_rays        (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_cross:             do { cross            (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_x:                 do { xing             (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_dash:              do { dash             (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_dot:               do { dot              (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case marker_pixel:             do { pixel            (int(*x), int(*y), int(*r)); ++x; ++y; ++r; } while(--n); break;
                case end_of_markers: break;
            }                                                                                  
        }
        
         
        template<class T>
        void markers(int n, const T* x, const T* y, const T* r, const color_type* fc, marker_e type)
        {
            if(n <= 0) return;
            switch(type)
            {
                case marker_square:            do { base_type::fill_color(*fc); square           (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_diamond:           do { base_type::fill_color(*fc); diamond          (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_circle:            do { base_type::fill_color(*fc); circle           (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_crossed_circle:    do { base_type::fill_color(*fc); crossed_circle   (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_semiellipse_left:  do { base_type::fill_color(*fc); semiellipse_left (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_semiellipse_right: do { base_type::fill_color(*fc); semiellipse_right(int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_semiellipse_up:    do { base_type::fill_color(*fc); semiellipse_up   (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_semiellipse_down:  do { base_type::fill_color(*fc); semiellipse_down (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_triangle_left:     do { base_type::fill_color(*fc); triangle_left    (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_triangle_right:    do { base_type::fill_color(*fc); triangle_right   (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_triangle_up:       do { base_type::fill_color(*fc); triangle_up      (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_triangle_down:     do { base_type::fill_color(*fc); triangle_down    (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_four_rays:         do { base_type::fill_color(*fc); four_rays        (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_cross:             do { base_type::fill_color(*fc); cross            (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_x:                 do { base_type::fill_color(*fc); xing             (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_dash:              do { base_type::fill_color(*fc); dash             (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_dot:               do { base_type::fill_color(*fc); dot              (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case marker_pixel:             do { base_type::fill_color(*fc); pixel            (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; } while(--n); break;
                case end_of_markers: break;
            }
        }
        
         
        template<class T>
        void markers(int n, const T* x, const T* y, const T* r, const color_type* fc, const color_type* lc, marker_e type)
        {
            if(n <= 0) return;
            switch(type)
            {
                case marker_square:            do { base_type::fill_color(*fc); base_type::line_color(*lc); square           (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_diamond:           do { base_type::fill_color(*fc); base_type::line_color(*lc); diamond          (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_circle:            do { base_type::fill_color(*fc); base_type::line_color(*lc); circle           (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_crossed_circle:    do { base_type::fill_color(*fc); base_type::line_color(*lc); crossed_circle   (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_semiellipse_left:  do { base_type::fill_color(*fc); base_type::line_color(*lc); semiellipse_left (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_semiellipse_right: do { base_type::fill_color(*fc); base_type::line_color(*lc); semiellipse_right(int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_semiellipse_up:    do { base_type::fill_color(*fc); base_type::line_color(*lc); semiellipse_up   (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_semiellipse_down:  do { base_type::fill_color(*fc); base_type::line_color(*lc); semiellipse_down (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_triangle_left:     do { base_type::fill_color(*fc); base_type::line_color(*lc); triangle_left    (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_triangle_right:    do { base_type::fill_color(*fc); base_type::line_color(*lc); triangle_right   (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_triangle_up:       do { base_type::fill_color(*fc); base_type::line_color(*lc); triangle_up      (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_triangle_down:     do { base_type::fill_color(*fc); base_type::line_color(*lc); triangle_down    (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_four_rays:         do { base_type::fill_color(*fc); base_type::line_color(*lc); four_rays        (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_cross:             do { base_type::fill_color(*fc); base_type::line_color(*lc); cross            (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_x:                 do { base_type::fill_color(*fc); base_type::line_color(*lc); xing             (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_dash:              do { base_type::fill_color(*fc); base_type::line_color(*lc); dash             (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_dot:               do { base_type::fill_color(*fc); base_type::line_color(*lc); dot              (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case marker_pixel:             do { base_type::fill_color(*fc); base_type::line_color(*lc); pixel            (int(*x), int(*y), int(*r)); ++x; ++y; ++r; ++fc; ++lc; } while(--n); break;
                case end_of_markers: break;
            }
        }
    };

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_RENDERER_MCLIP_INCLUDED
#define AGG_RENDERER_MCLIP_INCLUDED


namespace agg
{

     
    template<class PixelFormat> class renderer_mclip
    {
    public:
        typedef PixelFormat pixfmt_type;
        typedef typename pixfmt_type::color_type color_type;
        typedef typename pixfmt_type::row_data row_data;
        typedef renderer_base<pixfmt_type> base_ren_type;

         
        explicit renderer_mclip(pixfmt_type& pixf) :
            m_ren(pixf),
            m_curr_cb(0),
            m_bounds(m_ren.xmin(), m_ren.ymin(), m_ren.xmax(), m_ren.ymax())
        {}
        void attach(pixfmt_type& pixf)
        {
            m_ren.attach(pixf);
            reset_clipping(true);
        }
          
         
        const pixfmt_type& ren() const { return m_ren.ren();  }
        pixfmt_type& ren() { return m_ren.ren();  }

         
        unsigned width()  const { return m_ren.width();  }
        unsigned height() const { return m_ren.height(); }

         
        const rect_i& clip_box() const { return m_ren.clip_box(); }
        int           xmin()     const { return m_ren.xmin(); }
        int           ymin()     const { return m_ren.ymin(); }
        int           xmax()     const { return m_ren.xmax(); }
        int           ymax()     const { return m_ren.ymax(); }

         
        const rect_i& bounding_clip_box() const { return m_bounds;    }
        int           bounding_xmin()     const { return m_bounds.x1; }
        int           bounding_ymin()     const { return m_bounds.y1; }
        int           bounding_xmax()     const { return m_bounds.x2; }
        int           bounding_ymax()     const { return m_bounds.y2; }

         
        void first_clip_box() 
        {
            m_curr_cb = 0;
            if(m_clip.size())
            {
                const rect_i& cb = m_clip[0];
                m_ren.clip_box_naked(cb.x1, cb.y1, cb.x2, cb.y2);
            }
        }

         
        bool next_clip_box() 
        { 
            if(++m_curr_cb < m_clip.size())
            {
                const rect_i& cb = m_clip[m_curr_cb];
                m_ren.clip_box_naked(cb.x1, cb.y1, cb.x2, cb.y2);
                return true;
            }
            return false; 
        }

         
        void reset_clipping(bool visibility)
        {
            m_ren.reset_clipping(visibility);
            m_clip.remove_all();
            m_curr_cb = 0;
            m_bounds = m_ren.clip_box();
        }
        
         
        void add_clip_box(int x1, int y1, int x2, int y2)
        {
            rect_i cb(x1, y1, x2, y2); 
            cb.normalize();
            if(cb.clip(rect_i(0, 0, width() - 1, height() - 1)))
            {
                m_clip.add(cb);
                if(cb.x1 < m_bounds.x1) m_bounds.x1 = cb.x1;
                if(cb.y1 < m_bounds.y1) m_bounds.y1 = cb.y1;
                if(cb.x2 > m_bounds.x2) m_bounds.x2 = cb.x2;
                if(cb.y2 > m_bounds.y2) m_bounds.y2 = cb.y2;
            }
        }

         
        void clear(const color_type& c)
        {
            m_ren.clear(c);
        }
          
         
        void copy_pixel(int x, int y, const color_type& c)
        {
            first_clip_box();
            do
            {
                if(m_ren.inbox(x, y))
                {
                    m_ren.ren().copy_pixel(x, y, c);
                    break;
                }
            }
            while(next_clip_box());
        }

         
        void blend_pixel(int x, int y, const color_type& c, cover_type cover)
        {
            first_clip_box();
            do
            {
                if(m_ren.inbox(x, y))
                {
                    m_ren.ren().blend_pixel(x, y, c, cover);
                    break;
                }
            }
            while(next_clip_box());
        }

         
        color_type pixel(int x, int y) const
        {
            first_clip_box();
            do
            {
                if(m_ren.inbox(x, y))
                {
                    return m_ren.ren().pixel(x, y);
                }
            }
            while(next_clip_box());
            return color_type::no_color();
        }

         
        void copy_hline(int x1, int y, int x2, const color_type& c)
        {
            first_clip_box();
            do
            {
                m_ren.copy_hline(x1, y, x2, c);
            }
            while(next_clip_box());
        }

         
        void copy_vline(int x, int y1, int y2, const color_type& c)
        {
            first_clip_box();
            do
            {
                m_ren.copy_vline(x, y1, y2, c);
            }
            while(next_clip_box());
        }

         
        void blend_hline(int x1, int y, int x2, 
                         const color_type& c, cover_type cover)
        {
            first_clip_box();
            do
            {
                m_ren.blend_hline(x1, y, x2, c, cover);
            }
            while(next_clip_box());
        }

         
        void blend_vline(int x, int y1, int y2, 
                         const color_type& c, cover_type cover)
        {
            first_clip_box();
            do
            {
                m_ren.blend_vline(x, y1, y2, c, cover);
            }
            while(next_clip_box());
        }

         
        void copy_bar(int x1, int y1, int x2, int y2, const color_type& c)
        {
            first_clip_box();
            do
            {
                m_ren.copy_bar(x1, y1, x2, y2, c);
            }
            while(next_clip_box());
        }

         
        void blend_bar(int x1, int y1, int x2, int y2, 
                       const color_type& c, cover_type cover)
        {
            first_clip_box();
            do
            {
                m_ren.blend_bar(x1, y1, x2, y2, c, cover);
            }
            while(next_clip_box());
        }

         
        void blend_solid_hspan(int x, int y, int len, 
                               const color_type& c, const cover_type* covers)
        {
            first_clip_box();
            do
            {
                m_ren.blend_solid_hspan(x, y, len, c, covers);
            }
            while(next_clip_box());
        }

         
        void blend_solid_vspan(int x, int y, int len, 
                               const color_type& c, const cover_type* covers)
        {
            first_clip_box();
            do
            {
                m_ren.blend_solid_vspan(x, y, len, c, covers);
            }
            while(next_clip_box());
        }


         
        void copy_color_hspan(int x, int y, int len, const color_type* colors)
        {
            first_clip_box();
            do
            {
                m_ren.copy_color_hspan(x, y, len, colors);
            }
            while(next_clip_box());
        }

         
        void blend_color_hspan(int x, int y, int len, 
                               const color_type* colors, 
                               const cover_type* covers,
                               cover_type cover = cover_full)
        {
            first_clip_box();
            do
            {
                m_ren.blend_color_hspan(x, y, len, colors, covers, cover);
            }
            while(next_clip_box());
        }

         
        void blend_color_vspan(int x, int y, int len, 
                               const color_type* colors, 
                               const cover_type* covers,
                               cover_type cover = cover_full)
        {
            first_clip_box();
            do
            {
                m_ren.blend_color_vspan(x, y, len, colors, covers, cover);
            }
            while(next_clip_box());
        }

         
        void copy_from(const rendering_buffer& from, 
                       const rect_i* rc=0, 
                       int x_to=0, 
                       int y_to=0)
        {
            first_clip_box();
            do
            {
                m_ren.copy_from(from, rc, x_to, y_to);
            }
            while(next_clip_box());
        }

         
        template<class SrcPixelFormatRenderer>
        void blend_from(const SrcPixelFormatRenderer& src, 
                        const rect_i* rect_src_ptr = 0, 
                        int dx = 0, 
                        int dy = 0,
                        cover_type cover = cover_full)
        {
            first_clip_box();
            do
            {
                m_ren.blend_from(src, rect_src_ptr, dx, dy, cover);
            }
            while(next_clip_box());
        }

        
    private:
        renderer_mclip(const renderer_mclip<PixelFormat>&);
        const renderer_mclip<PixelFormat>& 
            operator = (const renderer_mclip<PixelFormat>&);

        base_ren_type          m_ren;
        pod_bvector<rect_i, 4> m_clip;
        unsigned               m_curr_cb;
        rect_i                 m_bounds;
    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_RENDERER_OUTLINE_AA_INCLUDED
#define AGG_RENDERER_OUTLINE_AA_INCLUDED

#include <cstdlib>

namespace agg
{

     
    class distance_interpolator0
    {
    public:
         
        distance_interpolator0() {}
        distance_interpolator0(int x1, int y1, int x2, int y2, int x, int y) :
            m_dx(line_mr(x2) - line_mr(x1)),
            m_dy(line_mr(y2) - line_mr(y1)),
            m_dist((line_mr(x + line_subpixel_scale/2) - line_mr(x2)) * m_dy - 
                   (line_mr(y + line_subpixel_scale/2) - line_mr(y2)) * m_dx)
        {
            m_dx <<= line_mr_subpixel_shift;
            m_dy <<= line_mr_subpixel_shift;
        }

         
        void inc_x() { m_dist += m_dy; }
        int  dist() const { return m_dist; }

    private:
         
        int m_dx;
        int m_dy;
        int m_dist;
    };

     
    class distance_interpolator00
    {
    public:
         
        distance_interpolator00() {}
        distance_interpolator00(int xc, int yc, 
                                int x1, int y1, int x2, int y2, 
                                int x,  int y) :
            m_dx1(line_mr(x1) - line_mr(xc)),
            m_dy1(line_mr(y1) - line_mr(yc)),
            m_dx2(line_mr(x2) - line_mr(xc)),
            m_dy2(line_mr(y2) - line_mr(yc)),
            m_dist1((line_mr(x + line_subpixel_scale/2) - line_mr(x1)) * m_dy1 - 
                    (line_mr(y + line_subpixel_scale/2) - line_mr(y1)) * m_dx1),
            m_dist2((line_mr(x + line_subpixel_scale/2) - line_mr(x2)) * m_dy2 - 
                    (line_mr(y + line_subpixel_scale/2) - line_mr(y2)) * m_dx2)
        {
            m_dx1 <<= line_mr_subpixel_shift;
            m_dy1 <<= line_mr_subpixel_shift;
            m_dx2 <<= line_mr_subpixel_shift;
            m_dy2 <<= line_mr_subpixel_shift;
        }

         
        void inc_x() { m_dist1 += m_dy1; m_dist2 += m_dy2; }
        int  dist1() const { return m_dist1; }
        int  dist2() const { return m_dist2; }

    private:
         
        int m_dx1;
        int m_dy1;
        int m_dx2;
        int m_dy2;
        int m_dist1;
        int m_dist2;
    };

     
    class distance_interpolator1
    {
    public:
         
        distance_interpolator1() {}
        distance_interpolator1(int x1, int y1, int x2, int y2, int x, int y) :
            m_dx(x2 - x1),
            m_dy(y2 - y1),
            m_dist(iround(double(x + line_subpixel_scale/2 - x2) * double(m_dy) - 
                          double(y + line_subpixel_scale/2 - y2) * double(m_dx)))
        {
            m_dx <<= line_subpixel_shift;
            m_dy <<= line_subpixel_shift;
        }

         
        void inc_x() { m_dist += m_dy; }
        void dec_x() { m_dist -= m_dy; }
        void inc_y() { m_dist -= m_dx; }
        void dec_y() { m_dist += m_dx; }

         
        void inc_x(int dy)
        {
            m_dist += m_dy; 
            if(dy > 0) m_dist -= m_dx; 
            if(dy < 0) m_dist += m_dx; 
        }

         
        void dec_x(int dy)
        {
            m_dist -= m_dy; 
            if(dy > 0) m_dist -= m_dx; 
            if(dy < 0) m_dist += m_dx; 
        }

         
        void inc_y(int dx)
        {
            m_dist -= m_dx; 
            if(dx > 0) m_dist += m_dy; 
            if(dx < 0) m_dist -= m_dy; 
        }

        void dec_y(int dx)
         
        {
            m_dist += m_dx; 
            if(dx > 0) m_dist += m_dy; 
            if(dx < 0) m_dist -= m_dy; 
        }

         
        int dist() const { return m_dist; }
        int dx()   const { return m_dx;   }
        int dy()   const { return m_dy;   }

    private:
         
        int m_dx;
        int m_dy;
        int m_dist;
    };





     
    class distance_interpolator2
    {
    public:
         
        distance_interpolator2() {}
        distance_interpolator2(int x1, int y1, int x2, int y2,
                               int sx, int sy, int x,  int y) :
            m_dx(x2 - x1),
            m_dy(y2 - y1),
            m_dx_start(line_mr(sx) - line_mr(x1)),
            m_dy_start(line_mr(sy) - line_mr(y1)),

            m_dist(iround(double(x + line_subpixel_scale/2 - x2) * double(m_dy) - 
                          double(y + line_subpixel_scale/2 - y2) * double(m_dx))),

            m_dist_start((line_mr(x + line_subpixel_scale/2) - line_mr(sx)) * m_dy_start - 
                         (line_mr(y + line_subpixel_scale/2) - line_mr(sy)) * m_dx_start)
        {
            m_dx       <<= line_subpixel_shift;
            m_dy       <<= line_subpixel_shift;
            m_dx_start <<= line_mr_subpixel_shift;
            m_dy_start <<= line_mr_subpixel_shift;
        }

        distance_interpolator2(int x1, int y1, int x2, int y2,
                               int ex, int ey, int x,  int y, int) :
            m_dx(x2 - x1),
            m_dy(y2 - y1),
            m_dx_start(line_mr(ex) - line_mr(x2)),
            m_dy_start(line_mr(ey) - line_mr(y2)),

            m_dist(iround(double(x + line_subpixel_scale/2 - x2) * double(m_dy) - 
                          double(y + line_subpixel_scale/2 - y2) * double(m_dx))),

            m_dist_start((line_mr(x + line_subpixel_scale/2) - line_mr(ex)) * m_dy_start - 
                         (line_mr(y + line_subpixel_scale/2) - line_mr(ey)) * m_dx_start)
        {
            m_dx       <<= line_subpixel_shift;
            m_dy       <<= line_subpixel_shift;
            m_dx_start <<= line_mr_subpixel_shift;
            m_dy_start <<= line_mr_subpixel_shift;
        }


         
        void inc_x() { m_dist += m_dy; m_dist_start += m_dy_start; }
        void dec_x() { m_dist -= m_dy; m_dist_start -= m_dy_start; }
        void inc_y() { m_dist -= m_dx; m_dist_start -= m_dx_start; }
        void dec_y() { m_dist += m_dx; m_dist_start += m_dx_start; }

         
        void inc_x(int dy)
        {
            m_dist       += m_dy; 
            m_dist_start += m_dy_start; 
            if(dy > 0)
            {
                m_dist       -= m_dx; 
                m_dist_start -= m_dx_start; 
            }
            if(dy < 0)
            {
                m_dist       += m_dx; 
                m_dist_start += m_dx_start; 
            }
        }

         
        void dec_x(int dy)
        {
            m_dist       -= m_dy; 
            m_dist_start -= m_dy_start; 
            if(dy > 0)
            {
                m_dist       -= m_dx; 
                m_dist_start -= m_dx_start; 
            }
            if(dy < 0)
            {
                m_dist       += m_dx; 
                m_dist_start += m_dx_start; 
            }
        }

         
        void inc_y(int dx)
        {
            m_dist       -= m_dx; 
            m_dist_start -= m_dx_start; 
            if(dx > 0)
            {
                m_dist       += m_dy; 
                m_dist_start += m_dy_start; 
            }
            if(dx < 0)
            {
                m_dist       -= m_dy; 
                m_dist_start -= m_dy_start; 
            }
        }

         
        void dec_y(int dx)
        {
            m_dist       += m_dx; 
            m_dist_start += m_dx_start; 
            if(dx > 0)
            {
                m_dist       += m_dy; 
                m_dist_start += m_dy_start; 
            }
            if(dx < 0)
            {
                m_dist       -= m_dy; 
                m_dist_start -= m_dy_start; 
            }
        }

         
        int dist()       const { return m_dist;       }
        int dist_start() const { return m_dist_start; }
        int dist_end()   const { return m_dist_start; }

         
        int dx()       const { return m_dx;       }
        int dy()       const { return m_dy;       }
        int dx_start() const { return m_dx_start; }
        int dy_start() const { return m_dy_start; }
        int dx_end()   const { return m_dx_start; }
        int dy_end()   const { return m_dy_start; }

    private:
         
        int m_dx;
        int m_dy;
        int m_dx_start;
        int m_dy_start;

        int m_dist;
        int m_dist_start;
    };





     
    class distance_interpolator3
    {
    public:
         
        distance_interpolator3() {}
        distance_interpolator3(int x1, int y1, int x2, int y2,
                               int sx, int sy, int ex, int ey, 
                               int x,  int y) :
            m_dx(x2 - x1),
            m_dy(y2 - y1),
            m_dx_start(line_mr(sx) - line_mr(x1)),
            m_dy_start(line_mr(sy) - line_mr(y1)),
            m_dx_end(line_mr(ex) - line_mr(x2)),
            m_dy_end(line_mr(ey) - line_mr(y2)),

            m_dist(iround(double(x + line_subpixel_scale/2 - x2) * double(m_dy) - 
                          double(y + line_subpixel_scale/2 - y2) * double(m_dx))),

            m_dist_start((line_mr(x + line_subpixel_scale/2) - line_mr(sx)) * m_dy_start - 
                         (line_mr(y + line_subpixel_scale/2) - line_mr(sy)) * m_dx_start),

            m_dist_end((line_mr(x + line_subpixel_scale/2) - line_mr(ex)) * m_dy_end - 
                       (line_mr(y + line_subpixel_scale/2) - line_mr(ey)) * m_dx_end)
        {
            m_dx       <<= line_subpixel_shift;
            m_dy       <<= line_subpixel_shift;
            m_dx_start <<= line_mr_subpixel_shift;
            m_dy_start <<= line_mr_subpixel_shift;
            m_dx_end   <<= line_mr_subpixel_shift;
            m_dy_end   <<= line_mr_subpixel_shift;
        }

         
        void inc_x() { m_dist += m_dy; m_dist_start += m_dy_start; m_dist_end += m_dy_end; }
        void dec_x() { m_dist -= m_dy; m_dist_start -= m_dy_start; m_dist_end -= m_dy_end; }
        void inc_y() { m_dist -= m_dx; m_dist_start -= m_dx_start; m_dist_end -= m_dx_end; }
        void dec_y() { m_dist += m_dx; m_dist_start += m_dx_start; m_dist_end += m_dx_end; }

         
        void inc_x(int dy)
        {
            m_dist       += m_dy; 
            m_dist_start += m_dy_start; 
            m_dist_end   += m_dy_end;
            if(dy > 0)
            {
                m_dist       -= m_dx; 
                m_dist_start -= m_dx_start; 
                m_dist_end   -= m_dx_end;
            }
            if(dy < 0)
            {
                m_dist       += m_dx; 
                m_dist_start += m_dx_start; 
                m_dist_end   += m_dx_end;
            }
        }

         
        void dec_x(int dy)
        {
            m_dist       -= m_dy; 
            m_dist_start -= m_dy_start; 
            m_dist_end   -= m_dy_end;
            if(dy > 0)
            {
                m_dist       -= m_dx; 
                m_dist_start -= m_dx_start; 
                m_dist_end   -= m_dx_end;
            }
            if(dy < 0)
            {
                m_dist       += m_dx; 
                m_dist_start += m_dx_start; 
                m_dist_end   += m_dx_end;
            }
        }

         
        void inc_y(int dx)
        {
            m_dist       -= m_dx; 
            m_dist_start -= m_dx_start; 
            m_dist_end   -= m_dx_end;
            if(dx > 0)
            {
                m_dist       += m_dy; 
                m_dist_start += m_dy_start; 
                m_dist_end   += m_dy_end;
            }
            if(dx < 0)
            {
                m_dist       -= m_dy; 
                m_dist_start -= m_dy_start; 
                m_dist_end   -= m_dy_end;
            }
        }

         
        void dec_y(int dx)
        {
            m_dist       += m_dx; 
            m_dist_start += m_dx_start; 
            m_dist_end   += m_dx_end;
            if(dx > 0)
            {
                m_dist       += m_dy; 
                m_dist_start += m_dy_start; 
                m_dist_end   += m_dy_end;
            }
            if(dx < 0)
            {
                m_dist       -= m_dy; 
                m_dist_start -= m_dy_start; 
                m_dist_end   -= m_dy_end;
            }
        }

         
        int dist()       const { return m_dist;       }
        int dist_start() const { return m_dist_start; }
        int dist_end()   const { return m_dist_end;   }

         
        int dx()       const { return m_dx;       }
        int dy()       const { return m_dy;       }
        int dx_start() const { return m_dx_start; }
        int dy_start() const { return m_dy_start; }
        int dx_end()   const { return m_dx_end;   }
        int dy_end()   const { return m_dy_end;   }

    private:
         
        int m_dx;
        int m_dy;
        int m_dx_start;
        int m_dy_start;
        int m_dx_end;
        int m_dy_end;

        int m_dist;
        int m_dist_start;
        int m_dist_end;
    };




    
     
    template<class Renderer> class line_interpolator_aa_base
    {
    public:
        typedef Renderer renderer_type;
        typedef typename Renderer::color_type color_type;

         
        enum max_half_width_e
        { 
            max_half_width = 64
        };

         
        line_interpolator_aa_base(renderer_type& ren, line_parameters& lp) :
            m_lp(&lp),
            m_li(lp.vertical ? line_dbl_hr(lp.x2 - lp.x1) :
                               line_dbl_hr(lp.y2 - lp.y1),
                 lp.vertical ? std::abs(lp.y2 - lp.y1) : 
                               std::abs(lp.x2 - lp.x1) + 1),
            m_ren(ren),
            m_len((lp.vertical == (lp.inc > 0)) ? -lp.len : lp.len),
            m_x(lp.x1 >> line_subpixel_shift),
            m_y(lp.y1 >> line_subpixel_shift),
            m_old_x(m_x),
            m_old_y(m_y),
            m_count((lp.vertical ? std::abs((lp.y2 >> line_subpixel_shift) - m_y) :
                                   std::abs((lp.x2 >> line_subpixel_shift) - m_x))),
            m_width(ren.subpixel_width()),
             
            m_max_extent((m_width + line_subpixel_mask) >> line_subpixel_shift),
            m_step(0)
        {
            agg::dda2_line_interpolator li(0, lp.vertical ? 
                                              (lp.dy << agg::line_subpixel_shift) :
                                              (lp.dx << agg::line_subpixel_shift),
                                           lp.len);

            unsigned i;
            int stop = m_width + line_subpixel_scale * 2;
            for(i = 0; i < max_half_width; ++i)
            {
                m_dist[i] = li.y();
                if(m_dist[i] >= stop) break;
                ++li;
            }
            m_dist[i++] = 0x7FFF0000;
        }

         
        template<class DI> int step_hor_base(DI& di)
        {
            ++m_li;
            m_x += m_lp->inc;
            m_y = (m_lp->y1 + m_li.y()) >> line_subpixel_shift;

            if(m_lp->inc > 0) di.inc_x(m_y - m_old_y);
            else              di.dec_x(m_y - m_old_y);

            m_old_y = m_y;

            return di.dist() / m_len;
        }

         
        template<class DI> int step_ver_base(DI& di)
        {
            ++m_li;
            m_y += m_lp->inc;
            m_x = (m_lp->x1 + m_li.y()) >> line_subpixel_shift;

            if(m_lp->inc > 0) di.inc_y(m_x - m_old_x);
            else              di.dec_y(m_x - m_old_x);

            m_old_x = m_x;

            return di.dist() / m_len;
        }

         
        bool vertical() const { return m_lp->vertical; }
        int  width() const { return m_width; }
        int  count() const { return m_count; }

    private:
        line_interpolator_aa_base(const line_interpolator_aa_base<Renderer>&);
        const line_interpolator_aa_base<Renderer>& 
            operator = (const line_interpolator_aa_base<Renderer>&);

    protected:
        line_parameters* m_lp;
        dda2_line_interpolator m_li;
        renderer_type&         m_ren;
        int m_len;
        int m_x;
        int m_y;
        int m_old_x;
        int m_old_y;
        int m_count;
        int m_width;
        int m_max_extent;
        int m_step;
        int m_dist[max_half_width + 1];
        cover_type m_covers[max_half_width * 2 + 4];
    };







     
    template<class Renderer> class line_interpolator_aa0 :
    public line_interpolator_aa_base<Renderer>
    {
    public:
        typedef Renderer renderer_type;
        typedef typename Renderer::color_type color_type;
        typedef line_interpolator_aa_base<Renderer> base_type;

         
        line_interpolator_aa0(renderer_type& ren, line_parameters& lp) :
            line_interpolator_aa_base<Renderer>(ren, lp),
            m_di(lp.x1, lp.y1, lp.x2, lp.y2, 
                 lp.x1 & ~line_subpixel_mask, lp.y1 & ~line_subpixel_mask)
        {
            base_type::m_li.adjust_forward();
        }

         
        bool step_hor()
        {
            int dist;
            int dy;
            int s1 = base_type::step_hor_base(m_di);
            cover_type* p0 = base_type::m_covers + base_type::max_half_width + 2;
            cover_type* p1 = p0;

            *p1++ = (cover_type)base_type::m_ren.cover(s1);

            dy = 1;
            while((dist = base_type::m_dist[dy] - s1) <= base_type::m_width)
            {
                *p1++ = (cover_type)base_type::m_ren.cover(dist);
                ++dy;
            }

            dy = 1;
            while((dist = base_type::m_dist[dy] + s1) <= base_type::m_width)
            {
                *--p0 = (cover_type)base_type::m_ren.cover(dist);
                ++dy;
            }
            base_type::m_ren.blend_solid_vspan(base_type::m_x, 
                                               base_type::m_y - dy + 1, 
                                               unsigned(p1 - p0), 
                                               p0);
            return ++base_type::m_step < base_type::m_count;
        }

         
        bool step_ver()
        {
            int dist;
            int dx;
            int s1 = base_type::step_ver_base(m_di);
            cover_type* p0 = base_type::m_covers + base_type::max_half_width + 2;
            cover_type* p1 = p0;

            *p1++ = (cover_type)base_type::m_ren.cover(s1);

            dx = 1;
            while((dist = base_type::m_dist[dx] - s1) <= base_type::m_width)
            {
                *p1++ = (cover_type)base_type::m_ren.cover(dist);
                ++dx;
            }

            dx = 1;
            while((dist = base_type::m_dist[dx] + s1) <= base_type::m_width)
            {
                *--p0 = (cover_type)base_type::m_ren.cover(dist);
                ++dx;
            }
            base_type::m_ren.blend_solid_hspan(base_type::m_x - dx + 1, 
                                               base_type::m_y,
                                               unsigned(p1 - p0), 
                                               p0);
            return ++base_type::m_step < base_type::m_count;
        }

    private:
        line_interpolator_aa0(const line_interpolator_aa0<Renderer>&);
        const line_interpolator_aa0<Renderer>& 
            operator = (const line_interpolator_aa0<Renderer>&);

         
        distance_interpolator1 m_di; 
    };






     
    template<class Renderer> class line_interpolator_aa1 :
    public line_interpolator_aa_base<Renderer>
    {
    public:
        typedef Renderer renderer_type;
        typedef typename Renderer::color_type color_type;
        typedef line_interpolator_aa_base<Renderer> base_type;

         
        line_interpolator_aa1(renderer_type& ren, line_parameters& lp, 
                              int sx, int sy) :
            line_interpolator_aa_base<Renderer>(ren, lp),
            m_di(lp.x1, lp.y1, lp.x2, lp.y2, sx, sy,
                 lp.x1 & ~line_subpixel_mask, lp.y1 & ~line_subpixel_mask)
        {
            int dist1_start;
            int dist2_start;

            int npix = 1;

            if(lp.vertical)
            {
                do
                {
                    --base_type::m_li;
                    base_type::m_y -= lp.inc;
                    base_type::m_x = (base_type::m_lp->x1 + base_type::m_li.y()) >> line_subpixel_shift;

                    if(lp.inc > 0) m_di.dec_y(base_type::m_x - base_type::m_old_x);
                    else           m_di.inc_y(base_type::m_x - base_type::m_old_x);

                    base_type::m_old_x = base_type::m_x;

                    dist1_start = dist2_start = m_di.dist_start(); 

                    int dx = 0;
                    if(dist1_start < 0) ++npix;
                    do
                    {
                        dist1_start += m_di.dy_start();
                        dist2_start -= m_di.dy_start();
                        if(dist1_start < 0) ++npix;
                        if(dist2_start < 0) ++npix;
                        ++dx;
                    }
                    while(base_type::m_dist[dx] <= base_type::m_width);
                    --base_type::m_step;
                    if(npix == 0) break;
                    npix = 0;
                }
                while(base_type::m_step >= -base_type::m_max_extent);
            }
            else
            {
                do
                {
                    --base_type::m_li;
                    base_type::m_x -= lp.inc;
                    base_type::m_y = (base_type::m_lp->y1 + base_type::m_li.y()) >> line_subpixel_shift;

                    if(lp.inc > 0) m_di.dec_x(base_type::m_y - base_type::m_old_y);
                    else           m_di.inc_x(base_type::m_y - base_type::m_old_y);

                    base_type::m_old_y = base_type::m_y;

                    dist1_start = dist2_start = m_di.dist_start(); 

                    int dy = 0;
                    if(dist1_start < 0) ++npix;
                    do
                    {
                        dist1_start -= m_di.dx_start();
                        dist2_start += m_di.dx_start();
                        if(dist1_start < 0) ++npix;
                        if(dist2_start < 0) ++npix;
                        ++dy;
                    }
                    while(base_type::m_dist[dy] <= base_type::m_width);
                    --base_type::m_step;
                    if(npix == 0) break;
                    npix = 0;
                }
                while(base_type::m_step >= -base_type::m_max_extent);
            }
            base_type::m_li.adjust_forward();
        }

         
        bool step_hor()
        {
            int dist_start;
            int dist;
            int dy;
            int s1 = base_type::step_hor_base(m_di);

            dist_start = m_di.dist_start();
            cover_type* p0 = base_type::m_covers + base_type::max_half_width + 2;
            cover_type* p1 = p0;

            *p1 = 0;
            if(dist_start <= 0)
            {
                *p1 = (cover_type)base_type::m_ren.cover(s1);
            }
            ++p1;

            dy = 1;
            while((dist = base_type::m_dist[dy] - s1) <= base_type::m_width)
            {
                dist_start -= m_di.dx_start();
                *p1 = 0;
                if(dist_start <= 0)
                {   
                    *p1 = (cover_type)base_type::m_ren.cover(dist);
                }
                ++p1;
                ++dy;
            }

            dy = 1;
            dist_start = m_di.dist_start();
            while((dist = base_type::m_dist[dy] + s1) <= base_type::m_width)
            {
                dist_start += m_di.dx_start();
                *--p0 = 0;
                if(dist_start <= 0)
                {   
                    *p0 = (cover_type)base_type::m_ren.cover(dist);
                }
                ++dy;
            }

            base_type::m_ren.blend_solid_vspan(base_type::m_x, 
                                               base_type::m_y - dy + 1,
                                               unsigned(p1 - p0), 
                                               p0);
            return ++base_type::m_step < base_type::m_count;
        }

         
        bool step_ver()
        {
            int dist_start;
            int dist;
            int dx;
            int s1 = base_type::step_ver_base(m_di);
            cover_type* p0 = base_type::m_covers + base_type::max_half_width + 2;
            cover_type* p1 = p0;

            dist_start = m_di.dist_start();

            *p1 = 0;
            if(dist_start <= 0)
            {
                *p1 = (cover_type)base_type::m_ren.cover(s1);
            }
            ++p1;

            dx = 1;
            while((dist = base_type::m_dist[dx] - s1) <= base_type::m_width)
            {
                dist_start += m_di.dy_start();
                *p1 = 0;
                if(dist_start <= 0)
                {   
                    *p1 = (cover_type)base_type::m_ren.cover(dist);
                }
                ++p1;
                ++dx;
            }

            dx = 1;
            dist_start = m_di.dist_start();
            while((dist = base_type::m_dist[dx] + s1) <= base_type::m_width)
            {
                dist_start -= m_di.dy_start();
                *--p0 = 0;
                if(dist_start <= 0)
                {   
                    *p0 = (cover_type)base_type::m_ren.cover(dist);
                }
                ++dx;
            }
            base_type::m_ren.blend_solid_hspan(base_type::m_x - dx + 1, 
                                               base_type::m_y,
                                               unsigned(p1 - p0), 
                                               p0);
            return ++base_type::m_step < base_type::m_count;
        }

    private:
        line_interpolator_aa1(const line_interpolator_aa1<Renderer>&);
        const line_interpolator_aa1<Renderer>& 
            operator = (const line_interpolator_aa1<Renderer>&);

         
        distance_interpolator2 m_di; 
    };












     
    template<class Renderer> class line_interpolator_aa2 :
    public line_interpolator_aa_base<Renderer>
    {
    public:
        typedef Renderer renderer_type;
        typedef typename Renderer::color_type color_type;
        typedef line_interpolator_aa_base<Renderer> base_type;

         
        line_interpolator_aa2(renderer_type& ren, line_parameters& lp, 
                              int ex, int ey) :
            line_interpolator_aa_base<Renderer>(ren, lp),
            m_di(lp.x1, lp.y1, lp.x2, lp.y2, ex, ey, 
                 lp.x1 & ~line_subpixel_mask, lp.y1 & ~line_subpixel_mask,
                 0)
        {
            base_type::m_li.adjust_forward();
            base_type::m_step -= base_type::m_max_extent;
        }

         
        bool step_hor()
        {
            int dist_end;
            int dist;
            int dy;
            int s1 = base_type::step_hor_base(m_di);
            cover_type* p0 = base_type::m_covers + base_type::max_half_width + 2;
            cover_type* p1 = p0;

            dist_end = m_di.dist_end();

            int npix = 0;
            *p1 = 0;
            if(dist_end > 0)
            {
                *p1 = (cover_type)base_type::m_ren.cover(s1);
                ++npix;
            }
            ++p1;

            dy = 1;
            while((dist = base_type::m_dist[dy] - s1) <= base_type::m_width)
            {
                dist_end -= m_di.dx_end();
                *p1 = 0;
                if(dist_end > 0)
                {   
                    *p1 = (cover_type)base_type::m_ren.cover(dist);
                    ++npix;
                }
                ++p1;
                ++dy;
            }

            dy = 1;
            dist_end = m_di.dist_end();
            while((dist = base_type::m_dist[dy] + s1) <= base_type::m_width)
            {
                dist_end += m_di.dx_end();
                *--p0 = 0;
                if(dist_end > 0)
                {   
                    *p0 = (cover_type)base_type::m_ren.cover(dist);
                    ++npix;
                }
                ++dy;
            }
            base_type::m_ren.blend_solid_vspan(base_type::m_x,
                                               base_type::m_y - dy + 1, 
                                               unsigned(p1 - p0), 
                                               p0);
            return npix && ++base_type::m_step < base_type::m_count;
        }

         
        bool step_ver()
        {
            int dist_end;
            int dist;
            int dx;
            int s1 = base_type::step_ver_base(m_di);
            cover_type* p0 = base_type::m_covers + base_type::max_half_width + 2;
            cover_type* p1 = p0;

            dist_end = m_di.dist_end();

            int npix = 0;
            *p1 = 0;
            if(dist_end > 0)
            {
                *p1 = (cover_type)base_type::m_ren.cover(s1);
                ++npix;
            }
            ++p1;

            dx = 1;
            while((dist = base_type::m_dist[dx] - s1) <= base_type::m_width)
            {
                dist_end += m_di.dy_end();
                *p1 = 0;
                if(dist_end > 0)
                {   
                    *p1 = (cover_type)base_type::m_ren.cover(dist);
                    ++npix;
                }
                ++p1;
                ++dx;
            }

            dx = 1;
            dist_end = m_di.dist_end();
            while((dist = base_type::m_dist[dx] + s1) <= base_type::m_width)
            {
                dist_end -= m_di.dy_end();
                *--p0 = 0;
                if(dist_end > 0)
                {   
                    *p0 = (cover_type)base_type::m_ren.cover(dist);
                    ++npix;
                }
                ++dx;
            }
            base_type::m_ren.blend_solid_hspan(base_type::m_x - dx + 1,
                                               base_type::m_y, 
                                               unsigned(p1 - p0), 
                                               p0);
            return npix && ++base_type::m_step < base_type::m_count;
        }

    private:
        line_interpolator_aa2(const line_interpolator_aa2<Renderer>&);
        const line_interpolator_aa2<Renderer>& 
            operator = (const line_interpolator_aa2<Renderer>&);

         
        distance_interpolator2 m_di; 
    };










     
    template<class Renderer> class line_interpolator_aa3 :
    public line_interpolator_aa_base<Renderer>
    {
    public:
        typedef Renderer renderer_type;
        typedef typename Renderer::color_type color_type;
        typedef line_interpolator_aa_base<Renderer> base_type;

         
        line_interpolator_aa3(renderer_type& ren, line_parameters& lp, 
                              int sx, int sy, int ex, int ey) :
            line_interpolator_aa_base<Renderer>(ren, lp),
            m_di(lp.x1, lp.y1, lp.x2, lp.y2, sx, sy, ex, ey, 
                 lp.x1 & ~line_subpixel_mask, lp.y1 & ~line_subpixel_mask)
        {
            int dist1_start;
            int dist2_start;
            int npix = 1;
            if(lp.vertical)
            {
                do
                {
                    --base_type::m_li;
                    base_type::m_y -= lp.inc;
                    base_type::m_x = (base_type::m_lp->x1 + base_type::m_li.y()) >> line_subpixel_shift;

                    if(lp.inc > 0) m_di.dec_y(base_type::m_x - base_type::m_old_x);
                    else           m_di.inc_y(base_type::m_x - base_type::m_old_x);

                    base_type::m_old_x = base_type::m_x;

                    dist1_start = dist2_start = m_di.dist_start(); 

                    int dx = 0;
                    if(dist1_start < 0) ++npix;
                    do
                    {
                        dist1_start += m_di.dy_start();
                        dist2_start -= m_di.dy_start();
                        if(dist1_start < 0) ++npix;
                        if(dist2_start < 0) ++npix;
                        ++dx;
                    }
                    while(base_type::m_dist[dx] <= base_type::m_width);
                    if(npix == 0) break;
                    npix = 0;
                }
                while(--base_type::m_step >= -base_type::m_max_extent);
            }
            else
            {
                do
                {
                    --base_type::m_li;
                    base_type::m_x -= lp.inc;
                    base_type::m_y = (base_type::m_lp->y1 + base_type::m_li.y()) >> line_subpixel_shift;

                    if(lp.inc > 0) m_di.dec_x(base_type::m_y - base_type::m_old_y);
                    else           m_di.inc_x(base_type::m_y - base_type::m_old_y);

                    base_type::m_old_y = base_type::m_y;

                    dist1_start = dist2_start = m_di.dist_start(); 

                    int dy = 0;
                    if(dist1_start < 0) ++npix;
                    do
                    {
                        dist1_start -= m_di.dx_start();
                        dist2_start += m_di.dx_start();
                        if(dist1_start < 0) ++npix;
                        if(dist2_start < 0) ++npix;
                        ++dy;
                    }
                    while(base_type::m_dist[dy] <= base_type::m_width);
                    if(npix == 0) break;
                    npix = 0;
                }
                while(--base_type::m_step >= -base_type::m_max_extent);
            }
            base_type::m_li.adjust_forward();
            base_type::m_step -= base_type::m_max_extent;
        }


         
        bool step_hor()
        {
            int dist_start;
            int dist_end;
            int dist;
            int dy;
            int s1 = base_type::step_hor_base(m_di);
            cover_type* p0 = base_type::m_covers + base_type::max_half_width + 2;
            cover_type* p1 = p0;

            dist_start = m_di.dist_start();
            dist_end   = m_di.dist_end();

            int npix = 0;
            *p1 = 0;
            if(dist_end > 0)
            {
                if(dist_start <= 0)
                {
                    *p1 = (cover_type)base_type::m_ren.cover(s1);
                }
                ++npix;
            }
            ++p1;

            dy = 1;
            while((dist = base_type::m_dist[dy] - s1) <= base_type::m_width)
            {
                dist_start -= m_di.dx_start();
                dist_end   -= m_di.dx_end();
                *p1 = 0;
                if(dist_end > 0 && dist_start <= 0)
                {   
                    *p1 = (cover_type)base_type::m_ren.cover(dist);
                    ++npix;
                }
                ++p1;
                ++dy;
            }

            dy = 1;
            dist_start = m_di.dist_start();
            dist_end   = m_di.dist_end();
            while((dist = base_type::m_dist[dy] + s1) <= base_type::m_width)
            {
                dist_start += m_di.dx_start();
                dist_end   += m_di.dx_end();
                *--p0 = 0;
                if(dist_end > 0 && dist_start <= 0)
                {   
                    *p0 = (cover_type)base_type::m_ren.cover(dist);
                    ++npix;
                }
                ++dy;
            }
            base_type::m_ren.blend_solid_vspan(base_type::m_x,
                                               base_type::m_y - dy + 1, 
                                               unsigned(p1 - p0), 
                                               p0);
            return npix && ++base_type::m_step < base_type::m_count;
        }

         
        bool step_ver()
        {
            int dist_start;
            int dist_end;
            int dist;
            int dx;
            int s1 = base_type::step_ver_base(m_di);
            cover_type* p0 = base_type::m_covers + base_type::max_half_width + 2;
            cover_type* p1 = p0;

            dist_start = m_di.dist_start();
            dist_end   = m_di.dist_end();

            int npix = 0;
            *p1 = 0;
            if(dist_end > 0)
            {
                if(dist_start <= 0)
                {
                    *p1 = (cover_type)base_type::m_ren.cover(s1);
                }
                ++npix;
            }
            ++p1;

            dx = 1;
            while((dist = base_type::m_dist[dx] - s1) <= base_type::m_width)
            {
                dist_start += m_di.dy_start();
                dist_end   += m_di.dy_end();
                *p1 = 0;
                if(dist_end > 0 && dist_start <= 0)
                {   
                    *p1 = (cover_type)base_type::m_ren.cover(dist);
                    ++npix;
                }
                ++p1;
                ++dx;
            }

            dx = 1;
            dist_start = m_di.dist_start();
            dist_end   = m_di.dist_end();
            while((dist = base_type::m_dist[dx] + s1) <= base_type::m_width)
            {
                dist_start -= m_di.dy_start();
                dist_end   -= m_di.dy_end();
                *--p0 = 0;
                if(dist_end > 0 && dist_start <= 0)
                {   
                    *p0 = (cover_type)base_type::m_ren.cover(dist);
                    ++npix;
                }
                ++dx;
            }
            base_type::m_ren.blend_solid_hspan(base_type::m_x - dx + 1,
                                               base_type::m_y, 
                                               unsigned(p1 - p0), 
                                               p0);
            return npix && ++base_type::m_step < base_type::m_count;
        }

    private:
        line_interpolator_aa3(const line_interpolator_aa3<Renderer>&);
        const line_interpolator_aa3<Renderer>& 
            operator = (const line_interpolator_aa3<Renderer>&);

         
        distance_interpolator3 m_di; 
    };




     
     
     
     
    class line_profile_aa
    {
    public:
         
        typedef int8u value_type;
        enum subpixel_scale_e
        {
            subpixel_shift = line_subpixel_shift,
            subpixel_scale = 1 << subpixel_shift,
            subpixel_mask  = subpixel_scale - 1
        };

        enum aa_scale_e
        {
            aa_shift = 8,
            aa_scale = 1 << aa_shift,
            aa_mask  = aa_scale - 1
        };
        
         
        line_profile_aa() : 
            m_subpixel_width(0),
            m_min_width(1.0),
            m_smoother_width(1.0)
        {
            int i;
            for(i = 0; i < aa_scale; i++) m_gamma[i] = (value_type)i;
        }

         
        template<class GammaF> 
        line_profile_aa(double w, const GammaF& gamma_function) : 
            m_subpixel_width(0),
            m_min_width(1.0),
            m_smoother_width(1.0)
        {
            gamma(gamma_function);
            width(w);
        }

         
        void min_width(double w) { m_min_width = w; }
        void smoother_width(double w) { m_smoother_width = w; }

         
        template<class GammaF> void gamma(const GammaF& gamma_function)
        { 
            int i;
            for(i = 0; i < aa_scale; i++)
            {
                m_gamma[i] = value_type(
                    uround(gamma_function(double(i) / (double)aa_mask) * (double)aa_mask));
            }
        }

        void width(double w);

        unsigned profile_size() const { return m_profile.size(); }
        int subpixel_width() const { return m_subpixel_width; }

         
        double min_width() const { return m_min_width; }
        double smoother_width() const { return m_smoother_width; }

         
        value_type value(int dist) const
        {
            return m_profile[dist + subpixel_scale*2];
        }

    private:
        line_profile_aa(const line_profile_aa&);
        const line_profile_aa& operator = (const line_profile_aa&);

        value_type* profile(double w);
        void set(double center_width, double smoother_width);

         
        pod_array<value_type> m_profile;
        value_type            m_gamma[aa_scale];
        int                   m_subpixel_width;
        double                m_min_width;
        double                m_smoother_width;
    };


     
    template<class BaseRenderer> class renderer_outline_aa
    {
    public:
         
        typedef BaseRenderer base_ren_type;
        typedef renderer_outline_aa<base_ren_type> self_type;
        typedef typename base_ren_type::color_type color_type;

         
        renderer_outline_aa(base_ren_type& ren, line_profile_aa& prof) :
            m_ren(&ren),
            m_profile(&prof),
            m_clip_box(0,0,0,0),
            m_clipping(false)
        {}
        void attach(base_ren_type& ren) { m_ren = &ren; }

         
        void color(const color_type& c) { m_color = c; }
        const color_type& color() const { return m_color; }

         
        void profile(line_profile_aa& prof) { m_profile = &prof; }
        const line_profile_aa& profile() const { return *m_profile; }
        line_profile_aa& profile() { return *m_profile; }

         
        int subpixel_width() const { return m_profile->subpixel_width(); }

         
        void reset_clipping() { m_clipping = false; }
        void clip_box(double x1, double y1, double x2, double y2)
        {
            m_clip_box.x1 = line_coord_sat::conv(x1);
            m_clip_box.y1 = line_coord_sat::conv(y1);
            m_clip_box.x2 = line_coord_sat::conv(x2);
            m_clip_box.y2 = line_coord_sat::conv(y2);
            m_clipping = true;
        }

         
        int cover(int d) const
        {
            return m_profile->value(d);
        }

         
        void blend_solid_hspan(int x, int y, unsigned len, const cover_type* covers)
        {
            m_ren->blend_solid_hspan(x, y, len, m_color, covers);
        }

         
        void blend_solid_vspan(int x, int y, unsigned len, const cover_type* covers)
        {
            m_ren->blend_solid_vspan(x, y, len, m_color, covers);
        }

         
        static bool accurate_join_only() { return false; }

         
        template<class Cmp>
        void semidot_hline(Cmp cmp,
                           int xc1, int yc1, int xc2, int yc2, 
                           int x1,  int y1,  int x2)
        {
            cover_type covers[line_interpolator_aa_base<self_type>::max_half_width * 2 + 4];
            cover_type* p0 = covers;
            cover_type* p1 = covers;
            int x = x1 << line_subpixel_shift;
            int y = y1 << line_subpixel_shift;
            int w = subpixel_width();
            distance_interpolator0 di(xc1, yc1, xc2, yc2, x, y);
            x += line_subpixel_scale/2;
            y += line_subpixel_scale/2;

            int x0 = x1;
            int dx = x - xc1;
            int dy = y - yc1;
            do
            {
                int d = int(fast_sqrt(dx*dx + dy*dy));
                *p1 = 0;
                if(cmp(di.dist()) && d <= w)
                {
                    *p1 = (cover_type)cover(d);
                }
                ++p1;
                dx += line_subpixel_scale;
                di.inc_x();
            }
            while(++x1 <= x2);
            m_ren->blend_solid_hspan(x0, y1, 
                                     unsigned(p1 - p0), 
                                     color(), 
                                     p0);
        }

         
        template<class Cmp> 
        void semidot(Cmp cmp, int xc1, int yc1, int xc2, int yc2)
        {
            if(m_clipping && clipping_flags(xc1, yc1, m_clip_box)) return;

            int r = ((subpixel_width() + line_subpixel_mask) >> line_subpixel_shift);
            if(r < 1) r = 1;
            ellipse_bresenham_interpolator ei(r, r);
            int dx = 0;
            int dy = -r;
            int dy0 = dy;
            int dx0 = dx;
            int x = xc1 >> line_subpixel_shift;
            int y = yc1 >> line_subpixel_shift;

            do
            {
                dx += ei.dx();
                dy += ei.dy();

                if(dy != dy0)
                {
                    semidot_hline(cmp, xc1, yc1, xc2, yc2, x-dx0, y+dy0, x+dx0);
                    semidot_hline(cmp, xc1, yc1, xc2, yc2, x-dx0, y-dy0, x+dx0);
                }
                dx0 = dx;
                dy0 = dy;
                ++ei;
            }
            while(dy < 0);
            semidot_hline(cmp, xc1, yc1, xc2, yc2, x-dx0, y+dy0, x+dx0);
        }

         
        void pie_hline(int xc, int yc, int xp1, int yp1, int xp2, int yp2, 
                       int xh1, int yh1, int xh2)
        {
            if(m_clipping && clipping_flags(xc, yc, m_clip_box)) return;
           
            cover_type covers[line_interpolator_aa_base<self_type>::max_half_width * 2 + 4];
            cover_type* p0 = covers;
            cover_type* p1 = covers;
            int x = xh1 << line_subpixel_shift;
            int y = yh1 << line_subpixel_shift;
            int w = subpixel_width();

            distance_interpolator00 di(xc, yc, xp1, yp1, xp2, yp2, x, y);
            x += line_subpixel_scale/2;
            y += line_subpixel_scale/2;

            int xh0 = xh1;
            int dx = x - xc;
            int dy = y - yc;
            do
            {
                int d = int(fast_sqrt(dx*dx + dy*dy));
                *p1 = 0;
                if(di.dist1() <= 0 && di.dist2() > 0 && d <= w)
                {
                    *p1 = (cover_type)cover(d);
                }
                ++p1;
                dx += line_subpixel_scale;
                di.inc_x();
            }
            while(++xh1 <= xh2);
            m_ren->blend_solid_hspan(xh0, yh1, 
                                     unsigned(p1 - p0), 
                                     color(), 
                                     p0);
        }


         
        void pie(int xc, int yc, int x1, int y1, int x2, int y2)
        {
            int r = ((subpixel_width() + line_subpixel_mask) >> line_subpixel_shift);
            if(r < 1) r = 1;
            ellipse_bresenham_interpolator ei(r, r);
            int dx = 0;
            int dy = -r;
            int dy0 = dy;
            int dx0 = dx;
            int x = xc >> line_subpixel_shift;
            int y = yc >> line_subpixel_shift;

            do
            {
                dx += ei.dx();
                dy += ei.dy();

                if(dy != dy0)
                {
                    pie_hline(xc, yc, x1, y1, x2, y2, x-dx0, y+dy0, x+dx0);
                    pie_hline(xc, yc, x1, y1, x2, y2, x-dx0, y-dy0, x+dx0);
                }
                dx0 = dx;
                dy0 = dy;
                ++ei;
            }
            while(dy < 0);
            pie_hline(xc, yc, x1, y1, x2, y2, x-dx0, y+dy0, x+dx0);
        }

         
        void line0_no_clip(line_parameters& lp)
        {
            if(lp.len > line_max_length)
            {
                line_parameters lp1, lp2;
                lp.divide(lp1, lp2);
                line0_no_clip(lp1);
                line0_no_clip(lp2);
                return;
            }

            line_interpolator_aa0<self_type> li(*this, lp);
            if(li.count())
            {
                if(li.vertical())
                {
                    while(li.step_ver());
                }
                else
                {
                    while(li.step_hor());
                }
            }
        }

         
        void line0(line_parameters& lp)
        {
            if(m_clipping)
            {
                int x1 = lp.x1;
                int y1 = lp.y1;
                int x2 = lp.x2;
                int y2 = lp.y2;
                unsigned flags = clip_line_segment(&x1, &y1, &x2, &y2, m_clip_box);
                if((flags & 4) == 0)
                {
                    if(flags)
                    {
                        line_parameters lp2(x1, y1, x2, y2, 
                                           uround(calc_distance(x1, y1, x2, y2)));
                        line0_no_clip(lp2);
                    }
                    else
                    {
                        line0_no_clip(lp);
                    }
                }
            }
            else
            {
                line0_no_clip(lp);
            }
        }

         
        void line1_no_clip(line_parameters& lp, int sx, int sy)
        {
            if(lp.len > line_max_length)
            {
                line_parameters lp1, lp2;
                lp.divide(lp1, lp2);
                line1_no_clip(lp1, (lp.x1 + sx) >> 1, (lp.y1 + sy) >> 1);
                line1_no_clip(lp2, lp1.x2 + (lp1.y2 - lp1.y1), lp1.y2 - (lp1.x2 - lp1.x1));
                return;
            }

            fix_degenerate_bisectrix_start(lp, &sx, &sy);
            line_interpolator_aa1<self_type> li(*this, lp, sx, sy);
            if(li.vertical())
            {
                while(li.step_ver());
            }
            else
            {
                while(li.step_hor());
            }
        }


         
        void line1(line_parameters& lp, int sx, int sy)
        {
            if(m_clipping)
            {
                int x1 = lp.x1;
                int y1 = lp.y1;
                int x2 = lp.x2;
                int y2 = lp.y2;
                unsigned flags = clip_line_segment(&x1, &y1, &x2, &y2, m_clip_box);
                if((flags & 4) == 0)
                {
                    if(flags)
                    {
                        line_parameters lp2(x1, y1, x2, y2, 
                                           uround(calc_distance(x1, y1, x2, y2)));
                        if(flags & 1)
                        {
                            sx = x1 + (y2 - y1); 
                            sy = y1 - (x2 - x1);
                        }
                        else
                        {
                            while(std::abs(sx - lp.x1) + std::abs(sy - lp.y1) > lp2.len)
                            {
                                sx = (lp.x1 + sx) >> 1;
                                sy = (lp.y1 + sy) >> 1;
                            }
                        }
                        line1_no_clip(lp2, sx, sy);
                    }
                    else
                    {
                        line1_no_clip(lp, sx, sy);
                    }
                }
            }
            else
            {
                line1_no_clip(lp, sx, sy);
            }
        }

         
        void line2_no_clip(line_parameters& lp, int ex, int ey)
        {
            if(lp.len > line_max_length)
            {
                line_parameters lp1, lp2;
                lp.divide(lp1, lp2);
                line2_no_clip(lp1, lp1.x2 + (lp1.y2 - lp1.y1), lp1.y2 - (lp1.x2 - lp1.x1));
                line2_no_clip(lp2, (lp.x2 + ex) >> 1, (lp.y2 + ey) >> 1);
                return;
            }

            fix_degenerate_bisectrix_end(lp, &ex, &ey);
            line_interpolator_aa2<self_type> li(*this, lp, ex, ey);
            if(li.vertical())
            {
                while(li.step_ver());
            }
            else
            {
                while(li.step_hor());
            }
        }

         
        void line2(line_parameters& lp, int ex, int ey)
        {
            if(m_clipping)
            {
                int x1 = lp.x1;
                int y1 = lp.y1;
                int x2 = lp.x2;
                int y2 = lp.y2;
                unsigned flags = clip_line_segment(&x1, &y1, &x2, &y2, m_clip_box);
                if((flags & 4) == 0)
                {
                    if(flags)
                    {
                        line_parameters lp2(x1, y1, x2, y2, 
                                           uround(calc_distance(x1, y1, x2, y2)));
                        if(flags & 2)
                        {
                            ex = x2 + (y2 - y1); 
                            ey = y2 - (x2 - x1);
                        }
                        else
                        {
                            while(std::abs(ex - lp.x2) + std::abs(ey - lp.y2) > lp2.len)
                            {
                                ex = (lp.x2 + ex) >> 1;
                                ey = (lp.y2 + ey) >> 1;
                            }
                        }
                        line2_no_clip(lp2, ex, ey);
                    }
                    else
                    {
                        line2_no_clip(lp, ex, ey);
                    }
                }
            }
            else
            {
                line2_no_clip(lp, ex, ey);
            }
        }

         
        void line3_no_clip(line_parameters& lp, 
                           int sx, int sy, int ex, int ey)
        {
            if(lp.len > line_max_length)
            {
                line_parameters lp1, lp2;
                lp.divide(lp1, lp2);
                int mx = lp1.x2 + (lp1.y2 - lp1.y1);
                int my = lp1.y2 - (lp1.x2 - lp1.x1);
                line3_no_clip(lp1, (lp.x1 + sx) >> 1, (lp.y1 + sy) >> 1, mx, my);
                line3_no_clip(lp2, mx, my, (lp.x2 + ex) >> 1, (lp.y2 + ey) >> 1);
                return;
            }

            fix_degenerate_bisectrix_start(lp, &sx, &sy);
            fix_degenerate_bisectrix_end(lp, &ex, &ey);
            line_interpolator_aa3<self_type> li(*this, lp, sx, sy, ex, ey);
            if(li.vertical())
            {
                while(li.step_ver());
            }
            else
            {
                while(li.step_hor());
            }
        }

         
        void line3(line_parameters& lp, 
                   int sx, int sy, int ex, int ey)
        {
            if(m_clipping)
            {
                int x1 = lp.x1;
                int y1 = lp.y1;
                int x2 = lp.x2;
                int y2 = lp.y2;
                unsigned flags = clip_line_segment(&x1, &y1, &x2, &y2, m_clip_box);
                if((flags & 4) == 0)
                {
                    if(flags)
                    {
                        line_parameters lp2(x1, y1, x2, y2, 
                                           uround(calc_distance(x1, y1, x2, y2)));
                        if(flags & 1)
                        {
                            sx = x1 + (y2 - y1); 
                            sy = y1 - (x2 - x1);
                        }
                        else
                        {
                            while(std::abs(sx - lp.x1) + std::abs(sy - lp.y1) > lp2.len)
                            {
                                sx = (lp.x1 + sx) >> 1;
                                sy = (lp.y1 + sy) >> 1;
                            }
                        }
                        if(flags & 2)
                        {
                            ex = x2 + (y2 - y1); 
                            ey = y2 - (x2 - x1);
                        }
                        else
                        {
                            while(std::abs(ex - lp.x2) + std::abs(ey - lp.y2) > lp2.len)
                            {
                                ex = (lp.x2 + ex) >> 1;
                                ey = (lp.y2 + ey) >> 1;
                            }
                        }
                        line3_no_clip(lp2, sx, sy, ex, ey);
                    }
                    else
                    {
                        line3_no_clip(lp, sx, sy, ex, ey);
                    }
                }
            }
            else
            {
                line3_no_clip(lp, sx, sy, ex, ey);
            }
        }


    private:
        base_ren_type*         m_ren;
        line_profile_aa* m_profile; 
        color_type             m_color;
        rect_i                 m_clip_box;
        bool                   m_clipping;
    };



}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_RENDERER_OUTLINE_IMAGE_INCLUDED
#define AGG_RENDERER_OUTLINE_IMAGE_INCLUDED

#include <cstdlib>


namespace agg
{
     
    template<class Source> class line_image_scale
    {
    public:
        typedef typename Source::color_type color_type;

        line_image_scale(const Source& src, double height) :
            m_source(src), 
            m_height(height),
            m_scale(src.height() / height),
            m_scale_inv(height / src.height())
        {
        }

        double width()  const { return m_source.width(); }
        double height() const { return m_height; }

        color_type pixel(int x, int y) const 
        { 
            if (m_scale < 1.0)
            {
                 
                double src_y = (y + 0.5) * m_scale - 0.5;
                int h  = m_source.height() - 1;
                int y1 = ifloor(src_y);
                int y2 = y1 + 1;
                rgba pix1 = (y1 < 0) ? rgba::no_color() : rgba(m_source.pixel(x, y1));
                rgba pix2 = (y2 > h) ? rgba::no_color() : rgba(m_source.pixel(x, y2));
                return pix1.gradient(pix2, src_y - y1);
            }
            else
            {
                 
                double src_y1 = (y + 0.5) * m_scale - 0.5;
                double src_y2 = src_y1 + m_scale;
                int h  = m_source.height() - 1;
                int y1 = ifloor(src_y1);
                int y2 = ifloor(src_y2);
                rgba c = rgba::no_color();
                if (y1 >= 0) c += rgba(m_source.pixel(x, y1)) *= y1 + 1 - src_y1;
                while (++y1 < y2)
                {
                    if (y1 <= h) c += m_source.pixel(x, y1);
                }
                if (y2 <= h) c += rgba(m_source.pixel(x, y2)) *= src_y2 - y2;
                return c *= m_scale_inv;
            }
        }

    private:
        line_image_scale(const line_image_scale<Source>&);
        const line_image_scale<Source>& operator = (const line_image_scale<Source>&);

        const Source& m_source;
        double        m_height;
        double        m_scale;
        double        m_scale_inv;
    };



     
    template<class Filter> class line_image_pattern
    {
    public:
        typedef Filter filter_type;
        typedef typename filter_type::color_type color_type;

         
        line_image_pattern(Filter& filter) :
            m_filter(&filter),
            m_dilation(filter.dilation() + 1),
            m_dilation_hr(m_dilation << line_subpixel_shift),
            m_data(),
            m_width(0),
            m_height(0),
            m_width_hr(0),
            m_half_height_hr(0),
            m_offset_y_hr(0)
        {
        }

         
         
        template<class Source> 
        line_image_pattern(Filter& filter, const Source& src) :
            m_filter(&filter),
            m_dilation(filter.dilation() + 1),
            m_dilation_hr(m_dilation << line_subpixel_shift),
            m_data(),
            m_width(0),
            m_height(0),
            m_width_hr(0),
            m_half_height_hr(0),
            m_offset_y_hr(0)
        {
            create(src);
        }

         
         
        template<class Source> void create(const Source& src)
        {
            m_height = uceil(src.height());
            m_width  = uceil(src.width());
            m_width_hr = uround(src.width() * (double)line_subpixel_scale);
            m_half_height_hr = uround(src.height() * (double)line_subpixel_scale/2);
            m_offset_y_hr = m_dilation_hr + m_half_height_hr - line_subpixel_scale/2;
            m_half_height_hr += line_subpixel_scale/2;

            m_data.resize((m_width + m_dilation * 2) * (m_height + m_dilation * 2));

            m_buf.attach(&m_data[0], m_width  + m_dilation * 2, 
                                     m_height + m_dilation * 2, 
                                     m_width  + m_dilation * 2);
            unsigned x, y;
            color_type* d1;
            color_type* d2;
            for(y = 0; y < m_height; y++)
            {
                d1 = m_buf.row_ptr(y + m_dilation) + m_dilation;
                for(x = 0; x < m_width; x++)
                {
                    *d1++ = src.pixel(x, y);
                }
            }

            const color_type* s1;
            const color_type* s2;
            for(y = 0; y < m_dilation; y++)
            {
                 
                 
                d1 = m_buf.row_ptr(m_dilation + m_height + y) + m_dilation;
                d2 = m_buf.row_ptr(m_dilation - y - 1) + m_dilation;
                for(x = 0; x < m_width; x++)
                {
                     
                     
                    *d1++ = color_type::no_color();
                    *d2++ = color_type::no_color();
                }
            }

            unsigned h = m_height + m_dilation * 2;
            for(y = 0; y < h; y++)
            {
                s1 = m_buf.row_ptr(y) + m_dilation;
                s2 = m_buf.row_ptr(y) + m_dilation + m_width;
                d1 = m_buf.row_ptr(y) + m_dilation + m_width;
                d2 = m_buf.row_ptr(y) + m_dilation;

                for(x = 0; x < m_dilation; x++)
                {
                    *d1++ = *s1++;
                    *--d2 = *--s2;
                }
            }
        }

         
        int pattern_width() const { return m_width_hr; }
        int line_width()    const { return m_half_height_hr; }
        double width()      const { return m_height; }

         
        void pixel(color_type* p, int x, int y) const
        {
            m_filter->pixel_high_res(m_buf.rows(), 
                                     p, 
                                     x % m_width_hr + m_dilation_hr,
                                     y + m_offset_y_hr);
        }

         
        const filter_type& filter() const { return *m_filter; }

    private:
        line_image_pattern(const line_image_pattern<filter_type>&);
        const line_image_pattern<filter_type>& 
            operator = (const line_image_pattern<filter_type>&);

    protected:
        row_ptr_cache<color_type> m_buf;
        const filter_type*        m_filter;
        unsigned                  m_dilation;
        int                       m_dilation_hr;
        pod_array<color_type>     m_data;
        unsigned                  m_width;
        unsigned                  m_height;
        int                       m_width_hr;
        int                       m_half_height_hr;
        int                       m_offset_y_hr;
    };






     
    template<class Filter> class line_image_pattern_pow2 : 
    public line_image_pattern<Filter>
    {
    public:
        typedef Filter filter_type;
        typedef typename filter_type::color_type color_type;
        typedef line_image_pattern<Filter> base_type;

         
        line_image_pattern_pow2(Filter& filter) :
            line_image_pattern<Filter>(filter), m_mask(line_subpixel_mask) {}

         
        template<class Source> 
        line_image_pattern_pow2(Filter& filter, const Source& src) :
            line_image_pattern<Filter>(filter), m_mask(line_subpixel_mask)
        {
            create(src);
        }
            
         
        template<class Source> void create(const Source& src)
        {
            line_image_pattern<Filter>::create(src);
            m_mask = 1;
            while(m_mask < base_type::m_width) 
            {
                m_mask <<= 1;
                m_mask |= 1;
            }
            m_mask <<= line_subpixel_shift - 1;
            m_mask |=  line_subpixel_mask;
            base_type::m_width_hr = m_mask + 1;
        }

         
        void pixel(color_type* p, int x, int y) const
        {
            base_type::m_filter->pixel_high_res(
                    base_type::m_buf.rows(), 
                    p,
                    (x & m_mask) + base_type::m_dilation_hr,
                    y + base_type::m_offset_y_hr);
        }
    private:
        unsigned m_mask;
    };
    
    
    
    
    
    
    
     
    class distance_interpolator4
    {
    public:
         
        distance_interpolator4() {}
        distance_interpolator4(int x1,  int y1, int x2, int y2,
                               int sx,  int sy, int ex, int ey, 
                               int len, double scale, int x, int y) :
            m_dx(x2 - x1),
            m_dy(y2 - y1),
            m_dx_start(line_mr(sx) - line_mr(x1)),
            m_dy_start(line_mr(sy) - line_mr(y1)),
            m_dx_end(line_mr(ex) - line_mr(x2)),
            m_dy_end(line_mr(ey) - line_mr(y2)),

            m_dist(iround(double(x + line_subpixel_scale/2 - x2) * double(m_dy) - 
                          double(y + line_subpixel_scale/2 - y2) * double(m_dx))),

            m_dist_start((line_mr(x + line_subpixel_scale/2) - line_mr(sx)) * m_dy_start - 
                         (line_mr(y + line_subpixel_scale/2) - line_mr(sy)) * m_dx_start),

            m_dist_end((line_mr(x + line_subpixel_scale/2) - line_mr(ex)) * m_dy_end - 
                       (line_mr(y + line_subpixel_scale/2) - line_mr(ey)) * m_dx_end),
            m_len(uround(len / scale))
        {
            double d = len * scale;
            int dx = iround(((x2 - x1) << line_subpixel_shift) / d);
            int dy = iround(((y2 - y1) << line_subpixel_shift) / d);
            m_dx_pict   = -dy;
            m_dy_pict   =  dx;
            m_dist_pict =  ((x + line_subpixel_scale/2 - (x1 - dy)) * m_dy_pict - 
                            (y + line_subpixel_scale/2 - (y1 + dx)) * m_dx_pict) >> 
                           line_subpixel_shift;

            m_dx       <<= line_subpixel_shift;
            m_dy       <<= line_subpixel_shift;
            m_dx_start <<= line_mr_subpixel_shift;
            m_dy_start <<= line_mr_subpixel_shift;
            m_dx_end   <<= line_mr_subpixel_shift;
            m_dy_end   <<= line_mr_subpixel_shift;
        }

         
        void inc_x() 
        { 
            m_dist += m_dy; 
            m_dist_start += m_dy_start; 
            m_dist_pict += m_dy_pict; 
            m_dist_end += m_dy_end; 
        }

         
        void dec_x() 
        { 
            m_dist -= m_dy; 
            m_dist_start -= m_dy_start; 
            m_dist_pict -= m_dy_pict; 
            m_dist_end -= m_dy_end; 
        }

         
        void inc_y() 
        { 
            m_dist -= m_dx; 
            m_dist_start -= m_dx_start; 
            m_dist_pict -= m_dx_pict; 
            m_dist_end -= m_dx_end; 
        }

         
        void dec_y() 
        { 
            m_dist += m_dx; 
            m_dist_start += m_dx_start; 
            m_dist_pict += m_dx_pict; 
            m_dist_end += m_dx_end; 
        }

         
        void inc_x(int dy)
        {
            m_dist       += m_dy; 
            m_dist_start += m_dy_start; 
            m_dist_pict  += m_dy_pict; 
            m_dist_end   += m_dy_end;
            if(dy > 0)
            {
                m_dist       -= m_dx; 
                m_dist_start -= m_dx_start; 
                m_dist_pict  -= m_dx_pict; 
                m_dist_end   -= m_dx_end;
            }
            if(dy < 0)
            {
                m_dist       += m_dx; 
                m_dist_start += m_dx_start; 
                m_dist_pict  += m_dx_pict; 
                m_dist_end   += m_dx_end;
            }
        }

         
        void dec_x(int dy)
        {
            m_dist       -= m_dy; 
            m_dist_start -= m_dy_start; 
            m_dist_pict  -= m_dy_pict; 
            m_dist_end   -= m_dy_end;
            if(dy > 0)
            {
                m_dist       -= m_dx; 
                m_dist_start -= m_dx_start; 
                m_dist_pict  -= m_dx_pict; 
                m_dist_end   -= m_dx_end;
            }
            if(dy < 0)
            {
                m_dist       += m_dx; 
                m_dist_start += m_dx_start; 
                m_dist_pict  += m_dx_pict; 
                m_dist_end   += m_dx_end;
            }
        }

         
        void inc_y(int dx)
        {
            m_dist       -= m_dx; 
            m_dist_start -= m_dx_start; 
            m_dist_pict  -= m_dx_pict; 
            m_dist_end   -= m_dx_end;
            if(dx > 0)
            {
                m_dist       += m_dy; 
                m_dist_start += m_dy_start; 
                m_dist_pict  += m_dy_pict; 
                m_dist_end   += m_dy_end;
            }
            if(dx < 0)
            {
                m_dist       -= m_dy; 
                m_dist_start -= m_dy_start; 
                m_dist_pict  -= m_dy_pict; 
                m_dist_end   -= m_dy_end;
            }
        }

         
        void dec_y(int dx)
        {
            m_dist       += m_dx; 
            m_dist_start += m_dx_start; 
            m_dist_pict  += m_dx_pict; 
            m_dist_end   += m_dx_end;
            if(dx > 0)
            {
                m_dist       += m_dy; 
                m_dist_start += m_dy_start; 
                m_dist_pict  += m_dy_pict; 
                m_dist_end   += m_dy_end;
            }
            if(dx < 0)
            {
                m_dist       -= m_dy; 
                m_dist_start -= m_dy_start; 
                m_dist_pict  -= m_dy_pict; 
                m_dist_end   -= m_dy_end;
            }
        }

         
        int dist()       const { return m_dist;       }
        int dist_start() const { return m_dist_start; }
        int dist_pict()  const { return m_dist_pict;  }
        int dist_end()   const { return m_dist_end;   }

         
        int dx()       const { return m_dx;       }
        int dy()       const { return m_dy;       }
        int dx_start() const { return m_dx_start; }
        int dy_start() const { return m_dy_start; }
        int dx_pict()  const { return m_dx_pict;  }
        int dy_pict()  const { return m_dy_pict;  }
        int dx_end()   const { return m_dx_end;   }
        int dy_end()   const { return m_dy_end;   }
        int len()      const { return m_len;      }

    private:
         
        int m_dx;
        int m_dy;
        int m_dx_start;
        int m_dy_start;
        int m_dx_pict;
        int m_dy_pict;
        int m_dx_end;
        int m_dy_end;

        int m_dist;
        int m_dist_start;
        int m_dist_pict;
        int m_dist_end;
        int m_len;
    };





     
    template<class Renderer> class line_interpolator_image
    {
    public:
        typedef Renderer renderer_type;
        typedef typename Renderer::color_type color_type;

         
        enum max_half_width_e
        { 
            max_half_width = 64
        };

         
        line_interpolator_image(renderer_type& ren, const line_parameters& lp,
                                int sx, int sy, int ex, int ey, 
                                int pattern_start,
                                double scale_x) :
            m_lp(lp),
            m_li(lp.vertical ? line_dbl_hr(lp.x2 - lp.x1) :
                               line_dbl_hr(lp.y2 - lp.y1),
                 lp.vertical ? std::abs(lp.y2 - lp.y1) : 
                               std::abs(lp.x2 - lp.x1) + 1),
            m_di(lp.x1, lp.y1, lp.x2, lp.y2, sx, sy, ex, ey, lp.len, scale_x,
                 lp.x1 & ~line_subpixel_mask, lp.y1 & ~line_subpixel_mask),
            m_ren(ren),
            m_x(lp.x1 >> line_subpixel_shift),
            m_y(lp.y1 >> line_subpixel_shift),
            m_old_x(m_x),
            m_old_y(m_y),
            m_count((lp.vertical ? std::abs((lp.y2 >> line_subpixel_shift) - m_y) :
                                   std::abs((lp.x2 >> line_subpixel_shift) - m_x))),
            m_width(ren.subpixel_width()),
             
            m_max_extent((m_width + line_subpixel_scale) >> line_subpixel_shift),
            m_start(pattern_start + (m_max_extent + 2) * ren.pattern_width()),
            m_step(0)
        {
            agg::dda2_line_interpolator li(0, lp.vertical ? 
                                              (lp.dy << agg::line_subpixel_shift) :
                                              (lp.dx << agg::line_subpixel_shift),
                                           lp.len);

            unsigned i;
            int stop = m_width + line_subpixel_scale * 2;
            for(i = 0; i < max_half_width; ++i)
            {
                m_dist_pos[i] = li.y();
                if(m_dist_pos[i] >= stop) break;
                ++li;
            }
            m_dist_pos[i] = 0x7FFF0000;

            int dist1_start;
            int dist2_start;
            int npix = 1;

            if(lp.vertical)
            {
                do
                {
                    --m_li;
                    m_y -= lp.inc;
                    m_x = (m_lp.x1 + m_li.y()) >> line_subpixel_shift;

                    if(lp.inc > 0) m_di.dec_y(m_x - m_old_x);
                    else           m_di.inc_y(m_x - m_old_x);

                    m_old_x = m_x;

                    dist1_start = dist2_start = m_di.dist_start(); 

                    int dx = 0;
                    if(dist1_start < 0) ++npix;
                    do
                    {
                        dist1_start += m_di.dy_start();
                        dist2_start -= m_di.dy_start();
                        if(dist1_start < 0) ++npix;
                        if(dist2_start < 0) ++npix;
                        ++dx;
                    }
                    while(m_dist_pos[dx] <= m_width);
                    if(npix == 0) break;

                    npix = 0;
                }
                while(--m_step >= -m_max_extent);
            }
            else
            {
                do
                {
                    --m_li;

                    m_x -= lp.inc;
                    m_y = (m_lp.y1 + m_li.y()) >> line_subpixel_shift;

                    if(lp.inc > 0) m_di.dec_x(m_y - m_old_y);
                    else           m_di.inc_x(m_y - m_old_y);

                    m_old_y = m_y;

                    dist1_start = dist2_start = m_di.dist_start(); 

                    int dy = 0;
                    if(dist1_start < 0) ++npix;
                    do
                    {
                        dist1_start -= m_di.dx_start();
                        dist2_start += m_di.dx_start();
                        if(dist1_start < 0) ++npix;
                        if(dist2_start < 0) ++npix;
                        ++dy;
                    }
                    while(m_dist_pos[dy] <= m_width);
                    if(npix == 0) break;

                    npix = 0;
                }
                while(--m_step >= -m_max_extent);
            }
            m_li.adjust_forward();
            m_step -= m_max_extent;
        }

         
        bool step_hor()
        {
            ++m_li;
            m_x += m_lp.inc;
            m_y = (m_lp.y1 + m_li.y()) >> line_subpixel_shift;

            if(m_lp.inc > 0) m_di.inc_x(m_y - m_old_y);
            else             m_di.dec_x(m_y - m_old_y);

            m_old_y = m_y;

            int s1 = m_di.dist() / m_lp.len;
            int s2 = -s1;

            if(m_lp.inc < 0) s1 = -s1;

            int dist_start;
            int dist_pict;
            int dist_end;
            int dy;
            int dist;

            dist_start = m_di.dist_start();
            dist_pict  = m_di.dist_pict() + m_start;
            dist_end   = m_di.dist_end();
            color_type* p0 = m_colors + max_half_width + 2;
            color_type* p1 = p0;

            int npix = 0;
            p1->clear();
            if(dist_end > 0)
            {
                if(dist_start <= 0)
                {
                    m_ren.pixel(p1, dist_pict, s2);
                }
                ++npix;
            }
            ++p1;

            dy = 1;
            while((dist = m_dist_pos[dy]) - s1 <= m_width)
            {
                dist_start -= m_di.dx_start();
                dist_pict  -= m_di.dx_pict();
                dist_end   -= m_di.dx_end();
                p1->clear();
                if(dist_end > 0 && dist_start <= 0)
                {   
                    if(m_lp.inc > 0) dist = -dist;
                    m_ren.pixel(p1, dist_pict, s2 - dist);
                    ++npix;
                }
                ++p1;
                ++dy;
            }

            dy = 1;
            dist_start = m_di.dist_start();
            dist_pict  = m_di.dist_pict() + m_start;
            dist_end   = m_di.dist_end();
            while((dist = m_dist_pos[dy]) + s1 <= m_width)
            {
                dist_start += m_di.dx_start();
                dist_pict  += m_di.dx_pict();
                dist_end   += m_di.dx_end();
                --p0;
                p0->clear();
                if(dist_end > 0 && dist_start <= 0)
                {   
                    if(m_lp.inc > 0) dist = -dist;
                    m_ren.pixel(p0, dist_pict, s2 + dist);
                    ++npix;
                }
                ++dy;
            }
            m_ren.blend_color_vspan(m_x, 
                                    m_y - dy + 1, 
                                    unsigned(p1 - p0), 
                                    p0); 
            return npix && ++m_step < m_count;
        }



         
        bool step_ver()
        {
            ++m_li;
            m_y += m_lp.inc;
            m_x = (m_lp.x1 + m_li.y()) >> line_subpixel_shift;

            if(m_lp.inc > 0) m_di.inc_y(m_x - m_old_x);
            else             m_di.dec_y(m_x - m_old_x);

            m_old_x = m_x;

            int s1 = m_di.dist() / m_lp.len;
            int s2 = -s1;

            if(m_lp.inc > 0) s1 = -s1;

            int dist_start;
            int dist_pict;
            int dist_end;
            int dist;
            int dx;

            dist_start = m_di.dist_start();
            dist_pict  = m_di.dist_pict() + m_start;
            dist_end   = m_di.dist_end();
            color_type* p0 = m_colors + max_half_width + 2;
            color_type* p1 = p0;

            int npix = 0;
            p1->clear();
            if(dist_end > 0)
            {
                if(dist_start <= 0)
                {
                    m_ren.pixel(p1, dist_pict, s2);
                }
                ++npix;
            }
            ++p1;

            dx = 1;
            while((dist = m_dist_pos[dx]) - s1 <= m_width)
            {
                dist_start += m_di.dy_start();
                dist_pict  += m_di.dy_pict();
                dist_end   += m_di.dy_end();
                p1->clear();
                if(dist_end > 0 && dist_start <= 0)
                {   
                    if(m_lp.inc > 0) dist = -dist;
                    m_ren.pixel(p1, dist_pict, s2 + dist);
                    ++npix;
                }
                ++p1;
                ++dx;
            }

            dx = 1;
            dist_start = m_di.dist_start();
            dist_pict  = m_di.dist_pict() + m_start;
            dist_end   = m_di.dist_end();
            while((dist = m_dist_pos[dx]) + s1 <= m_width)
            {
                dist_start -= m_di.dy_start();
                dist_pict  -= m_di.dy_pict();
                dist_end   -= m_di.dy_end();
                --p0;
                p0->clear();
                if(dist_end > 0 && dist_start <= 0)
                {   
                    if(m_lp.inc > 0) dist = -dist;
                    m_ren.pixel(p0, dist_pict, s2 - dist);
                    ++npix;
                }
                ++dx;
            }
            m_ren.blend_color_hspan(m_x - dx + 1, 
                                    m_y, 
                                    unsigned(p1 - p0), 
                                    p0);
            return npix && ++m_step < m_count;
        }


         
        int  pattern_end() const { return m_start + m_di.len(); }

         
        bool vertical() const { return m_lp.vertical; }
        int  width() const { return m_width; }
        int  count() const { return m_count; }

    private:
        line_interpolator_image(const line_interpolator_image<Renderer>&);
        const line_interpolator_image<Renderer>&
            operator = (const line_interpolator_image<Renderer>&);

    protected:
        const line_parameters& m_lp;
        dda2_line_interpolator m_li;
        distance_interpolator4 m_di; 
        renderer_type&         m_ren;
        int m_plen;
        int m_x;
        int m_y;
        int m_old_x;
        int m_old_y;
        int m_count;
        int m_width;
        int m_max_extent;
        int m_start;
        int m_step;
        int m_dist_pos[max_half_width + 1];
        color_type m_colors[max_half_width * 2 + 4];
    };








     
    template<class BaseRenderer, class ImagePattern> 
    class renderer_outline_image
    {
    public:
         
        typedef BaseRenderer base_ren_type;
        typedef renderer_outline_image<BaseRenderer, ImagePattern> self_type;
        typedef typename base_ren_type::color_type color_type;
        typedef ImagePattern pattern_type;


         
        renderer_outline_image(base_ren_type& ren, pattern_type& patt) :
            m_ren(&ren),
            m_pattern(&patt),
            m_start(0),
            m_scale_x(1.0),
            m_clip_box(0,0,0,0),
            m_clipping(false)
        {}
        void attach(base_ren_type& ren) { m_ren = &ren; }

         
        void pattern(pattern_type& p) { m_pattern = &p; }
        pattern_type& pattern() const { return *m_pattern; }

         
        void reset_clipping() { m_clipping = false; }
        void clip_box(double x1, double y1, double x2, double y2)
        {
            m_clip_box.x1 = line_coord_sat::conv(x1);
            m_clip_box.y1 = line_coord_sat::conv(y1);
            m_clip_box.x2 = line_coord_sat::conv(x2);
            m_clip_box.y2 = line_coord_sat::conv(y2);
            m_clipping = true;
        }

         
        void   scale_x(double s) { m_scale_x = s; }
        double scale_x() const   { return m_scale_x; }

         
        void   start_x(double s) { m_start = iround(s * (double)line_subpixel_scale); }
        double start_x() const   { return double(m_start) / (double)line_subpixel_scale; }

         
        int subpixel_width() const { return m_pattern->line_width(); }
        int pattern_width() const { return m_pattern->pattern_width(); }
        double width() const { return double(subpixel_width()) / (double)line_subpixel_scale; }

         
        void pixel(color_type* p, int x, int y) const
        {
            m_pattern->pixel(p, x, y);
        }

         
        void blend_color_hspan(int x, int y, unsigned len, const color_type* colors)
        {
            m_ren->blend_color_hspan(x, y, len, colors, 0);
        }

         
        void blend_color_vspan(int x, int y, unsigned len, const color_type* colors)
        {
            m_ren->blend_color_vspan(x, y, len, colors, 0);
        }

         
        static bool accurate_join_only() { return true; }

         
        template<class Cmp> 
        void semidot(Cmp, int, int, int, int)
        {
        }

         
        void pie(int, int, int, int, int, int)
        {
        }

         
        void line0(const line_parameters&)
        {
        }

         
        void line1(const line_parameters&, int, int)
        {
        }

         
        void line2(const line_parameters&, int, int)
        {
        }

         
        void line3_no_clip(const line_parameters& lp, 
                           int sx, int sy, int ex, int ey)
        {
            if(lp.len > line_max_length)
            {
                line_parameters lp1, lp2;
                lp.divide(lp1, lp2);
                int mx = lp1.x2 + (lp1.y2 - lp1.y1);
                int my = lp1.y2 - (lp1.x2 - lp1.x1);
                line3_no_clip(lp1, (lp.x1 + sx) >> 1, (lp.y1 + sy) >> 1, mx, my);
                line3_no_clip(lp2, mx, my, (lp.x2 + ex) >> 1, (lp.y2 + ey) >> 1);
                return;
            }
            
            fix_degenerate_bisectrix_start(lp, &sx, &sy);
            fix_degenerate_bisectrix_end(lp, &ex, &ey);
            line_interpolator_image<self_type> li(*this, lp, 
                                                  sx, sy, 
                                                  ex, ey, 
                                                  m_start, m_scale_x);
            if(li.vertical())
            {
                while(li.step_ver());
            }
            else
            {
                while(li.step_hor());
            }
            m_start += uround(lp.len / m_scale_x);
        }

         
        void line3(const line_parameters& lp, 
                   int sx, int sy, int ex, int ey)
        {
            if(m_clipping)
            {
                int x1 = lp.x1;
                int y1 = lp.y1;
                int x2 = lp.x2;
                int y2 = lp.y2;
                unsigned flags = clip_line_segment(&x1, &y1, &x2, &y2, m_clip_box);
                int start = m_start;
                if((flags & 4) == 0)
                {
                    if(flags)
                    {
                        line_parameters lp2(x1, y1, x2, y2, 
                                           uround(calc_distance(x1, y1, x2, y2)));
                        if(flags & 1)
                        {
                            m_start += uround(calc_distance(lp.x1, lp.y1, x1, y1) / m_scale_x);
                            sx = x1 + (y2 - y1); 
                            sy = y1 - (x2 - x1);
                        }
                        else
                        {
                            while(std::abs(sx - lp.x1) + std::abs(sy - lp.y1) > lp2.len)
                            {
                                sx = (lp.x1 + sx) >> 1;
                                sy = (lp.y1 + sy) >> 1;
                            }
                        }
                        if(flags & 2)
                        {
                            ex = x2 + (y2 - y1); 
                            ey = y2 - (x2 - x1);
                        }
                        else
                        {
                            while(std::abs(ex - lp.x2) + std::abs(ey - lp.y2) > lp2.len)
                            {
                                ex = (lp.x2 + ex) >> 1;
                                ey = (lp.y2 + ey) >> 1;
                            }
                        }
                        line3_no_clip(lp2, sx, sy, ex, ey);
                    }
                    else
                    {
                        line3_no_clip(lp, sx, sy, ex, ey);
                    }
                }
                m_start = start + uround(lp.len / m_scale_x);
            }
            else
            {
                line3_no_clip(lp, sx, sy, ex, ey);
            }
        }

    private:
        base_ren_type*      m_ren;
        pattern_type* m_pattern;
        int                 m_start;
        double              m_scale_x;
        rect_i              m_clip_box;
        bool                m_clipping;
    };





}



#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_RENDERER_RASTER_TEXT_INCLUDED
#define AGG_RENDERER_RASTER_TEXT_INCLUDED


namespace agg
{

     
    template<class BaseRenderer, class GlyphGenerator> 
    class renderer_raster_htext_solid
    {
    public:
        typedef BaseRenderer ren_type;
        typedef GlyphGenerator glyph_gen_type;
        typedef typename glyph_gen_type::glyph_rect glyph_rect;
        typedef typename ren_type::color_type color_type;

        renderer_raster_htext_solid(ren_type& ren, glyph_gen_type& glyph) :
            m_ren(&ren),
            m_glyph(&glyph)
        {}
        void attach(ren_type& ren) { m_ren = &ren; }

         
        void color(const color_type& c) { m_color = c; }
        const color_type& color() const { return m_color; }

         
        template<class CharT>
        void render_text(double x, double y, const CharT* str, bool flip=false)
        {
            glyph_rect r;
            while(*str)
            {
                m_glyph->prepare(&r, x, y, *str, flip);
                if(r.x2 >= r.x1)
                {
                    int i;
                    if(flip)
                    {
                        for(i = r.y1; i <= r.y2; i++)
                        {
                            m_ren->blend_solid_hspan(r.x1, i, (r.x2 - r.x1 + 1),
                                                     m_color,
                                                     m_glyph->span(r.y2 - i));
                        }
                    }
                    else
                    {
                        for(i = r.y1; i <= r.y2; i++)
                        {
                            m_ren->blend_solid_hspan(r.x1, i, (r.x2 - r.x1 + 1),
                                                     m_color,
                                                     m_glyph->span(i - r.y1));
                        }
                    }
                }
                x += r.dx;
                y += r.dy;
                ++str;
            }
        }

    private:
        ren_type* m_ren;
        glyph_gen_type* m_glyph;
        color_type m_color;
    };



     
    template<class BaseRenderer, class GlyphGenerator> 
    class renderer_raster_vtext_solid
    {
    public:
        typedef BaseRenderer ren_type;
        typedef GlyphGenerator glyph_gen_type;
        typedef typename glyph_gen_type::glyph_rect glyph_rect;
        typedef typename ren_type::color_type color_type;

        renderer_raster_vtext_solid(ren_type& ren, glyph_gen_type& glyph) :
            m_ren(&ren),
            m_glyph(&glyph)
        {
        }

         
        void color(const color_type& c) { m_color = c; }
        const color_type& color() const { return m_color; }

         
        template<class CharT>
        void render_text(double x, double y, const CharT* str, bool flip=false)
        {
            glyph_rect r;
            while(*str)
            {
                m_glyph->prepare(&r, x, y, *str, !flip);
                if(r.x2 >= r.x1)
                {
                    int i;
                    if(flip)
                    {
                        for(i = r.y1; i <= r.y2; i++)
                        {
                            m_ren->blend_solid_vspan(i, r.x1, (r.x2 - r.x1 + 1),
                                                     m_color,
                                                     m_glyph->span(i - r.y1));
                        }
                    }
                    else
                    {
                        for(i = r.y1; i <= r.y2; i++)
                        {
                            m_ren->blend_solid_vspan(i, r.x1, (r.x2 - r.x1 + 1),
                                                     m_color,
                                                     m_glyph->span(r.y2 - i));
                        }
                    }
                }
                x += r.dx;
                y += r.dy;
                ++str;
            }
        }

    private:
        ren_type* m_ren;
        glyph_gen_type* m_glyph;
        color_type m_color;
    };






     
    template<class ScanlineRenderer, class GlyphGenerator> 
    class renderer_raster_htext
    {
    public:
        typedef ScanlineRenderer ren_type;
        typedef GlyphGenerator glyph_gen_type;
        typedef typename glyph_gen_type::glyph_rect glyph_rect;

        class scanline_single_span
        {
        public:
            typedef agg::cover_type cover_type;

             
            struct const_span
            {
                int x;
                unsigned len;
                const cover_type* covers;

                const_span() {}
                const_span(int x_, unsigned len_, const cover_type* covers_) :
                    x(x_), len(len_), covers(covers_) 
                {}
            };

            typedef const const_span* const_iterator;

             
            scanline_single_span(int x, int y, unsigned len, 
                                 const cover_type* covers) :
                m_y(y),
                m_span(x, len, covers)
            {}

             
            int      y()           const { return m_y; }
            unsigned num_spans()   const { return 1;   }
            const_iterator begin() const { return &m_span; }

        private:
             
            int m_y;
            const_span m_span;
        };



         
        renderer_raster_htext(ren_type& ren, glyph_gen_type& glyph) :
            m_ren(&ren),
            m_glyph(&glyph)
        {
        }


         
        template<class CharT>
        void render_text(double x, double y, const CharT* str, bool flip=false)
        {
            glyph_rect r;
            while(*str)
            {
                m_glyph->prepare(&r, x, y, *str, flip);
                if(r.x2 >= r.x1)
                {
                    m_ren->prepare();
                    int i;
                    if(flip)
                    {
                        for(i = r.y1; i <= r.y2; i++)
                        {
                            m_ren->render(
                                scanline_single_span(r.x1, 
                                                     i, 
                                                     (r.x2 - r.x1 + 1),
                                                     m_glyph->span(r.y2 - i)));
                        }
                    }
                    else
                    {
                        for(i = r.y1; i <= r.y2; i++)
                        {
                            m_ren->render(
                                scanline_single_span(r.x1, 
                                                     i, 
                                                     (r.x2 - r.x1 + 1),
                                                     m_glyph->span(i - r.y1)));
                        }
                    }
                }
                x += r.dx;
                y += r.dy;
                ++str;
            }
        }

    private:
        ren_type* m_ren;
        glyph_gen_type* m_glyph;
    };




}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_RENDERER_SCANLINE_INCLUDED
#define AGG_RENDERER_SCANLINE_INCLUDED

#include <limits>
#include <cstdlib>
#include <cstring>

namespace agg
{

     
    template<class Scanline, class BaseRenderer, class ColorT> 
    void render_scanline_aa_solid(const Scanline& sl, 
                                  BaseRenderer& ren, 
                                  const ColorT& color)
    {
        int y = sl.y();
        unsigned num_spans = sl.num_spans();
        typename Scanline::const_iterator span = sl.begin();

        for(;;)
        {
            int x = span->x;
            if(span->len > 0)
            {
                ren.blend_solid_hspan(x, y, (unsigned)span->len, 
                                      color, 
                                      span->covers);
            }
            else
            {
                ren.blend_hline(x, y, (unsigned)(x - span->len - 1), 
                                color, 
                                *(span->covers));
            }
            if(--num_spans == 0) break;
            ++span;
        }
    }

     
    template<class Rasterizer, class Scanline, 
             class BaseRenderer, class ColorT>
    void render_scanlines_aa_solid(Rasterizer& ras, Scanline& sl, 
                                   BaseRenderer& ren, const ColorT& color)
    {
        if(ras.rewind_scanlines())
        {
             
             
             
             
             
            typename BaseRenderer::color_type ren_color = color;

            sl.reset(ras.min_x(), ras.max_x());
            while(ras.sweep_scanline(sl))
            {
                 

                 
                 
                 
                 
                int y = sl.y();
                unsigned num_spans = sl.num_spans();
                typename Scanline::const_iterator span = sl.begin();

                for(;;)
                {
                    int x = span->x;
                    if(span->len > 0)
                    {
                        ren.blend_solid_hspan(x, y, (unsigned)span->len, 
                                              ren_color, 
                                              span->covers);
                    }
                    else
                    {
                        ren.blend_hline(x, y, (unsigned)(x - span->len - 1), 
                                        ren_color, 
                                        *(span->covers));
                    }
                    if(--num_spans == 0) break;
                    ++span;
                }
            }
        }
    }

     
    template<class BaseRenderer> class renderer_scanline_aa_solid
    {
    public:
        typedef BaseRenderer base_ren_type;
        typedef typename base_ren_type::color_type color_type;

         
        renderer_scanline_aa_solid() : m_ren(0) {}
        explicit renderer_scanline_aa_solid(base_ren_type& ren) : m_ren(&ren) {}
        void attach(base_ren_type& ren)
        {
            m_ren = &ren;
        }
        
         
        void color(const color_type& c) { m_color = c; }
        const color_type& color() const { return m_color; }

         
        void prepare() {}

         
        template<class Scanline> void render(const Scanline& sl)
        {
            render_scanline_aa_solid(sl, *m_ren, m_color);
        }
        
    private:
        base_ren_type* m_ren;
        color_type m_color;
    };













     
    template<class Scanline, class BaseRenderer, 
             class SpanAllocator, class SpanGenerator> 
    void render_scanline_aa(const Scanline& sl, BaseRenderer& ren, 
                            SpanAllocator& alloc, SpanGenerator& span_gen)
    {
        int y = sl.y();

        unsigned num_spans = sl.num_spans();
        typename Scanline::const_iterator span = sl.begin();
        for(;;)
        {
            int x = span->x;
            int len = span->len;
            const typename Scanline::cover_type* covers = span->covers;

            if(len < 0) len = -len;
            typename BaseRenderer::color_type* colors = alloc.allocate(len);
            span_gen.generate(colors, x, y, len);
            ren.blend_color_hspan(x, y, len, colors, 
                                  (span->len < 0) ? 0 : covers, *covers);

            if(--num_spans == 0) break;
            ++span;
        }
    }

     
    template<class Rasterizer, class Scanline, class BaseRenderer, 
             class SpanAllocator, class SpanGenerator>
    void render_scanlines_aa(Rasterizer& ras, Scanline& sl, BaseRenderer& ren, 
                             SpanAllocator& alloc, SpanGenerator& span_gen)
    {
        if(ras.rewind_scanlines())
        {
            sl.reset(ras.min_x(), ras.max_x());
            span_gen.prepare();
            while(ras.sweep_scanline(sl))
            {
                render_scanline_aa(sl, ren, alloc, span_gen);
            }
        }
    }

     
    template<class BaseRenderer, class SpanAllocator, class SpanGenerator> 
    class renderer_scanline_aa
    {
    public:
        typedef BaseRenderer  base_ren_type;
        typedef SpanAllocator alloc_type;
        typedef SpanGenerator span_gen_type;

         
        renderer_scanline_aa() : m_ren(0), m_alloc(0), m_span_gen(0) {}
        renderer_scanline_aa(base_ren_type& ren, 
                             alloc_type& alloc, 
                             span_gen_type& span_gen) :
            m_ren(&ren),
            m_alloc(&alloc),
            m_span_gen(&span_gen)
        {}
        void attach(base_ren_type& ren, 
                    alloc_type& alloc, 
                    span_gen_type& span_gen)
        {
            m_ren = &ren;
            m_alloc = &alloc;
            m_span_gen = &span_gen;
        }
        
         
        void prepare() { m_span_gen->prepare(); }

         
        template<class Scanline> void render(const Scanline& sl)
        {
            render_scanline_aa(sl, *m_ren, *m_alloc, *m_span_gen);
        }

    private:
        base_ren_type* m_ren;
        alloc_type*    m_alloc;
        span_gen_type* m_span_gen;
    };






     
    template<class Scanline, class BaseRenderer, class ColorT> 
    void render_scanline_bin_solid(const Scanline& sl, 
                                   BaseRenderer& ren, 
                                   const ColorT& color)
    {
        unsigned num_spans = sl.num_spans();
        typename Scanline::const_iterator span = sl.begin();
        for(;;)
        {
            ren.blend_hline(span->x, 
                            sl.y(), 
                            span->x - 1 + ((span->len < 0) ? 
                                              -span->len : 
                                               span->len), 
                               color, 
                               cover_full);
            if(--num_spans == 0) break;
            ++span;
        }
    }

     
    template<class Rasterizer, class Scanline, 
             class BaseRenderer, class ColorT>
    void render_scanlines_bin_solid(Rasterizer& ras, Scanline& sl, 
                                    BaseRenderer& ren, const ColorT& color)
    {
        if(ras.rewind_scanlines())
        {
             
             
             
             
             
            typename BaseRenderer::color_type ren_color(color);

            sl.reset(ras.min_x(), ras.max_x());
            while(ras.sweep_scanline(sl))
            {
                 

                 
                 
                 
                 
                unsigned num_spans = sl.num_spans();
                typename Scanline::const_iterator span = sl.begin();
                for(;;)
                {
                    ren.blend_hline(span->x, 
                                    sl.y(), 
                                    span->x - 1 + ((span->len < 0) ? 
                                                      -span->len : 
                                                       span->len), 
                                       ren_color, 
                                       cover_full);
                    if(--num_spans == 0) break;
                    ++span;
                }
            }
        }
    }

     
    template<class BaseRenderer> class renderer_scanline_bin_solid
    {
    public:
        typedef BaseRenderer base_ren_type;
        typedef typename base_ren_type::color_type color_type;

         
        renderer_scanline_bin_solid() : m_ren(0) {}
        explicit renderer_scanline_bin_solid(base_ren_type& ren) : m_ren(&ren) {}
        void attach(base_ren_type& ren)
        {
            m_ren = &ren;
        }
        
         
        void color(const color_type& c) { m_color = c; }
        const color_type& color() const { return m_color; }

         
        void prepare() {}

         
        template<class Scanline> void render(const Scanline& sl)
        {
            render_scanline_bin_solid(sl, *m_ren, m_color);
        }
        
    private:
        base_ren_type* m_ren;
        color_type m_color;
    };








     
    template<class Scanline, class BaseRenderer, 
             class SpanAllocator, class SpanGenerator> 
    void render_scanline_bin(const Scanline& sl, BaseRenderer& ren, 
                             SpanAllocator& alloc, SpanGenerator& span_gen)
    {
        int y = sl.y();

        unsigned num_spans = sl.num_spans();
        typename Scanline::const_iterator span = sl.begin();
        for(;;)
        {
            int x = span->x;
            int len = span->len;
            if(len < 0) len = -len;
            typename BaseRenderer::color_type* colors = alloc.allocate(len);
            span_gen.generate(colors, x, y, len);
            ren.blend_color_hspan(x, y, len, colors, 0, cover_full); 
            if(--num_spans == 0) break;
            ++span;
        }
    }

     
    template<class Rasterizer, class Scanline, class BaseRenderer, 
             class SpanAllocator, class SpanGenerator>
    void render_scanlines_bin(Rasterizer& ras, Scanline& sl, BaseRenderer& ren, 
                              SpanAllocator& alloc, SpanGenerator& span_gen)
    {
        if(ras.rewind_scanlines())
        {
            sl.reset(ras.min_x(), ras.max_x());
            span_gen.prepare();
            while(ras.sweep_scanline(sl))
            {
                render_scanline_bin(sl, ren, alloc, span_gen);
            }
        }
    }

     
    template<class BaseRenderer, class SpanAllocator, class SpanGenerator> 
    class renderer_scanline_bin
    {
    public:
        typedef BaseRenderer  base_ren_type;
        typedef SpanAllocator alloc_type;
        typedef SpanGenerator span_gen_type;

         
        renderer_scanline_bin() : m_ren(0), m_alloc(0), m_span_gen(0) {}
        renderer_scanline_bin(base_ren_type& ren, 
                              alloc_type& alloc, 
                              span_gen_type& span_gen) :
            m_ren(&ren),
            m_alloc(&alloc),
            m_span_gen(&span_gen)
        {}
        void attach(base_ren_type& ren, 
                    alloc_type& alloc, 
                    span_gen_type& span_gen)
        {
            m_ren = &ren;
            m_alloc = &alloc;
            m_span_gen = &span_gen;
        }
        
         
        void prepare() { m_span_gen->prepare(); }

         
        template<class Scanline> void render(const Scanline& sl)
        {
            render_scanline_bin(sl, *m_ren, *m_alloc, *m_span_gen);
        }

    private:
        base_ren_type* m_ren;
        alloc_type*    m_alloc;
        span_gen_type* m_span_gen;
    };










     
    template<class Rasterizer, class Scanline, class Renderer>
    void render_scanlines(Rasterizer& ras, Scanline& sl, Renderer& ren)
    {
        if(ras.rewind_scanlines())
        {
            sl.reset(ras.min_x(), ras.max_x());
            ren.prepare();
            while(ras.sweep_scanline(sl))
            {
                ren.render(sl);
            }
        }
    }

     
    template<class Rasterizer, class Scanline, class Renderer, 
             class VertexSource, class ColorStorage, class PathId>
    void render_all_paths(Rasterizer& ras, 
                          Scanline& sl,
                          Renderer& r, 
                          VertexSource& vs, 
                          const ColorStorage& as, 
                          const PathId& path_id,
                          unsigned num_paths)
    {
        for(unsigned i = 0; i < num_paths; i++)
        {
            ras.reset();
            ras.add_path(vs, path_id[i]);
            r.color(as[i]);
            render_scanlines(ras, sl, r);
        }
    }






     
    template<class Rasterizer, 
             class ScanlineAA, 
             class ScanlineBin, 
             class BaseRenderer, 
             class SpanAllocator,
             class StyleHandler>
    void render_scanlines_compound(Rasterizer& ras, 
                                   ScanlineAA& sl_aa,
                                   ScanlineBin& sl_bin,
                                   BaseRenderer& ren,
                                   SpanAllocator& alloc,
                                   StyleHandler& sh)
    {
        if(ras.rewind_scanlines())
        {
            int min_x = ras.min_x();
            int len = ras.max_x() - min_x + 2;
            sl_aa.reset(min_x, ras.max_x());
            sl_bin.reset(min_x, ras.max_x());

            typedef typename BaseRenderer::color_type color_type;
            color_type* color_span = alloc.allocate(len * 2);
            color_type* mix_buffer = color_span + len;
            unsigned num_spans;

            unsigned num_styles;
            unsigned style;
            bool     solid;
            while((num_styles = ras.sweep_styles()) > 0)
            {
                typename ScanlineAA::const_iterator span_aa;
                if(num_styles == 1)
                {
                     
                     
                    if(ras.sweep_scanline(sl_aa, 0))
                    {
                        style = ras.style(0);
                        if(sh.is_solid(style))
                        {
                             
                             
                            render_scanline_aa_solid(sl_aa, ren, sh.color(style));
                        }
                        else
                        {
                             
                             
                            span_aa   = sl_aa.begin();
                            num_spans = sl_aa.num_spans();
                            for(;;)
                            {
                                len = span_aa->len;
                                sh.generate_span(color_span, 
                                                 span_aa->x, 
                                                 sl_aa.y(), 
                                                 len, 
                                                 style);

                                ren.blend_color_hspan(span_aa->x, 
                                                      sl_aa.y(), 
                                                      span_aa->len,
                                                      color_span,
                                                      span_aa->covers);
                                if(--num_spans == 0) break;
                                ++span_aa;
                            }
                        }
                    }
                }
                else
                {
                    if(ras.sweep_scanline(sl_bin, -1))
                    {
                         
                         
                        typename ScanlineBin::const_iterator span_bin = sl_bin.begin();
                        num_spans = sl_bin.num_spans();
                        for(;;)
                        {
                            std::memset(mix_buffer + span_bin->x - min_x, 
                                   0, 
                                   span_bin->len * sizeof(color_type));

                            if(--num_spans == 0) break;
                            ++span_bin;
                        }

                        unsigned i;
                        for(i = 0; i < num_styles; i++)
                        {
                            style = ras.style(i);
                            solid = sh.is_solid(style);

                            if(ras.sweep_scanline(sl_aa, i))
                            {
                                color_type* colors;
                                color_type* cspan;
                                typename ScanlineAA::cover_type* covers;
                                span_aa   = sl_aa.begin();
                                num_spans = sl_aa.num_spans();
                                if(solid)
                                {
                                     
                                     
                                    for(;;)
                                    {
                                        color_type c = sh.color(style);
                                        len    = span_aa->len;
                                        colors = mix_buffer + span_aa->x - min_x;
                                        covers = span_aa->covers;
                                        do
                                        {
                                            if(*covers == cover_full) 
                                            {
                                                *colors = c;
                                            }
                                            else
                                            {
                                                colors->add(c, *covers);
                                            }
                                            ++colors;
                                            ++covers;
                                        }
                                        while(--len);
                                        if(--num_spans == 0) break;
                                        ++span_aa;
                                    }
                                }
                                else
                                {
                                     
                                     
                                    for(;;)
                                    {
                                        len = span_aa->len;
                                        colors = mix_buffer + span_aa->x - min_x;
                                        cspan  = color_span;
                                        sh.generate_span(cspan, 
                                                         span_aa->x, 
                                                         sl_aa.y(), 
                                                         len, 
                                                         style);
                                        covers = span_aa->covers;
                                        do
                                        {
                                            if(*covers == cover_full) 
                                            {
                                                *colors = *cspan;
                                            }
                                            else
                                            {
                                                colors->add(*cspan, *covers);
                                            }
                                            ++cspan;
                                            ++colors;
                                            ++covers;
                                        }
                                        while(--len);
                                        if(--num_spans == 0) break;
                                        ++span_aa;
                                    }
                                }
                            }
                        }

                         
                         
                        span_bin = sl_bin.begin();
                        num_spans = sl_bin.num_spans();
                        for(;;)
                        {
                            ren.blend_color_hspan(span_bin->x, 
                                                  sl_bin.y(), 
                                                  span_bin->len,
                                                  mix_buffer + span_bin->x - min_x,
                                                  0,
                                                  cover_full);
                            if(--num_spans == 0) break;
                            ++span_bin;
                        }
                    }  
                }  
            }  
        }  
    }

     
    template<class Rasterizer, 
             class ScanlineAA, 
             class BaseRenderer, 
             class SpanAllocator,
             class StyleHandler>
    void render_scanlines_compound_layered(Rasterizer& ras, 
                                           ScanlineAA& sl_aa,
                                           BaseRenderer& ren,
                                           SpanAllocator& alloc,
                                           StyleHandler& sh)
    {
        if(ras.rewind_scanlines())
        {
            int min_x = ras.min_x();
            int len = ras.max_x() - min_x + 2;
            sl_aa.reset(min_x, ras.max_x());

            typedef typename BaseRenderer::color_type color_type;
            color_type* color_span   = alloc.allocate(len * 2);
            color_type* mix_buffer   = color_span + len;
            cover_type* cover_buffer = ras.allocate_cover_buffer(len);
            unsigned num_spans;

            unsigned num_styles;
            unsigned style;
            bool     solid;
            while((num_styles = ras.sweep_styles()) > 0)
            {
                typename ScanlineAA::const_iterator span_aa;
                if(num_styles == 1)
                {
                     
                     
                    if(ras.sweep_scanline(sl_aa, 0))
                    {
                        style = ras.style(0);
                        if(sh.is_solid(style))
                        {
                             
                             
                            render_scanline_aa_solid(sl_aa, ren, sh.color(style));
                        }
                        else
                        {
                             
                             
                            span_aa   = sl_aa.begin();
                            num_spans = sl_aa.num_spans();
                            for(;;)
                            {
                                len = span_aa->len;
                                sh.generate_span(color_span, 
                                                 span_aa->x, 
                                                 sl_aa.y(), 
                                                 len, 
                                                 style);

                                ren.blend_color_hspan(span_aa->x, 
                                                      sl_aa.y(), 
                                                      span_aa->len,
                                                      color_span,
                                                      span_aa->covers);
                                if(--num_spans == 0) break;
                                ++span_aa;
                            }
                        }
                    }
                }
                else
                {
                    int      sl_start = ras.scanline_start();
                    unsigned sl_len   = ras.scanline_length();

                    if(sl_len)
                    {
                        std::memset(mix_buffer + sl_start - min_x, 
                               0, 
                               sl_len * sizeof(color_type));

                        std::memset(cover_buffer + sl_start - min_x, 
                               0, 
                               sl_len * sizeof(cover_type));

                        int sl_y = std::numeric_limits<int>::max();
                        unsigned i;
                        for(i = 0; i < num_styles; i++)
                        {
                            style = ras.style(i);
                            solid = sh.is_solid(style);

                            if(ras.sweep_scanline(sl_aa, i))
                            {
                                unsigned    cover;
                                color_type* colors;
                                color_type* cspan;
                                cover_type* src_covers;
                                cover_type* dst_covers;
                                span_aa   = sl_aa.begin();
                                num_spans = sl_aa.num_spans();
                                sl_y      = sl_aa.y();
                                if(solid)
                                {
                                     
                                     
                                    for(;;)
                                    {
                                        color_type c = sh.color(style);
                                        len    = span_aa->len;
                                        colors = mix_buffer + span_aa->x - min_x;
                                        src_covers = span_aa->covers;
                                        dst_covers = cover_buffer + span_aa->x - min_x;
                                        do
                                        {
                                            cover = *src_covers;
                                            if(*dst_covers + cover > cover_full)
                                            {
                                                cover = cover_full - *dst_covers;
                                            }
                                            if(cover)
                                            {
                                                colors->add(c, cover);
                                                *dst_covers += cover;
                                            }
                                            ++colors;
                                            ++src_covers;
                                            ++dst_covers;
                                        }
                                        while(--len);
                                        if(--num_spans == 0) break;
                                        ++span_aa;
                                    }
                                }
                                else
                                {
                                     
                                     
                                    for(;;)
                                    {
                                        len = span_aa->len;
                                        colors = mix_buffer + span_aa->x - min_x;
                                        cspan  = color_span;
                                        sh.generate_span(cspan, 
                                                         span_aa->x, 
                                                         sl_aa.y(), 
                                                         len, 
                                                         style);
                                        src_covers = span_aa->covers;
                                        dst_covers = cover_buffer + span_aa->x - min_x;
                                        do
                                        {
                                            cover = *src_covers;
                                            if(*dst_covers + cover > cover_full)
                                            {
                                                cover = cover_full - *dst_covers;
                                            }
                                            if(cover)
                                            {
                                                colors->add(*cspan, cover);
                                                *dst_covers += cover;
                                            }
                                            ++cspan;
                                            ++colors;
                                            ++src_covers;
                                            ++dst_covers;
                                        }
                                        while(--len);
                                        if(--num_spans == 0) break;
                                        ++span_aa;
                                    }
                                }
                            }
                        }
                        ren.blend_color_hspan(sl_start, 
                                              sl_y, 
                                              sl_len,
                                              mix_buffer + sl_start - min_x,
                                              0,
                                              cover_full);
                    }  
                }  
            }  
        }  
    }


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_RENDERING_BUFFER_DYNAROW_INCLUDED
#define AGG_RENDERING_BUFFER_DYNAROW_INCLUDED

#include <cstring>

namespace agg
{

     
     
     
     
     
     
     
    class rendering_buffer_dynarow
    {
    public:
        typedef row_info<int8u> row_data;

         
        ~rendering_buffer_dynarow()
        {
            init(0,0,0);
        }

         
        rendering_buffer_dynarow() :
            m_rows(),
            m_width(0),
            m_height(0),
            m_byte_width(0)
        {
        }

         
         
        rendering_buffer_dynarow(unsigned width, unsigned height, 
                                 unsigned byte_width) :
            m_rows(height),
            m_width(width),
            m_height(height),
            m_byte_width(byte_width)
        {
            std::memset(&m_rows[0], 0, sizeof(row_data) * height);
        }

         
         
        void init(unsigned width, unsigned height, unsigned byte_width)
        {
            unsigned i;
            for(i = 0; i < m_height; ++i) 
            {
                pod_allocator<int8u>::deallocate((int8u*)m_rows[i].ptr, m_byte_width);
            }
            if(width && height)
            {
                m_width  = width;
                m_height = height;
                m_byte_width = byte_width;
                m_rows.resize(height);
                std::memset(&m_rows[0], 0, sizeof(row_data) * height);
            }
        }

         
        unsigned width()      const { return m_width;  }
        unsigned height()     const { return m_height; }
        unsigned byte_width() const { return m_byte_width; }

         
         
         
        int8u* row_ptr(int x, int y, unsigned len)
        {
            row_data* r = &m_rows[y];
            int x2 = x + len - 1;
            if(r->ptr)
            {
                if(x  < r->x1) { r->x1 = x;  }
                if(x2 > r->x2) { r->x2 = x2; }
            }
            else
            {
                int8u* p = pod_allocator<int8u>::allocate(m_byte_width);
                r->ptr = p;
                r->x1  = x;
                r->x2  = x2;
                std::memset(p, 0, m_byte_width);
            }
            return (int8u*)r->ptr;
        }

         
        const int8u* row_ptr(int y) const { return m_rows[y].ptr; }
              int8u* row_ptr(int y)       { return row_ptr(0, y, m_width); }
        row_data     row    (int y) const { return m_rows[y]; }

    private:
         
         
        rendering_buffer_dynarow(const rendering_buffer_dynarow&);
        const rendering_buffer_dynarow& operator = (const rendering_buffer_dynarow&);

    private:
         
        pod_array<row_data> m_rows;        
        unsigned            m_width;       
        unsigned            m_height;      
        unsigned            m_byte_width;  
    };


}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_ROUNDED_RECT_INCLUDED
#define AGG_ROUNDED_RECT_INCLUDED


namespace agg
{
     
     
     
     
    class rounded_rect
    {
    public:
        rounded_rect() {}
        rounded_rect(double x1, double y1, double x2, double y2, double r);

        void rect(double x1, double y1, double x2, double y2);
        void radius(double r);
        void radius(double rx, double ry);
        void radius(double rx_bottom, double ry_bottom, double rx_top, double ry_top);
        void radius(double rx1, double ry1, double rx2, double ry2, 
                    double rx3, double ry3, double rx4, double ry4);
        void normalize_radius();

        void approximation_scale(double s) { m_arc.approximation_scale(s); }
        double approximation_scale() const { return m_arc.approximation_scale(); }

        void rewind(unsigned);
        unsigned vertex(double* x, double* y);

    private:
        double m_x1;
        double m_y1;
        double m_x2;
        double m_y2;
        double m_rx1;
        double m_ry1;
        double m_rx2;
        double m_ry2;
        double m_rx3;
        double m_ry3;
        double m_rx4;
        double m_ry4;
        unsigned m_status;
        arc      m_arc;
    };

}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SCANLINE_BIN_INCLUDED
#define AGG_SCANLINE_BIN_INCLUDED


namespace agg
{

     
     
     
     
     
     
     
    class scanline_bin
    {
    public:
        typedef int32 coord_type;

        struct span
        {
            int16 x;
            int16 len;
        };

        typedef const span* const_iterator;

         
        scanline_bin() :
            m_last_x(0x7FFFFFF0),
            m_spans(),
            m_cur_span(0)
        {
        }

         
        void reset(int min_x, int max_x)
        {
            unsigned max_len = max_x - min_x + 3;
            if(max_len > m_spans.size())
            {
                m_spans.resize(max_len);
            }
            m_last_x   = 0x7FFFFFF0;
            m_cur_span = &m_spans[0];
        }

         
        void add_cell(int x, unsigned)
        {
            if(x == m_last_x+1)
            {
                m_cur_span->len++;
            }
            else
            {
                ++m_cur_span;
                m_cur_span->x = (int16)x;
                m_cur_span->len = 1;
            }
            m_last_x = x;
        }

         
        void add_span(int x, unsigned len, unsigned)
        {
            if(x == m_last_x+1)
            {
                m_cur_span->len = (int16)(m_cur_span->len + len);
            }
            else
            {
                ++m_cur_span;
                m_cur_span->x = (int16)x;
                m_cur_span->len = (int16)len;
            }
            m_last_x = x + len - 1;
        }

         
        void add_cells(int x, unsigned len, const void*)
        {
            add_span(x, len, 0);
        }

         
        void finalize(int y) 
        { 
            m_y = y; 
        }

         
        void reset_spans()
        {
            m_last_x    = 0x7FFFFFF0;
            m_cur_span  = &m_spans[0];
        }

         
        int            y()         const { return m_y; }
        unsigned       num_spans() const { return unsigned(m_cur_span - &m_spans[0]); }
        const_iterator begin()     const { return &m_spans[1]; }

    private:
        scanline_bin(const scanline_bin&);
        const scanline_bin operator = (const scanline_bin&);

        int             m_last_x;
        int             m_y;
        pod_array<span> m_spans;
        span*           m_cur_span;
    };






     
    class scanline32_bin
    {
    public:
        typedef int32 coord_type;

         
        struct span
        {
            span() {}
            span(coord_type x_, coord_type len_) : x(x_), len(len_) {}

            coord_type x;
            coord_type len;
        };
        typedef pod_bvector<span, 4> span_array_type;


         
        class const_iterator
        {
        public:
            const_iterator(const span_array_type& spans) :
                m_spans(spans),
                m_span_idx(0)
            {}

            const span& operator*()  const { return m_spans[m_span_idx];  }
            const span* operator->() const { return &m_spans[m_span_idx]; }

            void operator ++ () { ++m_span_idx; }

        private:
            const span_array_type& m_spans;
            unsigned               m_span_idx;
        };


         
        scanline32_bin() : m_max_len(0), m_last_x(0x7FFFFFF0) {}

         
        void reset(int, int)
        {
            m_last_x = 0x7FFFFFF0;
            m_spans.remove_all();
        }

         
        void add_cell(int x, unsigned)
        {
            if(x == m_last_x+1)
            {
                m_spans.last().len++;
            }
            else
            {
                m_spans.add(span(coord_type(x), 1));
            }
            m_last_x = x;
        }

         
        void add_span(int x, unsigned len, unsigned)
        {
            if(x == m_last_x+1)
            {
                m_spans.last().len += coord_type(len);
            }
            else
            {
                m_spans.add(span(coord_type(x), coord_type(len)));
            }
            m_last_x = x + len - 1;
        }

         
        void add_cells(int x, unsigned len, const void*)
        {
            add_span(x, len, 0);
        }

         
        void finalize(int y) 
        { 
            m_y = y; 
        }

         
        void reset_spans()
        {
            m_last_x = 0x7FFFFFF0;
            m_spans.remove_all();
        }

         
        int            y()         const { return m_y; }
        unsigned       num_spans() const { return m_spans.size(); }
        const_iterator begin()     const { return const_iterator(m_spans); }

    private:
        scanline32_bin(const scanline32_bin&);
        const scanline32_bin operator = (const scanline32_bin&);

        unsigned        m_max_len;
        int             m_last_x;
        int             m_y;
        span_array_type m_spans;
    };





}


#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SCANLINE_BOOLEAN_ALGEBRA_INCLUDED
#define AGG_SCANLINE_BOOLEAN_ALGEBRA_INCLUDED

#include <cstdlib>


namespace agg
{

     
     
     
     
     
     
    template<class Scanline1, 
             class Scanline2, 
             class Scanline> 
    struct sbool_combine_spans_bin
    {
        void operator () (const typename Scanline1::const_iterator&, 
                          const typename Scanline2::const_iterator&, 
                          int x, unsigned len, 
                          Scanline& sl) const
        {
            sl.add_span(x, len, cover_full);
        }
    };



     
     
     
     
     
    template<class Scanline1, 
             class Scanline2, 
             class Scanline> 
    struct sbool_combine_spans_empty
    {
        void operator () (const typename Scanline1::const_iterator&, 
                          const typename Scanline2::const_iterator&, 
                          int, unsigned, 
                          Scanline&) const
        {}
    };



     
     
     
     
    template<class Scanline1, 
             class Scanline> 
    struct sbool_add_span_empty
    {
        void operator () (const typename Scanline1::const_iterator&, 
                          int, unsigned, 
                          Scanline&) const
        {}
    };


     
     
     
     
    template<class Scanline1, 
             class Scanline> 
    struct sbool_add_span_bin
    {
        void operator () (const typename Scanline1::const_iterator&, 
                          int x, unsigned len, 
                          Scanline& sl) const
        {
            sl.add_span(x, len, cover_full);
        }
    };

    


     
     
     
     
     
     
    template<class Scanline1, 
             class Scanline> 
    struct sbool_add_span_aa
    {
        void operator () (const typename Scanline1::const_iterator& span, 
                          int x, unsigned len, 
                          Scanline& sl) const
        {
            if(span->len < 0)
            {
                sl.add_span(x, len, *span->covers);
            }
            else
            if(span->len > 0)
            {
                const typename Scanline1::cover_type* covers = span->covers;
                if(span->x < x) covers += x - span->x;
                sl.add_cells(x, len, covers);
            }
        }
    };




     
     
     
     
     
    template<class Scanline1, 
             class Scanline2, 
             class Scanline, 
             unsigned CoverShift = cover_shift> 
    struct sbool_intersect_spans_aa
    {
        enum cover_scale_e
        {
            cover_shift = CoverShift,
            cover_size  = 1 << cover_shift,
            cover_mask  = cover_size - 1,
            cover_full  = cover_mask
        };
        

        void operator () (const typename Scanline1::const_iterator& span1, 
                          const typename Scanline2::const_iterator& span2, 
                          int x, unsigned len, 
                          Scanline& sl) const
        {
            unsigned cover;
            const typename Scanline1::cover_type* covers1;
            const typename Scanline2::cover_type* covers2;

             
             
             
             
             
             
             
            switch((span1->len < 0) | ((span2->len < 0) << 1))
            {
            case 0:       
                covers1 = span1->covers;
                covers2 = span2->covers;
                if(span1->x < x) covers1 += x - span1->x;
                if(span2->x < x) covers2 += x - span2->x;
                do
                {
                    cover = *covers1++ * *covers2++;
                    sl.add_cell(x++, 
                                (cover == +cover_full * cover_full) ?
                                cover_full : 
                                (cover >> cover_shift));
                }
                while(--len);
                break;

            case 1:       
                covers2 = span2->covers;
                if(span2->x < x) covers2 += x - span2->x;
                if(*(span1->covers) == cover_full)
                {
                    sl.add_cells(x, len, covers2);
                }
                else
                {
                    do
                    {
                        cover = *(span1->covers) * *covers2++;
                        sl.add_cell(x++, 
                                    (cover == +cover_full * cover_full) ?
                                    cover_full : 
                                    (cover >> cover_shift));
                    }
                    while(--len);
                }
                break;

            case 2:       
                covers1 = span1->covers;
                if(span1->x < x) covers1 += x - span1->x;
                if(*(span2->covers) == cover_full)
                {
                    sl.add_cells(x, len, covers1);
                }
                else
                {
                    do
                    {
                        cover = *covers1++ * *(span2->covers);
                        sl.add_cell(x++, 
                                    (cover == cover_full * cover_full) ?
                                    cover_full : 
                                    (cover >> cover_shift));
                    }
                    while(--len);
                }
                break;

            case 3:       
                cover = *(span1->covers) * *(span2->covers);
                sl.add_span(x, len, 
                            (cover == +cover_full * cover_full) ?
                            cover_full : 
                            (cover >> cover_shift));
                break;
            }
        }
    };






     
     
     
     
     
    template<class Scanline1, 
             class Scanline2, 
             class Scanline, 
             unsigned CoverShift = cover_shift> 
    struct sbool_unite_spans_aa
    {
        enum cover_scale_e
        {
            cover_shift = CoverShift,
            cover_size  = 1 << cover_shift,
            cover_mask  = cover_size - 1,
            cover_full  = cover_mask
        };
        

        void operator () (const typename Scanline1::const_iterator& span1, 
                          const typename Scanline2::const_iterator& span2, 
                          int x, unsigned len, 
                          Scanline& sl) const
        {
            unsigned cover;
            const typename Scanline1::cover_type* covers1;
            const typename Scanline2::cover_type* covers2;

             
             
             
             
             
             
             
            switch((span1->len < 0) | ((span2->len < 0) << 1))
            {
            case 0:       
                covers1 = span1->covers;
                covers2 = span2->covers;
                if(span1->x < x) covers1 += x - span1->x;
                if(span2->x < x) covers2 += x - span2->x;
                do
                {
                    cover = +cover_mask * cover_mask - 
                                (cover_mask - *covers1++) * 
                                (cover_mask - *covers2++);
                    sl.add_cell(x++, 
                                (cover == +cover_full * cover_full) ?
                                cover_full : 
                                (cover >> cover_shift));
                }
                while(--len);
                break;

            case 1:       
                covers2 = span2->covers;
                if(span2->x < x) covers2 += x - span2->x;
                if(*(span1->covers) == cover_full)
                {
                    sl.add_span(x, len, cover_full);
                }
                else
                {
                    do
                    {
                        cover = +cover_mask * cover_mask - 
                                    (cover_mask - *(span1->covers)) * 
                                    (cover_mask - *covers2++);
                        sl.add_cell(x++, 
                                    (cover == +cover_full * cover_full) ?
                                    cover_full : 
                                    (cover >> cover_shift));
                    }
                    while(--len);
                }
                break;

            case 2:       
                covers1 = span1->covers;
                if(span1->x < x) covers1 += x - span1->x;
                if(*(span2->covers) == cover_full)
                {
                    sl.add_span(x, len, cover_full);
                }
                else
                {
                    do
                    {
                        cover = +cover_mask * cover_mask - 
                                    (cover_mask - *covers1++) * 
                                    (cover_mask - *(span2->covers));
                        sl.add_cell(x++, 
                                    (cover == +cover_full * cover_full) ?
                                    cover_full : 
                                    (cover >> cover_shift));
                    }
                    while(--len);
                }
                break;

            case 3:       
                cover = +cover_mask * cover_mask - 
                            (cover_mask - *(span1->covers)) * 
                            (cover_mask - *(span2->covers));
                sl.add_span(x, len, 
                            (cover == +cover_full * cover_full) ?
                            cover_full : 
                            (cover >> cover_shift));
                break;
            }
        }
    };


     
    template<unsigned CoverShift = cover_shift> 
    struct sbool_xor_formula_linear
    {
        enum cover_scale_e
        {
            cover_shift = CoverShift,
            cover_size  = 1 << cover_shift,
            cover_mask  = cover_size - 1
        };

        static AGG_INLINE unsigned calculate(unsigned a, unsigned b)
        {
            unsigned cover = a + b;
            if(cover > cover_mask) cover = +cover_mask + cover_mask - cover;
            return cover;
        }
    };


     
    template<unsigned CoverShift = cover_shift> 
    struct sbool_xor_formula_saddle
    {
        enum cover_scale_e
        {
            cover_shift = CoverShift,
            cover_size  = 1 << cover_shift,
            cover_mask  = cover_size - 1
        };

        static AGG_INLINE unsigned calculate(unsigned a, unsigned b)
        {
            unsigned k = a * b;
            if(k == +cover_mask * cover_mask) return 0;

            a = (+cover_mask * cover_mask - (a << cover_shift) + k) >> cover_shift;
            b = (+cover_mask * cover_mask - (b << cover_shift) + k) >> cover_shift;
            return cover_mask - ((a * b) >> cover_shift);
        }
    };


     
    struct sbool_xor_formula_abs_diff
    {
        static AGG_INLINE unsigned calculate(unsigned a, unsigned b)
        {
            return unsigned(std::abs(int(a) - int(b)));
        }
    };



     
     
     
     
     
    template<class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class XorFormula,
             unsigned CoverShift = cover_shift> 
    struct sbool_xor_spans_aa
    {
        enum cover_scale_e
        {
            cover_shift = CoverShift,
            cover_size  = 1 << cover_shift,
            cover_mask  = cover_size - 1,
            cover_full  = cover_mask
        };
        

        void operator () (const typename Scanline1::const_iterator& span1, 
                          const typename Scanline2::const_iterator& span2, 
                          int x, unsigned len, 
                          Scanline& sl) const
        {
            unsigned cover;
            const typename Scanline1::cover_type* covers1;
            const typename Scanline2::cover_type* covers2;

             
             
             
             
             
             
             
            switch((span1->len < 0) | ((span2->len < 0) << 1))
            {
            case 0:       
                covers1 = span1->covers;
                covers2 = span2->covers;
                if(span1->x < x) covers1 += x - span1->x;
                if(span2->x < x) covers2 += x - span2->x;
                do
                {
                    cover = XorFormula::calculate(*covers1++, *covers2++);
                    if(cover) sl.add_cell(x, cover);
                    ++x;
                }
                while(--len);
                break;

            case 1:       
                covers2 = span2->covers;
                if(span2->x < x) covers2 += x - span2->x;
                do
                {
                    cover = XorFormula::calculate(*(span1->covers), *covers2++);
                    if(cover) sl.add_cell(x, cover);
                    ++x;
                }
                while(--len);
                break;

            case 2:       
                covers1 = span1->covers;
                if(span1->x < x) covers1 += x - span1->x;
                do
                {
                    cover = XorFormula::calculate(*covers1++, *(span2->covers));
                    if(cover) sl.add_cell(x, cover);
                    ++x;
                }
                while(--len);
                break;

            case 3:       
                cover = XorFormula::calculate(*(span1->covers), *(span2->covers));
                if(cover) sl.add_span(x, len, cover);
                break;

            }
        }
    };





     
     
     
     
     
    template<class Scanline1, 
             class Scanline2, 
             class Scanline, 
             unsigned CoverShift = cover_shift> 
    struct sbool_subtract_spans_aa
    {
        enum cover_scale_e
        {
            cover_shift = CoverShift,
            cover_size  = 1 << cover_shift,
            cover_mask  = cover_size - 1,
            cover_full  = cover_mask
        };
        

        void operator () (const typename Scanline1::const_iterator& span1, 
                          const typename Scanline2::const_iterator& span2, 
                          int x, unsigned len, 
                          Scanline& sl) const
        {
            unsigned cover;
            const typename Scanline1::cover_type* covers1;
            const typename Scanline2::cover_type* covers2;

             
             
             
             
             
             
             
            switch((span1->len < 0) | ((span2->len < 0) << 1))
            {
            case 0:       
                covers1 = span1->covers;
                covers2 = span2->covers;
                if(span1->x < x) covers1 += x - span1->x;
                if(span2->x < x) covers2 += x - span2->x;
                do
                {
                    cover = *covers1++ * (cover_mask - *covers2++);
                    if(cover)
                    {
                        sl.add_cell(x, 
                                    (cover == +cover_full * cover_full) ?
                                    cover_full : 
                                    (cover >> cover_shift));
                    }
                    ++x;
                }
                while(--len);
                break;

            case 1:       
                covers2 = span2->covers;
                if(span2->x < x) covers2 += x - span2->x;
                do
                {
                    cover = *(span1->covers) * (cover_mask - *covers2++);
                    if(cover)
                    {
                        sl.add_cell(x, 
                                    (cover == +cover_full * cover_full) ?
                                    cover_full : 
                                    (cover >> cover_shift));
                    }
                    ++x;
                }
                while(--len);
                break;

            case 2:       
                covers1 = span1->covers;
                if(span1->x < x) covers1 += x - span1->x;
                if(*(span2->covers) != cover_full)
                {
                    do
                    {
                        cover = *covers1++ * (cover_mask - *(span2->covers));
                        if(cover)
                        {
                            sl.add_cell(x, 
                                        (cover == +cover_full * cover_full) ?
                                        cover_full : 
                                        (cover >> cover_shift));
                        }
                        ++x;
                    }
                    while(--len);
                }
                break;

            case 3:       
                cover = *(span1->covers) * (cover_mask - *(span2->covers));
                if(cover)
                {
                    sl.add_span(x, len, 
                                (cover == +cover_full * cover_full) ?
                                cover_full : 
                                (cover >> cover_shift));
                }
                break;
            }
        }
    };






     
    template<class Scanline1, 
             class Scanline, 
             class Renderer, 
             class AddSpanFunctor>
    void sbool_add_spans_and_render(const Scanline1& sl1, 
                                    Scanline& sl, 
                                    Renderer& ren, 
                                    AddSpanFunctor add_span)
    {
        sl.reset_spans();
        typename Scanline1::const_iterator span = sl1.begin();
        unsigned num_spans = sl1.num_spans();
        for(;;)
        {
            add_span(span, span->x, std::abs((int)span->len), sl);
            if(--num_spans == 0) break;
            ++span;
        }
        sl.finalize(sl1.y());
        ren.render(sl);
    }







     
     
     
     
     
     
     
    template<class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class CombineSpansFunctor>
    void sbool_intersect_scanlines(const Scanline1& sl1, 
                                   const Scanline2& sl2, 
                                   Scanline& sl, 
                                   CombineSpansFunctor combine_spans)
    {
        sl.reset_spans();

        unsigned num1 = sl1.num_spans();
        if(num1 == 0) return;

        unsigned num2 = sl2.num_spans();
        if(num2 == 0) return;

        typename Scanline1::const_iterator span1 = sl1.begin();
        typename Scanline2::const_iterator span2 = sl2.begin();

        while(num1 && num2)
        {
            int xb1 = span1->x;
            int xb2 = span2->x;
            int xe1 = xb1 + std::abs((int)span1->len) - 1;
            int xe2 = xb2 + std::abs((int)span2->len) - 1;

             
             
             
             
             
            bool advance_span1 = xe1 <  xe2;
            bool advance_both  = xe1 == xe2;

             
             
             
            if(xb1 < xb2) xb1 = xb2;
            if(xe1 > xe2) xe1 = xe2;
            if(xb1 <= xe1)
            {
                combine_spans(span1, span2, xb1, xe1 - xb1 + 1, sl);
            }

             
             
            if(advance_both)
            {
                --num1;
                --num2;
                if(num1) ++span1;
                if(num2) ++span2;
            }
            else
            {
                if(advance_span1)
                {
                    --num1;
                    if(num1) ++span1;
                }
                else
                {
                    --num2;
                    if(num2) ++span2;
                }
            }
        }
    }








     
     
     
     
     
     
     
     
     
     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer,
             class CombineSpansFunctor>
    void sbool_intersect_shapes(ScanlineGen1& sg1, ScanlineGen2& sg2,
                                Scanline1& sl1, Scanline2& sl2,
                                Scanline& sl, Renderer& ren, 
                                CombineSpansFunctor combine_spans)
    {
         
         
         
         
        if(!sg1.rewind_scanlines()) return;
        if(!sg2.rewind_scanlines()) return;

         
         
        rect_i r1(sg1.min_x(), sg1.min_y(), sg1.max_x(), sg1.max_y());
        rect_i r2(sg2.min_x(), sg2.min_y(), sg2.max_x(), sg2.max_y());

         
         
         
        rect_i ir = intersect_rectangles(r1, r2);
        if(!ir.is_valid()) return;

         
         
        sl.reset(ir.x1, ir.x2);
        sl1.reset(sg1.min_x(), sg1.max_x());
        sl2.reset(sg2.min_x(), sg2.max_x());
        if(!sg1.sweep_scanline(sl1)) return;
        if(!sg2.sweep_scanline(sl2)) return;

        ren.prepare();

         
         
         
         
         
         
        for(;;)
        {
            while(sl1.y() < sl2.y())
            {
                if(!sg1.sweep_scanline(sl1)) return;
            }
            while(sl2.y() < sl1.y())
            {
                if(!sg2.sweep_scanline(sl2)) return;
            }

            if(sl1.y() == sl2.y())
            {
                 
                 
                 
                 
                sbool_intersect_scanlines(sl1, sl2, sl, combine_spans);
                if(sl.num_spans())
                {
                    sl.finalize(sl1.y());
                    ren.render(sl);
                }
                if(!sg1.sweep_scanline(sl1)) return;
                if(!sg2.sweep_scanline(sl2)) return;
            }
        }
    }







     
     
     
     
     
     
     
    template<class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class AddSpanFunctor1,
             class AddSpanFunctor2,
             class CombineSpansFunctor>
    void sbool_unite_scanlines(const Scanline1& sl1, 
                               const Scanline2& sl2, 
                               Scanline& sl, 
                               AddSpanFunctor1 add_span1,
                               AddSpanFunctor2 add_span2,
                               CombineSpansFunctor combine_spans)
    {
        sl.reset_spans();

        unsigned num1 = sl1.num_spans();
        unsigned num2 = sl2.num_spans();

        typename Scanline1::const_iterator span1; 
        typename Scanline2::const_iterator span2; 

        enum invalidation_e 
        { 
            invalid_b = 0xFFFFFFF, 
            invalid_e = invalid_b - 1 
        };

         
         
        int xb1 = invalid_b;
        int xb2 = invalid_b;
        int xe1 = invalid_e;
        int xe2 = invalid_e;

         
         
        if(num1)
        {
            span1 = sl1.begin();
            xb1 = span1->x;
            xe1 = xb1 + std::abs((int)span1->len) - 1;
            --num1;
        }

         
         
        if(num2)
        {
            span2 = sl2.begin();
            xb2 = span2->x;
            xe2 = xb2 + std::abs((int)span2->len) - 1;
            --num2;
        }


        for(;;)
        {
             
             
            if(num1 && xb1 > xe1) 
            {
                --num1;
                ++span1;
                xb1 = span1->x;
                xe1 = xb1 + std::abs((int)span1->len) - 1;
            }

             
             
            if(num2 && xb2 > xe2) 
            {
                --num2;
                ++span2;
                xb2 = span2->x;
                xe2 = xb2 + std::abs((int)span2->len) - 1;
            }

            if(xb1 > xe1 && xb2 > xe2) break;

             
             
            int xb = xb1;
            int xe = xe1;
            if(xb < xb2) xb = xb2;
            if(xe > xe2) xe = xe2;
            int len = xe - xb + 1;  
            if(len > 0)
            {
                 
                 
                 
                if(xb1 < xb2)
                {
                    add_span1(span1, xb1, xb2 - xb1, sl);
                    xb1 = xb2;
                }
                else
                if(xb2 < xb1)
                {
                    add_span2(span2, xb2, xb1 - xb2, sl);
                    xb2 = xb1;
                }

                 
                 
                combine_spans(span1, span2, xb, len, sl);


                 
                 
                if(xe1 < xe2)
                {
                     
                     
                     
                    xb1 = invalid_b;    
                    xe1 = invalid_e;
                    xb2 += len;
                }
                else
                if(xe2 < xe1)
                {
                     
                     
                     
                    xb2 = invalid_b;  
                    xe2 = invalid_e;
                    xb1 += len;
                }
                else
                {
                    xb1 = invalid_b;   
                    xb2 = invalid_b;
                    xe1 = invalid_e;
                    xe2 = invalid_e;
                }
            }
            else
            {
                 
                 
                if(xb1 < xb2) 
                {
                     
                     
                    if(xb1 <= xe1)
                    {
                        add_span1(span1, xb1, xe1 - xb1 + 1, sl);
                    }
                    xb1 = invalid_b;  
                    xe1 = invalid_e;
                }
                else
                {
                     
                     
                    if(xb2 <= xe2)
                    {
                        add_span2(span2, xb2, xe2 - xb2 + 1, sl);
                    }
                    xb2 = invalid_b;  
                    xe2 = invalid_e;
                }
            }
        }
    }




     
     
     
     
     
     
     
     
     
     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer,
             class AddSpanFunctor1,
             class AddSpanFunctor2,
             class CombineSpansFunctor>
    void sbool_unite_shapes(ScanlineGen1& sg1, ScanlineGen2& sg2,
                            Scanline1& sl1, Scanline2& sl2,
                            Scanline& sl, Renderer& ren, 
                            AddSpanFunctor1 add_span1,
                            AddSpanFunctor2 add_span2,
                            CombineSpansFunctor combine_spans)
    {
         
         
         
         
        bool flag1 = sg1.rewind_scanlines();
        bool flag2 = sg2.rewind_scanlines();
        if(!flag1 && !flag2) return;

         
         
        rect_i r1(sg1.min_x(), sg1.min_y(), sg1.max_x(), sg1.max_y());
        rect_i r2(sg2.min_x(), sg2.min_y(), sg2.max_x(), sg2.max_y());

         
         
        rect_i ur(1,1,0,0);
             if(flag1 && flag2) ur = unite_rectangles(r1, r2);
        else if(flag1)          ur = r1;
        else if(flag2)          ur = r2;

        if(!ur.is_valid()) return;

        ren.prepare();

         
         
        sl.reset(ur.x1, ur.x2);
        if(flag1) 
        {
            sl1.reset(sg1.min_x(), sg1.max_x());
            flag1 = sg1.sweep_scanline(sl1);
        }

        if(flag2) 
        {
            sl2.reset(sg2.min_x(), sg2.max_x());
            flag2 = sg2.sweep_scanline(sl2);
        }

         
         
         
         
        while(flag1 || flag2)
        {
            if(flag1 && flag2)
            {
                if(sl1.y() == sl2.y())
                {
                     
                     
                     
                     
                    sbool_unite_scanlines(sl1, sl2, sl, 
                                          add_span1, add_span2, combine_spans);
                    if(sl.num_spans())
                    {
                        sl.finalize(sl1.y());
                        ren.render(sl);
                    }
                    flag1 = sg1.sweep_scanline(sl1);
                    flag2 = sg2.sweep_scanline(sl2);
                }
                else
                {
                    if(sl1.y() < sl2.y())
                    {
                        sbool_add_spans_and_render(sl1, sl, ren, add_span1);
                        flag1 = sg1.sweep_scanline(sl1);
                    }
                    else
                    {
                        sbool_add_spans_and_render(sl2, sl, ren, add_span2);
                        flag2 = sg2.sweep_scanline(sl2);
                    }
                }
            }
            else
            {
                if(flag1)
                {
                    sbool_add_spans_and_render(sl1, sl, ren, add_span1);
                    flag1 = sg1.sweep_scanline(sl1);
                }
                if(flag2)
                {
                    sbool_add_spans_and_render(sl2, sl, ren, add_span2);
                    flag2 = sg2.sweep_scanline(sl2);
                }
            }
        }
    }








     
     
     
     
     
     
     
     
     
     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer,
             class AddSpanFunctor1,
             class CombineSpansFunctor>
    void sbool_subtract_shapes(ScanlineGen1& sg1, ScanlineGen2& sg2,
                               Scanline1& sl1, Scanline2& sl2,
                               Scanline& sl, Renderer& ren, 
                               AddSpanFunctor1 add_span1,
                               CombineSpansFunctor combine_spans)
    {
         
         
         
        if(!sg1.rewind_scanlines()) return;
        bool flag2 = sg2.rewind_scanlines();

         
         
        rect_i r1(sg1.min_x(), sg1.min_y(), sg1.max_x(), sg1.max_y());

         
         
        sl.reset(sg1.min_x(), sg1.max_x());
        sl1.reset(sg1.min_x(), sg1.max_x());
        sl2.reset(sg2.min_x(), sg2.max_x());
        if(!sg1.sweep_scanline(sl1)) return;

        if(flag2) flag2 = sg2.sweep_scanline(sl2);

        ren.prepare();

         
        sbool_add_span_empty<Scanline2, Scanline> add_span2;

         
         
         
         
         
         
        bool flag1 = true;
        do
        {
             
             
            while(flag2 && sl2.y() < sl1.y())
            {
                flag2 = sg2.sweep_scanline(sl2);
            }


            if(flag2 && sl2.y() == sl1.y())
            {
                 
                 
                 
                sbool_unite_scanlines(sl1, sl2, sl, add_span1, add_span2, combine_spans);
                if(sl.num_spans())
                {
                    sl.finalize(sl1.y());
                    ren.render(sl);
                }
            }
            else
            {
                sbool_add_spans_and_render(sl1, sl, ren, add_span1);
            }

             
            flag1 = sg1.sweep_scanline(sl1);
        }
        while(flag1);
    }







     
     
     
     
     
     
     
     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_intersect_shapes_aa(ScanlineGen1& sg1, ScanlineGen2& sg2,
                                   Scanline1& sl1, Scanline2& sl2,
                                   Scanline& sl, Renderer& ren)
    {
        sbool_intersect_spans_aa<Scanline1, Scanline2, Scanline> combine_functor;
        sbool_intersect_shapes(sg1, sg2, sl1, sl2, sl, ren, combine_functor);
    }





     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_intersect_shapes_bin(ScanlineGen1& sg1, ScanlineGen2& sg2,
                                    Scanline1& sl1, Scanline2& sl2,
                                    Scanline& sl, Renderer& ren)
    {
        sbool_combine_spans_bin<Scanline1, Scanline2, Scanline> combine_functor;
        sbool_intersect_shapes(sg1, sg2, sl1, sl2, sl, ren, combine_functor);
    }





     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_unite_shapes_aa(ScanlineGen1& sg1, ScanlineGen2& sg2,
                               Scanline1& sl1, Scanline2& sl2,
                               Scanline& sl, Renderer& ren)
    {
        sbool_add_span_aa<Scanline1, Scanline> add_functor1;
        sbool_add_span_aa<Scanline2, Scanline> add_functor2;
        sbool_unite_spans_aa<Scanline1, Scanline2, Scanline> combine_functor;
        sbool_unite_shapes(sg1, sg2, sl1, sl2, sl, ren, 
                           add_functor1, add_functor2, combine_functor);
    }





     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_unite_shapes_bin(ScanlineGen1& sg1, ScanlineGen2& sg2,
                                Scanline1& sl1, Scanline2& sl2,
                                Scanline& sl, Renderer& ren)
    {
        sbool_add_span_bin<Scanline1, Scanline> add_functor1;
        sbool_add_span_bin<Scanline2, Scanline> add_functor2;
        sbool_combine_spans_bin<Scanline1, Scanline2, Scanline> combine_functor;
        sbool_unite_shapes(sg1, sg2, sl1, sl2, sl, ren, 
                           add_functor1, add_functor2, combine_functor);
    }









     
     
     
     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_xor_shapes_aa(ScanlineGen1& sg1, ScanlineGen2& sg2,
                             Scanline1& sl1, Scanline2& sl2,
                             Scanline& sl, Renderer& ren)
    {
        sbool_add_span_aa<Scanline1, Scanline> add_functor1;
        sbool_add_span_aa<Scanline2, Scanline> add_functor2;
        sbool_xor_spans_aa<Scanline1, Scanline2, Scanline, 
                           sbool_xor_formula_linear<> > combine_functor;
        sbool_unite_shapes(sg1, sg2, sl1, sl2, sl, ren, 
                           add_functor1, add_functor2, combine_functor);
    }



     
     
     
     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_xor_shapes_saddle_aa(ScanlineGen1& sg1, ScanlineGen2& sg2,
                                    Scanline1& sl1, Scanline2& sl2,
                                    Scanline& sl, Renderer& ren)
    {
        sbool_add_span_aa<Scanline1, Scanline> add_functor1;
        sbool_add_span_aa<Scanline2, Scanline> add_functor2;
        sbool_xor_spans_aa<Scanline1, 
                           Scanline2, 
                           Scanline, 
                           sbool_xor_formula_saddle<> > combine_functor;
        sbool_unite_shapes(sg1, sg2, sl1, sl2, sl, ren, 
                           add_functor1, add_functor2, combine_functor);
    }


     
     
     
     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_xor_shapes_abs_diff_aa(ScanlineGen1& sg1, ScanlineGen2& sg2,
                                      Scanline1& sl1, Scanline2& sl2,
                                      Scanline& sl, Renderer& ren)
    {
        sbool_add_span_aa<Scanline1, Scanline> add_functor1;
        sbool_add_span_aa<Scanline2, Scanline> add_functor2;
        sbool_xor_spans_aa<Scanline1, 
                           Scanline2, 
                           Scanline, 
                           sbool_xor_formula_abs_diff> combine_functor;
        sbool_unite_shapes(sg1, sg2, sl1, sl2, sl, ren, 
                           add_functor1, add_functor2, combine_functor);
    }



     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_xor_shapes_bin(ScanlineGen1& sg1, ScanlineGen2& sg2,
                              Scanline1& sl1, Scanline2& sl2,
                              Scanline& sl, Renderer& ren)
    {
        sbool_add_span_bin<Scanline1, Scanline> add_functor1;
        sbool_add_span_bin<Scanline2, Scanline> add_functor2;
        sbool_combine_spans_empty<Scanline1, Scanline2, Scanline> combine_functor;
        sbool_unite_shapes(sg1, sg2, sl1, sl2, sl, ren, 
                           add_functor1, add_functor2, combine_functor);
    }






     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_subtract_shapes_aa(ScanlineGen1& sg1, ScanlineGen2& sg2,
                                  Scanline1& sl1, Scanline2& sl2,
                                  Scanline& sl, Renderer& ren)
    {
        sbool_add_span_aa<Scanline1, Scanline> add_functor;
        sbool_subtract_spans_aa<Scanline1, Scanline2, Scanline> combine_functor;
        sbool_subtract_shapes(sg1, sg2, sl1, sl2, sl, ren, 
                              add_functor, combine_functor);
    }





     
     
     
     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_subtract_shapes_bin(ScanlineGen1& sg1, ScanlineGen2& sg2,
                                   Scanline1& sl1, Scanline2& sl2,
                                   Scanline& sl, Renderer& ren)
    {
        sbool_add_span_bin<Scanline1, Scanline> add_functor;
        sbool_combine_spans_empty<Scanline1, Scanline2, Scanline> combine_functor;
        sbool_subtract_shapes(sg1, sg2, sl1, sl2, sl, ren, 
                              add_functor, combine_functor);
    }






     
    enum sbool_op_e
    {
        sbool_or,             
        sbool_and,            
        sbool_xor,            
        sbool_xor_saddle,     
        sbool_xor_abs_diff,   
        sbool_a_minus_b,      
        sbool_b_minus_a       
    };






     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_combine_shapes_bin(sbool_op_e op,
                                  ScanlineGen1& sg1, ScanlineGen2& sg2,
                                  Scanline1& sl1, Scanline2& sl2,
                                  Scanline& sl, Renderer& ren)
    {
        switch(op)
        {
        case sbool_or          : sbool_unite_shapes_bin    (sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_and         : sbool_intersect_shapes_bin(sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_xor         :
        case sbool_xor_saddle  : 
        case sbool_xor_abs_diff: sbool_xor_shapes_bin      (sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_a_minus_b   : sbool_subtract_shapes_bin (sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_b_minus_a   : sbool_subtract_shapes_bin (sg2, sg1, sl2, sl1, sl, ren); break;
        }
    }




     
    template<class ScanlineGen1, 
             class ScanlineGen2, 
             class Scanline1, 
             class Scanline2, 
             class Scanline, 
             class Renderer>
    void sbool_combine_shapes_aa(sbool_op_e op,
                                 ScanlineGen1& sg1, ScanlineGen2& sg2,
                                 Scanline1& sl1, Scanline2& sl2,
                                 Scanline& sl, Renderer& ren)
    {
        switch(op)
        {
        case sbool_or          : sbool_unite_shapes_aa       (sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_and         : sbool_intersect_shapes_aa   (sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_xor         : sbool_xor_shapes_aa         (sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_xor_saddle  : sbool_xor_shapes_saddle_aa  (sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_xor_abs_diff: sbool_xor_shapes_abs_diff_aa(sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_a_minus_b   : sbool_subtract_shapes_aa    (sg1, sg2, sl1, sl2, sl, ren); break;
        case sbool_b_minus_a   : sbool_subtract_shapes_aa    (sg2, sg1, sl2, sl1, sl, ren); break;
        }
    }

}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_SCANLINE_P_INCLUDED
#define AGG_SCANLINE_P_INCLUDED

#include <cstring>

namespace agg
{

     
     
     
     
     
     
     
    class scanline_p8
    {
    public:
        typedef scanline_p8 self_type;
        typedef int8u       cover_type;
        typedef int16       coord_type;

         
        struct span
        {
            coord_type        x;
            coord_type        len;  
            const cover_type* covers;
        };

        typedef span* iterator;
        typedef const span* const_iterator;

        scanline_p8() :
            m_last_x(0x7FFFFFF0),
            m_covers(),
            m_cover_ptr(0),
            m_spans(),
            m_cur_span(0)
        {
        }

         
        void reset(int min_x, int max_x)
        {
            unsigned max_len = max_x - min_x + 3;
            if(max_len > m_spans.size())
            {
                m_spans.resize(max_len);
                m_covers.resize(max_len);
            }
            m_last_x    = 0x7FFFFFF0;
            m_cover_ptr = &m_covers[0];
            m_cur_span  = &m_spans[0];
            m_cur_span->len = 0;
        }

         
        void add_cell(int x, unsigned cover)
        {
            *m_cover_ptr = (cover_type)cover;
            if(x == m_last_x+1 && m_cur_span->len > 0)
            {
                m_cur_span->len++;
            }
            else
            {
                m_cur_span++;
                m_cur_span->covers = m_cover_ptr;
                m_cur_span->x = (int16)x;
                m_cur_span->len = 1;
            }
            m_last_x = x;
            m_cover_ptr++;
        }

         
        void add_cells(int x, unsigned len, const cover_type* covers)
        {
            std::memcpy(m_cover_ptr, covers, len * sizeof(cover_type));
            if(x == m_last_x+1 && m_cur_span->len > 0)
            {
                m_cur_span->len += (int16)len;
            }
            else
            {
                m_cur_span++;
                m_cur_span->covers = m_cover_ptr;
                m_cur_span->x = (int16)x;
                m_cur_span->len = (int16)len;
            }
            m_cover_ptr += len;
            m_last_x = x + len - 1;
        }

         
        void add_span(int x, unsigned len, unsigned cover)
        {
            if(x == m_last_x+1 && 
               m_cur_span->len < 0 && 
               cover == *m_cur_span->covers)
            {
                m_cur_span->len -= (int16)len;
            }
            else
            {
                *m_cover_ptr = (cover_type)cover;
                m_cur_span++;
                m_cur_span->covers = m_cover_ptr++;
                m_cur_span->x      = (int16)x;
                m_cur_span->len    = (int16)(-int(len));
            }
            m_last_x = x + len - 1;
        }

         
        void finalize(int y) 
        { 
            m_y = y; 
        }

         
        void reset_spans()
        {
            m_last_x    = 0x7FFFFFF0;
            m_cover_ptr = &m_covers[0];
            m_cur_span  = &m_spans[0];
            m_cur_span->len = 0;
        }

         
        int            y()         const { return m_y; }
        unsigned       num_spans() const { return unsigned(m_cur_span - &m_spans[0]); }
        const_iterator begin()     const { return &m_spans[1]; }

    private:
        scanline_p8(const self_type&);
        const self_type& operator = (const self_type&);

        int                   m_last_x;
        int                   m_y;
        pod_array<cover_type> m_covers;
        cover_type*           m_cover_ptr;
        pod_array<span>       m_spans;
        span*                 m_cur_span;
    };








     
    class scanline32_p8
    {
    public:
        typedef scanline32_p8 self_type;
        typedef int8u         cover_type;
        typedef int32         coord_type;

        struct span
        {
            span() {}
            span(coord_type x_, coord_type len_, const cover_type* covers_) :
                x(x_), len(len_), covers(covers_) {}

            coord_type x;
            coord_type len;  
            const cover_type* covers;
        };
        typedef pod_bvector<span, 4> span_array_type;


         
        class const_iterator
        {
        public:
            const_iterator(const span_array_type& spans) :
                m_spans(spans),
                m_span_idx(0)
            {}

            const span& operator*()  const { return m_spans[m_span_idx];  }
            const span* operator->() const { return &m_spans[m_span_idx]; }

            void operator ++ () { ++m_span_idx; }

        private:
            const span_array_type& m_spans;
            unsigned               m_span_idx;
        };

         
        scanline32_p8() :
            m_max_len(0),
            m_last_x(0x7FFFFFF0),
            m_covers(),
            m_cover_ptr(0)
        {
        }

         
        void reset(int min_x, int max_x)
        {
            unsigned max_len = max_x - min_x + 3;
            if(max_len > m_covers.size())
            {
                m_covers.resize(max_len);
            }
            m_last_x    = 0x7FFFFFF0;
            m_cover_ptr = &m_covers[0];
            m_spans.remove_all();
        }

         
        void add_cell(int x, unsigned cover)
        {
            *m_cover_ptr = cover_type(cover);
            if(x == m_last_x+1 && m_spans.size() && m_spans.last().len > 0)
            {
                m_spans.last().len++;
            }
            else
            {
                m_spans.add(span(coord_type(x), 1, m_cover_ptr));
            }
            m_last_x = x;
            m_cover_ptr++;
        }

         
        void add_cells(int x, unsigned len, const cover_type* covers)
        {
            std::memcpy(m_cover_ptr, covers, len * sizeof(cover_type));
            if(x == m_last_x+1 && m_spans.size() && m_spans.last().len > 0)
            {
                m_spans.last().len += coord_type(len);
            }
            else
            {
                m_spans.add(span(coord_type(x), coord_type(len), m_cover_ptr));
            }
            m_cover_ptr += len;
            m_last_x = x + len - 1;
        }

         
        void add_span(int x, unsigned len, unsigned cover)
        {
            if(x == m_last_x+1 && 
               m_spans.size() &&
               m_spans.last().len < 0 && 
               cover == *m_spans.last().covers)
            {
                m_spans.last().len -= coord_type(len);
            }
            else
            {
                *m_cover_ptr = cover_type(cover);
                m_spans.add(span(coord_type(x), -coord_type(len), m_cover_ptr++));
            }
            m_last_x = x + len - 1;
        }

         
        void finalize(int y) 
        { 
            m_y = y; 
        }

         
        void reset_spans()
        {
            m_last_x    = 0x7FFFFFF0;
            m_cover_ptr = &m_covers[0];
            m_spans.remove_all();
        }

         
        int            y()         const { return m_y; }
        unsigned       num_spans() const { return m_spans.size(); }
        const_iterator begin()     const { return const_iterator(m_spans); }

    private:
        scanline32_p8(const self_type&);
        const self_type& operator = (const self_type&);

        unsigned              m_max_len;
        int                   m_last_x;
        int                   m_y;
        pod_array<cover_type> m_covers;
        cover_type*           m_cover_ptr;
        span_array_type       m_spans;
    };


}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SCANLINE_STORAGE_AA_INCLUDED
#define AGG_SCANLINE_STORAGE_AA_INCLUDED

#include <cstring>
#include <cstdlib>
#include <limits>


namespace agg
{

     
    template<class T> class scanline_cell_storage
    {
        struct extra_span
        {
            unsigned len;
            T*       ptr;
        };

    public:
        typedef T value_type;

         
        ~scanline_cell_storage()
        {
            remove_all();
        }

         
        scanline_cell_storage() :
            m_cells(128-2),
            m_extra_storage()
        {}


         
         
        scanline_cell_storage(const scanline_cell_storage<T>& v) :
            m_cells(v.m_cells),
            m_extra_storage()
        {
            copy_extra_storage(v);
        }

         
        const scanline_cell_storage<T>& 
        operator = (const scanline_cell_storage<T>& v)
        {
            remove_all();
            m_cells = v.m_cells;
            copy_extra_storage(v);
            return *this;
        }

         
        void remove_all()
        {
            int i;
            for(i = m_extra_storage.size()-1; i >= 0; --i)
            {
                pod_allocator<T>::deallocate(m_extra_storage[i].ptr,
                                             m_extra_storage[i].len);
            }
            m_extra_storage.remove_all();
            m_cells.remove_all();
        }

         
        int add_cells(const T* cells, unsigned num_cells)
        {
            int idx = m_cells.allocate_continuous_block(num_cells);
            if(idx >= 0)
            {
                T* ptr = &m_cells[idx];
                std::memcpy(ptr, cells, sizeof(T) * num_cells);
                return idx;
            }
            extra_span s;
            s.len = num_cells;
            s.ptr = pod_allocator<T>::allocate(num_cells);
            std::memcpy(s.ptr, cells, sizeof(T) * num_cells);
            m_extra_storage.add(s);
            return -int(m_extra_storage.size());
        }

         
        const T* operator [] (int idx) const
        {
            if(idx >= 0)
            {
                if((unsigned)idx >= m_cells.size()) return 0;
                return &m_cells[(unsigned)idx];
            }
            unsigned i = unsigned(-idx - 1);
            if(i >= m_extra_storage.size()) return 0;
            return m_extra_storage[i].ptr;
        }

         
        T* operator [] (int idx)
        {
            if(idx >= 0)
            {
                if((unsigned)idx >= m_cells.size()) return 0;
                return &m_cells[(unsigned)idx];
            }
            unsigned i = unsigned(-idx - 1);
            if(i >= m_extra_storage.size()) return 0;
            return m_extra_storage[i].ptr;
        }

    private:
        void copy_extra_storage(const scanline_cell_storage<T>& v)
        {
            unsigned i;
            for(i = 0; i < v.m_extra_storage.size(); ++i)
            {
                const extra_span& src = v.m_extra_storage[i];
                extra_span dst;
                dst.len = src.len;
                dst.ptr = pod_allocator<T>::allocate(dst.len);
                std::memcpy(dst.ptr, src.ptr, dst.len * sizeof(T));
                m_extra_storage.add(dst);
            }
        }

        pod_bvector<T, 12>         m_cells;
        pod_bvector<extra_span, 6> m_extra_storage;
    };






     
    template<class T> class scanline_storage_aa
    {
    public:
        typedef T cover_type;

         
        struct span_data
        {
            int32 x;
            int32 len;        
            int   covers_id;  
        };

         
        struct scanline_data
        {
            int      y;
            unsigned num_spans;
            unsigned start_span;
        };


         
        class embedded_scanline
        {
        public:

             
            class const_iterator
            {
            public:
                struct span
                {
                    int32    x;
                    int32    len;  
                    const T* covers;
                };

                const_iterator() : m_storage(0) {}
                const_iterator(embedded_scanline& sl) :
                    m_storage(sl.m_storage),
                    m_span_idx(sl.m_scanline.start_span)
                {
                    init_span();
                }

                const span& operator*()  const { return m_span;  }
                const span* operator->() const { return &m_span; }

                void operator ++ ()
                {
                    ++m_span_idx;
                    init_span();
                }

            private:
                void init_span()
                {
                    const span_data& s = m_storage->span_by_index(m_span_idx);
                    m_span.x      = s.x;
                    m_span.len    = s.len;
                    m_span.covers = m_storage->covers_by_index(s.covers_id);
                }

                scanline_storage_aa* m_storage;
                unsigned                   m_span_idx;
                span                       m_span;
            };

            friend class const_iterator;


             
            embedded_scanline(const scanline_storage_aa& storage) :
                m_storage(&storage)
            {
                init(0);
            }

             
            void     reset(int, int)     {}
            unsigned num_spans()   const { return m_scanline.num_spans;  }
            int      y()           const { return m_scanline.y;          }
            const_iterator begin() const { return const_iterator(*this); }

             
            void init(unsigned scanline_idx)
            {
                m_scanline_idx = scanline_idx;
                m_scanline = m_storage->scanline_by_index(m_scanline_idx);
            }

        private:
            const scanline_storage_aa* m_storage;
            scanline_data              m_scanline;
            unsigned                   m_scanline_idx;
        };


         
        scanline_storage_aa() :
            m_covers(),
            m_spans(256-2),          
            m_scanlines(),
            m_min_x(std::numeric_limits<int>::max()),
            m_min_y(std::numeric_limits<int>::max()),
            m_max_x(std::numeric_limits<int>::min()),
            m_max_y(std::numeric_limits<int>::min()),
            m_cur_scanline(0)
        {
            m_fake_scanline.y = 0;
            m_fake_scanline.num_spans = 0;
            m_fake_scanline.start_span = 0;
            m_fake_span.x = 0;
            m_fake_span.len = 0;
            m_fake_span.covers_id = 0;
        }

         
         
        void prepare()
        {
            m_covers.remove_all();
            m_scanlines.remove_all();
            m_spans.remove_all();
            m_min_x = std::numeric_limits<int>::max();
            m_min_y = std::numeric_limits<int>::max();
            m_max_x = std::numeric_limits<int>::min();
            m_max_y = std::numeric_limits<int>::min();
            m_cur_scanline = 0;
        }

         
        template<class Scanline> void render(const Scanline& sl)
        {
            scanline_data sl_this;

            int y = sl.y();
            if(y < m_min_y) m_min_y = y;
            if(y > m_max_y) m_max_y = y;

            sl_this.y = y;
            sl_this.num_spans = sl.num_spans();
            sl_this.start_span = m_spans.size();
            typename Scanline::const_iterator span_iterator = sl.begin();

            unsigned num_spans = sl_this.num_spans;
            for(;;)
            {
                span_data sp;

                sp.x         = span_iterator->x;
                sp.len       = span_iterator->len;
                int len      = std::abs(int(sp.len));
                sp.covers_id = 
                    m_covers.add_cells(span_iterator->covers, 
                                       unsigned(len));
                m_spans.add(sp);
                int x1 = sp.x;
                int x2 = sp.x + len - 1;
                if(x1 < m_min_x) m_min_x = x1;
                if(x2 > m_max_x) m_max_x = x2;
                if(--num_spans == 0) break;
                ++span_iterator;
            }
            m_scanlines.add(sl_this);
        }


         
         
        int min_x() const { return m_min_x; }
        int min_y() const { return m_min_y; }
        int max_x() const { return m_max_x; }
        int max_y() const { return m_max_y; }

         
        bool rewind_scanlines()
        {
            m_cur_scanline = 0;
            return m_scanlines.size() > 0;
        }


         
        template<class Scanline> bool sweep_scanline(Scanline& sl)
        {
            sl.reset_spans();
            for(;;)
            {
                if(m_cur_scanline >= m_scanlines.size()) return false;
                const scanline_data& sl_this = m_scanlines[m_cur_scanline];

                unsigned num_spans = sl_this.num_spans;
                unsigned span_idx  = sl_this.start_span;
                do
                {
                    const span_data& sp = m_spans[span_idx++];
                    const T* covers = covers_by_index(sp.covers_id);
                    if(sp.len < 0)
                    {
                        sl.add_span(sp.x, unsigned(-sp.len), *covers);
                    }
                    else
                    {
                        sl.add_cells(sp.x, sp.len, covers);
                    }
                }
                while(--num_spans);
                ++m_cur_scanline;
                if(sl.num_spans())
                {
                    sl.finalize(sl_this.y);
                    break;
                }
            }
            return true;
        }


         
         
        bool sweep_scanline(embedded_scanline& sl)
        {
            do
            {
                if(m_cur_scanline >= m_scanlines.size()) return false;
                sl.init(m_cur_scanline);
                ++m_cur_scanline;
            }
            while(sl.num_spans() == 0);
            return true;
        }

         
        unsigned byte_size() const
        {
            unsigned i;
            unsigned size = sizeof(int32) * 4;  

            for(i = 0; i < m_scanlines.size(); ++i)
            {
                size += sizeof(int32) * 3;  

                const scanline_data& sl_this = m_scanlines[i];

                unsigned num_spans = sl_this.num_spans;
                unsigned span_idx  = sl_this.start_span;
                do
                {
                    const span_data& sp = m_spans[span_idx++];

                    size += sizeof(int32) * 2;                 
                    if(sp.len < 0)
                    {
                        size += sizeof(T);                     
                    }
                    else
                    {
                        size += sizeof(T) * unsigned(sp.len);  
                    }
                }
                while(--num_spans);
            }
            return size;
        }


         
        static void write_int32(int8u* dst, int32 val)
        {
            dst[0] = ((const int8u*)&val)[0];
            dst[1] = ((const int8u*)&val)[1];
            dst[2] = ((const int8u*)&val)[2];
            dst[3] = ((const int8u*)&val)[3];
        }


         
        void serialize(int8u* data) const
        {
            unsigned i;

            write_int32(data, min_x());  
            data += sizeof(int32);
            write_int32(data, min_y());  
            data += sizeof(int32);
            write_int32(data, max_x());  
            data += sizeof(int32);
            write_int32(data, max_y());  
            data += sizeof(int32);

            for(i = 0; i < m_scanlines.size(); ++i)
            {
                const scanline_data& sl_this = m_scanlines[i];
                
                int8u* size_ptr = data;
                data += sizeof(int32);   

                write_int32(data, sl_this.y);             
                data += sizeof(int32);

                write_int32(data, sl_this.num_spans);     
                data += sizeof(int32);

                unsigned num_spans = sl_this.num_spans;
                unsigned span_idx  = sl_this.start_span;
                do
                {
                    const span_data& sp = m_spans[span_idx++];
                    const T* covers = covers_by_index(sp.covers_id);

                    write_int32(data, sp.x);             
                    data += sizeof(int32);

                    write_int32(data, sp.len);           
                    data += sizeof(int32);

                    if(sp.len < 0)
                    {
                        std::memcpy(data, covers, sizeof(T));
                        data += sizeof(T);
                    }
                    else
                    {
                        std::memcpy(data, covers, unsigned(sp.len) * sizeof(T));
                        data += sizeof(T) * unsigned(sp.len);
                    }
                }
                while(--num_spans);
                write_int32(size_ptr, int32(unsigned(data - size_ptr)));
            }
        }


         
        const scanline_data& scanline_by_index(unsigned i) const
        {
            return (i < m_scanlines.size()) ? m_scanlines[i] : m_fake_scanline;
        }

         
        const span_data& span_by_index(unsigned i) const
        {
            return (i < m_spans.size()) ? m_spans[i] : m_fake_span;
        }

         
        const T* covers_by_index(int i) const
        {
            return m_covers[i];
        }

    private:
        scanline_cell_storage<T>      m_covers;
        pod_bvector<span_data, 10>    m_spans;
        pod_bvector<scanline_data, 8> m_scanlines;
        span_data     m_fake_span;
        scanline_data m_fake_scanline;
        int           m_min_x;
        int           m_min_y;
        int           m_max_x;
        int           m_max_y;
        unsigned      m_cur_scanline;
    };


    typedef scanline_storage_aa<int8u>  scanline_storage_aa8;   
    typedef scanline_storage_aa<int16u> scanline_storage_aa16;  
    typedef scanline_storage_aa<int32u> scanline_storage_aa32;  




     
    template<class T> class serialized_scanlines_adaptor_aa
    {
    public:
        typedef T cover_type;

         
        class embedded_scanline
        {
        public:
            typedef T cover_type;

             
            class const_iterator
            {
            public:
                struct span
                {
                    int32    x;
                    int32    len;  
                    const T* covers; 
                };

                const_iterator() : m_ptr(0) {}
                const_iterator(const embedded_scanline* sl) :
                    m_ptr(sl->m_ptr),
                    m_dx(sl->m_dx)
                {
                    init_span();
                }

                const span& operator*()  const { return m_span;  }
                const span* operator->() const { return &m_span; }

                void operator ++ ()
                {
                    if(m_span.len < 0) 
                    {
                        m_ptr += sizeof(T);
                    }
                    else 
                    {
                        m_ptr += m_span.len * sizeof(T);
                    }
                    init_span();
                }

            private:
                int read_int32()
                {
                    int32 val;
                    ((int8u*)&val)[0] = *m_ptr++;
                    ((int8u*)&val)[1] = *m_ptr++;
                    ((int8u*)&val)[2] = *m_ptr++;
                    ((int8u*)&val)[3] = *m_ptr++;
                    return val;
                }

                void init_span()
                {
                    m_span.x      = read_int32() + m_dx;
                    m_span.len    = read_int32();
                    m_span.covers = m_ptr;
                }

                const int8u* m_ptr;
                span         m_span;
                int          m_dx;
            };

            friend class const_iterator;


             
            embedded_scanline() : m_ptr(0), m_y(0), m_num_spans(0) {}

             
            void     reset(int, int)     {}
            unsigned num_spans()   const { return m_num_spans;  }
            int      y()           const { return m_y;          }
            const_iterator begin() const { return const_iterator(this); }


        private:
             
            int read_int32()
            {
                int32 val;
                ((int8u*)&val)[0] = *m_ptr++;
                ((int8u*)&val)[1] = *m_ptr++;
                ((int8u*)&val)[2] = *m_ptr++;
                ((int8u*)&val)[3] = *m_ptr++;
                return val;
            }

        public:
             
            void init(const int8u* ptr, int dx, int dy)
            {
                m_ptr       = ptr;
                m_y         = read_int32() + dy;
                m_num_spans = unsigned(read_int32());
                m_dx        = dx;
            }

        private:
            const int8u* m_ptr;
            int          m_y;
            unsigned     m_num_spans;
            int          m_dx;
        };



    public:
         
        serialized_scanlines_adaptor_aa() :
            m_data(0),
            m_end(0),
            m_ptr(0),
            m_dx(0),
            m_dy(0),
            m_min_x(std::numeric_limits<int>::max()),
            m_min_y(std::numeric_limits<int>::max()),
            m_max_x(std::numeric_limits<int>::min()),
            m_max_y(std::numeric_limits<int>::min())
        {}

         
        serialized_scanlines_adaptor_aa(const int8u* data, unsigned size,
                                        double dx, double dy) :
            m_data(data),
            m_end(data + size),
            m_ptr(data),
            m_dx(iround(dx)),
            m_dy(iround(dy)),
            m_min_x(std::numeric_limits<int>::max()),
            m_min_y(std::numeric_limits<int>::max()),
            m_max_x(std::numeric_limits<int>::min()),
            m_max_y(std::numeric_limits<int>::min())
        {}

         
        void init(const int8u* data, unsigned size, double dx, double dy)
        {
            m_data  = data;
            m_end   = data + size;
            m_ptr   = data;
            m_dx    = iround(dx);
            m_dy    = iround(dy);
            m_min_x = std::numeric_limits<int>::max();
            m_min_y = std::numeric_limits<int>::max();
            m_max_x = std::numeric_limits<int>::min();
            m_max_y = std::numeric_limits<int>::min();
        }

    private:
         
        int read_int32()
        {
            int32 val;
            ((int8u*)&val)[0] = *m_ptr++;
            ((int8u*)&val)[1] = *m_ptr++;
            ((int8u*)&val)[2] = *m_ptr++;
            ((int8u*)&val)[3] = *m_ptr++;
            return val;
        }

         
        unsigned read_int32u()
        {
            int32u val;
            ((int8u*)&val)[0] = *m_ptr++;
            ((int8u*)&val)[1] = *m_ptr++;
            ((int8u*)&val)[2] = *m_ptr++;
            ((int8u*)&val)[3] = *m_ptr++;
            return val;
        }
        
    public:
         
         
        bool rewind_scanlines()
        {
            m_ptr = m_data;
            if(m_ptr < m_end)
            {
                m_min_x = read_int32() + m_dx; 
                m_min_y = read_int32() + m_dy;
                m_max_x = read_int32() + m_dx;
                m_max_y = read_int32() + m_dy;
            }
            return m_ptr < m_end;
        }

         
        int min_x() const { return m_min_x; }
        int min_y() const { return m_min_y; }
        int max_x() const { return m_max_x; }
        int max_y() const { return m_max_y; }

         
        template<class Scanline> bool sweep_scanline(Scanline& sl)
        {
            sl.reset_spans();
            for(;;)
            {
                if(m_ptr >= m_end) return false;

                read_int32();       
                int y = read_int32() + m_dy;
                unsigned num_spans = read_int32();

                do
                {
                    int x = read_int32() + m_dx;
                    int len = read_int32();

                    if(len < 0)
                    {
                        sl.add_span(x, unsigned(-len), *m_ptr);
                        m_ptr += sizeof(T);
                    }
                    else
                    {
                        sl.add_cells(x, len, m_ptr);
                        m_ptr += len * sizeof(T);
                    }
                }
                while(--num_spans);

                if(sl.num_spans())
                {
                    sl.finalize(y);
                    break;
                }
            }
            return true;
        }


         
         
        bool sweep_scanline(embedded_scanline& sl)
        {
            do
            {
                if(m_ptr >= m_end) return false;

                unsigned byte_size = read_int32u();
                sl.init(m_ptr, m_dx, m_dy);
                m_ptr += byte_size - sizeof(int32);
            }
            while(sl.num_spans() == 0);
            return true;
        }

    private:
        const int8u* m_data;
        const int8u* m_end;
        const int8u* m_ptr;
        int          m_dx;
        int          m_dy;
        int          m_min_x;
        int          m_min_y;
        int          m_max_x;
        int          m_max_y;
    };



    typedef serialized_scanlines_adaptor_aa<int8u>  serialized_scanlines_adaptor_aa8;   
    typedef serialized_scanlines_adaptor_aa<int16u> serialized_scanlines_adaptor_aa16;  
    typedef serialized_scanlines_adaptor_aa<int32u> serialized_scanlines_adaptor_aa32;  

}


#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 


#ifndef AGG_SCANLINE_STORAGE_BIN_INCLUDED
#define AGG_SCANLINE_STORAGE_BIN_INCLUDED

#include <cstdlib>
#include <limits>


namespace agg
{

     
    class scanline_storage_bin
    {
    public:
         
        struct span_data
        {
            int32 x;
            int32 len;
        };

         
        struct scanline_data
        {
            int      y;
            unsigned num_spans;
            unsigned start_span;
        };


         
        class embedded_scanline
        {
        public:

             
            class const_iterator
            {
            public:
                const_iterator() : m_storage(0) {}
                const_iterator(const embedded_scanline* sl) :
                    m_storage(sl->m_storage),
                    m_span_idx(sl->m_scanline.start_span)
                {
                    m_span = m_storage->span_by_index(m_span_idx);
                }

                const span_data& operator*()  const { return m_span;  }
                const span_data* operator->() const { return &m_span; }

                void operator ++ ()
                {
                    ++m_span_idx;
                    m_span = m_storage->span_by_index(m_span_idx);
                }

            private:
                const scanline_storage_bin* m_storage;
                unsigned                    m_span_idx;
                span_data                   m_span;
            };

            friend class const_iterator;


             
            embedded_scanline(scanline_storage_bin& storage) :
                m_storage(&storage)
            {
                setup(0);
            }

             
            void     reset(int, int)     {}
            unsigned num_spans()   const { return m_scanline.num_spans;  }
            int      y()           const { return m_scanline.y;          }
            const_iterator begin() const { return const_iterator(this); }

             
            void setup(unsigned scanline_idx)
            {
                m_scanline_idx = scanline_idx;
                m_scanline = m_storage->scanline_by_index(m_scanline_idx);
            }

        private:
            scanline_storage_bin*       m_storage;
            scanline_data               m_scanline;
            unsigned                    m_scanline_idx;
        };


         
        scanline_storage_bin() :
            m_spans(256-2),          
            m_scanlines(),
            m_min_x(std::numeric_limits<int>::max()),
            m_min_y(std::numeric_limits<int>::max()),
            m_max_x(std::numeric_limits<int>::min()),
            m_max_y(std::numeric_limits<int>::min()),
            m_cur_scanline(0)
        {
            m_fake_scanline.y = 0;
            m_fake_scanline.num_spans = 0;
            m_fake_scanline.start_span = 0;
            m_fake_span.x = 0;
            m_fake_span.len = 0;
        }

         
         
        void prepare()
        {
            m_scanlines.remove_all();
            m_spans.remove_all();
            m_min_x = std::numeric_limits<int>::max();
            m_min_y = std::numeric_limits<int>::max();
            m_max_x = std::numeric_limits<int>::min();
            m_max_y = std::numeric_limits<int>::min();
            m_cur_scanline = 0;
        }

         
        template<class Scanline> void render(const Scanline& sl)
        {
            scanline_data sl_this;

            int y = sl.y();
            if(y < m_min_y) m_min_y = y;
            if(y > m_max_y) m_max_y = y;

            sl_this.y = y;
            sl_this.num_spans = sl.num_spans();
            sl_this.start_span = m_spans.size();
            typename Scanline::const_iterator span_iterator = sl.begin();

            unsigned num_spans = sl_this.num_spans;
            for(;;)
            {
                span_data sp;
                sp.x   = span_iterator->x;
                sp.len = (int32)std::abs((int)(span_iterator->len));
                m_spans.add(sp);
                int x1 = sp.x;
                int x2 = sp.x + sp.len - 1;
                if(x1 < m_min_x) m_min_x = x1;
                if(x2 > m_max_x) m_max_x = x2;
                if(--num_spans == 0) break;
                ++span_iterator;
            }
            m_scanlines.add(sl_this);
        }


         
         
        int min_x() const { return m_min_x; }
        int min_y() const { return m_min_y; }
        int max_x() const { return m_max_x; }
        int max_y() const { return m_max_y; }

         
        bool rewind_scanlines()
        {
            m_cur_scanline = 0;
            return m_scanlines.size() > 0;
        }


         
        template<class Scanline> bool sweep_scanline(Scanline& sl)
        {
            sl.reset_spans();
            for(;;)
            {
                if(m_cur_scanline >= m_scanlines.size()) return false;
                const scanline_data& sl_this = m_scanlines[m_cur_scanline];

                unsigned num_spans = sl_this.num_spans;
                unsigned span_idx  = sl_this.start_span;
                do
                {
                    const span_data& sp = m_spans[span_idx++];
                    sl.add_span(sp.x, sp.len, cover_full);
                }
                while(--num_spans);

                ++m_cur_scanline;
                if(sl.num_spans())
                {
                    sl.finalize(sl_this.y);
                    break;
                }
            }
            return true;
        }


         
         
        bool sweep_scanline(embedded_scanline& sl)
        {
            do
            {
                if(m_cur_scanline >= m_scanlines.size()) return false;
                sl.setup(m_cur_scanline);
                ++m_cur_scanline;
            }
            while(sl.num_spans() == 0);
            return true;
        }


         
        unsigned byte_size() const
        {
            unsigned i;
            unsigned size = sizeof(int32) * 4;  

            for(i = 0; i < m_scanlines.size(); ++i)
            {
                size += sizeof(int32) * 2 +  
                        unsigned(m_scanlines[i].num_spans) * sizeof(int32) * 2;  
            }
            return size;
        }


         
        static void write_int32(int8u* dst, int32 val)
        {
            dst[0] = ((const int8u*)&val)[0];
            dst[1] = ((const int8u*)&val)[1];
            dst[2] = ((const int8u*)&val)[2];
            dst[3] = ((const int8u*)&val)[3];
        }


         
        void serialize(int8u* data) const
        {
            unsigned i;

            write_int32(data, min_x());  
            data += sizeof(int32);
            write_int32(data, min_y());  
            data += sizeof(int32);
            write_int32(data, max_x());  
            data += sizeof(int32);
            write_int32(data, max_y());  
            data += sizeof(int32);

            for(i = 0; i < m_scanlines.size(); ++i)
            {
                const scanline_data& sl_this = m_scanlines[i];

                write_int32(data, sl_this.y);             
                data += sizeof(int32);

                write_int32(data, sl_this.num_spans);     
                data += sizeof(int32);

                unsigned num_spans = sl_this.num_spans;
                unsigned span_idx  = sl_this.start_span;
                do
                {
                    const span_data& sp = m_spans[span_idx++];

                    write_int32(data, sp.x);              
                    data += sizeof(int32);

                    write_int32(data, sp.len);            
                    data += sizeof(int32);
                }
                while(--num_spans);
            }
        }


         
        const scanline_data& scanline_by_index(unsigned i) const
        {
            return (i < m_scanlines.size()) ? m_scanlines[i] : m_fake_scanline;
        }

         
        const span_data& span_by_index(unsigned i) const
        {
            return (i < m_spans.size()) ? m_spans[i] : m_fake_span;
        }


    private:
        pod_bvector<span_data, 10>    m_spans;
        pod_bvector<scanline_data, 8> m_scanlines;
        span_data     m_fake_span;
        scanline_data m_fake_scanline;
        int           m_min_x;
        int           m_min_y;
        int           m_max_x;
        int           m_max_y;
        unsigned      m_cur_scanline;
    };













     
    class serialized_scanlines_adaptor_bin
    {
    public:
        typedef bool cover_type;

         
        class embedded_scanline
        {
        public:

             
            class const_iterator
            {
            public:
                struct span
                {
                    int32 x;
                    int32 len;
                };

                const_iterator() : m_ptr(0) {}
                const_iterator(const embedded_scanline* sl) :
                    m_ptr(sl->m_ptr),
                    m_dx(sl->m_dx)
                {
                    m_span.x   = read_int32() + m_dx;
                    m_span.len = read_int32();
                }

                const span& operator*()  const { return m_span;  }
                const span* operator->() const { return &m_span; }

                void operator ++ ()
                {
                    m_span.x   = read_int32() + m_dx;
                    m_span.len = read_int32();
                }

            private:
                int read_int32()
                {
                    int32 val;
                    ((int8u*)&val)[0] = *m_ptr++;
                    ((int8u*)&val)[1] = *m_ptr++;
                    ((int8u*)&val)[2] = *m_ptr++;
                    ((int8u*)&val)[3] = *m_ptr++;
                    return val;
                }

                const int8u* m_ptr;
                span         m_span;
                int          m_dx;
            };

            friend class const_iterator;


             
            embedded_scanline() : m_ptr(0), m_y(0), m_num_spans(0) {}

             
            void     reset(int, int)     {}
            unsigned num_spans()   const { return m_num_spans;  }
            int      y()           const { return m_y;          }
            const_iterator begin() const { return const_iterator(this); }


        private:
             
            int read_int32()
            {
                int32 val;
                ((int8u*)&val)[0] = *m_ptr++;
                ((int8u*)&val)[1] = *m_ptr++;
                ((int8u*)&val)[2] = *m_ptr++;
                ((int8u*)&val)[3] = *m_ptr++;
                return val;
            }

        public:
             
            void init(const int8u* ptr, int dx, int dy)
            {
                m_ptr       = ptr;
                m_y         = read_int32() + dy;
                m_num_spans = unsigned(read_int32());
                m_dx        = dx;
            }

        private:
            const int8u* m_ptr;
            int          m_y;
            unsigned     m_num_spans;
            int          m_dx;
        };



    public:
         
        serialized_scanlines_adaptor_bin() :
            m_data(0),
            m_end(0),
            m_ptr(0),
            m_dx(0),
            m_dy(0),
            m_min_x(std::numeric_limits<int>::max()),
            m_min_y(std::numeric_limits<int>::max()),
            m_max_x(std::numeric_limits<int>::min()),
            m_max_y(std::numeric_limits<int>::min())
        {}

         
        serialized_scanlines_adaptor_bin(const int8u* data, unsigned size,
                                         double dx, double dy) :
            m_data(data),
            m_end(data + size),
            m_ptr(data),
            m_dx(iround(dx)),
            m_dy(iround(dy)),
            m_min_x(std::numeric_limits<int>::max()),
            m_min_y(std::numeric_limits<int>::max()),
            m_max_x(std::numeric_limits<int>::min()),
            m_max_y(std::numeric_limits<int>::min())
        {}

         
        void init(const int8u* data, unsigned size, double dx, double dy)
        {
            m_data  = data;
            m_end   = data + size;
            m_ptr   = data;
            m_dx    = iround(dx);
            m_dy    = iround(dy);
            m_min_x = std::numeric_limits<int>::max();
            m_min_y = std::numeric_limits<int>::max();
            m_max_x = std::numeric_limits<int>::min();
            m_max_y = std::numeric_limits<int>::min();
        }

    private:
         
        int read_int32()
        {
            int32 val;
            ((int8u*)&val)[0] = *m_ptr++;
            ((int8u*)&val)[1] = *m_ptr++;
            ((int8u*)&val)[2] = *m_ptr++;
            ((int8u*)&val)[3] = *m_ptr++;
            return val;
        }
       
    public:
         
         
        bool rewind_scanlines()
        {
            m_ptr = m_data;
            if(m_ptr < m_end)
            {
                m_min_x = read_int32() + m_dx; 
                m_min_y = read_int32() + m_dy;
                m_max_x = read_int32() + m_dx;
                m_max_y = read_int32() + m_dy;
            }
            return m_ptr < m_end;
        }

         
        int min_x() const { return m_min_x; }
        int min_y() const { return m_min_y; }
        int max_x() const { return m_max_x; }
        int max_y() const { return m_max_y; }

         
        template<class Scanline> bool sweep_scanline(Scanline& sl)
        {
            sl.reset_spans();
            for(;;)
            {
                if(m_ptr >= m_end) return false;

                int y = read_int32() + m_dy;
                unsigned num_spans = read_int32();

                do
                {
                    int x = read_int32() + m_dx;
                    int len = read_int32();

                    if(len < 0) len = -len;
                    sl.add_span(x, unsigned(len), cover_full);
                }
                while(--num_spans);

                if(sl.num_spans())
                {
                    sl.finalize(y);
                    break;
                }
            }
            return true;
        }


         
         
        bool sweep_scanline(embedded_scanline& sl)
        {
            do
            {
                if(m_ptr >= m_end) return false;

                sl.init(m_ptr, m_dx, m_dy);

                 
                 
                read_int32();                     
                int num_spans = read_int32();     
                m_ptr += num_spans * sizeof(int32) * 2;
            }
            while(sl.num_spans() == 0);
            return true;
        }

    private:
        const int8u* m_data;
        const int8u* m_end;
        const int8u* m_ptr;
        int          m_dx;
        int          m_dy;
        int          m_min_x;
        int          m_min_y;
        int          m_max_x;
        int          m_max_y;
    };



}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SCANLINE_U_INCLUDED
#define AGG_SCANLINE_U_INCLUDED

#include <cstring>

namespace agg
{
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
    class scanline_u8
    {
    public:
        typedef scanline_u8 self_type;
        typedef int8u       cover_type;
        typedef int16       coord_type;

         
        struct span
        {
            coord_type  x;
            coord_type  len;
            cover_type* covers;
        };

        typedef span* iterator;
        typedef const span* const_iterator;

         
        scanline_u8() :
            m_min_x(0),
            m_last_x(0x7FFFFFF0),
            m_cur_span(0)
        {}

         
        void reset(int min_x, int max_x)
        {
            unsigned max_len = max_x - min_x + 2;
            if(max_len > m_spans.size())
            {
                m_spans.resize(max_len);
                m_covers.resize(max_len);
            }
            m_last_x   = 0x7FFFFFF0;
            m_min_x    = min_x;
            m_cur_span = &m_spans[0];
        }

         
        void add_cell(int x, unsigned cover)
        {
            x -= m_min_x;
            m_covers[x] = (cover_type)cover;
            if(x == m_last_x+1)
            {
                m_cur_span->len++;
            }
            else
            {
                m_cur_span++;
                m_cur_span->x      = (coord_type)(x + m_min_x);
                m_cur_span->len    = 1;
                m_cur_span->covers = &m_covers[x];
            }
            m_last_x = x;
        }

         
        void add_cells(int x, unsigned len, const cover_type* covers)
        {
            x -= m_min_x;
            std::memcpy(&m_covers[x], covers, len * sizeof(cover_type));
            if(x == m_last_x+1)
            {
                m_cur_span->len += (coord_type)len;
            }
            else
            {
                m_cur_span++;
                m_cur_span->x      = (coord_type)(x + m_min_x);
                m_cur_span->len    = (coord_type)len;
                m_cur_span->covers = &m_covers[x];
            }
            m_last_x = x + len - 1;
        }

         
        void add_span(int x, unsigned len, unsigned cover)
        {
            x -= m_min_x;
            std::memset(&m_covers[x], cover, len);
            if(x == m_last_x+1)
            {
                m_cur_span->len += (coord_type)len;
            }
            else
            {
                m_cur_span++;
                m_cur_span->x      = (coord_type)(x + m_min_x);
                m_cur_span->len    = (coord_type)len;
                m_cur_span->covers = &m_covers[x];
            }
            m_last_x = x + len - 1;
        }

         
        void finalize(int y) 
        { 
            m_y = y; 
        }

         
        void reset_spans()
        {
            m_last_x    = 0x7FFFFFF0;
            m_cur_span  = &m_spans[0];
        }

         
        int      y()           const { return m_y; }
        unsigned num_spans()   const { return unsigned(m_cur_span - &m_spans[0]); }
        const_iterator begin() const { return &m_spans[1]; }
        iterator       begin()       { return &m_spans[1]; }

    private:
        scanline_u8(const self_type&);
        const self_type& operator = (const self_type&);

    private:
        int                   m_min_x;
        int                   m_last_x;
        int                   m_y;
        pod_array<cover_type> m_covers;
        pod_array<span>       m_spans;
        span*                 m_cur_span;
    };




     
     
     
     
     
    template<class AlphaMask> 
    class scanline_u8_am : public scanline_u8
    {
    public:
        typedef scanline_u8           base_type;
        typedef AlphaMask             alpha_mask_type;
        typedef base_type::cover_type cover_type;
        typedef base_type::coord_type coord_type;

        scanline_u8_am() : base_type(), m_alpha_mask(0) {}
        scanline_u8_am(AlphaMask& am) : base_type(), m_alpha_mask(&am) {}

         
        void finalize(int span_y)
        {
            base_type::finalize(span_y);
            if(m_alpha_mask)
            {
                typename base_type::iterator span = base_type::begin();
                unsigned count = base_type::num_spans();
                do
                {
                    m_alpha_mask->combine_hspan(span->x, 
                                                base_type::y(), 
                                                span->covers, 
                                                span->len);
                    ++span;
                }
                while(--count);
            }
        }

    private:
        AlphaMask* m_alpha_mask;
    };




     
    class scanline32_u8
    {
    public:
        typedef scanline32_u8 self_type;
        typedef int8u         cover_type;
        typedef int32         coord_type;

         
        struct span
        {
            span() {}
            span(coord_type x_, coord_type len_, cover_type* covers_) :
                x(x_), len(len_), covers(covers_) {}

            coord_type  x;
            coord_type  len;
            cover_type* covers;
        };

        typedef pod_bvector<span, 4> span_array_type;

         
        class const_iterator
        {
        public:
            const_iterator(const span_array_type& spans) :
                m_spans(spans),
                m_span_idx(0)
            {}

            const span& operator*()  const { return m_spans[m_span_idx];  }
            const span* operator->() const { return &m_spans[m_span_idx]; }

            void operator ++ () { ++m_span_idx; }

        private:
            const span_array_type& m_spans;
            unsigned               m_span_idx;
        };

         
        class iterator
        {
        public:
            iterator(span_array_type& spans) :
                m_spans(spans),
                m_span_idx(0)
            {}

            span& operator*()  { return m_spans[m_span_idx];  }
            span* operator->() { return &m_spans[m_span_idx]; }

            void operator ++ () { ++m_span_idx; }

        private:
            span_array_type& m_spans;
            unsigned         m_span_idx;
        };



         
        scanline32_u8() :
            m_min_x(0),
            m_last_x(0x7FFFFFF0),
            m_covers()
        {}

         
        void reset(int min_x, int max_x)
        {
            unsigned max_len = max_x - min_x + 2;
            if(max_len > m_covers.size())
            {
                m_covers.resize(max_len);
            }
            m_last_x = 0x7FFFFFF0;
            m_min_x  = min_x;
            m_spans.remove_all();
        }

         
        void add_cell(int x, unsigned cover)
        {
            x -= m_min_x;
            m_covers[x] = cover_type(cover);
            if(x == m_last_x+1)
            {
                m_spans.last().len++;
            }
            else
            {
                m_spans.add(span(coord_type(x + m_min_x), 1, &m_covers[x]));
            }
            m_last_x = x;
        }

         
        void add_cells(int x, unsigned len, const cover_type* covers)
        {
            x -= m_min_x;
            std::memcpy(&m_covers[x], covers, len * sizeof(cover_type));
            if(x == m_last_x+1)
            {
                m_spans.last().len += coord_type(len);
            }
            else
            {
                m_spans.add(span(coord_type(x + m_min_x), 
                                 coord_type(len), 
                                 &m_covers[x]));
            }
            m_last_x = x + len - 1;
        }

         
        void add_span(int x, unsigned len, unsigned cover)
        {
            x -= m_min_x;
            std::memset(&m_covers[x], cover, len);
            if(x == m_last_x+1)
            {
                m_spans.last().len += coord_type(len);
            }
            else
            {
                m_spans.add(span(coord_type(x + m_min_x), 
                                 coord_type(len), 
                                 &m_covers[x]));
            }
            m_last_x = x + len - 1;
        }

         
        void finalize(int y) 
        { 
            m_y = y; 
        }

         
        void reset_spans()
        {
            m_last_x = 0x7FFFFFF0;
            m_spans.remove_all();
        }

         
        int      y()           const { return m_y; }
        unsigned num_spans()   const { return m_spans.size(); }
        const_iterator begin() const { return const_iterator(m_spans); }
        iterator       begin()       { return iterator(m_spans); }

    private:
        scanline32_u8(const self_type&);
        const self_type& operator = (const self_type&);

    private:
        int                   m_min_x;
        int                   m_last_x;
        int                   m_y;
        pod_array<cover_type> m_covers;
        span_array_type       m_spans;
    };




     
     
     
     
     
    template<class AlphaMask> 
    class scanline32_u8_am : public scanline32_u8
    {
    public:
        typedef scanline32_u8         base_type;
        typedef AlphaMask             alpha_mask_type;
        typedef base_type::cover_type cover_type;
        typedef base_type::coord_type coord_type;


        scanline32_u8_am() : base_type(), m_alpha_mask(0) {}
        scanline32_u8_am(AlphaMask& am) : base_type(), m_alpha_mask(&am) {}

         
        void finalize(int span_y)
        {
            base_type::finalize(span_y);
            if(m_alpha_mask)
            {
                typename base_type::iterator span = base_type::begin();
                unsigned count = base_type::num_spans();
                do
                {
                    m_alpha_mask->combine_hspan(span->x, 
                                                base_type::y(), 
                                                span->covers, 
                                                span->len);
                    ++span;
                }
                while(--count);
            }
        }

    private:
        AlphaMask* m_alpha_mask;
    };



}

#endif

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
#ifndef AGG_SIMUL_EQ_INCLUDED
#define AGG_SIMUL_EQ_INCLUDED

#include <cmath>

namespace agg
{

     
    template<class T> void swap_arrays(T* a1, T* a2, unsigned n)
    {
        unsigned i;
        for(i = 0; i < n; i++)
        {
            T tmp = *a1;
            *a1++ = *a2;
            *a2++ = tmp;
        }
    }


     
    template<unsigned Rows, unsigned Cols>
    struct matrix_pivot
    {
        static int pivot(double m[Rows][Cols], unsigned row)
        {
            int k = int(row);
            double max_val, tmp;

            max_val = -1.0;
            unsigned i;
            for(i = row; i < Rows; i++)
            {
                if((tmp = std::fabs(m[i][row])) > max_val && tmp != 0.0)
                {
                    max_val = tmp;
                    k = i;
                }
            }

            if(m[k][row] == 0.0)
            {
                return -1;
            }

            if(k != int(row))
            {
                swap_arrays(m[k], m[row], Cols);
                return k;
            }
            return 0;
        }
    };
    


     
    template<unsigned Size, unsigned RightCols>
    struct simul_eq
    {
        static bool solve(const double left[Size][Size], 
                          const double right[Size][RightCols],
                          double result[Size][RightCols])
        {
            unsigned i, j, k;
            double a1;

            double tmp[Size][Size + RightCols];

            for(i = 0; i < Size; i++)
            {
                for(j = 0; j < Size; j++)
                {
                    tmp[i][j] = left[i][j];
                } 
                for(j = 0; j < RightCols; j++)
                {
                    tmp[i][Size + j] = right[i][j];
                }
            }

            for(k = 0; k < Size; k++)
            {
                if(matrix_pivot<Size, Size + RightCols>::pivot(tmp, k) < 0)
                {
                    return false;  
                }

                a1 = tmp[k][k];

                for(j = k; j < Size + RightCols; j++)
                {
                    tmp[k][j] /= a1;
                }

                for(i = k + 1; i < Size; i++)
                {
                    a1 = tmp[i][k];
                    for (j = k; j < Size + RightCols; j++)
                    {
                        tmp[i][j] -= a1 * tmp[k][j];
                    }
                }
            }


            for(k = 0; k < RightCols; k++)
            {
                int m;
                for(m = int(Size - 1); m >= 0; m--)
                {
                    result[m][k] = tmp[m][Size + k];
                    for(j = m + 1; j < Size; j++)
                    {
                        result[m][k] -= tmp[m][j] * result[j][k];
                    }
                }
            }
            return true;
        }

    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SPAN_ALLOCATOR_INCLUDED
#define AGG_SPAN_ALLOCATOR_INCLUDED


namespace agg
{
     
    template<class ColorT> class span_allocator
    {
    public:
        typedef ColorT color_type;

         
        AGG_INLINE color_type* allocate(unsigned span_len)
        {
            if(span_len > m_span.size())
            {
                 
                 
                 
                 
                m_span.resize(((span_len + 255) >> 8) << 8);
            }
            return &m_span[0];
        }

        AGG_INLINE color_type* span()               { return &m_span[0]; }
        AGG_INLINE unsigned    max_span_len() const { return m_span.size(); }

    private:
        pod_array<color_type> m_span;
    };
}


#endif


 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SPAN_CONVERTER_INCLUDED
#define AGG_SPAN_CONVERTER_INCLUDED


namespace agg
{
     
    template<class SpanGenerator, class SpanConverter> class span_converter
    {
    public:
        typedef typename SpanGenerator::color_type color_type;

        span_converter(SpanGenerator& span_gen, SpanConverter& span_cnv) : 
            m_span_gen(&span_gen), m_span_cnv(&span_cnv) {}

        void attach_generator(SpanGenerator& span_gen) { m_span_gen = &span_gen; }
        void attach_converter(SpanConverter& span_cnv) { m_span_cnv = &span_cnv; }

         
        void prepare() 
        { 
            m_span_gen->prepare(); 
            m_span_cnv->prepare();
        }

         
        void generate(color_type* span, int x, int y, unsigned len)
        {
            m_span_gen->generate(span, x, y, len);
            m_span_cnv->generate(span, x, y, len);
        }

    private:
        SpanGenerator* m_span_gen;
        SpanConverter* m_span_cnv;
    };

}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SPAN_GOURAUD_GRAY_INCLUDED
#define AGG_SPAN_GOURAUD_GRAY_INCLUDED

#include <cmath>
#include <cstdlib>
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SPAN_GOURAUD_INCLUDED
#define AGG_SPAN_GOURAUD_INCLUDED


namespace agg
{

     
    template<class ColorT> class span_gouraud
    {
    public:
        typedef ColorT color_type;

        struct coord_type
        {
            double x;
            double y;
            color_type color;
        };

         
        span_gouraud() : 
            m_vertex(0)
        {
            m_cmd[0] = path_cmd_stop;
        }

         
        span_gouraud(const color_type& c1,
                     const color_type& c2,
                     const color_type& c3,
                     double x1, double y1,
                     double x2, double y2,
                     double x3, double y3,
                     double d) : 
            m_vertex(0)
        {
            colors(c1, c2, c3);
            triangle(x1, y1, x2, y2, x3, y3, d);
        }

         
        void colors(ColorT c1, ColorT c2, ColorT c3)
        {
            m_coord[0].color = c1;
            m_coord[1].color = c2;
            m_coord[2].color = c3;
        }

         
         
         
         
         
         
         
        void triangle(double x1, double y1, 
                      double x2, double y2,
                      double x3, double y3,
                      double d)
        {
            m_coord[0].x = m_x[0] = x1; 
            m_coord[0].y = m_y[0] = y1;
            m_coord[1].x = m_x[1] = x2; 
            m_coord[1].y = m_y[1] = y2;
            m_coord[2].x = m_x[2] = x3; 
            m_coord[2].y = m_y[2] = y3;
            m_cmd[0] = path_cmd_move_to;
            m_cmd[1] = path_cmd_line_to;
            m_cmd[2] = path_cmd_line_to;
            m_cmd[3] = path_cmd_stop;

            if(d != 0.0)
            {   
                dilate_triangle(m_coord[0].x, m_coord[0].y,
                                m_coord[1].x, m_coord[1].y,
                                m_coord[2].x, m_coord[2].y,
                                m_x, m_y, d);

                calc_intersection(m_x[4], m_y[4], m_x[5], m_y[5],
                                  m_x[0], m_y[0], m_x[1], m_y[1],
                                  &m_coord[0].x, &m_coord[0].y);

                calc_intersection(m_x[0], m_y[0], m_x[1], m_y[1],
                                  m_x[2], m_y[2], m_x[3], m_y[3],
                                  &m_coord[1].x, &m_coord[1].y);

                calc_intersection(m_x[2], m_y[2], m_x[3], m_y[3],
                                  m_x[4], m_y[4], m_x[5], m_y[5],
                                  &m_coord[2].x, &m_coord[2].y);
                m_cmd[3] = path_cmd_line_to;
                m_cmd[4] = path_cmd_line_to;
                m_cmd[5] = path_cmd_line_to;
                m_cmd[6] = path_cmd_stop;
            }
        }

         
         
        void rewind(unsigned)
        {
            m_vertex = 0;
        }

         
        unsigned vertex(double* x, double* y)
        {
            *x = m_x[m_vertex];
            *y = m_y[m_vertex];
            return m_cmd[m_vertex++];
        }

    protected:
         
        void arrange_vertices(coord_type* coord) const
        {
            coord[0] = m_coord[0];
            coord[1] = m_coord[1];
            coord[2] = m_coord[2];

            if(m_coord[0].y > m_coord[2].y)
            {
                coord[0] = m_coord[2]; 
                coord[2] = m_coord[0];
            }

            coord_type tmp;
            if(coord[0].y > coord[1].y)
            {
                tmp      = coord[1];
                coord[1] = coord[0];
                coord[0] = tmp;
            }

            if(coord[1].y > coord[2].y)
            {
                tmp      = coord[2];
                coord[2] = coord[1];
                coord[1] = tmp;
            }
       }

    private:
         
        coord_type m_coord[3];
        double m_x[8];
        double m_y[8];
        unsigned m_cmd[8];
        unsigned m_vertex;
    };

}

#endif


namespace agg
{

     
    template<class ColorT> class span_gouraud_gray : public span_gouraud<ColorT>
    {
    public:
        typedef ColorT color_type;
        typedef typename color_type::value_type value_type;
        typedef span_gouraud<color_type> base_type;
        typedef typename base_type::coord_type coord_type;
        enum subpixel_scale_e
        { 
            subpixel_shift = 4, 
            subpixel_scale = 1 << subpixel_shift
        };

    private:
         
        struct gray_calc
        {
            void init(const coord_type& c1, const coord_type& c2)
            {
                m_x1  = c1.x - 0.5;
                m_y1  = c1.y - 0.5;
                m_dx  = c2.x - c1.x;
                double dy = c2.y - c1.y;
                m_1dy = (std::fabs(dy) < 1e-10) ? 1e10 : 1.0 / dy;
                m_v1 = c1.color.v;
                m_a1 = c1.color.a;
                m_dv = c2.color.v - m_v1;
                m_da = c2.color.a - m_a1;
            }

            void calc(double y)
            {
                double k = (y - m_y1) * m_1dy;
                if(k < 0.0) k = 0.0;
                if(k > 1.0) k = 1.0;
                m_v = m_v1 + iround(m_dv * k);
                m_a = m_a1 + iround(m_da * k);
                m_x = iround((m_x1 + m_dx * k) * (double)subpixel_scale);
            }

            double m_x1;
            double m_y1;
            double m_dx;
            double m_1dy;
            int    m_v1;
            int    m_a1;
            int    m_dv;
            int    m_da;
            int    m_v;
            int    m_a;
            int    m_x;
        };


    public:
         
        span_gouraud_gray() {}
        span_gouraud_gray(const color_type& c1, 
                          const color_type& c2, 
                          const color_type& c3,
                          double x1, double y1, 
                          double x2, double y2,
                          double x3, double y3, 
                          double d = 0) : 
            base_type(c1, c2, c3, x1, y1, x2, y2, x3, y3, d)
        {}

         
        void prepare()
        {
            coord_type coord[3];
            base_type::arrange_vertices(coord);

            m_y2 = int(coord[1].y);

            m_swap = cross_product(coord[0].x, coord[0].y, 
                                   coord[2].x, coord[2].y,
                                   coord[1].x, coord[1].y) < 0.0;

            m_c1.init(coord[0], coord[2]);
            m_c2.init(coord[0], coord[1]);
            m_c3.init(coord[1], coord[2]);
        }

         
        void generate(color_type* span, int x, int y, unsigned len)
        {
            m_c1.calc(y);
            const gray_calc* pc1 = &m_c1;
            const gray_calc* pc2 = &m_c2;

            if(y < m_y2)
            {
                 
                 
                m_c2.calc(y + m_c2.m_1dy);
            }
            else
            {
                 
                 
                m_c3.calc(y - m_c3.m_1dy);
                pc2 = &m_c3;
            }

            if(m_swap)
            {
                 
                 
                 
                const gray_calc* t = pc2;
                pc2 = pc1;
                pc1 = t;
            }

             
             
             
            int nlen = std::abs(pc2->m_x - pc1->m_x);
            if(nlen <= 0) nlen = 1;

            dda_line_interpolator<14> v(pc1->m_v, pc2->m_v, nlen);
            dda_line_interpolator<14> a(pc1->m_a, pc2->m_a, nlen);

             
             
             
             
             
            int start = pc1->m_x - (x << subpixel_shift);
            v    -= start; 
            a    -= start;
            nlen += start;

            int vv, va;
            enum lim_e { lim = color_type::base_mask };

             
             
             
             
             
             
            while(len && start > 0)
            {
                vv = v.y();
                va = a.y();
                if(vv < 0) vv = 0; 
                if(vv > lim) vv = lim;
                if(va < 0) va = 0; 
                if(va > lim) va = lim;
                span->v = (value_type)vv;
                span->a = (value_type)va;
                v     += subpixel_scale; 
                a     += subpixel_scale;
                nlen  -= subpixel_scale;
                start -= subpixel_scale;
                ++span;
                --len;
            }

             
             
             
             
             
            while(len && nlen > 0)
            {
                span->v = (value_type)v.y();
                span->a = (value_type)a.y();
                v    += subpixel_scale; 
                a    += subpixel_scale;
                nlen -= subpixel_scale;
                ++span;
                --len;
            }

             
             
             
            while(len)
            {
                vv = v.y();
                va = a.y();
                if(vv < 0) vv = 0; 
                if(vv > lim) vv = lim;
                if(va < 0) va = 0; 
                if(va > lim) va = lim;
                span->v = (value_type)vv;
                span->a = (value_type)va;
                v += subpixel_scale; 
                a += subpixel_scale;
                ++span;
                --len;
            }
        }


    private:
        bool      m_swap;
        int       m_y2;
        gray_calc m_c1;
        gray_calc m_c2;
        gray_calc m_c3;
    };


}

#endif
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 

#ifndef AGG_SPAN_GOURAUD_RGBA_INCLUDED
#define AGG_SPAN_GOURAUD_RGBA_INCLUDED

#include <cstdlib>

namespace agg
{

     
    template<class ColorT> class span_gouraud_rgba : public span_gouraud<ColorT>
    {
    public:
        typedef ColorT color_type;
        typedef typename ColorT::value_type value_type;
        typedef span_gouraud<color_type> base_type;
        typedef typename base_type::coord_type coord_type;
        enum subpixel_scale_e
        { 
            subpixel_shift = 4, 
            subpixel_scale = 1 << subpixel_shift
        };

    private:
         
        struct rgba_calc
        {
            void init(const coord_type& c1, const coord_type& c2)
            {
                m_x1  = c1.x - 0.5; 
                m_y1  = c1.y - 0.5;
                m_dx  = c2.x - c1.x;
                double dy = c2.y - c1.y;
                m_1dy = (dy < 1e-5) ? 1e5 : 1.0 / dy;
                m_r1  = c1.color.r;
                m_g1  = c1.color.g;
                m_b1  = c1.color.b;
                m_a1  = c1.color.a;
                m_dr  = c2