/**
Copyright 2026 Sukru Cinar

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation 
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 'AS IS'
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.

   This is sinfil-6jan2026  see 
 
       kt8216.unixcab.org/sinfil
  
   for more details.
 **/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdint.h>

typedef unsigned char uchar;
typedef unsigned int  uint;

int getspec(uchar **arg);
void specs(int *argc, char **argv);

enum {
  sp_source, sp_bpref, sp_qpath, sp_outfile,
  sp_defmac, sp_var, sp_end, sp_braw, sp_unprotect,
  sp_replace, sp_copyright, sp_print,
};

#ifndef CRITBIT_H_INCLUDED
#define CRITBIT_H_INCLUDED

#ifdef __cplusplus
extern "C" {
#endif

typedef struct {
  void *root;
} strmap_t;

strmap_t *strmap_new();
void *strmap_find(strmap_t *map, char *key);
void *strmap_insert(strmap_t *map, char *key,void *data);
void *strmap_remove(strmap_t *map, char *key);
void *strmap_destroy(strmap_t *map);

#ifdef __cplusplus
};
#endif

#endif

typedef struct ipt
{
  char *fn;
  uchar* data;
  uchar* cur;
  struct ipt *up;
} IPT;

IPT *iptup(IPT *P);
IPT *source(IPT *par, char *fn, char *afn);

uchar *getword(uchar **Ra);

enum  { endo, binc, qinc, minc, raw };

int line(IPT *ipt, uchar **arg);


void readfile(uchar **Rf, uint *Rl, char *fn, int nzero);
int openout(char *fn);
void writeln(int fd, char *str);
uchar *mkpath(uchar *dir, uchar *fil);
int file_exists(uchar *fn);
uchar *file_sibling(uchar *fn, uchar *sibnam);
void file_append(int fd, char *fn);


void define_var(uchar *name, uchar *value);
uchar *find_var(uchar *name);
uchar* expand_vars(uchar* os);
void cmdlinevars(int *argc,char**argv);

int cfile, logg;
IPT *ipt;

typedef struct SLnode {
  struct SLnode *next; 
  uchar *data;
  uchar *name;
} SLnode_t;

typedef struct
{
  SLnode_t *head, *tail;
} SLIST;

void slistput(SLIST *S, uchar *name, uchar *data)
{
  SLnode_t *n;
  n= malloc(sizeof(*n));
  n->data= data;
  n->name= name;
  if (S->head) S->tail->next= n;
           else S->head= n;
  S->tail= n;  
}

SLIST* slistnew() { return calloc(1,sizeof(SLIST)); }

strmap_t *macros;
strmap_t *bprefs;
strmap_t *included;
strmap_t *unprotected;
strmap_t *braws;
strmap_t *repla;
SLIST *qpaths;

void qpath(uchar *arg)
{
  if (!qpaths) qpaths= slistnew();
  slistput(qpaths, NULL, expand_vars(arg));
}

void dobraw(uchar *arg)
{
  uchar *name, *value;
  name= getword(&arg);
  value= expand_vars(getword(&arg));    
  if (strmap_find(braws,(char*)name))
     strmap_remove(braws, (char*)name);
  strmap_insert(braws, (char*) name, value);
}

void defmac(uchar *arg)
{
  uchar *name, *value;
  name= getword(&arg);
  value= expand_vars(getword(&arg));    
  if (strmap_find(macros, (char*) name))
     strmap_remove(macros, (char*) name);
  strmap_insert(macros, (char*) name, value);
}

void bpref(uchar *arg)
{
  uchar *pref, *path;
  SLIST *ll;
  pref= getword(&arg);
  path= expand_vars(getword(&arg));
  ll= strmap_find(bprefs, (char*) pref);
  if (!ll)
  {
    ll= slistnew();
    strmap_insert(bprefs, (char*) pref, ll);
  }
  slistput(ll, NULL, path);
}

void dovar(uchar *arg)
{
  uchar *name, *value;
  name= getword(&arg);
  value= expand_vars(getword(&arg));  
  define_var(name, value);
}

static uchar *ustrchr(uchar *k, uchar v)
{ return (uchar*) strchr((char*) k, v); }


static void include_file(uchar *fn)
{
  if (strmap_find(unprotected, (char*) fn))
  {
    dprintf(logg,"including(unprotected) %s\n", fn);
    ipt= source(ipt, (char*) fn, strmap_find(repla, (char*) fn));
    return ;
  }
  if (strmap_find(included,(char*) fn))
  {
    dprintf(logg,"rejected %s\n", fn);
  }
  else
  {
    strmap_insert(included, (char*)fn, fn);
    dprintf(logg,"including %s\n", fn);
    ipt= source(ipt, (char*) fn, strmap_find(repla,(char*)fn));
  }
}


void dobinc(uchar *arg)
{
  uchar *z;
  SLIST *pa;
  uchar *fn;

  fn= strmap_find(braws,(char*)arg);
  if (fn)
  {
    include_file(fn);
    return ;
  }
  z= ustrchr(arg,'/');
  if (!z) goto bailout;

  *z= 0;
  pa= strmap_find(bprefs, (char*) arg);
  *z= '/';
  if (!pa) goto bailout;
  for(SLnode_t *n=pa->head;n;n=n->next)
  {
    fn= mkpath(n->data, arg);
    if (file_exists(fn))
    {
      include_file(fn);
      return ;
    }
    free(fn);
  }
bailout:
  dprintf(cfile, "#include <%s>\n",arg);
}

void doqinc(uchar *arg)
{
  uchar *path;
  if (arg[0]=='/') { include_file(arg); return ; }
  path= file_sibling((uchar*) ipt->fn,arg);
  if (file_exists(path)) { include_file(path); return ; }
  free(path);
  for(SLnode_t *n=qpaths->head;n;n=n->next)
  {
    path= mkpath(n->data, arg);
    if (file_exists(path)) { include_file(path); return ; }
    free(path);
  }
  dprintf(cfile, "#include \"%s\"\n",arg);
}

void dominc(uchar *arg)
{
  uchar *val;
  val= strmap_find(macros, (char*) arg);
  if (val)
    doqinc(val);
  else
    dprintf(cfile, "#include %s\n",arg);
}

void dosource(uchar *arg)
{
  include_file(expand_vars(arg));

  while(ipt)
    switch(line(ipt, &arg))
    {
    case endo: ipt=iptup(ipt); break;
    case raw: writeln(cfile, (char*) arg); break;
    case binc: dobinc(arg); break;
    case qinc: doqinc(arg); break;
    case minc: dominc(arg); break;
    }
}

int main(int argc, char **argv)
{
  int q;
  uchar *arg;
  if (argc<2) exit(fprintf(stderr,"usage: %s spec-files\n", argv[0]));
  logg= openout("sinfil.log");
  specs(&argc,argv);
  cmdlinevars(&argc,argv);

  macros= strmap_new();
  bprefs= strmap_new();   
  included= strmap_new();
  braws= strmap_new();
  repla= strmap_new();
  unprotected= strmap_new();
  qpaths= slistnew();
  q= 0;
                    while(1)
                      {
  switch(getspec(&arg))
  {
  case sp_end: q= 1; break;
  case sp_outfile:  cfile= openout((char*)expand_vars(arg)); break;
  case sp_bpref:  bpref(arg);break;
  case sp_qpath: qpath(arg); break;
  case sp_defmac: defmac(arg); break;
  case sp_var: dovar(arg); break;
  case sp_source: dosource(arg); break;
  case sp_braw: dobraw(arg);
  case sp_unprotect: strmap_insert(unprotected,
                      (char*) expand_vars(arg), arg); break;
  case sp_replace:
  { uchar *fil, *rep;
    fil= getword(&arg); rep= getword(&arg);
    strmap_insert(repla, (char*) expand_vars(fil), expand_vars(rep));
    break; }
  case sp_print: writeln(cfile, (char*) expand_vars(arg)); break;
  case sp_copyright: file_append(cfile, (char*) expand_vars(arg)); break;
  }    
  if (q) break;
                      }

  close(cfile);
  close(logg);
}


void cmdline_shift(int *argc,char **argv, int idx, int amo)
{
  for(int i=idx;i+amo<*argc;i++)
     argv[i]= argv[i+amo];
  *argc-= amo;
}

uchar *getword(uchar **Ra)
{
  uchar *r;
  while(**Ra<=' ' || **Ra==127)  if (**Ra) (*Ra)++; else return *Ra;
  r= *Ra;
  while(**Ra>' ' && **Ra!=127) (*Ra)++;
  if (**Ra)
  {
    **Ra= 0;
    (*Ra)++; 
  }
  return r;
}

typedef struct var
{
  uchar *name;
  uchar *value;
  struct var *next;
} var_t;

static var_t *vhead;

static uchar *ustrdup(uchar *s) { return (uchar*) strdup((char*) s); }
static int ustrcmp(uchar *a,uchar *b) { return strcmp((char*)a,(char*)b); }
static uint ustrlen(uchar *s) { uint l; for(l=0;s[l];l++) ; return l; }

void define_var(uchar *name, uchar *value)
{
  var_t *V;
  V= malloc(sizeof(*V));
  V->name= ustrdup( name);
  V->value= ustrdup( value);  
  V->next= vhead;
  vhead= V;
}

uchar *find_var(uchar *name)
{
  var_t *V;
  for(V=vhead;V;V=V->next)
    if (!ustrcmp(V->name, name)) 
       return V->value;
  return NULL;
}

static uint namelen(uchar *name)
{
  uint l;
  static unsigned char k[256];
  if (!k['a'])
  {
    int i;
    for(i='a';i<='z';i++) k[i]= 1;
    for(i='A';i<='Z';i++) k[i]= 1;
    for(i='0';i<='9';i++) k[i]= 1;
    k['_']= 1;
  }
  
  for(l=0;name[l];l++) if (!k[name[l]]) break;
  return l;
}


static uchar *findat(uchar *name, uint l)
{
  for(var_t *V=vhead;V;V=V->next)
  {
    uint L;
    L= ustrlen(V->name);
    if (L!=l) continue;
    if (!memcmp(V->name, name,L))  return V->value;
  }
  return NULL; 
}

static uchar *expandv(uchar *s, uchar *r)
{
  uchar *z, *v;
  uint len,l, vl, rep;

  len= 0;
  rep= 0;
  while(s[0])
  {
    z= ustrchr(s, '$');
    if (!z) { l= ustrlen(s); if (r) memcpy(r+len,s,l); len+=l; break; }
    if (z>s) { if (r) { memcpy(r+len, s, z-s); } len+= z-s; }
    z++;
    v= findat(z, l=namelen(z));
    if (v)  { rep++; vl= ustrlen(v);
              if (r) { memcpy(r+len, v, vl); } len+= vl; s= z+l; }
      else  { if (r) r[len]='$'; len++; s= z; }
  }
  if (!rep) return s;
  if (r) r[len]= 0;
    else r= malloc(len+1);
  return r;
}

uchar* expand_vars(uchar* s)
{
  uchar *r;
  while(1)
    if ((r= expandv(s, NULL))==s) return r; 
                             else s= expandv(s, r);
}

void cmdlinevars(int *argc,char **argv)
{
  int i;
  for(i=0;i<*argc;i++)
  {
    char *z, *name, *value;
    z= strchr(argv[i], '=');
    if (!z) continue;
    name= strdup(argv[i]);
    name[z-argv[i]]= 0;
    value= strdup(z+1);
    define_var((uchar*) name, (uchar*) value);
    cmdline_shift(argc,argv,i,1);
    i--;
  }    
}


typedef struct spec
{
  int kind;
  uchar *arg;
  struct spec *next;
} SPEC;

static SPEC *shead, *stail;

static struct
{
  char *name;
  int val;
} kwlist[]=
{
  { "copyright", sp_copyright },
  { "print", sp_print },
  { "replace", sp_replace },
  { "unprotect", sp_unprotect },
  { "braw", sp_braw },
  { "var", sp_var },
  { "source", sp_source },
  { "bpref", sp_bpref },
  { "qpath", sp_qpath },
  { "outfile", sp_outfile },
  { "defmac", sp_defmac },
  { "end", sp_end },
  { NULL, 0 },
};

static uchar *trimspc(uchar *s)
{
  int i,l;
  for(i=0;s[i];i++)
    if (s[i]=='\n') { s[i]= 0; break; }
  while(*s && (*s<=' ' || *s==127)) s++;
  l= strlen((char*)s);
  while(l && s[l-1])
  {
    if (s[l-1]<=' ' || s[l-1]==127) s[--l]= 0;
                               else break;
  }
  return s;
}

static uchar *skipword(uchar *s)
{
  while(*s && (*s>' ' && *s!=127)) s++;
  if (*s)
  {
    *s= 0;
    s++;
    while(*s && (*s<=' ' || *s==127)) s++;
  }
  return s;
}


static void parse_spec(char *fn)
{
  static char *line;
  static size_t linesz;
  uchar *s, *a;
  int i;
  FILE *f;
  SPEC *S;
  f= fopen(fn,"rb");
  if (!f) exit(fprintf(stderr,"open(%s): %s\n", fn, strerror(errno)));
  while(getline(&line,&linesz,f)>=0)
  {
    if (line[0]=='#') continue;
    s= trimspc((uchar*) line); 
    if (!s[0]) continue;
    a= skipword(s);
    for(i=0;kwlist[i].name;i++)
    {
      if (!strcmp((char*)s,kwlist[i].name)) break;
    }
    if (!kwlist[i].name)
       exit(fprintf(stderr,"unknown keyword %s in %s\n", s, fn));
    S= calloc(1,sizeof(*S));
    S->kind= kwlist[i].val;
    S->arg= (uchar*) strdup((char*) a);
    if (shead) stail->next= S;
          else shead= S;
    stail= S;
  }
  fclose(f);
}

void specs(int *argc, char **argv)
{
  for(int i=1;i<*argc;i++) 
  {
    if (!strchr(argv[i],'='))
    {
      parse_spec(argv[i]);
      cmdline_shift(argc, argv, i,1);
      i--;
    }
  }
  dprintf(logg,"SPECS\n");
  for(SPEC *S=shead;S;S=S->next)
  {
   char *ks;
   char buf[200];
   switch(S->kind)
   {
   case sp_source: ks= "source"; break;
   case sp_bpref: ks= "bpref"; break;
   case sp_qpath: ks= "qpath"; break;
   case sp_outfile: ks= "outfile"; break;
   case sp_defmac: ks="defmac"; break;
   case sp_var: ks= "var"; break;
   case sp_end: ks= "end"; break;
   case sp_braw: ks= "braw"; break;
   case sp_unprotect: ks= "unprotect"; break;
   case sp_replace: ks= "replace"; break;
   case sp_copyright: ks= "copyright"; break;
   case sp_print: ks= "print"; break;
   default: sprintf(buf,"unk%d", S->kind); ks= buf; 
   }
   dprintf(logg, "%s <%s>\n", ks, (char*) S->arg);
  }
  dprintf(logg,"END\n");
}

int getspec(uchar **arg)
{
  SPEC *S;
  int rv;
  if (!shead) return sp_end;
  S= shead;
  shead= S->next;
  rv= S->kind;
  *arg= S->arg;
  free(S);
  return rv;
}


static void failsys(int cond, char *op, char *fn)
{
  if (!cond) return ;
  fprintf(stderr,"%s(%s): %s\n", op, fn, strerror(errno));
  exit(1);
}

void readfile(uchar **Rf, uint *Rl, char *fn, int nzero)
{
  int fd;
  struct stat sb;
  uchar *co;
  ssize_t R, t;
  fd= open(fn, O_RDONLY);
  failsys(fd<0, "open", fn);
  failsys(fstat(fd, &sb),"stat",fn);
  co= malloc(sb.st_size+nzero);
  t= 0;
  while(t<sb.st_size)
  {
    R= read(fd, co+t, sb.st_size-t);
    if (R>0) t+= R;
        else failsys(errno!=EINTR, "read", fn);
  }
  close(fd);
  memset(co+t, 0, nzero);
  *Rf= co;
  *Rl= t;
}

int openout(char *fn)
{
  int fd;
  fd= open(fn, O_WRONLY | O_CREAT | O_TRUNC, 0600);
  failsys(fd<0, "open", fn);
  return fd;
}

void writebytes(int fd, void *p, uint sz)
{
  uint t;
  ssize_t R;
  uchar *str;
  str= p;
  t= 0;
  while(t<sz)
  {
    R= write(fd, str+t, sz-t);
    if (R>0) t+= R;
        else failsys(errno!=EINTR, "write", "out");
  }
}

void writeln(int fd, char *str)
{
  uint l;
  l= strlen(str);
  str[l]= '\n';
  writebytes(fd, str, l+1);
  str[l]= 0;
}

uchar *mkpath(uchar *dir, uchar *fil)
{
  uint l1, l2;
  uchar *p;
  l1= strlen((char*)dir);
  l2= strlen((char*)fil);
  p= malloc(l1+l2+2);
  snprintf((char*)p,l1+l2+2,"%s/%s",dir,fil);
  return p;
}

int file_exists(uchar *fn)
{
  struct stat sb;
  if (stat((char*)fn,&sb)) return 0;
  return S_ISREG(sb.st_mode);
}

uchar *file_sibling(uchar *fn, uchar *sibnam)
{
  char *z;
  uchar *r;
  z= strrchr((char*)fn,'/');
  if (!z) return (uchar*) strdup((char*) sibnam);
  *z= 0;
  r= mkpath(fn, sibnam);
  *z= '/';
  return r;
}

void file_append(int fd, char *fn)
{
  uchar *B;
  uint L;
  readfile(&B, &L, fn, 0);
  writebytes(fd, B, L);
  free(B);
}


typedef struct
{
  uchar *text;
  uint in, out, len;
} CTEXT;

static inline void tcpy(CTEXT *t) { t->text[t->out++]= t->text[t->in++]; }
static inline uchar tchar(CTEXT *t) { return t->text[t->in]; }
static inline uchar tchar1(CTEXT *t) { return t->text[t->in+1]; }
static inline int tmore(CTEXT *t) { return t->in<t->len; }
static inline void tput(CTEXT *t, uchar chr, int adv)
{
   t->text[t->out++]= chr;
   t->in+= adv; 
}
static inline void tadv(CTEXT *t, uint adv) { t->in+= adv; }
static inline void tbegin(CTEXT *t) { t->in= 0; t->out= 0; }
static inline void tend(CTEXT *t) 
{ 
  memset(t->text+t->out, 0, t->len-t->out);
  t->len= t->out;
}

static void skip_strlit(CTEXT *t)
{
  tcpy(t);
           while(tmore(t))
              {
  switch(tchar(t))
  {
  case '\"': tcpy(t); return ;
  case '\\': tcpy(t); break;
  }
  tcpy(t);
              }
}

static void skip_charconst(CTEXT *t)
{
  tcpy(t);
           while(tmore(t))
              {
  switch(tchar(t))
  {
  case '\'': tcpy(t); return ;
  case '\\': tcpy(t); break;
  }
  tcpy(t);
              }
}

static void skip_ccom(CTEXT *t)
{
  tadv(t,2);
  while(tmore(t))
  {
    if (tchar(t)=='*' && tchar1(t)=='/') { tput(t,' ',2); return ; }
                                    else { tadv(t,1); }
  }
}

static void skip_lcom(CTEXT *t)
{
  tadv(t,2);
  while(tmore(t))
    if (tchar(t)=='\n') { tput(t,' ',0); return ; }
                   else { tadv(t,1); }
}


static void xlate_chars(CTEXT *t)
{
  tbegin(t);
          while(tmore(t))
               {
  switch(tchar(t))
  {
  case     0: tput(t,' ',1); continue; 
  case  '\r': if (tchar1(t)=='\n') tput(t,'\n',2); 
                              else tput(t,'\n', 1);
              continue;
  case '\n':  if (tchar1(t)=='\r') tput(t, '\n',2); 
                              else tput(t,'\n', 1);
              continue;
  default:    tcpy(t);
  }
               }
  tend(t);
}

static void fix_lines(CTEXT *t)
{
  tbegin(t);
  while(tmore(t))
    if (tchar(t)=='\\' && tchar1(t)=='\n') tadv(t,2);
                                      else tcpy(t);
  tend(t);
}


static void remove_comments(CTEXT *t)
{
  tbegin(t);
             while(tmore(t))
              {
  switch(tchar(t))
  {
  case '\"': skip_strlit(t); continue;
  case '\'': skip_charconst(t); continue;
  case '/':  if (tchar1(t)=='*') { skip_ccom(t); continue; }
             if (tchar1(t)=='/') { skip_lcom(t); continue; }
             break;
  }
  tcpy(t);
              }
  tend(t);
}

void prepro(uchar *src, uint len)
{
  CTEXT TT;
  TT.text= src;
  TT.len= len;

  xlate_chars(&TT);
  fix_lines(&TT);
  remove_comments(&TT);
}

IPT *source(IPT *par, char *fn, char *acfn)
{
  IPT *p;
  uint len;
  p= malloc(sizeof(*p));
  p->up= par;
  p->fn= fn;
  readfile(&p->data, &len, acfn ? acfn : fn, 6); 
  prepro(p->data, len);
  p->cur= p->data;
  return p;
}

IPT *iptup(IPT *P)
{
  IPT *R;
  R= P->up;
  free(P->data);
  free(P);
  return R;
}

static int ispc(uchar *s)
{
  if (s[0]==0 || s[0]=='\n') return 0;
  if (s[0]<=' ' || s[0]==127) return 1;
  return 0;
}

static int smatch(uchar *d, char *_w)
{
  uchar *w;
  w= (uchar*) _w; 
  while(*w)
  {
    if (*d!=*w) return 0;
           else d++, w++;
  }
  return 1;
}

static int doraw(IPT *ipt, uchar **arg)
{
  uchar *c;
  *arg= c= ipt->cur;
  if (!c[0]) return endo;
  while(c[0]!=0 && c[0]!='\n') c++;
  if (c[0]=='\n') ipt->cur= c+1, *c= 0;
             else ipt->cur= c;
  return raw;
}

static int parseinc(IPT *ipt, uchar **arg, uchar *c, uchar ec, int rv)
{
  *arg= c;
  while(c[0]!=0 && c[0]!='\n' && c[0]!=ec) c++;
  switch(c[0])
  {
  case 0: ipt->cur= c; break;
  case '\n': ipt->cur= c+1; c[0]= 0; break;
  default: 
    c[0]= 0;
    c++;
    while(c[0]!=0 && c[0]!='\n') c++;
    if (c[0]=='\n') ipt->cur= c+1;
               else ipt->cur= c;
  }
  return rv;
}

static void skipq(uchar **Rd, uchar **Rc)
{
  uchar *c, *d, ec;
  c= *Rc, d= *Rd, ec= *c;

  *d++= *c++;
  while(c[0]!=0 && c[0]!='\n') 
  {
    if (c[0]<=' ' || c[0]==127) { *d++= ' '; c++; continue; }
    if (c[0]==ec) { *d++= *c++; break; }
    if (c[0]!='\\') { *d++= *c++; continue; }
    *d++= *c++;
    if (c[0]=='\n' || c[0]==0) break;
    if (c[0]<=' ' || c[0]==127) { *d++= ' '; c++; continue; }
    *d++= *c++;
  }
  *Rd= d, *Rc= c;
}
  
static int parseminc(IPT *ipt, uchar **arg, uchar *c)
{
/* this is for things like #include FOO where FOO is a macro and we need
   the expansion for it. it's really tough to do so, because the macro could
   be anything, even one with arguments, strings etc. so, here's my plan.
   I will compact it, removing spaces but keeping the ones within string
   literals and character constants. then I can use the whole thing as an
   identifier to be defined manually within the sinfil input file. it's
   not foolproof. i.e. if you have raw tab characters in string literals
   (not encoded like \t), this will fail badly, but I don't think I will
   encounter that too much.. we'll see **/

  uchar *d;

  *arg= c;
  d= c;
           while(1)
               {
  switch(c[0])
  {
  case 1: case 2: case 3: case 4: case 5: case 6: case 7: case 8:
  case 9:         case 11: case 12: case 13: case 14: case 15: case 16:
  case 17: case 18: case 19: case 20: case 21: case 22: case 23: case 24:
  case 25: case 26: case 27: case 28: case 29: case 30: case 31:
  case ' ': c++; continue;
  case '\n': *d=0; ipt->cur= c+1; return minc; 
  case 0: *d= 0; ipt->cur= c; return minc; 
  case '\'':
  case '"': skipq(&d, &c); continue;
  }
  *d++= *c++; 
                }
  return minc; // never reached
}

int line(IPT *ipt, uchar **arg)
{
  uchar *c;

  for(c= ipt->cur;ispc(c);c++) ;
  if (c[0]!='#') return doraw(ipt,arg);
  for(c++;ispc(c);c++) ;
  if (!smatch(c,"include")) return doraw(ipt,arg);
  c+= 7;
  if (!ispc(c)) return doraw(ipt,arg);
  for(;ispc(c);c++) ;

  switch(c[0])
  {
  case 0:
  case '\n': return doraw(ipt, arg);// just an #include on a line by itself :)
  case '<': return parseinc(ipt, arg, ++c, '>', binc);
  case '"': return parseinc(ipt, arg, ++c, '"', qinc);
  default:  return parseminc(ipt, arg, c);
  }
}



typedef struct
{
  void* child[2];
  unsigned internal:1, byte:23, mask:8;
} strmap_node_t;


static inline int strmap_dir
    (uint8_t *key,uint32_t len_key,strmap_node_t *node)
{
  uint8_t c;

  if (node->byte<len_key) c= key[node->byte];
                     else c= 0;
  return (1+(node->mask|c))>>8;
}

void* strmap_find(strmap_t *map,char *skey)
{
  uint8_t *key; unsigned int len_key;
  strmap_node_t *node;

  node= map->root;
  if (!node) return 0;

  key= (void*) skey;
  len_key= strlen(skey);

  while(node->internal)
     node= node->child[strmap_dir(key,len_key,node)];

  if (strcmp(skey, node->child[0])) return 0;
                               else return node->child[1];
}


static inline uint32_t strmap_make_mask
    (uint32_t A, uint32_t B)
{
  uint8_t diff;
  diff= A ^ B;
  diff|= diff>>1;
  diff|= diff>>2;
  diff|= diff>>4;
  return (diff&~(diff>>1))^255;
}

static inline strmap_node_t* strmap_leaf_node
    (uint8_t *key, void *data)
{
  strmap_node_t *N;
  N= malloc(sizeof(*N));
  N->internal= 0;
  N->child[0]= key;
  N->child[1]= data;
  return N;
}

static inline strmap_node_t* strmap_internal_node
    (uint32_t byte, uint32_t mask)
{
  strmap_node_t *N;
  N= malloc(sizeof(*N));
  N->internal= 1;
  N->byte= byte;
  N->mask= mask;
  return N;
}

void* strmap_insert(strmap_t* map, char* skey, void *data)
{
  uint8_t *key; uint32_t len_key;
  strmap_node_t *node_elt;

  key= (void*) skey;

  if (!map->root)
  {
     map->root= strmap_leaf_node(key,data);
     return 0;
  }

  len_key= strlen(skey);

  node_elt= map->root;
  while(node_elt->internal)
    node_elt= node_elt->child[strmap_dir(key,len_key,node_elt)];

  uint32_t crit_byte;
  uint8_t  mask;
  uint8_t  *elt;
  int dir_elt;
  strmap_node_t *node_key;

  elt= node_elt->child[0];

  for(crit_byte=0;
      crit_byte<=len_key && elt[crit_byte]==key[crit_byte];
      crit_byte++)
          ;

  if (crit_byte>len_key) return node_elt->child[1];

  mask= strmap_make_mask(elt[crit_byte],key[crit_byte]);

  dir_elt= (1+(mask|elt[crit_byte]))>>8;

  node_key= strmap_internal_node(crit_byte,mask);
  node_key->child[1-dir_elt]= strmap_leaf_node(key,data);

  strmap_node_t *parent;
  strmap_node_t *node;

  parent= 0;
  node= map->root;

  while(node->internal)
  {
     if (node->byte > crit_byte) break;
     if (node->byte==crit_byte && node->mask > mask) break;
     parent= node;
     node= node->child[strmap_dir(key,len_key,node)];
  }

  node_key->child[dir_elt]= node;
  if (parent) { if (parent->child[0]==node) parent->child[0]= node_key;
                                    else    parent->child[1]= node_key; }
  else map->root= node_key;
  return 0;
}

void *strmap_remove(strmap_t* map, char *skey)
{
  uint8_t *key; uint32_t len_key;
  void *retv;

  if (!map->root) return 0;

  key= (void*) skey;
  len_key= strlen(skey);
   
  strmap_node_t *node;
  strmap_node_t *sibling;
  strmap_node_t *parent;
  strmap_node_t *gparent;

  node= map->root;
  gparent= 0;
  sibling= 0;
  parent= 0;

  while(node->internal)
  {
    int dir;
    dir= strmap_dir(key, len_key, node);
    gparent= parent;
    parent= node;
    sibling= node->child[1-dir];
    node= node->child[dir];
  }

  if (strcmp(node->child[0],skey)) return 0;
  retv= node->child[1];
  free(node);

  if (!gparent)
  {
     map->root= sibling;
     if (parent) free(parent);
     return retv;
  }

  if (gparent->child[0]==parent)
    gparent->child[0]= sibling;
  else
    gparent->child[1]= sibling;

  free(parent);

  return retv;
}

void *strmap_destroy(strmap_t *map)
{
  strmap_node_t *node;
  strmap_node_t *parent;
  strmap_node_t *gparent;
  strmap_node_t *sibling;
  void *retv;

  if (!map->root) return 0;

  parent= 0;
  gparent= 0;
  sibling= 0;
  node= map->root;
  while(node->internal)
  {
    gparent= parent;
    parent= node;
    sibling= node->child[1];
    node= node->child[0];
  }
  retv= node->child[1];
  free(node);
  if (!gparent) 
  {
    map->root= sibling;
    if (parent) free(parent);
    return retv;
  }
  if (gparent->child[0]==parent)
    gparent->child[0]= sibling;
  else
    gparent->child[1]= sibling;

  free(parent);

  return retv;
}

strmap_t  *strmap_new()
{
  strmap_t *s;
  s= malloc(sizeof(*s));
  s->root= 0;
  return s;
}

